/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_andromeda.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ird_instrument_model.h"
#include "sph_framecombination.h"
#include "sph_ird_common_science.h"
#include "sph_keyword_manager.h"
#include "sph_fft.h"
#include "sph_psfcube.h"
#include "sph_andromeda_support.h"
#include "sph_common_science.h"
#include "sph_fits.h"
/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_ANDROMEDA_GENERAL;
extern sph_error_code SPH_IRD_ANDROMEDA_FRAMES_MISSING;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_andromeda_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_andromeda.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_andromeda recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_andromeda_run(sph_ird_andromeda* self) {
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_master_frame* raw = NULL;
    sph_master_frame* raw2 = NULL;
    cpl_frame* frame = NULL;
    sph_master_frame* result = NULL;
    int ff = 0;
    cpl_imagelist* leftlist = NULL;
    cpl_imagelist* leftvlist = NULL;
    cpl_image* rawimage_cut = NULL;
    cpl_image* varimage = NULL;
    sph_fft* fft = NULL;
    sph_fctable* centtab = NULL;
    int psfdivs = 10;
    cpl_vector* angles = NULL;
    cpl_bivector* indices = NULL;
    sph_psfcube* psfcube = NULL;
    cpl_image* psfim_big = NULL;
    cpl_image* psfim = NULL;
    int psfim_nx = 0;
    int psfim_ny = 0;
    cpl_imagelist* varimlist = NULL;
    cpl_imagelist* weightimlist = NULL;
    cpl_vector* gamma = NULL;
    cpl_imagelist* diffims = NULL;
    cpl_image* fluximage = NULL;
    cpl_vector* cent_left_x = NULL;
    cpl_vector* cent_left_y = NULL;
    cpl_vector* cent_right_x = NULL;
    cpl_vector* cent_right_y = NULL;
    cpl_frameset* left_frameset = NULL;
    cpl_frameset* right_frameset = NULL;
    cpl_frameset* total_frameset = NULL;
    cpl_frameset* fctable_frames = NULL;
    sph_ird_common_science* sci = NULL;
    int pp = 0;
    sph_fctable* fctable = NULL;
    cpl_propertylist* pl = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (cpl_frameset_get_size(self->rawframes) > 5000) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "Sorry but I cant handle more than 5000 raw input frames.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    /*------------------------------------------------------------------
     -  Reading in of PSF reference and constructon of PSF reference cube
     --------------------------------------------------------------------*/

    psfim_big = cpl_image_load(cpl_frame_get_filename(self->psf_frame),
            CPL_TYPE_DOUBLE, 0, 0);
    if (!psfim_big) {
        SPH_ERR("Error loading PSF reference image file.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if (self->only_prep == 0
            && (self->rhomax + self->psf_size) * 2
                    > self->window_maxx - self->window_minx) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_INPUT,
                "The rhomax value of %f and/or the PSF size of %d is too big to "
                "always fit into the window of x=%d, %d and y=%d,%d specified."
                "Please either reduce the rhomax value or the PSF size or"
                "change the window so that"
                "(rhomax + psf_size ) * 2 < window_maxx - window_minx.", self->rhomax, self->psf_size, self->window_minx, self->window_maxx, self->window_miny, self->window_maxy);
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    psfim_nx = (int) cpl_image_get_size_x(psfim_big);
    psfim_ny = (int) cpl_image_get_size_y(psfim_big);
    psfim = cpl_image_extract(psfim_big, (psfim_nx - self->psf_size) / 2 + 1,
            (psfim_ny - self->psf_size) / 2 + 1,
            (psfim_nx + self->psf_size) / 2, (psfim_ny + self->psf_size) / 2);
    cpl_image_delete(psfim_big);
    psfim_big = NULL;
    if (cpl_image_get_size_x(psfim) != self->psf_size
            || cpl_image_get_size_y(psfim) != self->psf_size) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_INPUT,
                "Error extracting the PSF image cutout. "
                "Sizes were %d and %d instead of %d", (int)cpl_image_get_size_x(psfim), (int)cpl_image_get_size_y(psfim), self->psf_size);
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Calculating the PSF cube.");
    SPH_RAISE_CPL;
    psfcube = sph_andromeda_support_calc_psf_shift_subpix(psfim, psfdivs);
    if (!psfcube) {
        SPH_ERR("Could not create PSF cube.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    /*------------------------------------------------------------------
     -  Preprocessing for ANDROMEDA
     --------------------------------------------------------------------*/
    if (self->flag_sdi) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Starting Andromeda pre-processing"
                " (re-centering, scaling, SDI)");
    } else {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Starting Andromeda pre-processing "
                "(re-centering, scaling, adding left and right FOV.)");
    }

    if (self->fcasciis != NULL) {
        fctable_frames = cpl_frameset_duplicate(self->fcasciis);
        if (fctable_frames == NULL) {
            goto EXIT;
        }
    } else {
        fctable_frames = sph_ird_common_science_create_fctables(self->rawframes,
                self->fcframe, self->keep_fctable, 1);
    }

    sci = sph_ird_common_science_new(self->inframes, self->inparams,
            self->dark_frame, self->flat_frame, self->static_badpixel_frame,
            self->distmap_frame, self->filter_frame, self->flat_frame,
            fctable_frames,
            SPH_RECIPE_NAME_IRD_ANDROMEDA, SPH_IRD_TAG_ANDROMEDA_CALIB);

    sci->minr = self->rhomin;
    sci->maxr = self->rhomax;
    sci->flag_sdi = self->flag_sdi;

    sph_ird_common_science_process_cubes(sci, self->rawframes, fctable_frames,
                               &left_frameset, &right_frameset, &total_frameset,
                                         NULL, NULL, NULL);
    sph_ird_common_science_delete(sci);
    sci = NULL;
    SPH_RAISE_CPL;

    fft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    leftlist = cpl_imagelist_new();
    leftvlist = cpl_imagelist_new();

    SPH_RAISE_CPL;
    angles = cpl_vector_new(sph_common_science_get_nraws(total_frameset, 0));
    SPH_RAISE_CPL;

    for (ff = 0; ff < cpl_frameset_get_size(total_frameset); ++ff) {
        frame = cpl_frameset_get_position(total_frameset, ff);

        pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(frame),
                0);

        if (fctable_frames) {
            fctable = sph_fctable_find_fctable(
                    cpl_frameset_get_position(self->rawframes, ff),
                    fctable_frames);
        }
        for (pp = 0;
                pp < sph_fits_get_nplanes(cpl_frame_get_filename(frame), 0);
                ++pp) {

            raw = sph_master_frame_load_(frame, pp);
            rawimage_cut = cpl_image_extract(raw->image, self->window_minx,
                    self->window_miny, self->window_maxx, self->window_maxy);
            if (!rawimage_cut) {
                SPH_RAISE_CPL
            }
            cpl_imagelist_set(leftlist, rawimage_cut,
                    cpl_imagelist_get_size(leftlist));
            sph_master_frame_delete(raw);
            raw = NULL;

            frame = cpl_frameset_get_position(left_frameset, ff);
            raw = sph_master_frame_load_(frame, 0);
            frame = cpl_frameset_get_position(right_frameset, ff);
            raw2 = sph_master_frame_load_(frame, 0);
            sph_master_frame_add_master_frame(raw, raw2);
            sph_master_frame_set_rms_poisson(raw, SPH_MASTER_FRAME_BAD_RMS, 1);
            varimage = sph_master_frame_get_rms(raw);
            sph_master_frame_delete(raw);
            raw = NULL;
            sph_master_frame_delete(raw2);
            raw2 = NULL;
            rawimage_cut = cpl_image_extract(varimage, self->window_minx,
                    self->window_miny, self->window_maxx, self->window_maxy);
            if (!rawimage_cut) {
                SPH_RAISE_CPL
            }
            cpl_imagelist_set(leftvlist, rawimage_cut,
                    cpl_imagelist_get_size(leftvlist));
            cpl_image_delete(varimage);
            varimage = NULL;
            if (fctable) {
                cpl_vector_set(angles, cpl_imagelist_get_size(leftlist) - 1,
                        sph_fctable_get_angle(fctable, pp));
            } else {
                cpl_vector_set(
                        angles,
                        cpl_imagelist_get_size(leftlist) - 1,
                        cpl_propertylist_get_double(pl,
                                SPH_IRD_KEYWORD_ROT_ANGLE));
            }
        }
        cpl_propertylist_delete(pl);
        pl = NULL;
        sph_fctable_delete(fctable);
        fctable = NULL;
    }

    if (self->only_prep) {
        cpl_imagelist_save(leftlist, self->leftlist_filename, CPL_TYPE_DOUBLE,
                NULL, CPL_IO_CREATE);
        cpl_imagelist_save(leftvlist, self->leftlist_filename, CPL_TYPE_DOUBLE,
                NULL, CPL_IO_EXTEND);
    } else {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Starting Andromeda...");
        indices = sph_andromeda_support_create_angpairs(angles,
                self->min_ang_dist, NULL);

        diffims = sph_andromeda_support_angular_difference(leftlist, leftvlist,
                cpl_bivector_get_x(indices), cpl_bivector_get_y(indices),
                self->rhomin, self->rhomax, 0, &gamma, &varimlist,
                &weightimlist);
        cpl_imagelist_save(diffims, "diffims.fits", CPL_TYPE_DOUBLE, NULL,
                CPL_IO_CREATE);
        cpl_imagelist_save(leftvlist, "varims.fits", CPL_TYPE_DOUBLE, NULL,
                CPL_IO_CREATE);
        cpl_imagelist_save(weightimlist, "weightims.fits", CPL_TYPE_DOUBLE,
                NULL, CPL_IO_CREATE);
        SPH_RAISE_CPL_RESET;

        fluximage = sph_andromeda_core_get_flux(diffims,
                cpl_bivector_get_x(indices), cpl_bivector_get_y(indices),
                angles, psfcube, 0, weightimlist, self->rhomin, self->rhomax,
                gamma, 0, NULL, NULL, NULL, NULL);

        SPH_RAISE_CPL_RESET;

        result = sph_master_frame_new_from_cpl_image(fluximage);

        cpl_imagelist_delete(diffims);
        diffims = NULL;
        cpl_imagelist_delete(weightimlist);
        weightimlist = NULL;
        cpl_imagelist_delete(varimlist);
        varimlist = NULL;
        cpl_vector_delete(gamma);
        gamma = NULL;
        cpl_image_delete(fluximage);
        fluximage = NULL;
        cpl_bivector_delete(indices);
        indices = NULL;

        if (result) {
            sph_master_frame_save_dfs(result, self->andromeda_outfilename,
                    self->rawframes, NULL, self->inparams,
                    SPH_IRD_TAG_ANDROMEDA_CALIB, SPH_RECIPE_NAME_IRD_ANDROMEDA,
                    SPH_PIPELINE_NAME_IRDIS, NULL);
            sph_master_frame_delete(result);
            result = NULL;
        }
    }
    if (dark) {
        sph_master_frame_delete(dark);
        dark = NULL;
    }
    if (flat) {
        sph_master_frame_delete(flat);
        flat = NULL;
    }
    EXIT: cpl_vector_delete(angles);
    angles = NULL;
    cpl_image_delete(psfim);
    psfim = NULL;
    cpl_imagelist_delete(leftlist);
    leftlist = NULL;
    cpl_imagelist_delete(leftvlist);
    leftvlist = NULL;
    sph_psfcube_delete(psfcube);
    cpl_vector_delete(cent_left_x);
    cent_left_x = NULL;
    cpl_vector_delete(cent_left_y);
    cent_left_y = NULL;
    cpl_vector_delete(cent_right_x);
    cent_right_x = NULL;
    cpl_vector_delete(cent_right_y);
    cent_right_y = NULL;
    cpl_frameset_delete(left_frameset);
    left_frameset = NULL;
    cpl_frameset_delete(right_frameset);
    right_frameset = NULL;
    cpl_frameset_delete(total_frameset);
    total_frameset = NULL;
    cpl_frameset_delete(fctable_frames);
    fctable_frames = NULL;
    sph_fctable_delete(centtab);
    centtab = NULL;
    sph_fft_delete(fft);
    fft = NULL;
    //sph_filemanager_clean();
    return SPH_RAISE_CPL;

}

/**@}*/

