/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_master_dark.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_shared_irdifs.h"

#include <math.h>

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_MASTER_DARK_GENERAL;
extern sph_error_code SPH_IRD_MASTER_DARK_FRAMES_MISSING;

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_master_dark_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_master_dark.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_master_dark recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_master_dark_run(sph_ird_master_dark* self) {
    int recipe_error = CPL_ERROR_NONE;
    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_image* bad_image = NULL;
    sph_master_frame* master_dark_mframe = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pli = NULL;

    double rms = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (self->master_dark_outfilename == NULL) {
        sph_error_raise(
                SPH_IRD_MASTER_DARK_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "No output filename was provided for the master dark. "
                        "Something must be wrong with the passed parameterlist.\n");
        recipe_error = CPL_ERROR_UNSPECIFIED;
        return recipe_error;
    }

    master_dark_mframe = sph_irdifs_master_dark_create(self->rawframes,
            self->coll_alg, self->framecomb_parameterlist, self->sigma_clip,
            self->smoothing, self->min_accept, self->max_accept, 0, &bad_image);

    if (master_dark_mframe) {
        const double ron = sph_irdifs_compute_ron(self->rawframes,
                                                  SPH_IRD_MASTER_DARK_RON_NIMG,
                                                  SPH_IRD_MASTER_DARK_RON_NLAST,
                                                  SPH_IRD_MASTER_DARK_RON_LLX,
                                                  SPH_IRD_MASTER_DARK_RON_LLY,
                                                  SPH_IRD_MASTER_DARK_RON_URX,
                                                  SPH_IRD_MASTER_DARK_RON_URY);

        if (cpl_error_get_code()) {
            SPH_ERR("Could not calculate RON estimate.");
        }

        pl = cpl_propertylist_new();
        cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_DARK_MEAN_RON,
                ron);

        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_DARK_MEAN_COUNT,
                sph_master_frame_get_mean(master_dark_mframe, &rms));
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_DARK_RMS_COUNT,
                rms);
        cpl_propertylist_update_int(pl, SPH_COMMON_KEYWORD_NUMBER_HOTPIXELS,
                sph_master_frame_get_nbads(master_dark_mframe));
        pli = cpl_propertylist_load(
                cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)),
                0);
        sph_utils_simple_copy_singular(pli, pl);

        if (sph_master_frame_save_dfs(master_dark_mframe,
                self->master_dark_outfilename, self->inframes,
                cpl_frameset_get_first(self->rawframes), self->inparams,
                SPH_IRD_TAG_DARK_CALIB, SPH_RECIPE_NAME_IRD_MASTER_DARK,
                SPH_PIPELINE_NAME_IRDIS, pl) != CPL_ERROR_NONE) {
            SPH_ERR("Could not save product master dark.");
        }
        cpl_propertylist_delete(pl);
        pl = NULL;

        if (self->save_addprod) {
        	if (bad_image) {
        		pl = cpl_propertylist_new();
        		cpl_propertylist_append_double(pl,
                    SPH_COMMON_KEYWORD_NUMBER_HOTPIXELS,
                    cpl_image_get_flux(bad_image));
        		cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_PRO_CATG,
                    SPH_IRD_TAG_STATIC_BADPIXELMAP_CALIB);
                        sph_utils_remove_wcs_3d(pl);
                        /* FIXME: Set WCS to dummy (pixel) value for now */
                        sph_utils_reset_wcs_12d(pl);

        		recipe_error = cpl_dfs_save_image(self->inframes, NULL,
                    self->inparams, self->inframes, NULL, bad_image,
                    CPL_BPP_8_UNSIGNED, SPH_RECIPE_NAME_IRD_MASTER_DARK, pl,
                    NULL, SPH_PIPELINE_NAME_IRDIS,
                    self->static_badpix_filename);

        		if (recipe_error != CPL_ERROR_NONE) {
        			SPH_RAISE_CPL;SPH_ERR("Could not save product hotpixel frame.");
        		}

        		cpl_propertylist_delete(pl);
        		pl = NULL;
        	}
        }
    }
    if (pli) {
        cpl_propertylist_delete(pli);
        pli = NULL;
    }

    recipe_error = cpl_error_get_code();

    cpl_ensure_code( cpl_errorstate_is_equal( prestate ),
            cpl_error_get_code());

    if (master_dark_mframe != NULL) {
        sph_master_frame_delete(master_dark_mframe);
        master_dark_mframe = NULL;
    }

    if (bad_image != NULL) {
        cpl_image_delete(bad_image);
        bad_image = NULL;
    }

    return (int) cpl_error_get_code();
}

/**@}*/

