/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_sky_bg.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ird_instrument_model.h"
#include "sph_framecombination.h"
#include "sph_common_science.h"
#include "sph_ird_common_science.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_fitting.h"
#include "sph_master_frame.h"
#include "sph_shared_irdifs.h"
#include "sph_ird_sky_bg.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <cpl.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_SKY_BG_GENERAL;
extern sph_error_code SPH_IRD_SKY_BG_FRAMES_MISSING;

static sph_master_frame*
sph_ird_sky_bg_create_polyfit(sph_master_frame* mframe, int order, int nsamples) {
    sph_master_frame* polframe = NULL;
    int nx = 0;
    int ny = 0;
    cpl_vector* xvals = NULL;
    cpl_vector* yvals = NULL;
    cpl_vector* zvals = NULL;
    int xx = 0;
    int yy = 0;
    double val = 0.0;
    gsl_rng* pRNG = NULL;
    int cc = 0;
    int bpix = 0;
    cpl_polynomial* poly = NULL;

    nx = cpl_image_get_size_x(mframe->image);
    ny = cpl_image_get_size_y(mframe->image);

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (!pRNG)
        return NULL;
    polframe = sph_master_frame_duplicate(mframe);
    if (!polframe)
        return NULL;
    xvals = cpl_vector_new(nsamples);
    yvals = cpl_vector_new(nsamples);
    zvals = cpl_vector_new(nsamples);
    if (!xvals || !yvals || !zvals)
        return NULL;

    cpl_image_delete(polframe->ncombmap);
    polframe->ncombmap = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_delete(polframe->badpixelmap);
    polframe->badpixelmap = cpl_image_new(nx, ny, CPL_TYPE_INT);

    xx = (int) gsl_ran_flat(pRNG, 0.0, nx);
    yy = (int) gsl_ran_flat(pRNG, 0.0, ny);

    while (cc < nsamples) {
        xx = (int) gsl_ran_flat(pRNG, 0.0, nx);
        yy = (int) gsl_ran_flat(pRNG, 0.0, ny);
        while (cpl_image_get(mframe->badpixelmap, xx + 1, yy + 1, &bpix) > 0) {
            xx = (int) gsl_ran_flat(pRNG, 0.0, nx);
            yy = (int) gsl_ran_flat(pRNG, 0.0, ny);
        }
        val = cpl_image_get(mframe->image, xx + 1, yy + 1, &bpix);
        cpl_vector_set(xvals, cc, xx);
        cpl_vector_set(yvals, cc, yy);
        cpl_vector_set(zvals, cc, val);
        val = cpl_image_get(polframe->ncombmap, xx + 1, yy + 1, &bpix);
        cpl_image_set(polframe->ncombmap, xx + 1, yy + 1, val + 1);
        cc++;
    }

    poly = sph_fitting_fit_poly2d(xvals, yvals, zvals, NULL, order, order);
    if (poly) {
        double vvxy[2];
        cpl_vector* vv = cpl_vector_wrap(2, vvxy);
        for (yy = 0; yy < ny; ++yy) {
            for (xx = 0; xx < nx; ++xx) {
                vvxy[0] = xx;
                vvxy[1] = yy;
                val = cpl_polynomial_eval(poly, vv);
                cpl_image_set(polframe->image, xx + 1, yy + 1, val);
                val -= cpl_image_get(mframe->image, xx + 1, yy + 1, &bpix);
                val *= val;
                cpl_image_set(polframe->rmsmap, xx + 1, yy + 1, val);
            }
        }
        cpl_vector_unwrap(vv);
    }
    SPH_RAISE_CPL_RESET;
    return polframe;
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_sky_bg_run Create Master Dark Recipe
 *
 * This module provides the algorithm implementation for the creation of the
 * sky background
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_sky_bg.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_sky_bg recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_sky_bg_run(sph_ird_sky_bg* self) {

    cpl_propertylist*          pl              = NULL;
    cpl_propertylist*          pli             = NULL;
    cpl_image*                 hot_image       = NULL;
    cpl_mask*                  badpix          = NULL;
    sph_master_frame*          sky_bg_mframe   = NULL;
    sph_master_frame*          fitresult_left  = NULL;
    sph_master_frame*          fitresult_right = NULL;
    sph_master_frame*          fitresult       = NULL;
    sph_master_frame*          right_frame     = NULL;
    sph_master_frame*          left_frame      = NULL;
    sph_ird_instrument_model*  model           = NULL;

    double                     rms;


    sky_bg_mframe = sph_irdifs_master_dark_create(self->rawframes,
                self->coll_alg, self->framecomb_parameterlist, 5.0,
                5.0, -999999.9, 999999.9, 0, NULL);

    if(self->static_badpixel_frame){
        hot_image = cpl_image_load(cpl_frame_get_filename(self->static_badpixel_frame),
                CPL_TYPE_INT, 0, 0);
        if (hot_image) {
            badpix = cpl_mask_threshold_image_create(hot_image, 0.5, 10000);
            cpl_image_delete(hot_image);
            hot_image = NULL;
        }
        if (!badpix){
        	cpl_msg_warning(cpl_func,"Could not read badpixel frame, continuing without. RMS and MEAN will be spoiled!");
        }
        else {
        	sph_master_frame_set_bads_from_mask(sky_bg_mframe, badpix );
        }

    }

    if (sky_bg_mframe) {
          pl = cpl_propertylist_new();

          cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_INSBG_MEAN_COUNT,
                  sph_master_frame_get_mean(sky_bg_mframe, &rms));
          cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_SKYBG_RMS_COUNT,
                  rms);
          cpl_propertylist_update_int(pl, SPH_COMMON_KEYWORD_NUMBER_HOTPIXELS,
                  sph_master_frame_get_nbads(sky_bg_mframe));
          pli = cpl_propertylist_load(
                  cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)),
                  0);
          sph_utils_simple_copy_singular(pli, pl);

          if (sph_master_frame_save_dfs(sky_bg_mframe,
                  self->sky_bg_outfilename, self->inframes,
                  cpl_frameset_get_first(self->rawframes), self->inparams,
                  SPH_IRD_TAG_SKY_BG_CALIB, SPH_RECIPE_NAME_IRD_SKY_BG,
                  SPH_PIPELINE_NAME_IRDIS, pl) != CPL_ERROR_NONE) {
              SPH_ERR("Could not save product sky background!");
          }


          if (self->save_addprod) {

        	  if (self->model_frame){
        		  model = sph_ird_instrument_model_load(
        		                  cpl_frame_get_filename(self->model_frame));
        		  if (!model)
        			  cpl_msg_warning(cpl_func,"Could not load model frame, falling back to default!");
        	  }
        	  if (!model)
        		  model = sph_ird_instrument_model_new();

        	  right_frame = sph_ird_instrument_model_extract_right_master_frame(model,sky_bg_mframe);
        	  left_frame = sph_ird_instrument_model_extract_left_master_frame(model,sky_bg_mframe);

        	  fitresult_left = sph_ird_sky_bg_create_polyfit(left_frame, self->order, 1000);
        	  fitresult_right = sph_ird_sky_bg_create_polyfit(right_frame, self->order, 1000);

        	  fitresult = sph_ird_instrument_model_assemble(model, fitresult_left, fitresult_right);
          }
          sph_master_frame_delete(sky_bg_mframe);
          sky_bg_mframe = NULL;

    }


    if (fitresult) {
        sph_master_frame_save_dfs(fitresult, self->lsf_outfilename,
                self->inframes, NULL, self->inparams,
                SPH_IRD_TAG_SKY_BG_FIT_CALIB, SPH_RECIPE_NAME_IRD_SKY_BG,
                SPH_PIPELINE_NAME_IRDIS, pl);

        sph_master_frame_delete(fitresult);
        fitresult = NULL;
    	}


    sph_filemanager_clean();

    cpl_propertylist_delete(pl); pl=NULL;
    cpl_propertylist_delete(pli); pli=NULL;

    return (int) cpl_error_get_code();
}

/**@}*/

