/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <cpl.h>
#include <math.h>
#include "sph_zpl_intensity_flat_imaging.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_zpl_exposure_imaging.h"
#include "sph_double_image.h"
#include "sph_create_flat.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_framecombination.h"
#include "sph_zpl_subtract_dark_scaled.h"

/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_intensity_flat_imaging_cunit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */

typedef struct _sph_zpl_intensity_flat_imaging_camera_unit {
    cpl_frameset* inframes; /* The recipe input frames */

    cpl_frameset* current_raw_frameset; /* The frameset for one product */

    cpl_parameterlist* inparams; /* The recipe input parameters */

    cpl_parameterlist* framecomb_parameterlist; /* The recipe input parameters */

    char* intensity_flat_imaging_outfilename; /* The parameter of zpl.intensity_flat_imaging.outfilename_cam1 */

    short subtract_overscan; /* The parameter of zpl.intensity_flat_imaging.subtract_overscan */

    short keep_intermediate; /* The parameter of zpl.intensity_flat_imaging.keep_intermediate */

    char* badpix_filename; /* The parameter of zpl.intensity_flat_imaging.badpixfilename_cam1 _cam2 */

    short robust_fit; /* The parameter of zpl.intensity_flat_imaging.robust_fit */

    short collapse; /* The parameter of zpl.intensity_flat_imaging.collapse */

    int coll_alg; /* The parameter of zpl.intensity_flat_imaging.coll_alg */

    int clean_mean_reject_high; /* The parameter of zpl.intensity_flat_imaging.coll_alg.clean_mean.reject_high */

    int clean_mean_reject_low; /* The parameter of zpl.intensity_flat_imaging.coll_alg.clean_mean.reject_low */

    double sigma_clip; /* The parameter of zpl.intensity_flat_imaging.sigma_clip */

    double badpix_lowtolerance; /* The parameter of zpl.intensity_flat_imaging.badpix_lowtolerance */

    double badpix_uptolerance; /* The parameter of zpl.intensity_flat_imaging.badpix_uptolerance */

    double badpix_chisqtolerance; /* The parameter of zpl.intensity_flat_imaging.badpix_chisqtolerance */

    char* outfilename_cam1; /* The parameter of zpl.preproc.outfilename_cam1 _cam2 */

    cpl_frameset* rawframes; /* The ZPL_INT_FLAT_FIELD_IMAGING_RAW frames */

    cpl_frameset* preproc_frames; /* The ZPL_INT_FLAT_FIELD_IMAGING_PREPROC_CAM1 or _CAM2 frames */

    cpl_frame* master_bias_frame; /* The ZPL_MASTER_BIAS_IMAGING_CAM1 or CAM2 frames */

    cpl_frame* master_dark_frame; /* The ZPL_MASTER_DARK_IMAGING_CAM1 or CAM2 frames */

    cpl_frame* static_badpixel_frame; /* The ZPL_STATIC_BADPIXELMAP_IMAGING_CAM1 or CAM2 frames */

    sph_double_image* intensity_flat_field_imaging; /* The ZPL_INT_FLAT_FIELD_IMAGING_CAM1 or CAM2 product */

    sph_zpl_exposure_imaging* linbadpix; /* The ZPL_NON_LINEAR_BADPIXELMAP_IMAGING_CAM1 or CAM2 product */

    char* eso_pro_catg; /* Here the ZPL_IFF_IMAGING_CAM1 or _CAM2 will be written */

    char* eso_pro_catg_nonlinbadmap; /* Here the ZPL_NON_LINEAR_BADPIXELMAP_IMAGING_CAM1 or _CAM2 will be written */

    char* eso_pro_catg_nonorm;

} sph_zpl_intensity_flat_imaging_cunit;

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_intensity_flat_imaging_cunit* _sph_zpl_intensity_flat_imaging_cunit_create(
        sph_zpl_intensity_flat_imaging* self, int camera);
static sph_error_code _sph_zpl_intensity_flat_imaging_cunit_delete(
        sph_zpl_intensity_flat_imaging_cunit* cunit);
static cpl_error_code _sph_zpl_intensity_flat_imaging_cunit_run(
        sph_zpl_intensity_flat_imaging_cunit* cunit);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_intensity_flat_imaging_run Create Intensity Flat Imaging Recipe
 *
 * This module provides the algorithm implementation for the creation of the
 * intensity flat field (imaging)
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_intensity_flat_imaging.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
cpl_error_code sph_zpl_intensity_flat_imaging_run(
        sph_zpl_intensity_flat_imaging* self) {
    sph_zpl_intensity_flat_imaging_cunit* cunit = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_intensity_flat_imaging_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_master_dark_preproc( self );
            recipe_error = sph_zpl_common_preproc_imaging_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }


        if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
            SPH_INFO_MSG("New style format of the raw data...")
            if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                        "The products will be created individually for each camera (off-line pipeline extended support!")
            } else {
                SPH_ERR("Pre-processing step is failed.");
                cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                return (int) recipe_error;
            }
        } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
            SPH_INFO_MSG("Old style format of the raw data...")
        } else {
            SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                    "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }
    }

    //reduce data for the camera-1
    //create cunit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create cunit structure for the camera-1...");
        cunit = _sph_zpl_intensity_flat_imaging_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

        if (cunit) {

            recipe_error = _sph_zpl_intensity_flat_imaging_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("redunction for the camera-1 is failed!");
            } else {
                //put the output product in the main self structure
                self->intensity_flat_field_imaging_cam1 =
                        sph_double_image_duplicate(
                                cunit->intensity_flat_field_imaging);
                if (cunit->linbadpix) {
                    self->linbadpix_cam1 = sph_zpl_exposure_imaging_duplicate(
                            cunit->linbadpix);
                }
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-1 is failed...trying for the camera-2")
        }
    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }
    if (cunit) {
        _sph_zpl_intensity_flat_imaging_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data for camera-2
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create cunit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create cunit structure for the camera-2...");
        cunit = _sph_zpl_intensity_flat_imaging_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);

        if (cunit) {
            recipe_error = _sph_zpl_intensity_flat_imaging_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("redunction for the camera-2 is failed!");
            } else {
                //put the output product in the main self structure
                self->intensity_flat_field_imaging_cam2 =
                        sph_double_image_duplicate(
                                cunit->intensity_flat_field_imaging);
                if (cunit->linbadpix) {
                    self->linbadpix_cam2 = sph_zpl_exposure_imaging_duplicate(
                            cunit->linbadpix);
                }
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-2 is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    if (cunit) {
        _sph_zpl_intensity_flat_imaging_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data
    SPH_RAISE_CPL_RESET;

    //reduce data for the new style data (on-line pipeline)
    //create cunit structure
    if (self->preproc_frames) {
        SPH_INFO_MSG("Create cunit structure for the new style format...");
        cunit = _sph_zpl_intensity_flat_imaging_cunit_create(self, SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID);

        if (cunit) {
            recipe_error = _sph_zpl_intensity_flat_imaging_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction is failed for the newstyle pre-processed data!");
            } else {
                //put the output product in the main self structure
                self->intensity_flat_field_imaging =
                        sph_double_image_duplicate(
                                cunit->intensity_flat_field_imaging);
                if (cunit->linbadpix) {
                    self->linbadpix = sph_zpl_exposure_imaging_duplicate(
                            cunit->linbadpix);
                }
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure newstyle pre-processed data is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the new style format -> nothing to reduce.");
    }

    if (cunit) {
        _sph_zpl_intensity_flat_imaging_cunit_delete(cunit);
        cunit = NULL;
    }


    if (self->intensity_flat_field_imaging_cam1 == NULL
            && self->intensity_flat_field_imaging_cam2 == NULL && self->intensity_flat_field_imaging == NULL) {
        SPH_ERR(" No outputs have been created !!!");
    }

    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }

    SPH_INFO_MSG("sph_zpl_intensity_flat_imaging_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

sph_zpl_intensity_flat_imaging_cunit* _sph_zpl_intensity_flat_imaging_cunit_create(
        sph_zpl_intensity_flat_imaging* self, int camera_id) {
    sph_zpl_intensity_flat_imaging_cunit* result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_intensity_flat_imaging_cunit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_masters_dark_imaging" structure into the cunit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->keep_intermediate = self->keep_intermediate;
    result->robust_fit = self->robust_fit;
    result->collapse = self->collapse;
    result->coll_alg = self->coll_alg;
    result->clean_mean_reject_high = self->clean_mean_reject_high;
    result->clean_mean_reject_low = self->clean_mean_reject_low;
    result->sigma_clip = self->sigma_clip;
    result->badpix_lowtolerance = self->badpix_lowtolerance;
    result->badpix_uptolerance = self->badpix_uptolerance;
    result->badpix_chisqtolerance = self->badpix_chisqtolerance;

    result->rawframes = self->rawframes;
    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);
        result->intensity_flat_imaging_outfilename = cpl_strdup(
                self->intensity_flat_imaging_outfilename_cam1);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_IFF_IMAGING_CALIB_CAM1);
        result->eso_pro_catg_nonlinbadmap = cpl_strdup(
                SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB_CAM1);
        result->badpix_filename = cpl_strdup(self->badpix_filename_cam1);
        result->eso_pro_catg_nonorm = cpl_strdup(
                SPH_ZPL_TAG_IFF_IMAGING_NONORM_CALIB_CAM1);

        if (self->master_bias_frame_cam1) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam1);
        }
        if (self->master_dark_frame_cam1) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam1);
        }
        if (self->static_badpixel_frame_cam1) {
            result->static_badpixel_frame = cpl_frame_duplicate(
                    self->static_badpixel_frame_cam1);
        }
    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);
        result->intensity_flat_imaging_outfilename = cpl_strdup(
                self->intensity_flat_imaging_outfilename_cam2);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_IFF_IMAGING_CALIB_CAM2);
        result->eso_pro_catg_nonlinbadmap = cpl_strdup(
                SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB_CAM2);
        result->badpix_filename = cpl_strdup(self->badpix_filename_cam2);
        result->eso_pro_catg_nonorm = cpl_strdup(
                SPH_ZPL_TAG_IFF_IMAGING_NONORM_CALIB_CAM2);

        if (self->master_bias_frame_cam2) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam2);
        }
        if (self->master_dark_frame_cam2) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam2);
        }
        if (self->static_badpixel_frame_cam2) {
            result->static_badpixel_frame = cpl_frame_duplicate(
                    self->static_badpixel_frame_cam2);
        }
    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames);
        result->intensity_flat_imaging_outfilename = cpl_strdup(
                self->intensity_flat_imaging_outfilename);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_IFF_IMAGING_CALIB);
        result->eso_pro_catg_nonlinbadmap = cpl_strdup(
                SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB);
        result->badpix_filename = cpl_strdup(self->badpix_filename);
        result->eso_pro_catg_nonorm = cpl_strdup(
                SPH_ZPL_TAG_IFF_IMAGING_NONORM_CALIB);

        if (self->master_bias_frame) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame);
        }
        if (self->master_dark_frame) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame);
        }
        if (self->static_badpixel_frame) {
            result->static_badpixel_frame = cpl_frame_duplicate(
                    self->static_badpixel_frame);
        }
    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2 for oldstyle format, 0 for newstyle format)",
                camera_id);
        _sph_zpl_intensity_flat_imaging_cunit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d",
                sph_error_get_last_code());
        _sph_zpl_intensity_flat_imaging_cunit_delete(result);
        result = NULL;
    }

    return result;

}

sph_error_code _sph_zpl_intensity_flat_imaging_cunit_delete(
        sph_zpl_intensity_flat_imaging_cunit* cunit) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (cunit == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return rerr;
    }

    /* Code to delete recipe pointers GENERATED DO NOT EDIT */

    if (cunit->preproc_frames != NULL) {
        cpl_frameset_delete(cunit->preproc_frames);
        cunit->preproc_frames = NULL;
    }

    if (cunit->intensity_flat_field_imaging) {
        sph_double_image_delete(cunit->intensity_flat_field_imaging);
        cunit->intensity_flat_field_imaging = NULL;
    }
    if (cunit->linbadpix) {
        sph_zpl_exposure_imaging_delete(cunit->linbadpix);
        cunit->linbadpix = NULL;
    }
    if (cunit->intensity_flat_imaging_outfilename)
        cpl_free(cunit->intensity_flat_imaging_outfilename);
    if (cunit->badpix_filename)
        cpl_free(cunit->badpix_filename);
    if (cunit->eso_pro_catg)
        cpl_free(cunit->eso_pro_catg);
    if (cunit->eso_pro_catg_nonlinbadmap)
        cpl_free(cunit->eso_pro_catg_nonlinbadmap);
    if (cunit->eso_pro_catg_nonorm)
        cpl_free(cunit->eso_pro_catg_nonorm);
    if (cunit->preproc_frames)
        cpl_frameset_delete(cunit->preproc_frames);

    //calibrations
    if (cunit->master_bias_frame)
        cpl_frame_delete(cunit->master_bias_frame);
    if (cunit->master_dark_frame)
        cpl_frame_delete(cunit->master_dark_frame);
    if (cunit->static_badpixel_frame)
        cpl_frame_delete(cunit->static_badpixel_frame);

    if (cunit->current_raw_frameset) {
        cpl_frameset_delete(cunit->current_raw_frameset);
        cunit->current_raw_frameset = NULL;
    }
    sph_polygon_free_all();
    cpl_free(cunit);
    return rerr;

}

static void sph_zpl_intensity_flat_imaging_delete__(
        sph_zpl_intensity_flat_imaging_cunit* self,
        sph_double_image* master_bias_doubleimage,
        sph_double_image* master_dark_doubleimage, cpl_mask* zplexp_odd_badpix,
        cpl_mask* zplexp_even_badpix, cpl_mask* curdoubleimage_mask,
        sph_double_image* curdoubleimage, cpl_vector* lampflux_odd,
        cpl_vector* lampflux_even, cpl_vector* lampcounts_odd,
        cpl_vector* lampcounts_even,
        cpl_frameset* rawframes_doubleimage) {
    if (self->linbadpix) {
        sph_zpl_exposure_imaging_delete(self->linbadpix); //it should be better deleted in the main.c file
        self->linbadpix = NULL;
    }
    if (master_bias_doubleimage) {
        sph_double_image_delete(master_bias_doubleimage);

    }
    if (master_dark_doubleimage) {
        sph_double_image_delete(master_dark_doubleimage);

    }
    if (zplexp_odd_badpix) {
        cpl_mask_delete(zplexp_odd_badpix);

    }
    if (zplexp_even_badpix) {
        cpl_mask_delete(zplexp_even_badpix);

    }
    if (curdoubleimage_mask) {
        cpl_mask_delete(curdoubleimage_mask);

    }
    if (curdoubleimage) {
        sph_double_image_delete(curdoubleimage);

    }
    if (lampflux_odd) {
        cpl_vector_delete(lampflux_odd);

    }
    if (lampflux_even) {
        cpl_vector_delete(lampflux_even);

    }
    if (lampcounts_odd) {
        cpl_vector_delete(lampcounts_odd);

    }
    if (lampcounts_even) {
        cpl_vector_delete(lampcounts_even);

    }

    if (rawframes_doubleimage) {
        cpl_frameset_delete(rawframes_doubleimage);

    }

}
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_intensity_flat_imaging recipe
 (imaging mode). The error  code returned is always a cpl error code
 (to allow maximal compatibility with  esorex, gasgano, etc.) even if during
 recipe execution an error in the SPHERE API is the cause. In this case
 (and if the underlying error is not a cpl error)  the cpl error code is set
 to the cpl_error_code that matches the failure reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code _sph_zpl_intensity_flat_imaging_cunit_run(
        sph_zpl_intensity_flat_imaging_cunit* self) {
    sph_double_image* master_bias_doubleimage = NULL;
    sph_double_image* master_dark_doubleimage = NULL;
    sph_double_image* curdoubleimage = NULL;
    sph_master_frame* mf_odd = NULL;
    sph_master_frame* mf_even = NULL;
    sph_master_frame* mf_tmp = NULL;
    double mean = 0.0;
    double rms = 0.0;

    sph_zpl_exposure_imaging* zplexp_badpixmap = NULL;
    cpl_mask* zplexp_odd_badpix = NULL;
    cpl_mask* zplexp_even_badpix = NULL;
    cpl_mask* curdoubleimage_mask = NULL;

    cpl_image* odd_linbadpix = NULL;
    cpl_image* even_linbadpix = NULL;
    cpl_propertylist* pl_linbadpix = NULL;

    cpl_frameset* rawframes_doubleimage = NULL;
    cpl_frame* curframe = NULL;
    //cpl_mask*           tmpmask 				= NULL;
    cpl_vector* lampflux_odd = NULL;
    cpl_vector* lampflux_even = NULL;
    cpl_vector* lampcounts_odd = NULL;
    cpl_vector* lampcounts_even = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pl_rawheader = NULL;
    cpl_frameset* ovsc_subtract_frames = NULL;

    char outfilename[256];

    char lampflux_keyname[256];
    int recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting static _sph_zpl_intensity_flat_imaging_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (!self->preproc_frames) {
        SPH_ERR("no preproc_frame are set up; please verify TAGS")
        return sph_error_get_last_code();
    }

    if (self->robust_fit) {
        if (cpl_frameset_get_size(self->rawframes) < 5) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "The robust fitting is enabled. "
                    "For this to work, a minimum of 5 raw frames needs "
                    "to be provided. Either provide more frames or "
                    "switch robust fitting off.");
            return CPL_ERROR_ILLEGAL_INPUT;
        }
    }

    if (self->static_badpixel_frame) {
        zplexp_badpixmap = sph_zpl_exposure_imaging_load(
                self->static_badpixel_frame, 0);

        if (zplexp_badpixmap) {
            zplexp_odd_badpix = cpl_mask_threshold_image_create(
                    zplexp_badpixmap->image_odd, 0.5, 10000);
            zplexp_even_badpix = cpl_mask_threshold_image_create(
                    zplexp_badpixmap->image_even, 0.5, 10000);
            sph_zpl_exposure_imaging_delete(zplexp_badpixmap);
            zplexp_badpixmap = NULL;
        } else {
            SPH_ERROR_RAISE_WARNING(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "The static badpixel map could not be read from the given input file.");
            return CPL_ERROR_ILLEGAL_INPUT;

        }
    }

    if (self->collapse) {
        /********************************************************************
         * Simple combining of the frames and normalizing, no linear fitting
         ********************************************************************/
        //self->intensity_flat_field_imaging = sph_framecombination_double_image_from_frameset(
        //						self->rawframes, self->coll_alg, self->framecomb_parameterlist );
        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from pre-processed frames...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(
                    self->preproc_frames);
            if (ovsc_subtract_frames) {
                self->intensity_flat_field_imaging =
                        sph_framecombination_double_image_from_frameset(
                                ovsc_subtract_frames, self->coll_alg,
                                self->framecomb_parameterlist);
                if (!self->keep_intermediate) {
                    sph_utils_frames_unlink(ovsc_subtract_frames);
                }
                cpl_frameset_delete(ovsc_subtract_frames); ovsc_subtract_frames = NULL;
            } else {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further.")
                self->intensity_flat_field_imaging =
                        sph_framecombination_double_image_from_frameset(
                                self->preproc_frames, self->coll_alg,
                                self->framecomb_parameterlist);
            }
        } else {
            self->intensity_flat_field_imaging =
                    sph_framecombination_double_image_from_frameset(
                            self->preproc_frames, self->coll_alg,
                            self->framecomb_parameterlist);
        }

        if (!self->intensity_flat_field_imaging) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol double intensity flat field:\n"
                            "-- sph_framecombination_double_image_from_frameset returns null pointer.\n"
                            "cpl error code is: %d", cpl_error_get_code());

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return cpl_error_get_code();
        }

        //if master bias exists load and subtract it from the master dark
        if (self->master_bias_frame) {
            master_bias_doubleimage = sph_double_image_load(
                    cpl_frame_get_filename(self->master_bias_frame), 0);
            if (master_bias_doubleimage) {
                recipe_error = sph_double_image_subtract_double_image(
                        self->intensity_flat_field_imaging,
                        master_bias_doubleimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                            __LINE__, SPH_ERROR_ERROR,
                            "Error is raised by subtracting master bias:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_imaging_delete__(self,
                            master_bias_doubleimage, master_dark_doubleimage,
                            zplexp_odd_badpix, zplexp_even_badpix,
                            curdoubleimage_mask, curdoubleimage, lampflux_odd,
                            lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

                    return cpl_error_get_code();
                }
            }
        }

        //if master dark exists load and subtract it from the master dark
        if (self->master_dark_frame) {
            master_dark_doubleimage = sph_double_image_load(
                    cpl_frame_get_filename(self->master_dark_frame), 0);
            if (master_dark_doubleimage) {
                recipe_error = sph_zpl_subtract_dark_double_image_scaled(
                        self->intensity_flat_field_imaging,
                        master_dark_doubleimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                            __LINE__, SPH_ERROR_ERROR,
                            "Error is raised by subtracting master dark:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_imaging_delete__(self,
                            master_bias_doubleimage, master_dark_doubleimage,
                            zplexp_odd_badpix, zplexp_even_badpix,
                            curdoubleimage_mask, curdoubleimage, lampflux_odd,
                            lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

                    return cpl_error_get_code();
                }
            }
        }

        if (zplexp_odd_badpix && zplexp_even_badpix) {

            //combine a new created self->intensity_flat_field_imaging badmap with a given one (as a separate input zpl exp frame)
            recipe_error = sph_master_frame_set_bads_from_mask(
                    self->intensity_flat_field_imaging->iframe,
                    zplexp_odd_badpix);
            recipe_error |= sph_master_frame_set_bads_from_mask(
                    self->intensity_flat_field_imaging->pframe,
                    zplexp_even_badpix);
            if (recipe_error) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                                "cpl error code is: %d", recipe_error);

                sph_zpl_intensity_flat_imaging_delete__(self,
                        master_bias_doubleimage, master_dark_doubleimage,
                        zplexp_odd_badpix, zplexp_even_badpix,
                        curdoubleimage_mask, curdoubleimage, lampflux_odd,
                        lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

                return recipe_error;
            }
        }

        //quality check (calculate statistical moments for the intensity flat)
        recipe_error = sph_double_image_quality_check_names(
                self->intensity_flat_field_imaging,
				NULL, &pframe_names_leakage);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }SPH_INFO_MSG("Calculated quality check.")

        //mask hot/bad pixels based on the given sigma_clip parameter
        recipe_error = sph_double_image_mask_sigma(
                self->intensity_flat_field_imaging, self->sigma_clip);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Could not create zimpol intensity flat:\n"
                            "sph_double_image_mask_sigma returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }SPH_INFO_MSG("Created mask using sigma clipping.")

        //re-calculate quality check (calculate statistical moments for the intensity flat) to exclude badpixels
        recipe_error = sph_double_image_quality_check_names(
                self->intensity_flat_field_imaging,
				NULL, &pframe_names_leakage);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }SPH_INFO_MSG(
                "Quality check re-done to exclude badpixels from the calculation...");

        /* Attention:
         * The following function includes a step to update SPH_COMMON_KEYWORD_SPH_TYPE by
         * SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE. It is necessary because cpl_dsf_save_image
         * updates the double image property list by the property list from the first input frames.
         * In the case of some misbehavior exclude this step for checking.
         */

        if (self->keep_intermediate) {
            SPH_INFO_MSG("Save non-normalized output filename:");
            sprintf(outfilename, "%s-qc-non-normalized.fits",
                    self->intensity_flat_imaging_outfilename);
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                    "outfilename = %s", outfilename)

            if (self->rawframes) {
                SPH_INFO_MSG(
                        "Save non-normalized iff double image as a dfs product: header info taken from the first raw frame.")
                pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                        self->rawframes);

                cpl_propertylist_update_string(pl, SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT,
                		SPH_IMAGING_PFRAME_NAME_LEAKAGE_NAME);

                cpl_propertylist_append(pl, self->intensity_flat_field_imaging->properties);
                recipe_error = sph_double_image_save_dfs(
                        self->intensity_flat_field_imaging, outfilename,
                        self->inframes, cpl_frameset_get_first(self->rawframes),
                        self->inparams,
                        self->eso_pro_catg_nonorm, //SPH_ZPL_TAG_IFF_IMAGING_NONORM_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF_IMAGING,
                        SPH_PIPELINE_NAME_ZIMPOL, pl);
                cpl_propertylist_delete(pl);
                pl = NULL;
            } else if (self->preproc_frames) {
                SPH_INFO_MSG(
                        "Save non-normalized iff double image as a dfs product: header info taken from the first pre-processed frame.")
                recipe_error = sph_double_image_save_dfs(
                        self->intensity_flat_field_imaging, outfilename,
                        self->inframes,
                        cpl_frameset_get_first(self->preproc_frames),
                        self->inparams,
                        self->eso_pro_catg_nonorm, //SPH_ZPL_TAG_IFF_IMAGING_NONORM_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF_IMAGING,
                        SPH_PIPELINE_NAME_ZIMPOL, pl);
            } else {
                SPH_ERR(
                        "Neither raw frames nor pre-processed frames provided!");
                recipe_error = sph_error_get_last_code();
            }
        }

        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "couldn't save non-normalized intensity flat field.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }

        //normalize each sub-frame individually (i.e. divided by corresponding sub-frame mean )
        recipe_error = sph_master_frame_divide_double(
                self->intensity_flat_field_imaging->iframe,
                cpl_propertylist_get_double(
                        self->intensity_flat_field_imaging->qclist,
                        SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME));
        recipe_error |= sph_master_frame_divide_double(
                self->intensity_flat_field_imaging->pframe,
                cpl_propertylist_get_double(
                        self->intensity_flat_field_imaging->qclist,
                        pframe_names_leakage.mean_name));

        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "normalization is failed.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }

        //Normalized quality check (calculate statistical moments for the intensity flat)
        recipe_error = sph_double_image_quality_check_names(
                self->intensity_flat_field_imaging, NULL,
				&pframe_names_leakage);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }SPH_INFO_MSG(
                "Calculate quality check for the normalized double image.")

        if (self->rawframes) {
            pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->rawframes);
            cpl_propertylist_append(pl, self->intensity_flat_field_imaging->properties);
            sph_zpl_qc_add_keys(pl, self->rawframes,
                                self->intensity_flat_field_imaging->properties);
            sph_zpl_qc_add_plane_keys(pl, self->rawframes, CPL_FALSE);
            SPH_INFO_MSG(
                    "Save normalized iff double image as a dfs product: header info taken from the first raw frame.")
            recipe_error = sph_double_image_save_dfs(
                    self->intensity_flat_field_imaging,
                    self->intensity_flat_imaging_outfilename, self->inframes,
                    cpl_frameset_get_first(self->rawframes),
                    self->inparams,
                    self->eso_pro_catg, //SPH_ZPL_TAG_IFF_IMAGING_CALIB,
                    SPH_RECIPE_NAME_ZPL_IFF_IMAGING, SPH_PIPELINE_NAME_ZIMPOL,
                    pl);
            cpl_propertylist_delete(pl);
            pl = NULL;
        } else if (self->preproc_frames) {
            sph_zpl_qc_add_keys(pl, self->preproc_frames,
                                self->intensity_flat_field_imaging->properties);
            sph_zpl_qc_add_plane_keys(pl, self->preproc_frames, CPL_FALSE);
            SPH_INFO_MSG(
                    "Save normalized iff double image as a dfs product: header info taken from the first pre-procesed frame.")
            recipe_error = sph_double_image_save_dfs(
                    self->intensity_flat_field_imaging,
                    self->intensity_flat_imaging_outfilename, self->inframes,
                    cpl_frameset_get_first(self->preproc_frames),
                    self->inparams,
                    self->eso_pro_catg, //SPH_ZPL_TAG_IFF_IMAGING_CALIB,
                    SPH_RECIPE_NAME_ZPL_IFF_IMAGING, SPH_PIPELINE_NAME_ZIMPOL,
                    pl);

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        cpl_propertylist_delete(pl);
        pl = NULL;

        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol intensity flat field imaging:\n"
                            "sph_double_image_save_dfs returns error code.\n"
                            "cpl error code is: %d", recipe_error);
        }

        if (master_bias_doubleimage) {
            sph_double_image_delete(master_bias_doubleimage);
            master_bias_doubleimage = NULL;
        }
        if (master_dark_doubleimage) {
            sph_double_image_delete(master_dark_doubleimage);
            master_dark_doubleimage = NULL;
        }

        sph_zpl_intensity_flat_imaging_delete__(self, master_bias_doubleimage,
                master_dark_doubleimage, zplexp_odd_badpix, zplexp_even_badpix,
                curdoubleimage_mask, curdoubleimage, lampflux_odd,
                lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

        SPH_INFO_MSG("static _sph_zpl_intensity_flat_imaging_run...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    /************************************************************************
     * Linear fitting method for the calculation of the intensity flat field
     ************************************************************************/

    if (self->subtract_overscan) {
        SPH_INFO_MSG("Subtract overscans from pre-processed frames...")
        ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(
                self->preproc_frames);
        if (ovsc_subtract_frames) {
            rawframes_doubleimage =
                    sph_double_image_create_double_image_frameset_from_zplexp_imaging_cubes(
                            ovsc_subtract_frames);
            if (!self->keep_intermediate) {
                sph_utils_frames_unlink(ovsc_subtract_frames);
            }
            cpl_frameset_delete(ovsc_subtract_frames); ovsc_subtract_frames = NULL;
        } else {
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_WARNING,
                    "Overscans could not be subtracted, trying to proceed further.")
            rawframes_doubleimage =
                    sph_double_image_create_double_image_frameset_from_zplexp_imaging_cubes(
                            self->preproc_frames);
        }
    } else {
        rawframes_doubleimage =
                sph_double_image_create_double_image_frameset_from_zplexp_imaging_cubes(
                        self->preproc_frames);
    }

    if (!rawframes_doubleimage) {
        SPH_ERR(
                "couldn't create rawframes_doubleimage from the intensity flat imaging rawframes");

        sph_zpl_intensity_flat_imaging_delete__(self, master_bias_doubleimage,
                master_dark_doubleimage, zplexp_odd_badpix, zplexp_even_badpix,
                curdoubleimage_mask, curdoubleimage, lampflux_odd,
                lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

        return sph_error_get_last_code();
    }
    //if master bias exists load it
    if (self->master_bias_frame) {
        master_bias_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->master_bias_frame), 0);
    }
    //if master dark exist load it
    if (self->master_dark_frame) {
        master_dark_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->master_dark_frame), 0);
    }

    if (master_bias_doubleimage || master_dark_doubleimage) {

        curframe = cpl_frameset_get_first(rawframes_doubleimage);
        while (curframe) {
            const char * filename = cpl_frame_get_filename(curframe);

            curdoubleimage = sph_double_image_load(filename, 0);
            if (!curdoubleimage) {
                SPH_ERR("couldn't load current double image from the file")
                return sph_error_get_last_code();
            }
            if (master_bias_doubleimage) {
                recipe_error = sph_double_image_subtract_double_image(
                        curdoubleimage, master_bias_doubleimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(
                            SPH_ERROR_GENERAL,
                            __FILE__,
                            __func__,
                            __LINE__,
                            SPH_ERROR_ERROR,
                            "Error is raised by subtracting master bias imaging:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_imaging_delete__(self,
                            master_bias_doubleimage, master_dark_doubleimage,
                            zplexp_odd_badpix, zplexp_even_badpix,
                            curdoubleimage_mask, curdoubleimage, lampflux_odd,
                            lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

                    return cpl_error_get_code();
                }
            }
            if (master_dark_doubleimage) {
                recipe_error = sph_zpl_subtract_dark_double_image_scaled(
                        curdoubleimage, master_dark_doubleimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(
                            SPH_ERROR_GENERAL,
                            __FILE__,
                            __func__,
                            __LINE__,
                            SPH_ERROR_ERROR,
                            "Error is raised by subtracting dark bias imaging:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_imaging_delete__(self,
                            master_bias_doubleimage, master_dark_doubleimage,
                            zplexp_odd_badpix, zplexp_even_badpix,
                            curdoubleimage_mask, curdoubleimage, lampflux_odd,
                            lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

                    return cpl_error_get_code();
                }
            }

            sph_double_image_save(curdoubleimage,
                    cpl_frame_get_filename(curframe), NULL, CPL_IO_CREATE);
            sph_double_image_delete(curdoubleimage);
            curframe = cpl_frameset_get_next(rawframes_doubleimage);

        } //end while
    } //enf if-block-with dark imaging and bias imaging

    for (int i = 0; i < 2; i++) {
        mf_tmp = sph_framecombination_master_frame_from_frameset(
                rawframes_doubleimage, self->coll_alg,
                self->framecomb_parameterlist, i);
        mean = sph_master_frame_get_mean(mf_tmp, &rms);
        if (mean < 1000.0) {
            SPH_ERROR_RAISE_WARNING(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "The mean of a flat is %f. "
                    "Using flats with mean counts less than 1000 is dangerous."
                    "The recipe will proceed but be aware that the master"
                    " flat created will most likely not be of high quality.", mean);
        }
        //sph_master_frame_mask_tolerance(mf_tmp, mean*0.1, FLT_MAX);
        //tmpmask = sph_master_frame_get_badpixelmask(mf_tmp);
        sph_master_frame_delete(mf_tmp);
        mf_tmp = NULL;
        mean = 0.0;
        rms = 0.0;
    }

    //load double image from the first rawframes_doubleimage in order to get badpixelmaps and then create badpixel masks
    //it is supposed that the statical bad pixel map is the same for all rawframes_doubleimage, so we retrieve it only from
    // from the first frame
    curdoubleimage = sph_double_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(rawframes_doubleimage)), 0);

    //create mask from iframe->badpixelmap and combine it (if exists) with
    //zplexp_odd_badpix given as an input badpixmap
    curdoubleimage_mask = cpl_mask_threshold_image_create(
            curdoubleimage->iframe->badpixelmap, 0.5, 10000);
    if (zplexp_odd_badpix) {
        recipe_error = cpl_mask_or(curdoubleimage_mask, zplexp_odd_badpix);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }
    }
    mf_odd = sph_create_flat(rawframes_doubleimage, self->robust_fit,
            curdoubleimage_mask, NULL, &odd_linbadpix,
            self->badpix_lowtolerance, self->badpix_uptolerance,
            self->badpix_chisqtolerance, &lampflux_odd, &lampcounts_odd, 0, NULL);
    cpl_mask_delete(curdoubleimage_mask);
    curdoubleimage_mask = NULL;

    //create mask from pframe->badpixelmap and combine it (if exists) with
    //zplexp_odd_badpix given as an input badpixmap
    curdoubleimage_mask = cpl_mask_threshold_image_create(
            curdoubleimage->pframe->badpixelmap, 0.5, 10000);
    if (zplexp_even_badpix) {
        recipe_error = cpl_mask_or(curdoubleimage_mask, zplexp_even_badpix);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_imaging_delete__(self,
                    master_bias_doubleimage, master_dark_doubleimage,
                    zplexp_odd_badpix, zplexp_even_badpix, curdoubleimage_mask,
                    curdoubleimage, lampflux_odd, lampflux_even, lampcounts_odd, lampcounts_even,
                    rawframes_doubleimage);

            return recipe_error;
        }
    }

    mf_even = sph_create_flat(rawframes_doubleimage, self->robust_fit,
            curdoubleimage_mask, NULL, &even_linbadpix,
            self->badpix_lowtolerance, self->badpix_uptolerance,
            self->badpix_chisqtolerance, &lampflux_even, &lampcounts_even, 4, NULL);

    if (mf_odd && mf_even) {
        self->intensity_flat_field_imaging =
                sph_double_image_new_from_master_frames(mf_odd, mf_even);
    } else {
        SPH_ERR("intensity flat field imaging product has NULL pointer");

        sph_zpl_intensity_flat_imaging_delete__(self, master_bias_doubleimage,
                master_dark_doubleimage, zplexp_odd_badpix, zplexp_even_badpix,
                curdoubleimage_mask, curdoubleimage, lampflux_odd,
                lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

        return (int) cpl_error_get_code();
    }

    //quality check (calculate statistical moments for the intensity flat)
    recipe_error = sph_double_image_quality_check_names(
            self->intensity_flat_field_imaging,
			NULL, &pframe_names_leakage);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "sph_double_image_quality_check returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);

        sph_zpl_intensity_flat_imaging_delete__(self, master_bias_doubleimage,
                master_dark_doubleimage, zplexp_odd_badpix, zplexp_even_badpix,
                curdoubleimage_mask, curdoubleimage, lampflux_odd,
                lampflux_even, lampcounts_odd, lampcounts_even, rawframes_doubleimage);

        return recipe_error;
    }

    //add some QC keywords to the pl of the intensity flat field imaging product
    pl = cpl_propertylist_new();
    //bad pixels
    cpl_propertylist_update_long(
            pl,
            SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD,
            (long) cpl_image_get_flux(
                    self->intensity_flat_field_imaging->iframe->badpixelmap));
    cpl_propertylist_update_long(
            pl,
            pframe_names_leakage.bad_pixels_name,
            (long) cpl_image_get_flux(
                    self->intensity_flat_field_imaging->pframe->badpixelmap));

    //mean value of each frame: odd (iframe)
    for (int ii = 0; ii < cpl_vector_get_size(lampflux_odd); ++ii) {
        sprintf(lampflux_keyname, "%s ODD %d",
                SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX, ii);
        cpl_propertylist_append_double(pl, lampflux_keyname,
                cpl_vector_get(lampflux_odd, ii));
    }

    //mean value of each frame: odd (iframe)
    for (int ii = 0; ii < cpl_vector_get_size(lampflux_even); ++ii) {
        sprintf(lampflux_keyname, "%s EVEN %d",
                SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX, ii);
        cpl_propertylist_append_double(pl, lampflux_keyname,
                cpl_vector_get(lampflux_even, ii));
    }

    /* Attention:
     * The following function includes a step to update SPH_COMMON_KEYWORD_SPH_TYPE by
     * SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE. It is necessary because cpl_dsf_save_image
     * updates the double image property list by the property list from the first input frames.
     * In the case of some misbehavior exclude this step for checking.
     */

    if (self->rawframes) {
        pl_rawheader = sph_zpl_utils_get_camera_header(self->preproc_frames,
                self->rawframes);
        cpl_propertylist_append(pl, pl_rawheader);
        SPH_INFO_MSG(
                "Save iff double image (linear fitting) as a dfs product: header info taken from the first raw frame.")
        recipe_error = sph_double_image_save_dfs(
                self->intensity_flat_field_imaging,
                self->intensity_flat_imaging_outfilename, self->inframes,
                cpl_frameset_get_first(self->rawframes), self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_IFF_IMAGING_CALIB,
                SPH_RECIPE_NAME_ZPL_IFF_IMAGING, SPH_PIPELINE_NAME_ZIMPOL, pl);
        cpl_propertylist_delete(pl_rawheader);
        pl_rawheader = NULL;
    } else if (self->preproc_frames) {
        SPH_INFO_MSG(
                "Save iff double image (linear fitting) as a dfs product: header info taken from the first pre-processed frame.")
        recipe_error = sph_double_image_save_dfs(
                self->intensity_flat_field_imaging,
                self->intensity_flat_imaging_outfilename, self->inframes,
                cpl_frameset_get_first(self->preproc_frames), self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_IFF_IMAGING_CALIB,
                SPH_RECIPE_NAME_ZPL_IFF_IMAGING, SPH_PIPELINE_NAME_ZIMPOL, pl);
    } else {
        SPH_ERR("Neither raw frames nor pre-processed frames provided!");
        recipe_error = sph_error_get_last_code();
    }
    cpl_propertylist_delete(pl);
    pl = NULL;

    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not create zimpol intensity flat field imaging:\n"
                        "sph_double_image_save returns error code.\n"
                        "cpl error code is: %d", recipe_error);
    }

    if (odd_linbadpix && even_linbadpix) {
        self->linbadpix = sph_zpl_exposure_imaging_new_from_cplimages(
                odd_linbadpix, even_linbadpix);
        cpl_image_delete(odd_linbadpix);
        odd_linbadpix = NULL;
        cpl_image_delete(even_linbadpix);
        even_linbadpix = NULL;

        pl_linbadpix = cpl_propertylist_new();

        cpl_propertylist_update_long(pl_linbadpix,
                SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_ZPLEXP_IMG_ODD,
                (long) cpl_image_get_flux(self->linbadpix->image_odd));
        cpl_propertylist_update_long(pl_linbadpix,
                SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_ZPLEXP_IMG_EVEN,
                (long) cpl_image_get_flux(self->linbadpix->image_even));
        cpl_propertylist_append_string(pl_linbadpix,
                SPH_COMMON_KEYWORD_PRO_CATG,
                SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_IMAGING_CALIB);

        if (self->keep_intermediate) {

            if (self->rawframes) {
                SPH_INFO_MSG(
                        "Save zpl exp linbadpix map as a dfs product: header info taken from the first raw frame.");
                pl = sph_zpl_utils_get_camera_header(self->preproc_frames, self->rawframes);
                cpl_propertylist_append(pl, pl_linbadpix);

                sph_zpl_exposure_imaging_save_dfs(self->linbadpix,
                        self->badpix_filename, self->inframes,
                        cpl_frameset_get_first(self->rawframes),
                        self->inparams,
                        self->eso_pro_catg_nonlinbadmap, //SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL,
                        pl);
                if ( pl ) {
                    cpl_propertylist_delete( pl ); pl = NULL;
                }

            } else if (self->preproc_frames) {
                SPH_INFO_MSG(
                        "Save zpl exp linbadpix map as a dfs product: header info taken from the first pre-processed frame.")
                sph_zpl_exposure_imaging_save_dfs(self->linbadpix,
                        self->badpix_filename, self->inframes,
                        cpl_frameset_get_first(self->preproc_frames),
                        self->inparams,
                        self->eso_pro_catg_nonlinbadmap, //SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL,
                        pl_linbadpix);
            } else {
                SPH_ERR(
                        "Neither raw frames nor pre-processed frames provided!");
                recipe_error = sph_error_get_last_code();
            }
        }

        cpl_propertylist_delete(pl_linbadpix);
        pl_linbadpix = NULL;

    } else {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Couldn't create self->linbadpix map of the zpl exp imaging type");
    }

    sph_zpl_intensity_flat_imaging_delete__(self, master_bias_doubleimage,
            master_dark_doubleimage, zplexp_odd_badpix, zplexp_even_badpix,
            curdoubleimage_mask, curdoubleimage, lampflux_odd, lampflux_even,
            lampcounts_odd, lampcounts_even,
            rawframes_doubleimage);

    //sph_keyword_manager_delete();

    SPH_INFO_MSG("static _sph_zpl_intensity_flat_imaging_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}
/**@}*/
