/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <cpl.h>

#include "sph_zpl_intensity_flat.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_exposure.h"
#include "sph_master_frame.h"
#include "sph_quad_image.h"
#include "sph_framecombination.h"
#include "sph_create_flat.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_zpl_subtract_dark_scaled.h"

/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_intensity_flat_cunit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */
typedef struct _sph_zpl_intensity_flat_camera_unit_ {
    cpl_frameset* inframes; /* The recipe input frames */

    cpl_frameset* current_raw_frameset; /* The frameset for one product */

    cpl_parameterlist* inparams; /* The recipe input parameters */

    cpl_parameterlist* framecomb_parameterlist; /* The recipe input parameters */

    char* master_intensity_flat_outfilename; /* The parameter of zpl.master_intensity_flat.outfilename_cam1 */

    char* intensity_flat_outfilename; /* The parameter of zpl.intensity_flat.outfilename_cam1 */

    short subtract_overscan; /* The parameter of zpl.intensity_flat.subtract_overscan */

    short keep_intermediate; /* The parameter of zpl.intensity_flat.keep_intermediate */

    char* badpix_filename; /* The parameter of zpl.intensity_flat.badpixfilename */

    short robust_fit; /* The parameter of zpl.intensity_flat.robust_fit */

    short collapse; /* The parameter of zpl.intensity_flat.collapse */

    int coll_alg; /* The parameter of zpl.intensity_flat.coll_alg */

    int clean_mean_reject_high; /* The parameter of zpl.intensity_flat.coll_alg.clean_mean.reject_high */

    int clean_mean_reject_low; /* The parameter of zpl.intensity_flat.coll_alg.clean_mean.reject_low */

    double sigma_clip; /* The parameter of zpl.intensity_flat.sigma_clip */

    double badpix_lowtolerance; /* The parameter of zpl.intensity_flat.badpix_lowtolerance */

    double badpix_uptolerance; /* The parameter of zpl.intensity_flat.badpix_uptolerance */

    double badpix_chisqtolerance; /* The parameter of zpl.intensity_flat.badpix_chisqtolerance */

    short quadimage_weight_mean; /* The parameter of zpl.intensity_flat.quadimage_weight_mean */

    cpl_frameset* rawframes; /* The ZPL_INT_FLAT_FIELD_RAW frames */

    cpl_frameset* preproc_frames; /* The ZPL_INT_FLAT_PREPROC_CAM1 frames */

    cpl_frame* master_bias_frame; /* The ZPL_MASTER_BIAS_CAM1 frames */

    cpl_frame* master_dark_frame; /* The ZPL_MASTER_DARK_CAM1 frames */

    cpl_frame* static_badpixel_frame; /* The ZPL_STATIC_BADPIXELMAP frames */

    sph_master_frame* master_iff; /* The ZPL_INT_FLAT_FIELD_MASTER_CAM1 product */

    sph_quad_image* intensity_flat_field; /* The ZPL_INT_FLAT_FIELD_CAM1 product */

    sph_zpl_exposure* linbadpix; /* The ZPL_NON_LINEAR_BADPIXELMAP_CAM1 product */

    char* eso_pro_catg; /* Here the ZPL_INT_FLAT_FIELD_CAM1 or _CAM2 will be written */

    char* eso_pro_catg_master;/* Here the ZPL_INT_FLAT_FIELD_MASTER_CAM1 or _CAM2 will be written */

    char* eso_pro_catg_nonlinbadmap;/* Here the ZPL_NON_LINEAR_BADPIXELMAP_CAM1 or _CAM2 will be written */

    char* eso_pro_catg_nonorm;

    char* eso_pro_catg_nonorm_master;
} sph_zpl_intensity_flat_cunit;

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_intensity_flat_cunit* _sph_zpl_intensity_flat_cunit_create(
        sph_zpl_intensity_flat* self, int camera);
static sph_error_code _sph_zpl_intensity_flat_cunit_delete(
        sph_zpl_intensity_flat_cunit* cunit);
static cpl_error_code _sph_zpl_intensity_flat_cunit_run(
        sph_zpl_intensity_flat_cunit* cunit);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_intensity_flat_run Create Intensity Flat Recipe
 *
 * This module provides the algorithm implementation for the creation of the
 * intensity flat field (polarimetry)
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_intensity_flat.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_intensity_flat recipe
 (polarimetric mode). The error  code returned is always a cpl error code
 (to allow maximal compatibility with  esorex, gasgano, etc.) even if during
 recipe execution an error in the SPHERE API is the cause. In this case
 (and if the underlying error is not a cpl error)  the cpl error code is set
 to the cpl_error_code that matches the failure reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_zpl_intensity_flat_run(sph_zpl_intensity_flat* self) {
    sph_zpl_intensity_flat_cunit* cunit = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_intensity_flat_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_master_dark_preproc( self );
            recipe_error = sph_zpl_common_preproc_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }


        if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
            SPH_INFO_MSG("New style format of the raw data...")
            if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                        "The products will be created individually for each camera (off-line pipeline extended support!")
            } else {
                SPH_ERR("Pre-processing step is failed.");
                cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                return (int) recipe_error;
            }
        } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
            SPH_INFO_MSG("Old style format of the raw data...")
        } else {
            SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                    "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }
    }

    //reduce data for the camera-1
    //create cunit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create cunit structure for the camera-1...");
        cunit = _sph_zpl_intensity_flat_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

        if (cunit) {

            recipe_error = _sph_zpl_intensity_flat_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction for the camera-1 is failed!");
            } else {
                //put the output product in the main self structure
                self->intensity_flat_field_cam1 = sph_quad_image_duplicate(
                        cunit->intensity_flat_field);
                self->master_iff_cam1 = sph_master_frame_duplicate(
                        cunit->master_iff);
                if (cunit->linbadpix) {
                    self->linbadpix_cam1 = sph_zpl_exposure_duplicate(
                            cunit->linbadpix);
                }
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-1 is failed...trying for the camera-2")
        }
    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }
    if (cunit) {
        _sph_zpl_intensity_flat_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data for camera-2
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create cunit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create cunit structure for the camera-2...");
        cunit = _sph_zpl_intensity_flat_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);

        if (cunit) {
            recipe_error = _sph_zpl_intensity_flat_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction for the camera-2 is failed!");
            } else {
                //put the output product in the main self structure
                self->intensity_flat_field_cam2 = sph_quad_image_duplicate(
                        cunit->intensity_flat_field);
                self->master_iff_cam2 = sph_master_frame_duplicate(
                        cunit->master_iff);
                if (cunit->linbadpix) {
                    self->linbadpix_cam2 = sph_zpl_exposure_duplicate(
                            cunit->linbadpix);
                }
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-2 is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    if (cunit) {
        _sph_zpl_intensity_flat_cunit_delete(cunit);
        cunit = NULL;
    }


    //reset an error log system to reduce data for the new style format data (indifferent to the cameras))
    SPH_RAISE_CPL_RESET;
    if (self->preproc_frames) {
        SPH_INFO_MSG("Create cunit structure for the new style format...");
        cunit = _sph_zpl_intensity_flat_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID);

        if (cunit) {
            recipe_error = _sph_zpl_intensity_flat_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction is failed for the newstyle pre-processed data!");
            } else {
                //put the output product in the main self structure
                self->intensity_flat_field = sph_quad_image_duplicate(
                        cunit->intensity_flat_field);
                self->master_iff = sph_master_frame_duplicate(
                        cunit->master_iff);
                if (cunit->linbadpix) {
                    self->linbadpix = sph_zpl_exposure_duplicate(
                            cunit->linbadpix);
                }
            }
        } else {
            SPH_ERR("Creation of the cunit structure newstyle pre-processed data is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the new style format -> nothing to reduce.");
    }

    if (cunit) {
        _sph_zpl_intensity_flat_cunit_delete(cunit);
        cunit = NULL;
    }



    if ( self->intensity_flat_field_cam1 == NULL
            && self->intensity_flat_field_cam2 == NULL
            && self->intensity_flat_field == NULL ) {
        SPH_ERR(" No quad image outputs have been created !!!");
    }

    if (self->master_iff_cam1 == NULL && self->master_iff_cam2 == NULL && self->master_iff == NULL) {
        SPH_ERR(" No master frame outputs have been created !!!");
    }

    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }

    SPH_INFO_MSG("sph_zpl_intensity_flat_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

sph_zpl_intensity_flat_cunit* _sph_zpl_intensity_flat_cunit_create(
        sph_zpl_intensity_flat* self, int camera_id) {
    sph_zpl_intensity_flat_cunit* result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_intensity_flat_cunit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_masters_dark" structure into the cunit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->keep_intermediate = self->keep_intermediate;
    result->robust_fit = self->robust_fit;
    result->collapse = self->collapse;
    result->coll_alg = self->coll_alg;
    result->clean_mean_reject_high = self->clean_mean_reject_high;
    result->clean_mean_reject_low = self->clean_mean_reject_low;
    result->sigma_clip = self->sigma_clip;
    result->badpix_lowtolerance = self->badpix_lowtolerance;
    result->badpix_uptolerance = self->badpix_uptolerance;
    result->badpix_chisqtolerance = self->badpix_chisqtolerance;
    result->quadimage_weight_mean = self->quadimage_weight_mean;

    result->rawframes = self->rawframes;
    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);
        result->badpix_filename = cpl_strdup(self->badpix_filename_cam1);
        result->intensity_flat_outfilename = cpl_strdup(
                self->intensity_flat_outfilename_cam1);
        result->master_intensity_flat_outfilename = cpl_strdup(
                self->master_intensity_flat_outfilename_cam1);
        result->eso_pro_catg = cpl_strdup(SPH_ZPL_TAG_IFF_CALIB_CAM1); //TAG for the quad image product
        result->eso_pro_catg_master = cpl_strdup(
                SPH_ZPL_TAG_IFFM_CALIB_CAM1); //TAG for the master frame product
        result->eso_pro_catg_nonlinbadmap = cpl_strdup(
                SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB_CAM1); //TAG for the nonlinear badpixel map
        result->eso_pro_catg_nonorm_master = cpl_strdup(
                SPH_ZPL_TAG_IFFM_NONORM_CALIB_CAM1);
        result->eso_pro_catg_nonorm = cpl_strdup(
                SPH_ZPL_TAG_IFF_NONORM_CALIB_CAM1);

        if (self->master_bias_frame_cam1) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam1);
        }
        if (self->master_dark_frame_cam1) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam1);
        }

    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);
        result->badpix_filename = cpl_strdup(self->badpix_filename_cam2);
        result->intensity_flat_outfilename = cpl_strdup(
                self->intensity_flat_outfilename_cam2);
        result->master_intensity_flat_outfilename = cpl_strdup(
                self->master_intensity_flat_outfilename_cam2);
        result->eso_pro_catg = cpl_strdup(SPH_ZPL_TAG_IFF_CALIB_CAM2);
        result->eso_pro_catg_master = cpl_strdup(
                SPH_ZPL_TAG_IFFM_CALIB_CAM2); //TAG for the master frame product
        result->eso_pro_catg_nonlinbadmap = cpl_strdup(
                SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB_CAM2); //TAG for the nonlinear badpixel map
        result->eso_pro_catg_nonorm_master = cpl_strdup(
                SPH_ZPL_TAG_IFFM_NONORM_CALIB_CAM2);
        result->eso_pro_catg_nonorm = cpl_strdup(
                SPH_ZPL_TAG_IFF_NONORM_CALIB_CAM2);

        if (self->master_bias_frame_cam2) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam2);
        }
        if (self->master_dark_frame_cam2) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam2);
        }
    } else if ( camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID ) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames);
        result->badpix_filename = cpl_strdup(self->badpix_filename);
        result->intensity_flat_outfilename = cpl_strdup(
                self->intensity_flat_outfilename);
        result->master_intensity_flat_outfilename = cpl_strdup(
                self->master_intensity_flat_outfilename);
        result->eso_pro_catg = cpl_strdup(SPH_ZPL_TAG_IFF_CALIB);
        result->eso_pro_catg_master = cpl_strdup(
                SPH_ZPL_TAG_IFFM_CALIB); //TAG for the master frame product
        result->eso_pro_catg_nonlinbadmap = cpl_strdup(
                SPH_ZPL_TAG_NON_LINEAR_BADPIXELMAP_CALIB); //TAG for the nonlinear badpixel map
        result->eso_pro_catg_nonorm_master = cpl_strdup(
                SPH_ZPL_TAG_IFFM_NONORM_CALIB);
        result->eso_pro_catg_nonorm = cpl_strdup(
                SPH_ZPL_TAG_IFF_NONORM_CALIB);

        if (self->master_bias_frame) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame);
        }
        if (self->master_dark_frame) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame);
        }
    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2 for oldstyle format, 0 for newstyle format)",
                camera_id);
        _sph_zpl_intensity_flat_cunit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d",
                sph_error_get_last_code());
        _sph_zpl_intensity_flat_cunit_delete(result);
        result = NULL;
    }

    return result;

}

sph_error_code _sph_zpl_intensity_flat_cunit_delete(
        sph_zpl_intensity_flat_cunit* cunit) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (cunit == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return rerr;
    }

    /* Code to delete recipe pointers GENERATED DO NOT EDIT */

    if (cunit->preproc_frames != NULL) {
        cpl_frameset_delete(cunit->preproc_frames);
        cunit->preproc_frames = NULL;
    }
    if (cunit->intensity_flat_field) {
        sph_quad_image_delete(cunit->intensity_flat_field);
        cunit->intensity_flat_field = NULL;
    }
    if (cunit->master_iff) {
        sph_master_frame_delete(cunit->master_iff);
    }
    if (cunit->badpix_filename)
        cpl_free(cunit->badpix_filename);
    if (cunit->intensity_flat_outfilename)
        cpl_free(cunit->intensity_flat_outfilename);
    if (cunit->master_intensity_flat_outfilename)
        cpl_free(cunit->master_intensity_flat_outfilename);
    if (cunit->eso_pro_catg)
        cpl_free(cunit->eso_pro_catg);
    if (cunit->eso_pro_catg_master)
        cpl_free(cunit->eso_pro_catg_master);
    if (cunit->eso_pro_catg_nonlinbadmap)
        cpl_free(cunit->eso_pro_catg_nonlinbadmap);
    if (cunit->eso_pro_catg_nonorm)
        cpl_free(cunit->eso_pro_catg_nonorm);
    if (cunit->eso_pro_catg_nonorm_master)
        cpl_free(cunit->eso_pro_catg_nonorm_master);
    if (cunit->preproc_frames)
        cpl_frameset_delete(cunit->preproc_frames);
    if (cunit->master_bias_frame)
        cpl_frame_delete(cunit->master_bias_frame);
    if (cunit->master_dark_frame)
        cpl_frame_delete(cunit->master_dark_frame);
    if (cunit->current_raw_frameset) {
        cpl_frameset_delete(cunit->current_raw_frameset);
        cunit->current_raw_frameset = NULL;
    }
    sph_polygon_free_all();
    cpl_free(cunit);
    return rerr;

}

//static sph_error_code
static void sph_zpl_intensity_flat_delete__(cpl_mask* zplexp_zero_odd_badpix,
        cpl_mask* zplexp_zero_even_badpix, cpl_mask* zplexp_pi_odd_badpix,
        cpl_mask* zplexp_pi_even_badpix, sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage, cpl_mask* curquadimage_mask,
        sph_quad_image* curquadimage) {
    //free memory of the local pointers
    if (zplexp_zero_odd_badpix) {
        cpl_mask_delete(zplexp_zero_odd_badpix);

    }
    if (zplexp_zero_even_badpix) {
        cpl_mask_delete(zplexp_zero_even_badpix);

    }
    if (zplexp_pi_odd_badpix) {
        cpl_mask_delete(zplexp_pi_odd_badpix);

    }
    if (zplexp_pi_even_badpix) {
        cpl_mask_delete(zplexp_pi_even_badpix);

    }
    if (master_bias_quadimage) {
        sph_quad_image_delete(master_bias_quadimage);

    }
    if (master_dark_quadimage) {
        sph_quad_image_delete(master_dark_quadimage);

    }
    if (curquadimage_mask) {
        cpl_mask_delete(curquadimage_mask);

    }
    if (curquadimage) {
        sph_quad_image_delete(curquadimage);

    }
    //SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_intensity_flat recipe
 (polarimetric mode). The error  code returned is always a cpl error code
 (to allow maximal compatibility with  esorex, gasgano, etc.) even if during
 recipe execution an error in the SPHERE API is the cause. In this case
 (and if the underlying error is not a cpl error)  the cpl error code is set
 to the cpl_error_code that matches the failure reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code _sph_zpl_intensity_flat_cunit_run(
        sph_zpl_intensity_flat_cunit* self) {
    sph_quad_image* master_bias_quadimage = NULL;
    sph_quad_image* master_dark_quadimage = NULL;
    sph_quad_image* curquadimage = NULL;

    sph_master_frame* mf_zero_odd = NULL;
    sph_master_frame* mf_zero_even = NULL;
    sph_master_frame* mf_pi_odd = NULL;
    sph_master_frame* mf_pi_even = NULL;

    //sph_zpl_exposure*	linbadpix				= NULL;
    cpl_image* zero_odd_linbadpix = NULL;
    cpl_image* zero_even_linbadpix = NULL;
    cpl_image* pi_odd_linbadpix = NULL;
    cpl_image* pi_even_linbadpix = NULL;

    cpl_mask* zplexp_zero_odd_badpix = NULL;
    cpl_mask* zplexp_zero_even_badpix = NULL;
    cpl_mask* zplexp_pi_odd_badpix = NULL;
    cpl_mask* zplexp_pi_even_badpix = NULL;
    cpl_mask* curquadimage_mask = NULL;

    cpl_frameset* ovsc_subtract_frames = NULL;
    cpl_frameset* rawframes_quadimage = NULL;
    //cpl_mask*           tmpmask 				= NULL;
    cpl_vector* lampflux_zero_odd = NULL;
    cpl_vector* lampflux_zero_even = NULL;
    cpl_vector* lampflux_pi_odd = NULL;
    cpl_vector* lampflux_pi_even = NULL;
    cpl_vector* lampcounts_zero_odd = NULL;
    cpl_vector* lampcounts_zero_even = NULL;
    cpl_vector* lampcounts_pi_odd = NULL;
    cpl_vector* lampcounts_pi_even = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pl_rawheader = NULL;

    char lampflux_keyname[256];
    int recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_intensity_flat_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (!self->preproc_frames) {
        SPH_ERR("no preproc frames are set up!")
        return sph_error_get_last_code();
    }

    if (self->robust_fit) {
        if (cpl_frameset_get_size(self->preproc_frames) < 5) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "The robust fitting is enabled. "
                    "For this to work, a minimum of 5 raw(preproc) frames needs "
                    "to be provided. Either provide more frames or "
                    "switch robust fitting off.");
            return CPL_ERROR_ILLEGAL_INPUT;
        }
    }

    if (self->static_badpixel_frame) {
        sph_zpl_exposure* zplexp_badpixmap = 
            sph_zpl_exposure_load(self->static_badpixel_frame, 0);

        if (zplexp_badpixmap) {
            zplexp_zero_odd_badpix = cpl_mask_threshold_image_create(
                    zplexp_badpixmap->image_zero_odd, 0.5, 10000);
            zplexp_zero_even_badpix = cpl_mask_threshold_image_create(
                    zplexp_badpixmap->image_zero_even, 0.5, 10000);
            zplexp_pi_odd_badpix = cpl_mask_threshold_image_create(
                    zplexp_badpixmap->image_pi_odd, 0.5, 10000);
            zplexp_pi_even_badpix = cpl_mask_threshold_image_create(
                    zplexp_badpixmap->image_pi_even, 0.5, 10000);
            sph_zpl_exposure_delete(zplexp_badpixmap);
        } else {
            SPH_ERROR_RAISE_WARNING(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "The static badpixel map could not be read from the given input file.");

            return CPL_ERROR_ILLEGAL_INPUT;

        }
    }

    if (self->collapse) {
        sph_quad_image* iff_quadimage = NULL;
        char outfilename[256];
        /********************************************************************
         * Simple combining of the frames and normalizing, no linear fitting
         ********************************************************************/
        SPH_INFO_MSG(
                "Simple combining of the frames and normalizing, no linear fitting...")

        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from pre-processed frames...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(
                    self->preproc_frames);
            if (ovsc_subtract_frames) {
                self->intensity_flat_field =
                        sph_framecombination_quad_image_from_frameset(
                                ovsc_subtract_frames, self->coll_alg,
                                self->framecomb_parameterlist);
                if (!self->keep_intermediate) {
                    sph_utils_frames_unlink(ovsc_subtract_frames);
                }
                cpl_frameset_delete(ovsc_subtract_frames);
                ovsc_subtract_frames = NULL;
            } else {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further.")
                cpl_error_reset();
                self->intensity_flat_field =
                        sph_framecombination_quad_image_from_frameset(
                                self->preproc_frames, self->coll_alg,
                                self->framecomb_parameterlist);
            }
        } else {
            self->intensity_flat_field =
                    sph_framecombination_quad_image_from_frameset(
                            self->preproc_frames, self->coll_alg,
                            self->framecomb_parameterlist);
        }

        if (!self->intensity_flat_field) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol quad intensity flat field:\n"
                            "-- sph_framecombination_quad_image_from_frameset returns null pointer.\n"
                            "cpl error code is: %d", cpl_error_get_code());

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return cpl_error_get_code();
        }

        //if master bias exists load and subtract it from the master dark
        if (self->master_bias_frame) {
            master_bias_quadimage = sph_quad_image_load(
                    cpl_frame_get_filename(self->master_bias_frame), 0);
            if (master_bias_quadimage) {
                recipe_error = sph_quad_image_subtract_quad_image(
                        self->intensity_flat_field, master_bias_quadimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                            __LINE__, SPH_ERROR_ERROR,
                            "Error is raised by subtracting master bias:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                            zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                            zplexp_pi_even_badpix, master_bias_quadimage,
                            master_dark_quadimage, curquadimage_mask,
                            curquadimage);
                    return cpl_error_get_code();
                }
            }
        }

        //if master dark exists load and subtract it from the master dark
        if (self->master_dark_frame) {
            master_dark_quadimage = sph_quad_image_load(
                    cpl_frame_get_filename(self->master_dark_frame), 0);
            if (master_dark_quadimage) {
                recipe_error = sph_zpl_subtract_dark_quad_image_scaled(
                        self->intensity_flat_field, master_dark_quadimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                            __LINE__, SPH_ERROR_ERROR,
                            "Error is raised by subtracting master dark:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                            zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                            zplexp_pi_even_badpix, master_bias_quadimage,
                            master_dark_quadimage, curquadimage_mask,
                            curquadimage);

                    return cpl_error_get_code();
                }
            }
        }

        if (zplexp_zero_odd_badpix && zplexp_zero_even_badpix
                && zplexp_pi_odd_badpix && zplexp_pi_even_badpix) {

            //combine a new created self->intensity_flat_field badmap with a given one (as a separate input zpl exp frame)
            recipe_error = sph_master_frame_set_bads_from_mask(
                    self->intensity_flat_field->zero_image->iframe,
                    zplexp_zero_odd_badpix);
            recipe_error |= sph_master_frame_set_bads_from_mask(
                    self->intensity_flat_field->zero_image->pframe,
                    zplexp_zero_even_badpix);
            recipe_error |= sph_master_frame_set_bads_from_mask(
                    self->intensity_flat_field->pi_image->iframe,
                    zplexp_pi_odd_badpix);
            recipe_error |= sph_master_frame_set_bads_from_mask(
                    self->intensity_flat_field->pi_image->pframe,
                    zplexp_pi_even_badpix);
            if (recipe_error) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                                "cpl error code is: %d", recipe_error);

                sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                        zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                        zplexp_pi_even_badpix, master_bias_quadimage,
                        master_dark_quadimage, curquadimage_mask, curquadimage);

                return recipe_error;
            }
        }

        //duplicate self->intensity_flat_field to keep it without badpixels and qc parameters
        iff_quadimage = sph_quad_image_duplicate(self->intensity_flat_field);

        //quality check (calculate statistical moments for the intensity flat)
        recipe_error = sph_quad_image_quality_check(iff_quadimage);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_quad_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            sph_quad_image_delete(iff_quadimage);
            return recipe_error;
        }

        //mask bad pixels based on the given sigma_clip parameter for the intensity non-normalized quad image
        recipe_error = sph_quad_image_mask_sigma(iff_quadimage,
                self->sigma_clip);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Could not create zimpol intensity flat:\n"
                            "sph_quad_image_mask_sigma returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            sph_quad_image_delete(iff_quadimage);
            return recipe_error;
        }

        //re-calculate quality check (statistical moments and # badpixels) to exclude badpixels from mean
        recipe_error = sph_quad_image_quality_check(iff_quadimage);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_quad_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            sph_quad_image_delete(iff_quadimage);
            return recipe_error;
        }

        /* Attention:
         * The following function includes a step to update SPH_COMMON_KEYWORD_SPH_TYPE by
         * SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE. It is necessary because cpl_dsf_save_image
         * updates the quad image property list by the property list from the first input frames.
         * In the case of some misbehavior exclude this step for checking.
         */

        if (self->keep_intermediate) {
            SPH_INFO_MSG("Save non-normalized output filename:");
            sprintf(outfilename, "%s-qc-non-normalized.fits",
                    basename(self->intensity_flat_outfilename));
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                    "outfilename = %s", outfilename)

            if (self->rawframes) {
                pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                        self->rawframes);
                cpl_propertylist_append(pl, iff_quadimage->properties);
                sph_zpl_qc_add_plane_keys(pl, self->rawframes, CPL_FALSE);
                sph_zpl_qc_add_odd_even_keys(pl, self->rawframes,
                                             iff_quadimage->properties);

                SPH_INFO_MSG(
                        "Save non-normalized iff quad image as a dfs product: header info taken from the first raw frame.")
                recipe_error = sph_quad_image_save_dfs(iff_quadimage,
                        outfilename, self->inframes,
                        cpl_frameset_get_first(self->rawframes), self->inparams,
                        self->eso_pro_catg_nonorm, //SPH_ZPL_TAG_IFF_NONORM_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
                if ( pl ) {
                    cpl_propertylist_delete(pl);
                    pl = NULL;
                }
            } else if (self->preproc_frames) {
                sph_zpl_qc_add_plane_keys(pl, self->rawframes, CPL_FALSE);
                sph_zpl_qc_add_odd_even_keys(pl, self->rawframes,
                                             iff_quadimage->properties);
                SPH_INFO_MSG(
                        "Save non-normalized iff quad image as a dfs product: header info taken from the first pre-processed frame.")
                recipe_error = sph_quad_image_save_dfs(iff_quadimage,
                        outfilename, self->inframes,
                        cpl_frameset_get_first(self->preproc_frames),
                        self->inparams, self->eso_pro_catg_nonorm, //SPH_ZPL_TAG_IFF_NONORM_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
            } else {
                SPH_ERR(
                        "Neither raw frames nor pre-processed frames provided!");
                recipe_error = sph_error_get_last_code();
            }
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol non-normalized quad image intensity flat field:\n"
                            "sph_quad_image_save_dfs returns error code.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            sph_quad_image_delete(iff_quadimage);
            return recipe_error;

        }

        //create master frame intensity flat product from quad image intensity flat

        if (self->quadimage_weight_mean) {
            //calculate final master frame product using weighted mean formula
            SPH_INFO_MSG(
                    "Combine quad image into single master frame using weighted mean.")
            self->master_iff =
                    sph_zpl_utils_calculate_master_frame_mean_weight_from_quad_image(
                            self->intensity_flat_field);
            if (!self->master_iff) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        " sph_zpl_utils_calculate_master_frame_mean_weight_from_quad_image returns NULL.\n"
                                "cpl error code is: %d", cpl_error_get_code());

                sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                        zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                        zplexp_pi_even_badpix, master_bias_quadimage,
                        master_dark_quadimage, curquadimage_mask, curquadimage);
                iff_quadimage = NULL;
                return cpl_error_get_code();
            }
        } else {
            //calculate final master frame product using standard mean formula
            SPH_INFO_MSG(
                    "Combine quad image into single master frame using standard mean.")
            self->master_iff = sph_master_frame_duplicate(
                    self->intensity_flat_field->zero_image->iframe);

            //next step is to exclude existing keywords from master frame, it is DIRTY for the moment!!!
            cpl_propertylist_empty(self->master_iff->qclist);
            cpl_propertylist_empty(self->master_iff->properties);

            sph_master_frame_add_master_frame(self->master_iff,
                    self->intensity_flat_field->zero_image->pframe);
            sph_master_frame_add_master_frame(self->master_iff,
                    self->intensity_flat_field->pi_image->iframe);
            sph_master_frame_add_master_frame(self->master_iff,
                    self->intensity_flat_field->pi_image->pframe);
            sph_master_frame_multiply_double(self->master_iff, 0.25);
        }
        //master frame: calculate quality check
        //quality check (calculate statistical moments for the master intensity flat)
        recipe_error = sph_master_frame_quality_check(self->master_iff);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_master_frame_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            sph_quad_image_delete(iff_quadimage);
            return recipe_error;
        }

        //master frame: mask bad pixels based on the given sigma_clip parameter for the intensity non-normalized master frame
        recipe_error = sph_master_frame_mask_sigma(self->master_iff,
                self->sigma_clip);
        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol intensity flat master frame product:\n"
                            "sph_master_frame_mask_sigma returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            sph_quad_image_delete(iff_quadimage);
            return recipe_error;
        }

        //master frame: re-calculate quality check to exclude bad pixels
        //quality check (calculate statistical moments for the master intensity flat)
        recipe_error = sph_master_frame_quality_check(self->master_iff);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_master_frame_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            sph_quad_image_delete(iff_quadimage);
            return recipe_error;
        }

        //master frame: calculate numder of badpixels and add it to the QC keywords of the master intensity flat field product
        cpl_propertylist_update_long(self->master_iff->qclist,
                SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
                (long) cpl_image_get_flux(self->master_iff->badpixelmap));

        if (self->keep_intermediate) {
            SPH_INFO_MSG("Save non-normalized master frame output filename:");
            sprintf(outfilename, "%s-qc-non-normalized.fits",
                    basename(self->master_intensity_flat_outfilename));
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                    "outfilename = %s", outfilename)

            if (self->rawframes) {
                pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                        self->rawframes);
                cpl_propertylist_append(pl, self->master_iff->properties);

                SPH_INFO_MSG(
                        "Save non-normalized iff master frame as a dfs product: header info taken from the first raw frame.")
                recipe_error = sph_master_frame_save_dfs(self->master_iff,
                        outfilename, self->inframes,
                        cpl_frameset_get_first(self->rawframes), self->inparams,
                        self->eso_pro_catg_nonorm_master, //SPH_ZPL_TAG_IFFM_NONORM_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
                if ( pl ) {
                    cpl_propertylist_delete(pl);
                    pl = NULL;
                }
            } else if (self->preproc_frames) {
                SPH_INFO_MSG(
                        "Save non-normalized iff master frame as a dfs product: header info taken from the first pre-processed frame.")
                recipe_error = sph_master_frame_save_dfs(self->master_iff,
                        outfilename, self->inframes,
                        cpl_frameset_get_first(self->preproc_frames),
                        self->inparams, self->eso_pro_catg_nonorm_master, //SPH_ZPL_TAG_IFFM_NONORM_CALIB,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
            } else {
                SPH_ERR(
                        "Neither raw frames nor pre-processed frames provided!");
                recipe_error = sph_error_get_last_code();
            }
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol non-normalized master intensity flat field:\n"
                            "sph_master_frame_save_dfs returns error code.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;

        }

        //master frame: normalize intensity master frame
        recipe_error = sph_master_frame_divide_double(
                self->master_iff,
                cpl_propertylist_get_double(self->master_iff->qclist,
                        SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME));

        //master frame: re-calculate quality check to exclude bad pixels
        //quality check (calculate statistical moments for the master intensity flat)
        recipe_error = sph_master_frame_quality_check(self->master_iff);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_master_frame_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;
        }

        //master frame: save normalized intensity master frame
        SPH_INFO_MSG("Save master normalized intensity flat product.");
        SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                "in outfilename = %s", self->master_intensity_flat_outfilename);

        if (self->rawframes) {
            pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->rawframes);
            cpl_propertylist_append(pl, self->master_iff->properties);

            SPH_INFO_MSG(
                    "Save normalized master frame as a dfs product: header info taken from the first raw frame.")
            recipe_error = sph_master_frame_save_dfs(self->master_iff,
                    self->master_intensity_flat_outfilename, self->inframes,
                    cpl_frameset_get_first(self->rawframes), self->inparams,
                    self->eso_pro_catg_master, // SPH_ZPL_TAG_IFFM_CALIB,
                    SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
            if ( pl ) {
                cpl_propertylist_delete(pl);  pl = NULL;
            }
        } else if (self->preproc_frames) {
            SPH_INFO_MSG(
                    "Save normalized iff master frame as a dfs product: header info taken from the first pre-processed frame.")
            recipe_error = sph_master_frame_save_dfs(self->master_iff,
                    self->master_intensity_flat_outfilename, self->inframes,
                    cpl_frameset_get_first(self->preproc_frames),
                    self->inparams, self->eso_pro_catg_master, // SPH_ZPL_TAG_IFFM_CALIB,
                    SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol normalized master intensity flat field:\n"
                            "sph_master_frame_save_dfs returns error code.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            sph_quad_image_delete(iff_quadimage);
            return recipe_error;

        }

        //create and save normalized quad image intensity flat field product
        //normalize each sub-frame individually (i.e. divided by corresponding sub-frame mean )
        recipe_error = sph_master_frame_divide_double(
                self->intensity_flat_field->zero_image->iframe,
                cpl_propertylist_get_double(iff_quadimage->qclist,
                        SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD));
        recipe_error |= sph_master_frame_divide_double(
                self->intensity_flat_field->zero_image->pframe,
                cpl_propertylist_get_double(iff_quadimage->qclist,
                        SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN));
        recipe_error |= sph_master_frame_divide_double(
                self->intensity_flat_field->pi_image->iframe,
                cpl_propertylist_get_double(iff_quadimage->qclist,
                        SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD));
        recipe_error |= sph_master_frame_divide_double(
                self->intensity_flat_field->pi_image->pframe,
                cpl_propertylist_get_double(iff_quadimage->qclist,
                        SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN));

        sph_quad_image_delete(iff_quadimage);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "normalization is failed.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            return recipe_error;
        }

        //quad image: normalized quality check (calculate statistical moments for the intensity flat)
        recipe_error = sph_quad_image_quality_check(self->intensity_flat_field);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_quad_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;
        }

        //quad image: mask
        //mask hot/bad pixels based on the given sigma_clip parameter
        recipe_error = sph_quad_image_mask_sigma(self->intensity_flat_field,
                self->sigma_clip);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Could not create zimpol intensity flat:\n"
                            "sph_quad_image_mask_sigma returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;
        }

        //quad image: re-calculate normalized quality check to exclude badpixels
        recipe_error = sph_quad_image_quality_check(self->intensity_flat_field);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_quad_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;
        }

        //quad image: save normalized
        SPH_INFO_MSG("Save quadimage normalized intensity flat product.");
        SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                "in outfilename = %s", self->intensity_flat_outfilename);

        if (self->rawframes) {
            pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->rawframes);
            cpl_propertylist_append(pl, self->intensity_flat_field->properties);
            sph_zpl_qc_add_plane_keys(pl, self->rawframes, CPL_FALSE);
            sph_zpl_qc_add_odd_even_keys(pl, self->rawframes,
                                         self->intensity_flat_field->properties);
            SPH_INFO_MSG(
                    "Save normalized iff quad image as a dfs product: header info taken from the first raw frame.")
            recipe_error = sph_quad_image_save_dfs(self->intensity_flat_field,
                    self->intensity_flat_outfilename, self->inframes,
                    cpl_frameset_get_first(self->rawframes), self->inparams,
                    self->eso_pro_catg, //SPH_ZPL_TAG_IFF_CALIB,
                    SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
            if ( pl ){
                cpl_propertylist_delete(pl);
                pl = NULL;
            }
        } else if (self->preproc_frames) {
            sph_zpl_qc_add_plane_keys(pl, self->preproc_frames, CPL_FALSE);
            sph_zpl_qc_add_odd_even_keys(pl, self->preproc_frames,
                                         self->intensity_flat_field->properties);
            SPH_INFO_MSG(
                    "Save normalized iff quad image as a dfs product: header info taken from the first pre-processed frame.")
            recipe_error = sph_quad_image_save_dfs(self->intensity_flat_field,
                    self->intensity_flat_outfilename, self->inframes,
                    cpl_frameset_get_first(self->preproc_frames),
                    self->inparams, self->eso_pro_catg, //SPH_ZPL_TAG_IFF_CALIB,
                    SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        cpl_propertylist_delete(pl);
        pl = NULL;

        if (master_bias_quadimage) {
            sph_quad_image_delete(master_bias_quadimage);
            master_bias_quadimage = NULL;
        }
        if (master_dark_quadimage) {
            sph_quad_image_delete(master_dark_quadimage);
            master_dark_quadimage = NULL;
        }

        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not create zimpol quad intensity flat field:\n"
                            "sph_quad_image_save_dfs returns error code.\n"
                            "cpl error code is: %d", recipe_error);
        }

        sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                zplexp_pi_even_badpix, master_bias_quadimage,
                master_dark_quadimage, curquadimage_mask, curquadimage);

        SPH_INFO_MSG("sph_zpl_intensity_flat_run...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    /************************************************************************
     * Linear fitting method for the calculation of the intensity flat field
     ************************************************************************/
    SPH_INFO_MSG("Linear fitting method...")
    if (self->subtract_overscan) {
        SPH_INFO_MSG("Subtract overscans from pre-processed frames ...")
        ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(
                self->preproc_frames);
        if (ovsc_subtract_frames) {
            rawframes_quadimage =
                    sph_quad_image_create_quad_image_frameset_from_zplexp_cubes(
                            ovsc_subtract_frames);
            if (!self->keep_intermediate) {
                sph_utils_delete_files(ovsc_subtract_frames);
            }
            cpl_frameset_delete(ovsc_subtract_frames);
        } else {
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_WARNING,
                    "Overscans could not be subtracted, trying to proceed further.")
            rawframes_quadimage =
                    sph_quad_image_create_quad_image_frameset_from_zplexp_cubes(
                            self->preproc_frames);

        }
    } else {
        rawframes_quadimage =
                sph_quad_image_create_quad_image_frameset_from_zplexp_cubes(
                        self->preproc_frames);
    }

    if (!rawframes_quadimage) {
        SPH_ERR(
                "Couldn't create quadimage frame from the input preproc frames.");
        sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                zplexp_pi_even_badpix, master_bias_quadimage,
                master_dark_quadimage, curquadimage_mask, curquadimage);

        return sph_error_get_last_code();
    }
    //if master bias exists load it
    if (self->master_bias_frame) {
        master_bias_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_bias_frame), 0);
    }
    //if master dark exist load it
    if (self->master_dark_frame) {
        master_dark_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_dark_frame), 0);
    }

    if (master_bias_quadimage || master_dark_quadimage) {
        const cpl_frame* curframe =
            cpl_frameset_get_first_const(rawframes_quadimage);

        while (curframe) {

            curquadimage = sph_quad_image_load(cpl_frame_get_filename(curframe),
                    0);
            if (!curquadimage) {
                SPH_ERR("couldn't load current quad image from the file")

                sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                        zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                        zplexp_pi_even_badpix, master_bias_quadimage,
                        master_dark_quadimage, curquadimage_mask, curquadimage);

                return sph_error_get_last_code();
            }
            if (master_bias_quadimage) {
                recipe_error = sph_quad_image_subtract_quad_image(curquadimage,
                        master_bias_quadimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                            __LINE__, SPH_ERROR_ERROR,
                            "Error is raised by subtracting master bias:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                            zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                            zplexp_pi_even_badpix, master_bias_quadimage,
                            master_dark_quadimage, curquadimage_mask,
                            curquadimage);

                    return cpl_error_get_code();
                }
            }
            if (master_dark_quadimage) {
                recipe_error = sph_zpl_subtract_dark_quad_image_scaled(curquadimage,
                        master_dark_quadimage);
                if (recipe_error != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                            __LINE__, SPH_ERROR_ERROR,
                            "Error is raised by subtracting dark:\n"
                                    "cpl error code is: %d",
                            cpl_error_get_code());

                    sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                            zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                            zplexp_pi_even_badpix, master_bias_quadimage,
                            master_dark_quadimage, curquadimage_mask,
                            curquadimage);

                    return cpl_error_get_code();
                }
            }

            sph_quad_image_save(curquadimage, cpl_frame_get_filename(curframe),
                    NULL);
            sph_quad_image_delete(curquadimage);
            curframe = cpl_frameset_get_next_const(rawframes_quadimage);

        } //end while
    } //enf if-block-with dark and bias

    for (int i = 0; i < 4; i++) {
        sph_master_frame* mf_tmp =
            sph_framecombination_master_frame_from_frameset
            (rawframes_quadimage, self->coll_alg,
             self->framecomb_parameterlist, i);
        double rms = 0.0;
        const double mean = sph_master_frame_get_mean(mf_tmp, &rms);

        if (mean < 1000.0) {
            SPH_ERROR_RAISE_WARNING(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "The mean of a flat is %f. "
                    "Using flats with mean counts less than 1000 is dangerous."
                    "The recipe will proceed but be aware that the master"
                    " flat created will most likely not be of high quality.", mean);
        }
        //sph_master_frame_mask_tolerance(mf_tmp, mean*0.1, FLT_MAX);
        //tmpmask = sph_master_frame_get_badpixelmask(mf_tmp);
        sph_master_frame_delete(mf_tmp);
    }

    //load quad image from the first rawframes_quadimage in order to get badpixelmaps and then create badpixel masks
    //it is supposed that the statical bad pixel map is the same for all rawframes_quadimage, so we retrieve it only from
    // from the first frame
    curquadimage = sph_quad_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(rawframes_quadimage)),
            0);

    //create mask from zero_image->iframe->badpixelmap and combine it (if exists) with
    //zplexp_zero_odd_badpix given as an input badpixmap
    curquadimage_mask = cpl_mask_threshold_image_create(
            curquadimage->zero_image->iframe->badpixelmap, 0.5, 10000);
    if (zplexp_zero_odd_badpix) {
        recipe_error = cpl_mask_or(curquadimage_mask, zplexp_zero_odd_badpix);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;
        }
    }
    mf_zero_odd = sph_create_flat(rawframes_quadimage, self->robust_fit,
            curquadimage_mask, NULL, &zero_odd_linbadpix,
            self->badpix_lowtolerance, self->badpix_uptolerance,
            self->badpix_chisqtolerance, &lampflux_zero_odd, &lampcounts_zero_odd, 0, NULL);

    //create mask from zero_image->pframe->badpixelmap and combine it (if exists) with
    //zplexp_zero_even_badpix given as an input badpixmap
    cpl_mask_delete(curquadimage_mask);
    curquadimage_mask = cpl_mask_threshold_image_create(
            curquadimage->zero_image->pframe->badpixelmap, 0.5, 10000);
    if (zplexp_zero_even_badpix) {
        recipe_error = cpl_mask_or(curquadimage_mask, zplexp_zero_even_badpix);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;
        }
    }
    mf_zero_even = sph_create_flat(rawframes_quadimage, self->robust_fit,
            curquadimage_mask, NULL, &zero_even_linbadpix,
            self->badpix_lowtolerance, self->badpix_uptolerance,
            self->badpix_chisqtolerance, &lampflux_zero_even, &lampcounts_zero_even, 4, NULL);

    //create mask from pi_image->iframe->badpixelmap and combine it (if exists) with
    //zplexp_pi_odd_badpix given as an input badpixmap
    cpl_mask_delete(curquadimage_mask);
    curquadimage_mask = cpl_mask_threshold_image_create(
            curquadimage->pi_image->iframe->badpixelmap, 0.5, 10000);
    if (zplexp_pi_odd_badpix) {
        recipe_error = cpl_mask_or(curquadimage_mask, zplexp_pi_odd_badpix);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);

            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);

            return recipe_error;
        }
    }
    mf_pi_odd = sph_create_flat(rawframes_quadimage, self->robust_fit,
            curquadimage_mask, NULL, &pi_odd_linbadpix,
            self->badpix_lowtolerance, self->badpix_uptolerance,
            self->badpix_chisqtolerance, &lampflux_pi_odd, &lampcounts_pi_odd, 8, NULL);

    //create mask from pi_image->pframe->badpixelmap and combine it (if exists) with
    //zplexp_pi_even_badpix given as an input badpixmap
    cpl_mask_delete(curquadimage_mask);
    curquadimage_mask = cpl_mask_threshold_image_create(
            curquadimage->pi_image->pframe->badpixelmap, 0.5, 10000);
    if (zplexp_pi_even_badpix) {
        recipe_error = cpl_mask_or(curquadimage_mask, zplexp_pi_even_badpix);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "cpl_mask_or returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                    zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                    zplexp_pi_even_badpix, master_bias_quadimage,
                    master_dark_quadimage, curquadimage_mask, curquadimage);
            return recipe_error;
        }
    }
    mf_pi_even = sph_create_flat(rawframes_quadimage, self->robust_fit,
            curquadimage_mask, NULL, &pi_even_linbadpix,
            self->badpix_lowtolerance, self->badpix_uptolerance,
            self->badpix_chisqtolerance, &lampflux_pi_even, &lampcounts_pi_even, 12, NULL);

    if (mf_zero_odd && mf_zero_even && mf_pi_odd && mf_pi_even) {
        self->intensity_flat_field = sph_quad_image_new_from_master_frames(
                mf_zero_odd, mf_zero_even, mf_pi_odd, mf_pi_even);
    } else {
        SPH_ERR(
                "Couldn't create intensity flat field quad image product, NULL pointers.")
        sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                zplexp_pi_even_badpix, master_bias_quadimage,
                master_dark_quadimage, curquadimage_mask, curquadimage);

        return (int) cpl_error_get_code();
    }

    if (!self->intensity_flat_field) {
        SPH_ERR("intensity flat field product has NULL pointer")
        sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                zplexp_pi_even_badpix, master_bias_quadimage,
                master_dark_quadimage, curquadimage_mask, curquadimage);

        return (int) cpl_error_get_code();
    }

    //add some QC keywords to the pl of the intensity flat field product
    pl = cpl_propertylist_new();
    //bad pixels
    cpl_propertylist_update_long(
            pl,
            SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD,
            (long) cpl_image_get_flux(
                    self->intensity_flat_field->zero_image->iframe->badpixelmap));
    cpl_propertylist_update_long(
            pl,
            SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN,
            (long) cpl_image_get_flux(
                    self->intensity_flat_field->zero_image->pframe->badpixelmap));
    cpl_propertylist_update_long(
            pl,
            SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD,
            (long) cpl_image_get_flux(
                    self->intensity_flat_field->pi_image->iframe->badpixelmap));
    cpl_propertylist_update_long(
            pl,
            SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN,
            (long) cpl_image_get_flux(
                    self->intensity_flat_field->pi_image->pframe->badpixelmap));

    //mean value of each frame: zero_odd
    for (int ii = 0; ii < cpl_vector_get_size(lampflux_zero_odd); ++ii) {
        sprintf(lampflux_keyname, "%s ZERO ODD %d",
                SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX, ii);
        cpl_propertylist_append_double(pl, lampflux_keyname,
                cpl_vector_get(lampflux_zero_odd, ii));
    }

    //mean value of each frame: zero_even
    for (int ii = 0; ii < cpl_vector_get_size(lampflux_zero_even); ++ii) {
        sprintf(lampflux_keyname, "%s ZERO EVEN %d",
                SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX, ii);
        cpl_propertylist_append_double(pl, lampflux_keyname,
                cpl_vector_get(lampflux_zero_even, ii));
    }

    //mean value of each frame: pi_odd
    for (int ii = 0; ii < cpl_vector_get_size(lampflux_pi_odd); ++ii) {
        sprintf(lampflux_keyname, "%s PI ODD %d",
                SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX, ii);
        cpl_propertylist_append_double(pl, lampflux_keyname,
                cpl_vector_get(lampflux_pi_odd, ii));
    }

    //mean value of each frame: zero_even
    for (int ii = 0; ii < cpl_vector_get_size(lampflux_pi_even); ++ii) {
        sprintf(lampflux_keyname, "%s PI EVEN %d",
                SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX, ii);
        cpl_propertylist_append_double(pl, lampflux_keyname,
                cpl_vector_get(lampflux_pi_even, ii));
    }

    //quality check (calculate statistical moments for the intensity flat)
    recipe_error = sph_quad_image_quality_check(self->intensity_flat_field);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "sph_quad_image_quality_check returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);
        sph_zpl_intensity_flat_delete__(zplexp_zero_odd_badpix,
                zplexp_zero_even_badpix, zplexp_pi_odd_badpix,
                zplexp_pi_even_badpix, master_bias_quadimage,
                master_dark_quadimage, curquadimage_mask, curquadimage);
        return recipe_error;
    }

    /* Attention:
     * The following function includes a step to update SPH_COMMON_KEYWORD_SPH_TYPE by
     * SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE. It is necessary because cpl_dsf_save_image
     * updates the quad image property list by the property list from the first input frames.
     * In the case of some misbehavior exclude this step for checking.
     */

    if (self->rawframes) {
        pl_rawheader = sph_zpl_utils_get_camera_header(self->preproc_frames,
                self->rawframes);
        cpl_propertylist_append(pl, pl_rawheader);
        cpl_propertylist_append(pl, self->intensity_flat_field->properties);
        sph_zpl_qc_add_plane_keys(pl, self->rawframes, CPL_FALSE);
        sph_zpl_qc_add_odd_even_keys(pl, self->rawframes,
                                self->intensity_flat_field->properties);
        SPH_INFO_MSG(
                "Save iff quad image (linear fitting) as a dfs product: header info taken from the first raw frame.")
        recipe_error = sph_quad_image_save_dfs(self->intensity_flat_field,
                self->intensity_flat_outfilename, self->inframes,
                cpl_frameset_get_first(self->rawframes), self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_IFF_CALIB,
                SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
        if (pl_rawheader) {
            cpl_propertylist_delete(pl_rawheader);
            pl_rawheader = NULL;
        }
    } else if (self->preproc_frames) {
        sph_zpl_qc_add_plane_keys(pl, self->preproc_frames, CPL_FALSE);
        sph_zpl_qc_add_odd_even_keys(pl, self->preproc_frames,
                                self->intensity_flat_field->properties);
        SPH_INFO_MSG(
                "Save iff quad image (linear fitting) as a dfs product: header info taken from the first pre-processed frame.")
        recipe_error = sph_quad_image_save_dfs(self->intensity_flat_field,
                self->intensity_flat_outfilename, self->inframes,
                cpl_frameset_get_first(self->preproc_frames), self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_IFF_CALIB,
                SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL, pl);
    } else {
        SPH_ERR("Neither raw frames nor pre-processed frames provided!");
        recipe_error = sph_error_get_last_code();
    }
    cpl_propertylist_delete(pl);

    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not create zimpol quad intensity flat field:\n"
                        "sph_quad_image_save_dfs returns error code.\n"
                        "cpl error code is: %d", recipe_error);
    }

    if (zero_odd_linbadpix && zero_even_linbadpix && pi_odd_linbadpix
            && pi_even_linbadpix) {
        cpl_propertylist* pl_linbadpix = NULL;
        self->linbadpix = sph_zpl_exposure_new_from_cplimages(
                zero_odd_linbadpix, zero_even_linbadpix, pi_odd_linbadpix,
                pi_even_linbadpix);
        cpl_image_delete(zero_odd_linbadpix);
        cpl_image_delete(zero_even_linbadpix);
        cpl_image_delete(pi_odd_linbadpix);
        cpl_image_delete(pi_even_linbadpix);

        pl_linbadpix = cpl_propertylist_new();

        cpl_propertylist_update_long(pl_linbadpix,
                SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_ZPLEXP_ZERO_ODD,
                (long) cpl_image_get_flux(self->linbadpix->image_zero_odd));
        cpl_propertylist_update_long(pl_linbadpix,
                SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_ZPLEXP_ZERO_EVEN,
                (long) cpl_image_get_flux(self->linbadpix->image_zero_even));
        cpl_propertylist_update_long(pl_linbadpix,
                SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_ZPLEXP_PI_ODD,
                (long) cpl_image_get_flux(self->linbadpix->image_pi_odd));
        cpl_propertylist_update_long(pl_linbadpix,
                SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_ZPLEXP_PI_EVEN,
                (long) cpl_image_get_flux(self->linbadpix->image_pi_even));
        cpl_propertylist_append_string(pl_linbadpix,
                SPH_COMMON_KEYWORD_PRO_CATG, self->eso_pro_catg_nonlinbadmap);

        //sph_zpl_exposure_save( linbadpix, self->badpix_filename, pl_linbadpix);

        if (self->keep_intermediate) {

            if (self->rawframes) {
                SPH_INFO_MSG(
                        "Save zpl exp linbadpix map as a dfs product: header info taken from the first raw frame.");
                pl_rawheader = sph_zpl_utils_get_camera_header(self->preproc_frames,
                        self->rawframes);
                cpl_propertylist_append(pl_linbadpix, pl_rawheader);

                sph_zpl_exposure_save_dfs(self->linbadpix,
                        self->badpix_filename, self->inframes,
                        cpl_frameset_get_first(self->rawframes), self->inparams,
                        self->eso_pro_catg_nonlinbadmap,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL,
                        pl_linbadpix);
                if( pl_rawheader ) {
                    cpl_propertylist_delete( pl_rawheader ); pl_rawheader = NULL;
                }
            } else if (self->preproc_frames) {
                SPH_INFO_MSG(
                        "Save zpl exp linbadpix map as a dfs product: header info taken from the first pre-processed frame.")
                sph_zpl_exposure_save_dfs(self->linbadpix,
                        self->badpix_filename, self->inframes,
                        cpl_frameset_get_first(self->preproc_frames),
                        self->inparams, self->eso_pro_catg_nonlinbadmap,
                        SPH_RECIPE_NAME_ZPL_IFF, SPH_PIPELINE_NAME_ZIMPOL,
                        pl_linbadpix);
            } else {
                SPH_ERR(
                        "Neither raw frames nor pre-processed frames provided!");
            }
            cpl_propertylist_delete(pl_linbadpix);

        } else {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't create self->linbadpix map of the zpl exp type");
        }
    }

    cpl_mask_delete(zplexp_zero_odd_badpix);
    cpl_mask_delete(zplexp_zero_even_badpix);
    cpl_mask_delete(zplexp_pi_odd_badpix);
    cpl_mask_delete(zplexp_pi_even_badpix);
    sph_quad_image_delete(master_bias_quadimage);
    sph_quad_image_delete(master_dark_quadimage);
    cpl_mask_delete(curquadimage_mask);
    sph_quad_image_delete(curquadimage);
    cpl_vector_delete(lampflux_pi_even);
    cpl_vector_delete(lampflux_pi_odd);
    cpl_vector_delete(lampflux_zero_even);
    cpl_vector_delete(lampflux_zero_odd);
    cpl_vector_delete(lampcounts_pi_even);
    cpl_vector_delete(lampcounts_pi_odd);
    cpl_vector_delete(lampcounts_zero_even);
    cpl_vector_delete(lampcounts_zero_odd);

    SPH_INFO_MSG("sph_zpl_intensity_flat_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}
/**@}*/
