/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <cpl.h>
#include <math.h>
#include "sph_zpl_master_bias.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_exposure.h"
#include "sph_zpl_utils.h"
#include "sph_quad_image.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_framecombination.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"

/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_master_bias_cunit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */
typedef struct _sph_zpl_master_bias_camera_unit_ {
    cpl_frameset* inframes; /* The recipe input frames */

    cpl_frameset* current_raw_frameset; /* The frameset for one product */

    cpl_parameterlist* inparams; /* The recipe input parameters */

    cpl_parameterlist* framecomb_parameterlist; /* The recipe input parameters */

    char* master_bias_outfilename; /* The parameter of zpl.master_bias.outfilename */

    short subtract_overscan; /* The parameter of zpl.master_bias.subtract_overscan */

    int coll_alg; /* The parameter of zpl.master_bias.coll_alg */

    int clean_mean_reject_high; /* The parameter of zpl.master_bias.coll_alg.clean_mean.reject_high */

    int clean_mean_reject_low; /* The parameter of zpl.master_bias.coll_alg.clean_mean.reject_low */

    short keep_intermediate; /* The parameter of zpl.master_bias.keep_intermediate */

    double clean_mean_sigma; /* The parameter of zpl.master_bias.clean_mean.sigma */

    double sigma_clip; /* The parameter of zpl.master_bias.sigma_clip */

    cpl_frameset* rawframes; /* The ZPL_BIAS_RAW frames */

    cpl_frameset* preproc_frames; /* The ZPL_BIAS_PREPROC_CAM1 frames */

    sph_quad_image* master_bias_quadimage; /* The ZPL_MASTER_BIAS_CAM1 or _2 product */

    char* eso_pro_catg; /*Here the ZPL_MASTER_BIAS_CAM1 or _2 will be written */

} sph_zpl_master_bias_cunit;

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_master_bias_cunit* _sph_zpl_master_bias_cunit_create(
        sph_zpl_master_bias* self, int camera);
static sph_error_code _sph_zpl_master_bias_cunit_delete(
        sph_zpl_master_bias_cunit* cunit);
static cpl_error_code _sph_zpl_master_bias_cunit_run(
        sph_zpl_master_bias_cunit* cunit);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_master_bias_run Create Master Bias Recipe
 *
 * This module provides the algorithm implementation for the creation of the
 * master bias
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_master_bias.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_master_bias recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_zpl_master_bias_run(sph_zpl_master_bias* self) {
    sph_zpl_master_bias_cunit* cunit = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_master_bias_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_master_dark_preproc( self );
            recipe_error = sph_zpl_common_preproc_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }


        if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
            SPH_INFO_MSG("New style format of the raw data...")
            if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                        "The products will be created individually for each camera (off-line pipeline extended support!")
            } else {
                SPH_ERR("Pre-processing step is failed.");
                cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                return (int) recipe_error;
            }
        } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
            SPH_INFO_MSG("Old style format of the raw data...")
        } else {
            SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                    "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }
    }

    //reduce data for the camera-1
    //create cunit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create cunit structure for the camera-1...");
        cunit = _sph_zpl_master_bias_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

        if (cunit) {

            recipe_error = _sph_zpl_master_bias_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction for the camera-1 is failed!");
            } else {
                //put the output product in the main self structure
                self->master_bias_quadimage_cam1 = sph_quad_image_duplicate(
                        cunit->master_bias_quadimage);
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-1 is failed...trying for the camera-2")
        }
    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }
    if (cunit) {
        _sph_zpl_master_bias_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data for camera-2
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create cunit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create cunit structure for the camera-2...");
        cunit = _sph_zpl_master_bias_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);

        if (cunit) {
            recipe_error = _sph_zpl_master_bias_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("redunction for the camera-2 is failed!");
            } else {
                //put the output product in the main self structure
                self->master_bias_quadimage_cam2 = sph_quad_image_duplicate(
                        cunit->master_bias_quadimage);
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-2 is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    if (cunit) {
        _sph_zpl_master_bias_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data for the new style format data (indifferent to the cameras))
    SPH_RAISE_CPL_RESET;

    //reduce data for the new style data (on-line pipeline)
    //create cunit structure
    if (self->preproc_frames) {
        SPH_INFO_MSG("Create cunit structure for the new style format...");
        cunit = _sph_zpl_master_bias_cunit_create(self, SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID);

        if (cunit) {
            recipe_error = _sph_zpl_master_bias_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction is failed for the newstyle pre-processed data!");
            } else {
                //put the output product in the main self structure
                self->master_bias_quadimage = sph_quad_image_duplicate(
                        cunit->master_bias_quadimage);
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure newstyle pre-processed data is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the new style format -> nothing to reduce.");
    }

    if (cunit) {
        _sph_zpl_master_bias_cunit_delete(cunit);
        cunit = NULL;
    }


    if (self->master_bias_quadimage_cam1 == NULL
            && self->master_bias_quadimage_cam2 == NULL && self->master_bias_quadimage == NULL) {
        SPH_ERR(" No outputs have been created !!!");
    }

    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }


    SPH_INFO_MSG("sph_zpl_master_bias_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

}

sph_zpl_master_bias_cunit* _sph_zpl_master_bias_cunit_create(
        sph_zpl_master_bias* self, int camera_id) {
    sph_zpl_master_bias_cunit* result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_master_bias_cunit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_masters_bias" structure into the cunit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->coll_alg = self->coll_alg;
    result->clean_mean_reject_high = self->clean_mean_reject_high;
    result->clean_mean_reject_low = self->clean_mean_reject_low;
    result->clean_mean_sigma = self->clean_mean_sigma;
    result->sigma_clip = self->sigma_clip;
    result->keep_intermediate = self->keep_intermediate;

    result->rawframes = self->rawframes;
    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);
        result->master_bias_outfilename = cpl_strdup(
                self->master_bias_outfilename_cam1);
        result->eso_pro_catg = cpl_strdup(SPH_ZPL_TAG_BIAS_CALIB_CAM1);
    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);
        result->master_bias_outfilename = cpl_strdup(
                self->master_bias_outfilename_cam2);
        result->eso_pro_catg = cpl_strdup(SPH_ZPL_TAG_BIAS_CALIB_CAM2);
    } else if ( camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID ) {
        result->preproc_frames = cpl_frameset_duplicate( self->preproc_frames );
        result->master_bias_outfilename = cpl_strdup( self->master_bias_outfilename );
        result->eso_pro_catg = cpl_strdup( SPH_ZPL_TAG_BIAS_CALIB );
    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2 for oldstyle format, 0 for newstyle format)",
                camera_id);
        _sph_zpl_master_bias_cunit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d",
                sph_error_get_last_code());
        _sph_zpl_master_bias_cunit_delete(result);
        result = NULL;
    }

    return result;

}

sph_error_code _sph_zpl_master_bias_cunit_delete(
        sph_zpl_master_bias_cunit* cunit) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (cunit == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return rerr;
    }

    /* Code to delete recipe pointers GENERATED DO NOT EDIT */

    if (cunit->preproc_frames != NULL) {
        cpl_frameset_delete(cunit->preproc_frames);
        cunit->preproc_frames = NULL;
    }

    if (cunit->master_bias_quadimage) {
        sph_quad_image_delete(cunit->master_bias_quadimage);
        cunit->master_bias_quadimage = NULL;
    }
    if (cunit->master_bias_outfilename)
        cpl_free(cunit->master_bias_outfilename);
    if (cunit->eso_pro_catg)
        cpl_free(cunit->eso_pro_catg);
    if (cunit->preproc_frames)
        cpl_frameset_delete(cunit->preproc_frames);
    if (cunit->current_raw_frameset) {
        cpl_frameset_delete(cunit->current_raw_frameset);
        cunit->current_raw_frameset = NULL;
    }
    sph_polygon_free_all();
    cpl_free(cunit);
    return rerr;

}

cpl_error_code _sph_zpl_master_bias_cunit_run(sph_zpl_master_bias_cunit* self) {
    int                 recipe_error = CPL_ERROR_NONE;
    cpl_frameset*       ovsc_subtract_frames = NULL;
    cpl_propertylist*   pl = NULL;

    SPH_INFO_MSG("Starting sph_zpl_master_bias_run_cunit...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self->preproc_frames) {

        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from pre-processed frames...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(
                    self->preproc_frames);
            if (ovsc_subtract_frames) {
                self->master_bias_quadimage =
                        sph_framecombination_quad_image_from_frameset(
                                ovsc_subtract_frames, self->coll_alg,
                                self->framecomb_parameterlist);
                if (!self->keep_intermediate) {
                    sph_utils_delete_files(ovsc_subtract_frames);
                }
                cpl_frameset_delete(ovsc_subtract_frames); ovsc_subtract_frames = NULL;
            } else {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further.")
                self->master_bias_quadimage =
                        sph_framecombination_quad_image_from_frameset(
                                self->preproc_frames, self->coll_alg,
                                self->framecomb_parameterlist);
            }
        } else {
            SPH_INFO_MSG(
                    "Combine pre-processed frames, no subtraction of the overscan regions.")
            self->master_bias_quadimage =
                    sph_framecombination_quad_image_from_frameset(
                            self->preproc_frames, self->coll_alg,
                            self->framecomb_parameterlist);
        }

    } else {
        SPH_ERR("no preproc frames are  set up!")
        return sph_error_get_last_code();
    }

    if (!self->master_bias_quadimage) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Could not create zimpol master bias:\n"
                        "-- sph_framecombination_quad_image_from_frameset returns null pointer.\n"
                        "cpl error code is: %d", cpl_error_get_code());
        return cpl_error_get_code();
    }

    recipe_error = sph_quad_image_quality_check(self->master_bias_quadimage);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create zimpol master bias:\n"
                        "-- sph_quad_image_quality_check returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);
        return recipe_error;
    }

    recipe_error = sph_quad_image_mask_sigma(self->master_bias_quadimage,
            self->sigma_clip);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create zimpol master bias:\n"
                        "-- sph_quad_image_mask_sigma returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);
        return recipe_error;
    }

    //re-calculate quality check to exclude badpixels from statistics
    recipe_error = sph_quad_image_quality_check(self->master_bias_quadimage);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create zimpol master bias:\n"
                        "-- sph_quad_image_quality_check returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);
        return recipe_error;
    }

    /* Attention:
     * The following function includes a step to update SPH_COMMON_KEYWORD_SPH_TYPE by
     * SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE. It is necessary because cpl_dsf_save_image
     * updates the quad image property list by the property list from the first input frames.
     * In the case of some misbehavior exclude this step for checking.
     */
    if (self->rawframes) {
        SPH_INFO_MSG(
                "Save master bias as a dfs product: header info taken from the first raw frame.");
        pl = sph_zpl_utils_get_camera_header(self->preproc_frames, self->rawframes);
        cpl_propertylist_append(pl, self->master_bias_quadimage->properties);
        sph_zpl_common_preproc_update_header_with_raw_stats(self->rawframes, pl);

        recipe_error = sph_quad_image_save_dfs(self->master_bias_quadimage,
                self->master_bias_outfilename, self->inframes,
                cpl_frameset_get_first(self->rawframes),
                self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_BIAS_CALIB,
                SPH_RECIPE_NAME_ZPL_MASTER_BIAS, SPH_PIPELINE_NAME_ZIMPOL,
                pl);
        if ( pl ) {
            cpl_propertylist_delete( pl ); pl = NULL;
        }
    } else if (self->preproc_frames) {
        SPH_INFO_MSG(
                "Save master bias as a dfs product: header info taken from the first pre-processed frame.")
        recipe_error = sph_quad_image_save_dfs(self->master_bias_quadimage,
                self->master_bias_outfilename, self->inframes,
                cpl_frameset_get_first(self->preproc_frames),
                self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_BIAS_CALIB,
                SPH_RECIPE_NAME_ZPL_MASTER_BIAS, SPH_PIPELINE_NAME_ZIMPOL,
                self->master_bias_quadimage->properties);
    } else {
        SPH_ERR("Neither raw frames nor pre-processed frames provided!");
        recipe_error = sph_error_get_last_code();
    }

    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create zimpol master bias:\n"
                        "-- sph_quad_image_save_dfs returns error code.\n"
                        "cpl error code is: %d", recipe_error);
    }

    SPH_INFO_MSG("sph_zpl_master_bias_run_cunit...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

/**@}*/
