/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <cpl.h>
#include <math.h>
#include "sph_zpl_preproc_imaging.h"
#include "sph_error.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_utils.h"
#include "sph_filemanager.h"

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_preproc_imaging_run Pre-processed zpl imaging raw data
 *
 * This module provides the algorithm implementation for the creation of the
 * pre-processed zimpol raw data, imaging mode.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_preproc_imaging.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the run recipe function for the sph_zpl_preproc_imaging recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_zpl_preproc_imaging_run(sph_zpl_preproc_imaging* self) {
    cpl_frameset* results = NULL;
    cpl_frame* currawframe = NULL;
    cpl_frameset* currawframes = NULL;
    cpl_frame* curframe = NULL;
    //cpl_frame*           outframe		= NULL;
    cpl_propertylist* pl = NULL;
    int camera_id = 0;
    char outfilename[256];
    char outfilename_tmp[256];
    int recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_preproc_imaging_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    self->preproc_frameset = cpl_frameset_new();

    currawframe = cpl_frameset_get_first(self->rawframes);
    while (currawframe) {

        currawframes = cpl_frameset_new();
        recipe_error = cpl_frameset_insert(currawframes,
                cpl_frame_duplicate(currawframe));
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not insert rawframe into the temporal frameset:\n"
                            "cpl error code is: %d", recipe_error);
        }

        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_WARNING,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_WARNING,
                    "Something wrong with the file %s. Trying to continue with others!",
                    cpl_frame_get_filename(currawframe));
            cpl_frameset_delete(currawframes);
            cpl_error_reset();
            currawframe = cpl_frameset_get_next(self->rawframes);
            continue;
        }

        SPH_INFO_MSG("Call sph_zpl_common_preproc_imaging... "
        "");
        results = sph_zpl_common_preproc_imaging(currawframes);
        if (!results || (cpl_frameset_get_size(results) != 2)) {
            SPH_ERR("sph_zpl_common_preproc is failed.");
            cpl_frameset_delete(currawframes);
            return CPL_ERROR_ILLEGAL_OUTPUT;
        }

        SPH_INFO_MSG("Save results");
        curframe = cpl_frameset_get_first(results);

        while (curframe) {
            pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe),
                    0, ".*ESO.*", 0);
            //camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);
            camera_id = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_DET_CHIP_INDEX);
            if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
                sprintf(outfilename_tmp, "_%s", self->outfilename_cam1);
            } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
                sprintf(outfilename_tmp, "_%s", self->outfilename_cam2);
            } else {
                sph_error_raise(
                        SPH_ERROR_ERROR,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Camera ID = <%d>  is wrong in the pre-processed file: %s",
                        camera_id, cpl_frame_get_filename(curframe));
                return CPL_ERROR_ILLEGAL_OUTPUT;

            }

            //write the results into the output file
            // if (!rename(cpl_frame_get_filename( curframe ), (const char*) outfilename) )
            SPH_INFO_MSG("Creating filename...");
            (void)strcpy(outfilename,
                    sph_filemanager_get_basename( cpl_frame_get_filename(currawframe)));
            strcat(outfilename, outfilename_tmp);
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                    "output filename = %s", outfilename);

            //rename the filename from the current result frame
            if (!rename(cpl_frame_get_filename(curframe), outfilename)) {
                cpl_frame_set_filename(curframe, outfilename);
                /*
                 * Can't propagate keywords from all input files.
                 * The preproc recipe should not be used anyway in the online pipeline,
                 * otherwise this problem must be solved. Header is not updated according to the eso standard!

                 outframe = sph_utils_create_calib_frame( outfilename,
                 CPL_FRAME_TYPE_IMAGE,
                 CPL_FRAME_LEVEL_FINAL,
                 self->inframes,
                 self->inparams,
                 SPH_ZPL_KEYWORD_VALUE_PRO_CATG_PREPROC_IMAGING,
                 SPH_RECIPE_NAME_ZPL_PREPROC,
                 SPH_PIPELINE_NAME_ZIMPOL,
                 currawframe);

                 recipe_error = cpl_frameset_insert( self->preproc_frameset, cpl_frame_duplicate( outframe ) );
                 */
                recipe_error = cpl_frameset_insert(self->preproc_frameset,
                        cpl_frame_duplicate(curframe));

                if (recipe_error) {
                    sph_error_raise(
                            SPH_ERROR_GENERAL,
                            __FILE__,
                            __func__,
                            __LINE__,
                            SPH_ERROR_ERROR,
                            "Could not insert ouframe into the preproc_frameset:\n"
                                    "cpl error code is: %d", recipe_error);

                    cpl_frameset_delete(currawframes);
                    cpl_frameset_delete(results);
                    return recipe_error;
                }
            } else {
                SPH_ERROR_RAISE_ERR(
                        SPH_ERROR_GENERAL,
                        "Couldn't rename the resulting file from the zpl common/utility to the "
                        "final product filename  = %s", outfilename);
                cpl_frameset_delete(currawframes);
                cpl_frameset_delete(results);
                return cpl_error_get_code();
            }
            cpl_propertylist_delete(pl);
            curframe = cpl_frameset_get_next(results);
        }
        cpl_frameset_delete(results);
        cpl_frameset_delete(currawframes);
        currawframe = cpl_frameset_get_next(self->rawframes);
    }

    SPH_INFO_MSG("Remove temporarily files: sph_filemanager_clean...");
    sph_filemanager_clean();

    SPH_INFO_MSG("sph_zpl_preproc_imaging_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

/**@}*/
