/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * sph_zpl_star_center_common.c
 *
 *  Created on: Nov 26, 2015
 *      Author: pavlov
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <cpl.h>
#include <math.h>
#include <stdbool.h>

#include "sph_zpl_star_center_common.h"
#include "sph_zpl_keywords.h"
#include "sph_master_frame.h"
#include "sph_double_image.h"
#include "sph_fitting.h"
#include "sph_fits.h"
#include "sph_unsharp_mask.h"
#include "sph_point_pattern.h"

/*------------------------------------------------------------------------------
 * The Structure Definition
 * ----------------------------------------------------------------------------*/

typedef struct _sph_zpl_star_center_star_image_ {
    double cx;
    double cy;
    int npix;
} sph_zpl_star_center_star_image;

/*-----------------------------------------------------------------------------
 Static variables
 -----------------------------------------------------------------------------*/
const int SPH_ZPL_STAR_CENTER_MINPIX = 10;
const int SPH_ZPL_STAR_CENTER_MAX_WAFFIMS = 4;
const int SPH_ZPL_STAR_CENTER_MAX_SEARCH_APERTURES = 20;
const double SPH_ZPL_STAR_CENTER_FIT_FWHM_GUESS = 3.0;

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static bool sph_zpl_star_center_lines_do_intersect__(
        double A0, double B0,
        double A1, double B1,
        double A2, double B2,
        double A3, double B3);
static cpl_image* sph_zpl_star_center_get_extraction_image__(
        sph_master_frame* mf, cpl_image** errim);
static sph_error_code sph_zpl_star_center_fit_center__(
        cpl_apertures* aps,
        int aa, cpl_image* im, cpl_image* errim,
        sph_zpl_star_center_star_image* sim);
static sph_error_code sph_zpl_star_center_get_square_centre__(
        sph_point_pattern* square, double* dx, double *dy);



cpl_propertylist* sph_zpl_star_center_common_get_center_double_image( sph_double_image* di,
                                                       double sigma, int unsharp_window){
    sph_error_code      rerr_i = CPL_ERROR_NONE;
    sph_error_code        rerr_p = CPL_ERROR_NONE;
    cpl_propertylist*   pl = NULL;
    sph_master_frame*    mf_p = NULL;
    double              cen_ix = -9999.0; double cen_px = -9999.0;
    double              cen_iy = -9999.0; double cen_py = -9999.0;

    cpl_ensure( di, CPL_ERROR_NULL_INPUT, NULL);

    pl = cpl_propertylist_new();

    SPH_INFO_MSG("Computer center for the iframe of the given double image...")
    rerr_i = sph_zpl_star_center_common_get_center( di->iframe,
                                                  sigma, unsharp_window,
                                                  &cen_ix, &cen_iy );


    SPH_INFO_MSG("Computer center for the pframe of the given double image...")
    rerr_p = sph_zpl_star_center_common_get_center( di->pframe,
                                                  sigma, unsharp_window,
                                                  &cen_px, &cen_py );

    if ( rerr_p != CPL_ERROR_NONE ){
        SPH_WARNING("Can't compute center from the di->pframe, trying to compute on abs(di->pframe)...");
        cpl_error_reset();
        mf_p = sph_master_frame_duplicate( di->pframe );
        cpl_image_abs(mf_p->image);
        rerr_p = sph_zpl_star_center_common_get_center( mf_p,
                                                      sigma, unsharp_window,
                                                      &cen_px, &cen_py );
        sph_master_frame_delete(mf_p); mf_p = NULL;
    }

    if (rerr_i == CPL_ERROR_NONE && rerr_p != CPL_ERROR_NONE){
        SPH_WARNING("Can't compute center either from the di->pframe or from abs(di->pframe), "
                "the computed center coordinates of the iframe are set up for the pframe as well!");
        cen_px = cen_ix;
        cen_py = cen_iy;
        rerr_p = CPL_ERROR_NONE;
        cpl_error_reset();
    }

    if ( rerr_i == CPL_ERROR_NONE && rerr_p == CPL_ERROR_NONE ){
        cpl_propertylist_append_double( pl, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD, cen_ix );
        cpl_propertylist_append_double( pl, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD, cen_iy );
        cpl_propertylist_append_double( pl, SPH_ZPL_STAR_CENTER_PFRAME_XCOORD, cen_px );
        cpl_propertylist_append_double( pl, SPH_ZPL_STAR_CENTER_PFRAME_YCOORD, cen_py );
    } else {
        SPH_ERR("Error occurs by calculating star center(s)! Return NULL...");
        if ( pl ) { cpl_propertylist_delete( pl ); pl = NULL; }
    }

    return pl;

}


sph_error_code sph_zpl_star_center_common_get_center( sph_master_frame* mf, double sigma, int unsharp_window,
                                                   double* dx, double* dy ){

    sph_error_code  rerr = CPL_ERROR_NONE;
    sph_point_pattern* ppattern = NULL;
    sph_point_pattern* square = NULL;

    cpl_apertures*  aps          = NULL;
    cpl_image*      im           = NULL;
    cpl_image*      unsharped = NULL;
    cpl_image*      errim       = NULL;

    double xw = 3.0;
    double yw = 3.0;
    int nsources = SPH_ZPL_STAR_CENTER_MAX_WAFFIMS;
    int maxaps   = SPH_ZPL_STAR_CENTER_MAX_SEARCH_APERTURES;
    sph_zpl_star_center_star_image   sim;

    cpl_ensure_code( mf, CPL_ERROR_NULL_INPUT);


    sim.cx = sim.cy = 0.0; sim.npix = 0;

    im = sph_zpl_star_center_get_extraction_image__( mf, &errim );

    if ( im ){
        unsharped = sph_unsharp_mask( im, unsharp_window );
        aps = cpl_apertures_extract_sigma( unsharped, sigma );

        if (aps == NULL) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "Could not find any sources on the input above a sigma "
                    "of %f. Check that the quality of the input is sufficient, "
                    "and that the flat field and dark have reasonable quality."
                    "", sigma);
            if ( im ) { cpl_image_delete(im); im = NULL; }
            if ( unsharped ) { cpl_image_delete( unsharped ); unsharped = NULL; }
            if ( errim ) { cpl_image_delete(errim); errim = NULL; }
            return cpl_error_get_code();
        }

    }

    cpl_apertures_sort_by_npix(aps);
    cpl_image_accept_all(unsharped);

    if (cpl_apertures_get_size(aps) < maxaps)  maxaps = cpl_apertures_get_size(aps);

    ppattern = sph_point_pattern_new_(maxaps);

    for (int aa = 0; aa < maxaps; ++aa) {
        if (cpl_apertures_get_npix(aps, aa + 1)  > SPH_ZPL_STAR_CENTER_MINPIX) {
            sph_zpl_star_center_fit_center__(aps, aa, unsharped, errim,
                    &sim);
            sph_point_pattern_add_point(ppattern, sim.cx, sim.cy);

            //nwaffles++;
            if ( nsources == 1 ) {
                *dx = sim.cx; *dy = sim.cy;
            }
        }
    }
    if (ppattern == NULL || sph_point_pattern_get_size(ppattern) < 1) {
         SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                 "Could not find any sources on the input above a sigma "
                 "of %f. Check that the quality of the input is sufficient, "
                 "and that the flat field and dark have reasonable quality."
                 "", sigma);
         goto ERROR_EXIT;
     }

     if (nsources == 1) {
         SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                 "Not using waffle. Just doing simple central fit. "
                 "Guess positions: %5.2f,%5.2f", *dx, *dy);
         if (sph_fitting_fit_gauss2D(im, errim, dx, dy, &xw, &yw, 6, 3)
                 != CPL_ERROR_NONE) {
             SPH_ERROR_RAISE_ERR( CPL_ERROR_ILLEGAL_INPUT,
                     "Could not perform simple central gauss fit.");
             goto ERROR_EXIT;
         }
         SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                 "Gauss fit positions: %5.2f,%5.2f", *dx, *dy);
     } else {
         square = sph_point_pattern_find_square(ppattern);
         if ( !square ) {
             SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                     "Using waffle, but could not find the expected cross shaped waffle pattern "
                     "in the raw input files. Can not proceed.");
             goto ERROR_EXIT;
         }
         else {
             SPH_ERROR_RAISE_INFO(
                     SPH_ERROR_GENERAL,
                     "Identified the following points as the square pattern of the waffle:");

             sph_zpl_star_center_get_square_centre__(square, dx, dy);
         }
     }
     cpl_image_delete(im);
     im = NULL;
     cpl_image_delete(unsharped);
     unsharped = NULL;
     sph_point_pattern_delete(ppattern);
     ppattern = NULL;
     sph_point_pattern_delete(square);
     square = NULL;

     cpl_apertures_delete(aps); aps = NULL;
     cpl_image_delete(errim);errim = NULL;

     return rerr;

     ERROR_EXIT: cpl_image_delete(im); im = NULL;
                 cpl_image_delete(errim); errim = NULL;
                 sph_point_pattern_delete(ppattern);  ppattern = NULL;
                 sph_point_pattern_delete(square); square = NULL;
                 cpl_apertures_delete(aps); aps = NULL;
                 cpl_image_delete(errim);errim = NULL;

                 return cpl_error_get_code();
}


static cpl_image*
sph_zpl_star_center_get_extraction_image__(sph_master_frame* mf, cpl_image** errim) {
    cpl_image* tmprms = NULL;
    cpl_image* im = NULL;

    cpl_ensure( mf, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( errim, CPL_ERROR_NULL_INPUT, NULL);

    im = sph_master_frame_extract_image(mf, 1);
    tmprms = sph_master_frame_get_rms(mf);
    sph_master_frame_set_rms_poisson(mf, 1e10, 0);
    *errim = sph_master_frame_get_rms(mf);
    cpl_image_delete(mf->rmsmap);
    mf->rmsmap = tmprms;
    cpl_image_fill_rejected(im, 0.0);
    cpl_image_accept_all(*errim);
    return im;
}


static sph_error_code sph_zpl_star_center_fit_center__(cpl_apertures* aps,
        int aa, cpl_image* im, cpl_image* errim,
        sph_zpl_star_center_star_image* sim) {
    double xfit = 0;
    double yfit = 0;
    double wx = 0;
    double wy = 0;

    cpl_ensure_code( aps, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( errim, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( sim, CPL_ERROR_NULL_INPUT);
    sim->cx = cpl_apertures_get_centroid_x(aps, aa + 1) - 0.5;
    sim->cy = cpl_apertures_get_centroid_y(aps, aa + 1) - 0.5;
    sim->npix = cpl_apertures_get_npix(aps, aa + 1);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Peak found: centroid: %f, %f and npix=%d, flux=%g, median=%g",
            sim->cx, sim->cy, sim->npix, cpl_apertures_get_flux(aps, aa + 1),
            cpl_apertures_get_median(aps, aa + 1));

    xfit = sim->cx + 0.5;
    yfit = sim->cy + 0.5;
    wx = wy = SPH_ZPL_STAR_CENTER_FIT_FWHM_GUESS;
    if (sph_fitting_fit_gauss2D(im, errim, &xfit, &yfit, &wx, &wy, 6, 3)
            == CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Centroid: %f, %f. "
        "Gauss fit :%f, %f.", sim->cx, sim->cy, xfit - 0.5, yfit - 0.5);

        sim->cx = xfit - 0.5;
        sim->cy = yfit - 0.5;
    } else {
        cpl_error_reset();
        SPH_ERROR_RAISE_WARNING( SPH_ERROR_GENERAL,
                "Could not do gauss fit. Am using centroid.");
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*
 * Helper function to find the centre of a square by finding the
 * intersection of the two lines connecting opposite points.
 *
 * TODO: the code is a bit verbose and could be tightened up...
 */
static
sph_error_code sph_zpl_star_center_get_square_centre__(sph_point_pattern* square, double* dx, double *dy)
{
    sph_point p11;
    sph_point p12;
    sph_point p21;
    sph_point p22;
    sph_point centrepoint;
    sph_line lineA;
    sph_line lineB;

    cpl_ensure_code(square != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sph_point_pattern_get_size(square) != 3,
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(dx != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(dy != NULL, CPL_ERROR_NULL_INPUT);

    sph_point_pattern_get_centre(square, dx, dy);
    p11.x = sph_point_pattern_get_x(square, 0);
    p11.y = sph_point_pattern_get_y(square, 0);
    p12.x = sph_point_pattern_get_x(square, 1);
    p12.y = sph_point_pattern_get_y(square, 1);
    p21.x = sph_point_pattern_get_x(square, 2);
    p21.y = sph_point_pattern_get_y(square, 2);
    p22.x = sph_point_pattern_get_x(square, 3);
    p22.y = sph_point_pattern_get_y(square, 3);

    if ( sph_zpl_star_center_lines_do_intersect__(p11.x,p11.y,
                                                  p12.x,p12.y,
                                                  p21.x,p21.y,
                                                  p22.x,p22.y) )
    {
        lineA.p1 = &p11;
        lineA.p2 = &p12;
        lineB.p1 = &p21;
        lineB.p2 = &p22;
    }
    else {
        if ( sph_zpl_star_center_lines_do_intersect__(p11.x,p11.y,
                p22.x,p22.y,
                p21.x,p21.y,
                p12.x,p12.y) )
        {
            lineA.p1 = &p11;
            lineA.p2 = &p22;
            lineB.p1 = &p21;
            lineB.p2 = &p12;
        }
        else {
            lineA.p1 = &p11;
            lineA.p2 = &p21;
            lineB.p1 = &p12;
            lineB.p2 = &p22;
        }
    }
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Line A %.2f, %.2f to %.2f, %.2f", lineA.p1->x, lineA.p1->y, lineA.p2->x, lineA.p2->y  );
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Line B %.2f, %.2f to %.2f, %.2f", lineB.p1->x, lineB.p1->y, lineB.p2->x, lineB.p2->y  );
    if ( sph_zpl_star_center_lines_do_intersect__(lineA.p1->x,lineA.p1->y,
                                                  lineA.p2->x,lineA.p2->y,
                                                  lineB.p1->x,lineB.p1->y,
                                                  lineB.p2->x,lineB.p2->y) )
    {
        sph_line_intersect_point(&centrepoint, &lineA, &lineB);
        *dx = centrepoint.x;
        *dy = centrepoint.y;
    }
    else {
        SPH_ERROR_RAISE_INFO(CPL_ERROR_ILLEGAL_OUTPUT,
                "Could not find intersection point of lines. "
                "Am using centre of square pattern instead.");
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static bool sph_zpl_star_center_lines_do_intersect__(double A0, double B0,
        double A1, double B1,
        double A2, double B2,
        double A3, double B3)
{
    return (
            ((A2-A0)*(B1-B0) - (B2-B0)*(A1-A0)) *
            ((A3-A0)*(B1-B0) - (B3-B0)*(A1-A0)) < 0)
    &&
            (((A0-A2)*(B3-B2) - (B0-B2)*(A3-A2)) *
            ((A1-A2)*(B3-B2) - (B1-B2)*(A3-A2)) < 0);
}

