/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include "sph_zpl_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_fits.h"
#include "sph_utils.h"
#include "sph_framecombination.h"
#include "sph_common_science.h"

/*-----------------------------------------------------------------------------
 Static variables
 -----------------------------------------------------------------------------*/
// GENERAL KEYWORDS FOR PREPROCESSING
// (presented here only temporarily)
#define SPH_ZPL_KEYWORD_PHASE                  "ESO DRS ZPL DPR PHASE"
#define SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO    "ZERO"
#define SPH_ZPL_KEYWORD_VALUE_PHASE_PI        "PI"
#define SPH_ZPL_KEYWORD_FRAME_ID             "ESO DRS ZPL DPR FRAME ID"
#define SPH_ZPL_KEYWORD_VALUE_FRAME_ID          -1
#define SPH_ZPL_KEYWORD_SPLIT                 "ESO DRS ZPL DPR SPLIT"
#define SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN     "EVEN"
#define SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD     "ODD"

#ifndef PRO_DATANCOM
#define PRO_DATANCOM          "ESO PRO DATANCOM"
#endif

//#define SPH_ZPL_KEYWORD_VALUE_PI_PRSCY          0
//#define SPH_ZPL_KEYWORD_VALUE_PI_OVSCY         52
#define PRESCAN                                 1 // 1- update the header with the correct keywords
#define OVERSCAN                             1 // 0- no update is needed
#define ADU_NX                                 1
#define ADU_NY                                 1
#define ADU_X                                 1

#define ZPL_OLDSTYLE_RAWFORMAT_NUMBEXT  2
#define ZPL_NEWSTYLE_RAWFORMAT_NUMBEXT  0

/*-----------------------------------------------------------------------------
 Function declaration
 -----------------------------------------------------------------------------*/
static cpl_frameset*
sph_zpl_utils_preproc_extract_frames(const cpl_frame* inframe)
 CPL_ATTR_ALLOC;

static cpl_frameset* sph_zpl_utils_preproc_combine_adus(cpl_frameset* exframes);

static cpl_frameset* sph_zpl_utils_preproc_junk_rows(cpl_frameset* adusframes);

static cpl_frameset* sph_zpl_utils_preproc_split_image(cpl_frameset* jrframes);
static cpl_frameset* sph_zpl_utils_preproc_split_image_imaging(
        cpl_frameset* jrframes);

static cpl_frameset* sph_zpl_utils_preproc_create_zpl_exposure(
        cpl_frameset* spliframes);
static cpl_frameset* sph_zpl_utils_preproc_get_frames_ccd(
        cpl_frameset* splitframes, int ccdID);
static cpl_frameset* sph_zpl_utils_preproc_get_frames_pair(
        cpl_frameset* inframes);
static cpl_frameset* sph_zpl_utils_preproc_sort_frames_phase(
        cpl_frameset* inframes);

static cpl_frameset* sph_zpl_utils_preproc_cube_zpl_exposures(
        cpl_frameset* zexpframes);

static cpl_frameset* sph_zpl_utils_preproc_cube_zpl_exposures_imaging(
        cpl_frameset* zexpframes);

/*-----------------------------------------------------------------------------
 Function code
 -----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 * @brief Check the format of the zimpol rawdata
 *
 * @param rawframes   the input rawframes to be checked
 *
 * @return return
 *  *
 * This function checks the format  of the zimpol data and return 0 if
 * the data format has an old style (2 extensions) or return 1 if the format
 * is new one (0 extension), otherwise the value = -1 is returned.
 * The check of the format is not detailed and based only on the expected numbers
 * of the extensions. If the frameset contains both the old and the new format,
 * the return value is -1, because the frameset should be homogeneous.
 *
 */
/*----------------------------------------------------------------------------*/
int sph_zpl_utils_check_format( const cpl_frameset* rawframes ){

    const cpl_frame* curframe = NULL;
    static int oldformat = 0;
    static int newformat = 1;
    int oldformat_index = -1;
    int newformat_index = -1;
    int result = -1;

    cpl_ensure( rawframes, CPL_ERROR_NULL_INPUT, -1);

    curframe = cpl_frameset_get_first_const( rawframes );
    while ( curframe ) {
        const char* filename = cpl_frame_get_filename( curframe );
        const int extn = cpl_frame_get_nextensions( curframe );

        SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "filename: %s,   extn = %d",
                filename, extn );

        switch ( extn )
        {
            case ZPL_OLDSTYLE_RAWFORMAT_NUMBEXT:  oldformat_index++;
                 break;
            case ZPL_NEWSTYLE_RAWFORMAT_NUMBEXT:  newformat_index++;
                 break;
            default:
                SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL, "Unsupported rawdata format, filename: %s",
                        filename );
                return result;
                break;
        }
        curframe = cpl_frameset_get_next_const( rawframes );
    }

    if ( oldformat_index >=0 && newformat_index < 0){
        result = oldformat;
    } else if ( oldformat_index < 0 && newformat_index >= 0) {
        result = newformat;
    } else {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_INCONSISTENT_INPUT, "The input rawdata should have the same format"
                " for the all rawframes (oldstyle: 2 extensions, newstyle 0 extensions");
    }

    return result;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief Check the format of the zimpol rawdata
 *
 * @param rawframes   the input rawframes to be checked
 *
 * @return return
 *  *
 * This function checks the format  of the zimpol data and return 0 if
 * the data format has an old style (2 extensions) or return 1 if the format
 * is new one (0 extension), otherwise the value = -1 is returned.
 * The check of the format is not detailed and based only on the expected numbers
 * of the extensions. If the frameset contains both the old and the new format,
 * the return value is -1, because the frameset should be homogeneous.
 *
 */
/*----------------------------------------------------------------------------*/
int sph_zpl_utils_check_format_frame( const cpl_frame* rawframe ){
    static int oldformat = 0;
    static int newformat = 1;
    int result = -1;
    int extn   = -1;

    cpl_ensure( rawframe, CPL_ERROR_NULL_INPUT, -1);

    extn = cpl_frame_get_nextensions( rawframe );
    SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "filename: %s,   extn = %d",
          cpl_frame_get_filename( rawframe ), extn );

     switch ( extn )
    {
            case ZPL_OLDSTYLE_RAWFORMAT_NUMBEXT:  result = oldformat;
                 break;
            case ZPL_NEWSTYLE_RAWFORMAT_NUMBEXT:  result = newformat;
                 break;
            default:
                SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL, "Unsupported rawdata format, filename: %s",
                        cpl_frame_get_filename( rawframe ) );
                break;
    }

    return result;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create weighted mean master frame from the quad image
 *
 * @param qi    the input quad image to combine
 *
 * @return pointer to the newly created master frame or NULL in case of error.
 *
 * This function creates a new sph_master_frame from the given quad image
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_zpl_utils_calculate_master_frame_mean_weight_from_quad_image(
        const sph_quad_image* qi) {
    sph_master_frame* result = NULL;
    cpl_frame* mf_zero_iframe = NULL;
    cpl_frame* mf_zero_pframe = NULL;
    cpl_frame* mf_pi_iframe = NULL;
    cpl_frame* mf_pi_pframe = NULL;
    cpl_frameset* mframes = NULL;

    if (!qi) {
        SPH_NO_SELF;
        return NULL;
    }

    //check image sizes in the quad image
    if (cpl_image_get_size_x(qi->zero_image->iframe->image)
            != cpl_image_get_size_x(qi->zero_image->pframe->image)
            || cpl_image_get_size_x(qi->zero_image->iframe->image)
                    != cpl_image_get_size_x(qi->pi_image->iframe->image)
            || cpl_image_get_size_x(qi->zero_image->iframe->image)
                    != cpl_image_get_size_x(qi->pi_image->pframe->image)
            || cpl_image_get_size_y(qi->zero_image->iframe->image)
                    != cpl_image_get_size_y(qi->zero_image->pframe->image)
            || cpl_image_get_size_y(qi->zero_image->iframe->image)
                    != cpl_image_get_size_y(qi->pi_image->iframe->image)
            || cpl_image_get_size_y(qi->zero_image->iframe->image)
                    != cpl_image_get_size_y(qi->pi_image->pframe->image)) {
        SPH_ERR(
                "Wrong quad image, i.e. master frames in the quad image have different image sizes");
        return NULL;
    }

    mframes = cpl_frameset_new();

    //zero iframe
    mf_zero_iframe = sph_filemanager_create_temp_frame("quad2master.fits",
            "NONE", CPL_FRAME_GROUP_NONE);
    sph_master_frame_save(qi->zero_image->iframe,
            cpl_frame_get_filename(mf_zero_iframe), NULL);
    cpl_frameset_insert(mframes, mf_zero_iframe);
    //zero pframe
    mf_zero_pframe = sph_filemanager_create_temp_frame("quad2master.fits",
            "NONE", CPL_FRAME_GROUP_NONE);
    sph_master_frame_save(qi->zero_image->pframe,
            cpl_frame_get_filename(mf_zero_pframe), NULL);
    cpl_frameset_insert(mframes, mf_zero_pframe);
    //pi iframe
    mf_pi_iframe = sph_filemanager_create_temp_frame("quad2master.fits", "NONE",
            CPL_FRAME_GROUP_NONE);
    sph_master_frame_save(qi->pi_image->iframe,
            cpl_frame_get_filename(mf_pi_iframe), NULL);
    cpl_frameset_insert(mframes, mf_pi_iframe);
    //pi pframe
    mf_pi_pframe = sph_filemanager_create_temp_frame("quad2master.fits", "NONE",
            CPL_FRAME_GROUP_NONE);
    sph_master_frame_save(qi->pi_image->pframe,
            cpl_frame_get_filename(mf_pi_pframe), NULL);
    cpl_frameset_insert(mframes, mf_pi_pframe);

    result = sph_framecombination_master_frame_new_mean_weight_from_cpl_framest(
            mframes);

    cpl_frameset_delete(mframes);
    mframes = NULL;
    if (cpl_error_get_code() != CPL_ERROR_NONE) {

        SPH_RAISE_CPL;
        return NULL;
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Calculate Stokes sph_double_image from sph_quad_image
 @return   a pointer to the newly created sph_double_image or NULL if
 unsuccessful.
 @param    qi        a pointer to the given quad image

 @note A CPL error is raised in case there was a problem.

 Construct a new sph_double_image from the given sph_quad_image, i.e. calculate
 stokes parameters as follows:
 IFRAME = zero_odd + zero_even + pi_odd + pi_even
 PFRAME = zero_odd - zero_even - pi_odd + pi_even
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_zpl_utils_calculate_stokes_param_double_image_from_quad_image(const sph_quad_image* qi) {
    sph_double_image* spmi = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int single_frame = FALSE;

    if (qi == NULL) {
        SPH_ERR(
                "Could not create double image from quad image: input quad image has null pointer.")
        return NULL;
    }
    spmi = cpl_calloc(1, sizeof(sph_double_image));

    if (spmi == NULL) {
        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not allocate double image");
        return NULL;
    }

    spmi->properties = NULL;
    spmi->qclist = NULL;

    single_frame = sph_utils_is_single_frame(qi->properties);

    if (qi->zero_image->iframe && qi->zero_image->pframe && qi->pi_image->iframe
            && qi->pi_image->pframe) {
        //calculate Stokes parameters
        // Calculate I = odd(0) + even(0) + odd(PI) + even (PI) = spmi->iframe
        spmi->iframe = sph_master_frame_duplicate(qi->zero_image->iframe);
        if (!spmi->iframe) {
            SPH_ERR("Could not create double image from quad image:\n"
            "duplicate of qi->zero_image->iframe is failed ")
            return NULL;
        }
        rerr |= sph_master_frame_add_master_frame(spmi->iframe,
                qi->zero_image->pframe);
        // Single frames do not have a valid pi_image
        if (!single_frame) {
          rerr |= sph_master_frame_add_master_frame(spmi->iframe,
                  qi->pi_image->iframe);
          rerr |= sph_master_frame_add_master_frame(spmi->iframe,
                  qi->pi_image->pframe);
        }

        // Calculate P = odd(0) - even(0) - odd(PI) + even (PI) = spmi->pframe
        spmi->pframe = sph_master_frame_duplicate(qi->zero_image->iframe);
        if (!spmi->pframe) {
            SPH_ERR("Could not create double image from quad image:\n"
            "duplicate of qi->zero_image->iframe is failed ")
            return NULL;
        }
        rerr |= sph_master_frame_subtract_master_frame(spmi->pframe,
                qi->zero_image->pframe);
        // Single frames do not have a valid pi_image
        if (!single_frame) {
          rerr |= sph_master_frame_subtract_master_frame(spmi->pframe,
                  qi->pi_image->iframe);
          rerr |= sph_master_frame_add_master_frame(spmi->pframe,
                  qi->pi_image->pframe);
        }

    } else {
        if (qi == NULL) {
            SPH_ERR("Could not create double image from quad image: \n "
            "input quad image has null master frame pointers .")
            return NULL;
        }
    }
    if (rerr != CPL_ERROR_NONE) {
        SPH_ERR("Could not create double image from quad image: \n"
        "calculation of the stokes parameters is failed")
        return NULL;
    }

    return spmi;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Apply polarization flat field (double image) to the given double image
 @param    doubleimage	a pointer to the given double image to apply pff to
 @param    polarization_flat  a pointer to the polarization flat

 @return   sphere error code


 @note   polarization_flat = pff
 result_IFRAME = doubleimage_IFRAME
 result_PFRAME = doubleimage_PFRAME - doubleimage_IFRAME*(pff_PFRAME/pff_IFRAME)
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_utils_apply_polarization_flat(sph_double_image* doubleimage,
                                      const sph_double_image* polarization_flat)
{
    sph_master_frame* pff_iframe = NULL;
    sph_master_frame* pff_pframe = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    pff_iframe = sph_double_image_get_iframe_master_frame_duplicate(
            polarization_flat);
    pff_pframe = sph_double_image_get_pframe_master_frame_duplicate(
            polarization_flat);

    rerr = sph_master_frame_divide_master_frame(pff_pframe, pff_iframe);
    rerr |= sph_master_frame_multiply_master_frame(pff_pframe,
            doubleimage->iframe);
    rerr |= sph_master_frame_subtract_master_frame(doubleimage->pframe,
            pff_pframe);

    sph_master_frame_delete(pff_iframe);
    sph_master_frame_delete(pff_pframe);

    if (rerr != CPL_ERROR_NONE) {
        SPH_ERR("Error is raised by applying polarization flat field.")
    }

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Calculate polarization efficiency/crosstalk from two double images of stokes
 parameters with opposite direction (e.g. +Q and -Q)
 @param    di_plus	  a pointer to the given double image of +Q
 @param    di_minus  a pointer to the given double image of -Q

 @return   sph_master_frame

 @note 	calculate p/i for +Q/+U and -Q/-U-> then subtract P/I(-Q/-U)
 from P/I(+U/+Q) and divide by 2.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_zpl_utils_calculate_polarization_efficiency(const sph_double_image* di_plus,
                                                const sph_double_image* di_minus)
{
    sph_master_frame* pslashi_plus = NULL;
    sph_master_frame* pslashi_minus = NULL;

    if (di_plus && di_minus) {
        pslashi_plus =
                sph_double_image_calculate_pframe_slash_iframe_master_frame(
                        di_plus);
        pslashi_minus =
                sph_double_image_calculate_pframe_slash_iframe_master_frame(
                        di_minus);
        sph_master_frame_subtract_master_frame(pslashi_plus, pslashi_minus);
        sph_master_frame_multiply_double(pslashi_plus, 0.5);
    } else {
        SPH_NULL_ERROR;
        return NULL;
    }

    sph_master_frame_delete(pslashi_minus);
    return pslashi_plus;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Calculate polarization offset from two double images of stokes
 parameters with opposite direction (e.g. +Q and -Q)
 @param    di_plus	  a pointer to the given double image of +Q
 @param    di_minus  a pointer to the given double image of -Q
 @return   sph_master_frame
 @note 	calculate p/i for +Q/+U and -Q/-U-> then add P/I(-Q/-U)
 to the P/I(-U/-Q) and divide it by 2.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_zpl_utils_calculate_polarization_offset(const sph_double_image* di_plus,
                                            const sph_double_image* di_minus)
{
    sph_master_frame* pslashi_plus = NULL;
    sph_master_frame* pslashi_minus = NULL;

    if (di_plus && di_minus) {
        pslashi_plus =
                sph_double_image_calculate_pframe_slash_iframe_master_frame(
                        di_plus);
        pslashi_minus =
                sph_double_image_calculate_pframe_slash_iframe_master_frame(
                        di_minus);
        sph_master_frame_add_master_frame(pslashi_plus, pslashi_minus);
        sph_master_frame_multiply_double(pslashi_plus, 0.5);
    } else {
        SPH_NULL_ERROR;
        return NULL;
    }

    sph_master_frame_delete(pslashi_minus);
    return pslashi_plus;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Apply modem efficiency(master frame) to the given stock parameters(double image)
 @param    doubleimage	a pointer to the given double image to apply modem efficiency to
 @param    modem	    a pointer to the modem efficiency to be applied

 @return   sphere error code


 @note   modem_efficiency = modem
 result_IFRAME = doubleimage_IFRAME
 result_PFRAME = doubleimage_PFRAME/(modem_PFRAME/modem_IFRAME)
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_utils_apply_modem_efficiency(sph_double_image* doubleimage,
                                     const sph_master_frame* modem) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (doubleimage && modem) {
        rerr = sph_master_frame_divide_master_frame(
                sph_double_image_get_pframe_master_frame(doubleimage), modem);
    } else {
        SPH_NULL_ERROR;
        return sph_error_get_last_code();
    }

    if (rerr != CPL_ERROR_NONE) {
        SPH_ERR("error is raised by applying modem efficiency");
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Find center coordinates from the first given rawframes
 @param    cx0		a pointer to the double of the x coord center
 @param    cy0		a pointer to the double of the y coord center
 @return   sphere error code
 */
/*----------------------------------------------------------------------------*/

sph_error_code sph_zpl_utils_get_frame_center(const cpl_frameset* inframes,
                                              double* cx0, double* cy0)
{
    const char* filename;
    cpl_propertylist* pl = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_ensure_code( inframes != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( cx0      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( cy0      != NULL, CPL_ERROR_NULL_INPUT);

    filename = cpl_frame_get_filename(cpl_frameset_get_first_const(inframes));

    pl = cpl_propertylist_load_regexp(filename, 0, "NAXIS", CPL_FALSE);

    *cx0 = (double)cpl_propertylist_get_int(pl, "NAXIS1") / 2.0;
    /* Not divided by factor 2, because it was not yet expanded in y */
    *cy0 = (double)cpl_propertylist_get_int(pl, "NAXIS2");

    cpl_propertylist_delete(pl);
    pl = NULL;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*
 cpl_frame* sph_zpl_utils_remove_planes_zpl_exposure( cpl_frame* inframe, cpl_vector* iplanes){
 cpl_frame*			result = NULL;
 sph_zpl_exposure	zplexp = NULL;


 if ( inframe == NULL || iplanes == NULL ) {
 SPH_NULL_ERROR;
 return NULL;
 }

 result = sph_filemanager_create_temp_frame("remove_planes.tmp.fits", SPH_ZPL_TAG_PREPROC_CALIB, CPL_FRAME_GROUP_NONE);

 sph_zpl_exposure_load()


 return result;
 }
 */

/*----------------------------------------------------------------------------*/
/**
 @brief    Extract the propertylist from the corresponding extension

 @param    preproc_frames   the frameset of the pre-processed data either of the cam1 or cam2

 @param 	rawframes	     the rawframes the propertylist to be extracted

 @return	pl 		   		 the extracted propertylist from the corresponding extension

 This function identify the chip index (corresponds to the ext in the rawdata)
 from the first frame of the given pre-processed frames. It returns a created propertylist extracted
 from the corresponding extension of the raw data.
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist* sph_zpl_utils_get_camera_header(const cpl_frameset* preproc_frames,
                                                  const cpl_frameset* rawframes) {
    const char* preproc_name;
    const char* rawframe_name;
    cpl_propertylist* pl = NULL;
    int chip_index = -1;

    cpl_ensure(preproc_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rawframes, CPL_ERROR_NULL_INPUT, NULL);

    preproc_name =
        cpl_frame_get_filename(cpl_frameset_get_first_const(preproc_frames));
    rawframe_name = 
        cpl_frame_get_filename(cpl_frameset_get_first_const(rawframes));

    if ( sph_zpl_utils_check_format( rawframes ) == 1 ){
        SPH_INFO_MSG("New style format of the raw data, no needs to add propertylist");
        return NULL;
    } else if ( sph_zpl_utils_check_format( rawframes ) == -1 ) {
        SPH_WARNING("Unsupported rawdata format" );
        return NULL;
    }

    SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, "Preproc filename to extract pl from = %s )",
             preproc_name);

    pl = cpl_propertylist_load(preproc_name, 0);
    if (pl) {
        chip_index = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_DET_CHIP_INDEX);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, "Chip index = %d from the first preproc", chip_index);
        cpl_propertylist_delete(pl);  pl = NULL;
        if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
            SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR, "Extracting chip index from the first preproc file "
                                "frame failed ( filename = %s )",
                                preproc_name );
            return NULL;
        }
    } else {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR, "Extracting header (pl) from the first preproc file "
                            "frame failed ( filename = %s )",
                            preproc_name );
    }
    pl = cpl_propertylist_load(rawframe_name, chip_index);
    if ( pl == NULL) {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR, "Extracting header (pl) from the first raw file "
                            "frame failed ( filename = %s )", rawframe_name);
    }
    return pl;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    subtract  overscan mean values from the pre-processed data

 @param    inframes   the framelist of the pre-processed  data

 @return	the frameslist of the subtracted pre-processed data

 This function subtracts overscan mean values from all pre-processed zpl_exposure
 and zpl_exposure_imaging data given either as frames or cubes.
 It returns a created frameset with subtracted  pre-processed data.
 The number of outframes equals to the input ones.
 */
/*----------------------------------------------------------------------------*/
cpl_frameset* sph_zpl_utils_subtract_overscans(const cpl_frameset* inframes) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frameset* results = NULL;
    const cpl_frame* curframe = NULL;
    cpl_frameset* curframes = NULL;
    cpl_frame* sframe = NULL;
    char* filename = NULL;
    int extn = 0;

    cpl_frame* (*subtract_overscan)(cpl_frameset* curframes,
            const char* filename);

    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);

    sph_error_reset();

    curframe = cpl_frameset_get_first_const (inframes);

    extn = cpl_frame_get_nextensions(curframe);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_INFO,
            "Number of extensions in the first frame(file = %s): %d\n", cpl_frame_get_filename( curframe ), extn)
    if (extn == 4) {
        subtract_overscan = sph_zpl_utils_subtract_overscans_zpl_exposure;
    } else if (extn == 2) {
        subtract_overscan =
                sph_zpl_utils_subtract_overscans_zpl_exposure_imaging;
    } else {
        SPH_ERROR_RAISE_ERR(
                SPH_ERROR_ERROR,
                "Wrong number of extensions (%d) in file: %s\n", extn, cpl_frame_get_filename( curframe ))
        return NULL;
    }

    results = cpl_frameset_new();
    while (curframe) {
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_INFO,
                "Subtract overscans from file: %s\n", cpl_frame_get_filename( curframe ))
        filename = sph_filemanager_new_filename_from_base(
                cpl_frame_get_filename(curframe), "overscan_subtracted");
        curframes = cpl_frameset_new();
        cpl_frameset_insert(curframes, cpl_frame_duplicate(curframe));
        //sframe = sph_zpl_utils_subtract_overscans_zpl_exposure( curframes, filename );
        sframe = subtract_overscan(curframes, filename);
        cpl_frameset_insert(results, sframe);
        cpl_frameset_delete(curframes);
        curframes = NULL;
        cpl_free((void*) filename);
        curframe = cpl_frameset_get_next_const(inframes);
    }

    rerr = sph_error_get_last_code();
    if (rerr != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "post-conditions: error has occured: %d",
                rerr);
        cpl_frameset_delete(results);
        results = NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    subtract  overscan mean values from the pre-processed zpl_exposure data

 @param    inframes   the framelist of the pre-processed zpl_exposure data
 @param	filename   the output filename

 @return	the frame of the overscan subtracted pre-processed zpl_exposure data

 This function subtracts overscan mean values from all pre-processed zpl exposure data
 given either as frames or cubes. It returns a created one frame of subtracted
 pre-processed data. If input data contains many frames (cubes) the output will be combined
 in one cpl_frame.
 To get one output per one input frame use sph_zpl_utils_subtract_overscans.
 */
/*----------------------------------------------------------------------------*/
cpl_frame* sph_zpl_utils_subtract_overscans_zpl_exposure(cpl_frameset* inframes,
                                                         const char* filename)
{
    sph_zpl_exposure* zplexp = NULL;
    cpl_table* ovsc_table = NULL;
    cpl_frame* result = NULL;

    cpl_ensure( inframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( filename, CPL_ERROR_NULL_INPUT, NULL);

    result = cpl_frame_new();
    cpl_frame_set_filename(result, filename);
    cpl_frame_set_tag(result,
            cpl_frame_get_tag(cpl_frameset_get_first_const(inframes)));
    zplexp = sph_common_science_get_next_zpl_exposure(inframes);
    sph_zpl_exposure_subtract_overscans(zplexp);

    ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
    sph_zpl_exposure_save_open(zplexp, result, NULL, ovsc_table);
    sph_zpl_exposure_delete(zplexp);
    zplexp = NULL;

    zplexp = sph_common_science_get_next_zpl_exposure(inframes);
    if (!zplexp) {
        sph_zpl_exposure_finalize_file(result, ovsc_table); //CORRECT in the same way as for IMAGING MODE
        cpl_table_delete(ovsc_table);
        SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
        return result;
    }
    while (zplexp) {
        sph_zpl_exposure_subtract_overscans(zplexp);
        sph_zpl_exposure_save_append(zplexp, result, ovsc_table);
        sph_zpl_exposure_delete(zplexp);
        zplexp = NULL;
        zplexp = sph_common_science_get_next_zpl_exposure(inframes);
    }
    sph_zpl_exposure_finalize_file(result, ovsc_table);
    cpl_table_delete(ovsc_table);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    subtract  overscan mean values from the pre-processed zpl_exposure imaging data

 @param    inframes   the framelist of the pre-processed zpl_exposure_imaging
 @param	filename   the output filename

 @return	the frame of the overscan subtracted pre-processed zpl_exposure_imaging

 This function subtracts overscan mean values from all pre-processed zpl exposure
 imaging data given either as frames or cubes. It returns a created frame of subtracted
 pre-processed data taken in imaging mode. If input data contains many frames (cubes)
 the output will be combined in one cpl_frame.
 To get one output per one input frame use sph_zpl_utils_subtract_overscans.
 */
/*----------------------------------------------------------------------------*/
cpl_frame* sph_zpl_utils_subtract_overscans_zpl_exposure_imaging(
        cpl_frameset* inframes, const char* filename) {
    sph_zpl_exposure_imaging* zplexp = NULL;
    cpl_table* ovsc_table = NULL;
    cpl_frame* result = NULL;

    cpl_ensure( inframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( filename, CPL_ERROR_NULL_INPUT, NULL);

    result = cpl_frame_new();
    cpl_frame_set_filename(result, filename);
    cpl_frame_set_tag(result,
            cpl_frame_get_tag(cpl_frameset_get_first_const(inframes)));
    zplexp = sph_common_science_get_next_zpl_exposure_imaging(inframes);
    sph_zpl_exposure_imaging_subtract_overscans(zplexp);

    ovsc_table = sph_zpl_exposure_imaging_ovsc_table_create_empty();
    sph_zpl_exposure_imaging_save_open(zplexp, result, NULL, ovsc_table);
    sph_zpl_exposure_imaging_delete(zplexp);
    zplexp = NULL;

    zplexp = sph_common_science_get_next_zpl_exposure_imaging(inframes);
    if (!zplexp) {
        sph_zpl_exposure_imaging_save_ovsc(result, ovsc_table);
        cpl_table_delete(ovsc_table);
        SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
        return result;
    }

    while (zplexp) {
        sph_zpl_exposure_imaging_subtract_overscans(zplexp);
        sph_zpl_exposure_imaging_save_append(zplexp, result, ovsc_table);
        sph_zpl_exposure_imaging_delete(zplexp);
        zplexp = NULL;
        zplexp = sph_common_science_get_next_zpl_exposure_imaging(inframes);
    }
    sph_zpl_exposure_imaging_finalize_file(result, ovsc_table);
    cpl_table_delete(ovsc_table);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Zimpol Wrapper to create a property list of the WCS keywords
 * @param frame         cpl_frame to exract the RA and DEC keywords
 * @param xpix          X coordinate of reference pixel
 * @param ypix          Y coordinate of reference pixel
 *
 * @return pl           cpl_propertylist that contains wcs keywords
 *
 * This function is a wrapper function for the sph_utils_astrometry_create_wcs_pl.
 * Reference RA and DEC are extracted from the given cpl_frame. The pixel plate scale,
 * xscale and yscale (mas), are set up here for the zimpol.
  */
/*----------------------------------------------------------------------------*/
cpl_propertylist*
sph_zpl_utils_astrometry_create_wcs_pl( const cpl_frame* frame,
                                        double xpix, double ypix)
{
    cpl_propertylist* pl = NULL;
    const double xscale = 3.63; //mas
    const double yscale = 3.63; //mas
    double ra = -9999.0;
    double dec = -9999.0;

    cpl_ensure( frame, CPL_ERROR_NULL_INPUT, NULL );
    cpl_ensure( xpix > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( ypix > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL );

    pl = cpl_propertylist_load( cpl_frame_get_filename( frame), 0);
    if ( pl ){
        if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_WCS_RA) &&
                cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_WCS_DEC) ) {
            ra = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_WCS_RA);
            dec = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_WCS_DEC);
        } else {
            SPH_WARNING("Input frame doesn't have RA and/or DEC keywords in the main header."
                    "WCS can't be provided in the science product.");
            cpl_propertylist_delete( pl ); pl = NULL;
            return NULL;
        }
    } else {
        SPH_WARNING("Can't read header from the given frame. RA and DEC not extracted."
                "WCS can't be provided in the science product.");
        return NULL;

    }
    if ( pl ) {cpl_propertylist_delete( pl ); pl = NULL;}

    return sph_utils_astrometry_create_wcs_pl( ra, dec,
                                               xpix, ypix, xscale, yscale);
}



/*----------------------------------------------------------------------------*/
/**
 @brief    Pre-processing of the cpl_frameset of the imaging raw zimpol data
 @return   a pointer to the cpl_frameset of the pre-processed data or NULL if
 unsucsessfull.
 @note     A CPL error is raised in case there was a problem.

 This is a main pre-processing function which takes cpl_frameset
 of zimpol rawframes as an argument and invokes consequently
 functions which perform needed pre-processing steps. Input imaging zimpol rawdata
 cpl_frameset are cubes and must follow certain convention (see docs).
 */
/*----------------------------------------------------------------------------*/

cpl_frameset* sph_zpl_utils_preproc_imaging(const cpl_frameset* inframes) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frameset* results = NULL;
    const cpl_size nz = cpl_frameset_get_size(inframes);

    cpl_error_reset();
    if (!inframes) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "No input frames to perform imaging pre-rpocessing steps.");
        return NULL;
    }

    results = cpl_frameset_new();
    for (cpl_size index = 0; index < nz; index++) {
        const cpl_frame* curframe = cpl_frameset_get_position_const(inframes,
                                                                    index);
        cpl_frameset* exframes = sph_zpl_utils_preproc_extract_frames(curframe);
        cpl_frame* czexpframe = NULL;
        cpl_frame* czexpframe_dupl = NULL;
        cpl_frameset* adusframes = NULL;
        cpl_frameset* splitframes = NULL;
        cpl_frameset* zexpcubes = NULL;

        if (!exframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "extract frames is failed; goto the next raw frame.");
            continue;
        }

        adusframes = sph_zpl_utils_preproc_combine_adus(exframes);
        if (!adusframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "combine adus-frames is failed; goto the next raw frame.");
            continue;
        }

        splitframes = sph_zpl_utils_preproc_split_image_imaging(adusframes); //be ware of the two Camera
        if (!splitframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "split images into even/odd is failed; goto the next raw frame.");
            continue;
        }

        zexpcubes = sph_zpl_utils_preproc_cube_zpl_exposures_imaging(
                splitframes);
        if (!zexpcubes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create cube of zimpol exposure imaging frames is failed; goto the next raw frame");
            continue;
        }

        //insert two frames -- zimpol exposure cubes (ccd-1 and ccd-2) into result cpl_frameset
        czexpframe = cpl_frame_new();
        czexpframe = cpl_frameset_get_first(zexpcubes);
        while (czexpframe) {
            czexpframe_dupl = cpl_frame_duplicate(czexpframe);
            cpl_frameset_insert(results, czexpframe_dupl);
            czexpframe = cpl_frameset_get_next(zexpcubes);
        }

        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Current size of the results cpl_frameset: %d",
                (int) cpl_frameset_get_size(results));

        //delete tmp frames for the treated raw frame
        cpl_frameset_delete(exframes);
        cpl_frameset_delete(adusframes);
        cpl_frameset_delete(splitframes);
        cpl_frameset_delete(zexpcubes);
        exframes = NULL;
        adusframes = NULL;
        splitframes = NULL;
        zexpcubes = NULL;

    } // end-for (index)

    rerr = sph_error_get_last_code();

    if (rerr != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "SPH error has occured: %d", rerr);
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Pre-processing of the cpl_frameset of the raw zimpol data

 @param    inframes the frame list

 @return   a pointer to the cpl_frameset of the pre-processed data or NULL if
 unsucsessfull.
 @note     A CPL error is raised in case there was a problem.

 This is a main pre-processing function which takes cpl_frameset
 of zimpol rawframes as an argument and invokes consequently
 functions which perform needed pre-processing steps. Input zimpol rawdata
 cpl_frameset are cubes and must follow certain convention (see docs).
 */
/*----------------------------------------------------------------------------*/

cpl_frameset* sph_zpl_utils_preproc(const cpl_frameset* inframes) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frameset* results = NULL;
    const cpl_size nz = cpl_frameset_get_size(inframes);

    if (!inframes) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "No input frames to perform pre-rpocessing steps.");
        return NULL;
    }

    results = cpl_frameset_new();

    for (cpl_size index = 0; index < nz; index++) {
        const cpl_frame* curframe = cpl_frameset_get_position_const(inframes,
                                                                    index);
        cpl_frameset* exframes = sph_zpl_utils_preproc_extract_frames(curframe);
        cpl_frameset* adusframes = NULL;
        cpl_frameset* jrframes = NULL;
        cpl_frame* czexpframe = NULL;
        cpl_frame* czexpframe_dupl = NULL;
        cpl_frameset* splitframes = NULL;
        cpl_frameset* zexpframes = NULL;
        cpl_frameset* zexpcubes = NULL;

        if (!exframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "extract frames is failed; goto the next raw frame.");
            continue;
        }

        adusframes = sph_zpl_utils_preproc_combine_adus(exframes);
        if (!adusframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "combine adus-frames is failed; goto the next raw frame.");
            continue;
        }

        jrframes = sph_zpl_utils_preproc_junk_rows(adusframes);
        if (!jrframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create trimmed images is failed; go to the next raw frame.");
            continue;
        }

        splitframes = sph_zpl_utils_preproc_split_image(jrframes);
        if (!splitframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "split images into even/odd is failed; goto the next raw frame.");
            continue;
        }

        zexpframes = sph_zpl_utils_preproc_create_zpl_exposure(splitframes);
        if (!zexpframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create zimpol exposure frames is failed; goto the next raw frame.");
            continue;
        }

        zexpcubes = sph_zpl_utils_preproc_cube_zpl_exposures(zexpframes);
        if (!zexpcubes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create cube of zimpol exposure frames is failed; goto the next raw frame");
            continue;
        }

        //insert two frames -- zimpol exposure cubes (ccd-1 and ccd-2) into result cpl_frameset
        czexpframe = cpl_frame_new();
        czexpframe = cpl_frameset_get_first(zexpcubes);
        while (czexpframe) {
            czexpframe_dupl = cpl_frame_duplicate(czexpframe);
            cpl_frameset_insert(results, czexpframe_dupl);
            czexpframe = cpl_frameset_get_next(zexpcubes);
        }

        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Current size of the results cpl_frameset: %d",
                (int) cpl_frameset_get_size(results));

        //delete tmp frames for the treated raw frame
        cpl_frameset_delete(exframes);
        cpl_frameset_delete(adusframes);
        cpl_frameset_delete(jrframes);
        cpl_frameset_delete(splitframes);
        cpl_frameset_delete(zexpframes);
        cpl_frameset_delete(zexpcubes);
        exframes = NULL;
        adusframes = NULL;
        jrframes = NULL;
        splitframes = NULL;
        zexpframes = NULL;
        zexpcubes = NULL;

    } // end-for (index)

    rerr = sph_error_get_last_code();

    if (rerr != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "SPH error has occured: %d", rerr);
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create cube fits-files for the zimpol data exposure imaging camera-1,2
 @return   a pointer to the cpl_frameset of two cube fits-files or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function create two cubes of the pre-processed zimpol imaging raw data.
 Input cpl_frameset is framesets of separate zimpol exposure fits files for
 camera-1,2
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_cube_zpl_exposures_imaging(
        cpl_frameset* zexpframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* frames_ccd = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_imagelist* imagelist = NULL;
    cpl_image* image = NULL;
    int ccdID;
    int io, extn;

    cpl_error_reset();
    if (!zexpframes)
        return NULL;

    results = cpl_frameset_new();
    for (ccdID = 1; ccdID < 3; ccdID++) {
        frames_ccd = sph_zpl_utils_preproc_get_frames_ccd(zexpframes, ccdID);
        if (frames_ccd == NULL) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Frames for this camera ID were not found: \n"
                            " camear ID: %d\n", ccdID);
            break;
        }

        curframe = cpl_frameset_get_first(frames_ccd);
        frame = cpl_frame_new();
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)),
                SPH_ZPL_TAG_PREPROC_IMAGING_CALIB, CPL_FRAME_GROUP_NONE);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Name of the pre-processed file: %s",
                cpl_frame_get_filename(frame));

        io = 0;
        while (curframe) {
            if (io == 0) {
                for (extn = 0; extn < 2; extn++) {
                    pl = cpl_propertylist_load_regexp(
                            cpl_frame_get_filename(curframe), extn, ".*ESO.*",
                            0);
                    cpl_propertylist_append_string(
                            pl,
                            SPH_COMMON_KEYWORD_SPH_TYPE,
                            SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP_IMG);
                    cpl_propertylist_erase(pl, SPH_ZPL_KEYWORD_FRAME_ID);
                    cpl_propertylist_erase(pl, SPH_ZPL_KEYWORD_PHASE);
                    cpl_propertylist_erase(pl, SPH_ZPL_KEYWORD_SPLIT);
                    /* Make any final header updates */
                    sph_utils_update_header(pl);
                    imagelist = cpl_imagelist_load(
                            cpl_frame_get_filename(curframe), CPL_TYPE_INT,
                            extn); //CPL_TYPE_INT ?
                    cpl_imagelist_save(imagelist, cpl_frame_get_filename(frame),
                            CPL_BPP_32_SIGNED, pl, 2 + io);
                    cpl_imagelist_delete(imagelist);
                    io = 2;
                }
            } else {
                for (extn = 0; extn < 2; extn++) {
                    pl = cpl_propertylist_load_regexp(
                            cpl_frame_get_filename(curframe), extn, ".*ESO.*",
                            0);
                    image = cpl_image_load(cpl_frame_get_filename(curframe),
                            CPL_TYPE_INT, 0, extn);
                    sph_cube_append_image(cpl_frame_get_filename(frame), image,
                            NULL, extn);
                    cpl_image_delete(image);
                }
            }
            curframe = cpl_frameset_get_next(frames_ccd);
        }
        cpl_frameset_insert(results, frame);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }
    sph_cube_finalise_all(results);
    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create cube fits-files for the zimpol camera-1,2
 @return   a pointer to the cpl_frameset of two cube fits-files or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function create two cubes of the pre-processed zimpol raw data.
 Input cpl_frameset is framesets of separate zimpol exposure fits files for
 camera-1,2
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_cube_zpl_exposures(
        cpl_frameset* zexpframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* frames_ccd = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_imagelist* imagelist = NULL;
    cpl_image* image = NULL;
    int ccdID;
    int io, extn;

    cpl_error_reset();
    if (!zexpframes)
        return NULL;

    results = cpl_frameset_new();
    for (ccdID = 1; ccdID < 3; ccdID++) {
        frames_ccd = sph_zpl_utils_preproc_get_frames_ccd(zexpframes, ccdID);
        if (frames_ccd == NULL) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Frames for this camera ID were not found: \n"
                            " camear ID: %d\n", ccdID);
            break;
        }

        curframe = cpl_frameset_get_first(frames_ccd);
        frame = cpl_frame_new();
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)),
                SPH_ZPL_TAG_PREPROC_CALIB, CPL_FRAME_GROUP_NONE);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Name of the pre-processed file: %s",
                cpl_frame_get_filename(frame));

        io = 0;
        while (curframe) {
            if (io == 0) {
                for (extn = 0; extn < 4; extn++) {
                    pl = cpl_propertylist_load_regexp(
                            cpl_frame_get_filename(curframe), extn, ".*ESO.*",
                            0);
                    cpl_propertylist_append_string(pl,
                            SPH_COMMON_KEYWORD_SPH_TYPE,
                            SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP);
                    cpl_propertylist_erase(pl, SPH_ZPL_KEYWORD_FRAME_ID);
                    cpl_propertylist_erase(pl, SPH_ZPL_KEYWORD_PHASE);
                    cpl_propertylist_erase(pl, SPH_ZPL_KEYWORD_SPLIT);
                    /* Make any final header updates */
                    sph_utils_update_header(pl);
                    imagelist = cpl_imagelist_load(
                            cpl_frame_get_filename(curframe), CPL_TYPE_INT,
                            extn); //CPL_TYPE_INT ?
                    cpl_imagelist_save(imagelist, cpl_frame_get_filename(frame),
                            CPL_BPP_32_SIGNED, pl, 2 + io);
                    cpl_imagelist_delete(imagelist);
                    io = 2;
                }
            } else {
                for (extn = 0; extn < 4; extn++) {
                    pl = cpl_propertylist_load_regexp(
                            cpl_frame_get_filename(curframe), extn, ".*ESO.*",
                            0);
                    image = cpl_image_load(cpl_frame_get_filename(curframe),
                            CPL_TYPE_INT, 0, extn);
                    sph_cube_append_image(cpl_frame_get_filename(frame), image,
                            NULL, extn);
                    cpl_image_delete(image);
                }
            }
            curframe = cpl_frameset_get_next(frames_ccd);
        }
        cpl_frameset_insert(results, frame);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }

    sph_cube_finalise_all(results);
    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Extract frames for a given CCD ID from the input frames
 @return   a pointer to the cpl_frameset with the same CCD ID or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_get_frames_ccd(
        cpl_frameset* inframes, int ccdID) {
    cpl_frame* curframe = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    int camera_id;

    cpl_error_reset();

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(inframes);
    while (curframe) {
        pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe), 0,
                ".*ESO.*", 0);
        camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);
        if (ccdID == camera_id) {
            cpl_frameset_insert(results, curframe);
        }
        curframe = cpl_frameset_get_next(inframes);
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Extract two frames with the same unique frame ID
 @return   a pointer to the cpl_frameset with 2 corresponding frames or with
 ini-frame if a pair was not found, or NULL if error is raised.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function extracts two frames based on the before generated ID
 from a given frameset. The frame ID (id_frame_ini) is defined from
 the first frame of the given frameset. As soon as a corresponding
 frame is found the result (frameset with 2 frames) will be returned.
 If the corresponding frame is not found the frameset with
 one ini-frame will be returned.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_get_frames_pair(
        cpl_frameset* inframes) {
    cpl_frameset* frames_pair = NULL;
    cpl_frameset* frames_phase = NULL;
    cpl_frame* curframe = NULL;
    cpl_propertylist* pl = NULL;
    int id_frame_ini, id_frame;

    cpl_error_reset();
    if (!inframes)
        return NULL;

    frames_pair = cpl_frameset_new();
    curframe = cpl_frameset_get_first(inframes);
    pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe), 0,
            ".*ESO.*", 0);
    id_frame_ini = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_FRAME_ID);
    cpl_frameset_insert(frames_pair, curframe);
    curframe = cpl_frameset_get_next(inframes);
    while (curframe) {
        pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe), 0,
                ".*ESO.*", 0);
        id_frame = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_FRAME_ID);
        //printf( "id_frame_ini = %d, id_frame = %d\n", id_frame_ini, id_frame);
        if (id_frame == id_frame_ini) {
            cpl_frameset_insert(frames_pair, curframe);
            break;
        }
        curframe = cpl_frameset_get_next(inframes);
    } // end while


    frames_phase = sph_zpl_utils_preproc_sort_frames_phase(frames_pair);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }
    return frames_phase;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    sorts pair-frames in the order of phase.
 @return   a pointer to the cpl_frameset with 2 ordered frames or
 NULL if error is raised.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function sorts pair-frames (with the same frame ID) in the order
 of phase, i.e. first the frame with phase = ZERO and then with the
 phase = PI are put into a frameset.
 */
/*----------------------------------------------------------------------------*/

static cpl_frameset* sph_zpl_utils_preproc_sort_frames_phase(
        cpl_frameset* inframes) {
    cpl_frame* frame_first = NULL;
    cpl_frame* frame_second = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl_first = NULL;
    cpl_propertylist* pl_second = NULL;
    const char* phase_first;
    const char* phase_second;

    cpl_error_reset();
    if (!inframes)
        return NULL;

    results = cpl_frameset_new();

    if (cpl_frameset_get_size(inframes) == 2) {
        frame_first = cpl_frameset_get_first(inframes);
        pl_first = cpl_propertylist_load_regexp(
                cpl_frame_get_filename(frame_first), 0, ".*ESO.*", 0);

        frame_second = cpl_frameset_get_next(inframes);
        pl_second = cpl_propertylist_load_regexp(
                cpl_frame_get_filename(frame_second), 0, ".*ESO.*", 0);

        if (cpl_propertylist_get_int(pl_first, SPH_ZPL_KEYWORD_FRAME_ID)
                == cpl_propertylist_get_int(pl_second, SPH_ZPL_KEYWORD_FRAME_ID)) {
            phase_first = cpl_propertylist_get_string(pl_first,
                    SPH_ZPL_KEYWORD_PHASE);
            phase_second = cpl_propertylist_get_string(pl_second,
                    SPH_ZPL_KEYWORD_PHASE);

            if (!strcmp(phase_first, SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)
                    && !strcmp(phase_second, SPH_ZPL_KEYWORD_VALUE_PHASE_PI)) {
                return inframes;
            } else if (!strcmp(phase_first, SPH_ZPL_KEYWORD_VALUE_PHASE_PI)
                    && !strcmp(phase_second, SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)) {
                cpl_frameset_insert(results, frame_second);
                cpl_frameset_insert(results, frame_first);
            } else {
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Phases IDs are not correct for this frame-pair: \n"
                                " phase_first: %s\n"
                                "phase_second: %s", phase_first, phase_second);
                return NULL;
            }

        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "IDs of frame are not the same.");

            return NULL;
        }

    } else {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Not expected number of frames (2): %d ",
                (int) cpl_frameset_get_size(inframes));
        return NULL;
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create zimpol expose frames
 @return   a pointer to the cpl_frameset with zimpol exposure frame or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This is a forth-B  step in pre-processing of the zimpol rawdata:
 This function combines frames of two different phases (ZERO, PI) into
 four extension fits file. The input frames are separate frames with even and
 odd images (split-images) for each phase.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_create_zpl_exposure(
        cpl_frameset* splitframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* curframe_duplicate = NULL;
    cpl_frame* curframe_pair = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* frames_ccd = NULL;
    cpl_frameset* frames_ccd_duplicate = NULL;
    cpl_frameset* frames_pair = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl_pair_odd = NULL;
    cpl_propertylist* pl_pair_even = NULL;
    cpl_image* image_pair_odd = NULL;
    cpl_image* image_pair_even = NULL;
    int io;
    int camera_id;

    cpl_error_reset();
    if (!splitframes)
        return NULL;

    results = cpl_frameset_new();
    for (int ccdID = 1; ccdID < 3; ccdID++) {
        if (ccdID == 1) {
            camera_id = SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID; //ccdID = 1 -> CAM1_ID
        } else {
            camera_id = SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID; //ccdID = 2 -> CAM2_ID
        }
        frames_ccd = sph_zpl_utils_preproc_get_frames_ccd(splitframes,
                camera_id);
        frames_ccd_duplicate = cpl_frameset_duplicate(frames_ccd);
        curframe = cpl_frameset_get_first(frames_ccd);
        while (curframe) {
            curframe_duplicate = cpl_frameset_get_first(frames_ccd_duplicate);
            frames_pair = sph_zpl_utils_preproc_get_frames_pair(
                    frames_ccd_duplicate);

            if (frames_pair) {
                if (cpl_frameset_get_size(frames_pair) == 2) {
                    curframe_pair = cpl_frameset_get_first(frames_pair);
                    frame = cpl_frame_new();
                    frame = sph_filemanager_create_temp_frame(
                            sph_filemanager_get_basename( cpl_frame_get_filename(curframe)),
                            "NONE", CPL_FRAME_GROUP_NONE);
                    io = 0;
                    while (curframe_pair) {
                        image_pair_odd = cpl_image_load(
                                cpl_frame_get_filename(curframe_pair),
                                CPL_TYPE_INT, 0, 0);
                        image_pair_even = cpl_image_load(
                                cpl_frame_get_filename(curframe_pair),
                                CPL_TYPE_INT, 0, 1);
                        pl_pair_odd = cpl_propertylist_load_regexp(
                                cpl_frame_get_filename(curframe_pair), 0,
                                ".*ESO.*", 0);
                        pl_pair_even = cpl_propertylist_load_regexp(
                                cpl_frame_get_filename(curframe_pair), 1,
                                ".*ESO.*", 0);
                        cpl_image_save(image_pair_odd,
                                cpl_frame_get_filename(frame),
                                CPL_BPP_32_SIGNED, pl_pair_odd, 2 + io);
                        cpl_image_save(image_pair_even,
                                cpl_frame_get_filename(frame),
                                CPL_BPP_32_SIGNED, pl_pair_even, 4);
                        cpl_image_delete(image_pair_odd);
                        cpl_image_delete(image_pair_even);
                        io = 2;
                        curframe_pair = cpl_frameset_get_next(frames_pair);
                    } // end while

                    cpl_frameset_insert(results, frame);
                } // size(frames_pair) == 2

            } // frames_pair

            cpl_frameset_erase_frame(frames_ccd_duplicate, curframe_duplicate);
            curframe = cpl_frameset_get_next(frames_ccd);
        }
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Split "trimmed" images of the frames into even and odd sub-images in imaging mode
 @return   a pointer to the cpl_frameset with even and odd sub-images or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This is a forth-A step in pre-processing of the zimpol rawdata:
 This function  splits "trimmed" images of the frames into even and
 odd sub-images in imaging mode. This function is identical to the sph_zpl_utils_preproc_split_image
 (polarimetric mode) except it doesn't make swap of the odd and even for the camera-2 as it
 is done in the case of the polarimetric mode. These two functions can be reduced to the one
 as soon as imaging keyword to identify mode is written in the header of the raw data.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_split_image_imaging(
        cpl_frameset* jrframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* image = NULL;
    cpl_image* image_odd = NULL;
    cpl_image* image_even = NULL;
    int nx, ny, pis_rejected, ynew;
    double val;

    cpl_error_reset();

    if (!jrframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(jrframes);
    while (curframe) {
        const char* det_read_curname;
        pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe), 0,
                ".*ESO.*", 0);
        //ftmp = fopen(sph_filemanager_get_tmp_filename_const("propertylist"), "w");
        //cpl_propertylist_dump(pl, ftmp);
        //fclose(ftmp);

        //pl = cpl_propertylist_new();
        if (pl == NULL) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "property list is NULL. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }

        det_read_curname = cpl_propertylist_get_string(pl,
                SPH_ZPL_KEYWORD_DET_READ_CURNAME);
        if (!det_read_curname) {
            det_read_curname = "Keyword is not defined";
        }SPH_ERROR_RAISE_WARNING(
                SPH_ERROR_INFO,
                "det_read_curname = %s => StandardImaging mode is set up as default", det_read_curname);

        image = cpl_image_load(cpl_frame_get_filename(curframe), CPL_TYPE_INT,
                0, 0);
        nx = cpl_image_get_size_x(image);
        ny = cpl_image_get_size_y(image);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "nx= %d, ny=%d", nx, ny);
        if (ny % 2 != 0) {
            sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "ny = %d not an even number.", ny);
            cpl_image_delete(image);
            return NULL;
        }
        image_odd = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        image_even = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        for (int xx = 0; xx < nx; ++xx) {
            for (int yy = 0; yy < ny; ++yy) {
                val = cpl_image_get(image, xx + 1, yy + 1, &pis_rejected);
                ynew = yy / 2;
                if (yy % 2 == 0) {
                    cpl_image_set(image_odd, xx + 1, ynew + 1, val); //with regard to the yy it is even
                                                                     //but cpl_image starts from 1,1 pixels (fits convention)
                } else {
                    cpl_image_set(image_even, xx + 1, ynew + 1, val); //with regard to the yy it is odd
                                                                      //but cpl_image starts from 1,1 pixels (fits convention)
                }
                if (cpl_error_get_code() != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__,
                            __LINE__, SPH_ERROR_WARNING,
                            "%s: nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d",
                            cpl_error_get_message_default(cpl_error_get_code()),
                            nx, ny, xx, yy, ynew);
                    /* temporarily commented
                     sph_error_raise(SPH_ERROR_ERROR,
                     __FILE__, __func__, __LINE__,
                     SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                     "Info message: %s \n"
                     "nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d", cpl_error_get_code(),
                     cpl_error_get_message_default(cpl_error_get_code()),
                     nx, ny, xx, yy, ynew);
                     */
                    cpl_error_reset(); //temporarily solution to avoid error arising.
                    //return NULL; //temporarily commented
                }
            }
        }

        frame = cpl_frame_new();
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);

        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Problem with appending propertylist. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }

        if (!strcmp(det_read_curname,
                SPH_ZPL_KEYWORD_VALUE_DET_READ_SNAPSHOT_IMAGING)) {

            // swap extentions if the detector mode is "SnapShotImaging"
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            cpl_image_save(image_even, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_even);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save(image_odd, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_odd);
            cpl_propertylist_delete(pl);

        } else {

            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            cpl_image_save(image_odd, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_odd);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save(image_even, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_even);
            cpl_propertylist_delete(pl);

        }

        image = NULL;
        image_odd = NULL;
        image_even = NULL;
        pl = NULL;

        cpl_frameset_insert(results, frame);

        curframe = cpl_frameset_get_next(jrframes);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                        "Error message: %s", cpl_error_get_code(),
                cpl_error_get_message_default(cpl_error_get_code()));
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Split "trimmed" images of the frames into even and odd sub-images
 @return   a pointer to the cpl_frameset with even and odd sub-images or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This is a forth-A step in pre-processing of the zimpol rawdata:
 This function  splits "trimmed" images of the frames into even and
 odd sub-images.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_split_image(cpl_frameset* jrframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* image = NULL;
    cpl_image* image_odd = NULL;
    cpl_image* image_even = NULL;
    int nx, ny, pis_rejected, ynew;
    double val;

    cpl_error_reset();

    if (!jrframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(jrframes);
    while (curframe) {
        pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe), 0,
                ".*ESO.*", 0);
        //ftmp = fopen(sph_filemanager_get_tmp_filename_const("propertylist"), "w");
        //cpl_propertylist_dump(pl, ftmp);
        //fclose(ftmp);

        //pl = cpl_propertylist_new();
        if (pl == NULL) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "property list is NULL. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }
        image = cpl_image_load(cpl_frame_get_filename(curframe), CPL_TYPE_INT,
                0, 0);
        nx = cpl_image_get_size_x(image);
        ny = cpl_image_get_size_y(image);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "nx= %d, ny=%d", nx, ny);
        if (ny % 2 != 0) {
            sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "ny = %d not an even number.", ny);
            cpl_image_delete(image);
            return NULL;
        }
        image_odd = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        image_even = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        for (int xx = 0; xx < nx; ++xx) {
            for (int yy = 0; yy < ny; ++yy) {
                val = cpl_image_get(image, xx + 1, yy + 1, &pis_rejected);
                ynew = yy / 2;
                if (yy % 2 == 0) {
                    cpl_image_set(image_odd, xx + 1, ynew + 1, val); //with regard to the yy it is even
                                                                     //but cpl_image starts from 1,1 pixels (fits convention)
                } else {
                    cpl_image_set(image_even, xx + 1, ynew + 1, val); //with regard to the yy it is odd
                                                                      //but cpl_image starts from 1,1 pixels (fits convention)
                }
                if (cpl_error_get_code() != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__,
                            __LINE__, SPH_ERROR_WARNING,
                            "%s: nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d",
                            cpl_error_get_message_default(cpl_error_get_code()),
                            nx, ny, xx, yy, ynew);
                    /* temporarily commented
                     sph_error_raise(SPH_ERROR_ERROR,
                     __FILE__, __func__, __LINE__,
                     SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                     "Info message: %s \n"
                     "nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d", cpl_error_get_code(),
                     cpl_error_get_message_default(cpl_error_get_code()),
                     nx, ny, xx, yy, ynew);
                     */
                    cpl_error_reset(); //temporarily solution to avoid error arising.
                    //return NULL; //temporarily commented
                }
            }
        }

        frame = cpl_frame_new();
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);

        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Problem with appending propertylist. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }

        if (cpl_propertylist_get_int(
                pl,
                SPH_ZPL_KEYWORD_CAMERA_ID) == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {

            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            //ftmp = fopen(sph_filemanager_get_tmp_filename_const("propertylist"), "w");
            //cpl_propertylist_dump(pl, ftmp);
            //fclose(ftmp);
            cpl_image_save(image_odd, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_odd);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save(image_even, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_even);
            cpl_propertylist_delete(pl);
        } else if (cpl_propertylist_get_int(
                pl,
                SPH_ZPL_KEYWORD_CAMERA_ID) == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {

            /* subframes for the CAMERA2 must be swapped, i.e. we save first even and then odd     */
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            cpl_image_save(image_even, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_even);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save(image_odd, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_odd);
            cpl_propertylist_delete(pl);
        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "CAMERA: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));

        }

        image = NULL;
        image_odd = NULL;
        image_even = NULL;
        pl = NULL;

        cpl_frameset_insert(results, frame);

        curframe = cpl_frameset_get_next(jrframes);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                        "Error message: %s", cpl_error_get_code(),
                cpl_error_get_message_default(cpl_error_get_code()));
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create trimmed images from the adus frames
 @return   a pointer to the cpl_frameset with the treamed images or NULL if
 unsuccessfull.
 @note     A SPH and CPL error is raised in case there was a problem.

 This is a third step in pre-processing of the zimpol rawdata:
 This function junk rows with regard to the phase (ZERO, PI) to create
 a "trimmed" image

 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_junk_rows(cpl_frameset* adusframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl;
    const char* phase;
    int lly_cut, ury_cut;
    cpl_image* image;
    cpl_image* image_new;

    cpl_error_reset();
    if (!adusframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(adusframes);
    while (curframe) {
        pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe), 0,
                ".*ESO.*", 0);
        phase = cpl_propertylist_get_string(pl, SPH_ZPL_KEYWORD_PHASE);

        if (!strcmp(phase, SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)) {
            lly_cut = 1; // bottom row(s) to be cut for phase = ZERO (in PYTHON) + 1
            ury_cut = 1; // upper row(s) to be cut for phase = ZERO
            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            image_new = cpl_image_extract(image, 1, 1 + lly_cut,
                    cpl_image_get_size_x(image),
                    cpl_image_get_size_y(image) - ury_cut);
        } else if (!strcmp(phase, SPH_ZPL_KEYWORD_VALUE_PHASE_PI)) {
            lly_cut = 0; // bottom row(s) to be cut for phase = PI
            ury_cut = 2; // upper row(s) to be cut for phase = PI (in PYTHON)
            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            image_new = cpl_image_extract(image, 1, 1 + lly_cut,
                    cpl_image_get_size_x(image),
                    cpl_image_get_size_y(image) - ury_cut);
        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Not expected phase value (string: ZERO/PI): %s", phase);
            return NULL;
        }

        frame = cpl_frame_new();
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);
        cpl_image_save(image_new, cpl_frame_get_filename(frame),
                CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE);

        cpl_image_delete(image);
        cpl_image_delete(image_new);
        cpl_frameset_insert(results, frame);
        curframe = cpl_frameset_get_next(adusframes);
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }
    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Combine images of the 2 detector segment into a single image
 @return   a pointer to the cpl_frameset with the combined ADU images or NULL if
 unsuccessfull.
 @note     A SPH and CPL error is raised in case there was a problem.

 This is a second step in pre-processing of the zimpol rawdata:
 This function combines 2 detector segments (ADU) for an each frame from the
 frameset (exframes) into a single image and it treams away overscan areas.

 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_utils_preproc_combine_adus(cpl_frameset* exframes) {
    cpl_frame* curframe = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl;
    cpl_image* image;
    cpl_image* image_adu1;
    cpl_image* image_adu2;
    cpl_image* image_adus;
    cpl_frame* frame;
    int camera_id = 0;
    int adu1_x, adu1_y, adu1_nx, adu1_ny;
    int adu1_prscx, adu1_prscy, adu1_ovscx, adu1_ovscy;
    int adu2_x, adu2_y, adu2_nx, adu2_ny;
    int adu2_prscx, adu2_prscy, adu2_ovscx, adu2_ovscy;
    int llx, lly, urx, ury;

    cpl_error_reset();
    if (!exframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(exframes);
    while (curframe) {
        int xsize_adu1;
        int ysize_adu1;
        int xsize_adu2;
        int ysize_adu2;

        pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe), 0,
                ".*ESO.*", 0);
        camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);

        if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {

            adu1_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_X);
            adu1_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_Y);
            adu1_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_NX);
            adu1_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_NY);
            adu1_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_PRSCX);
            adu1_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_PRSCY);
            adu1_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_OVSCX);
            adu1_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_OVSCY);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO, "ADU sizes: \n"
                            "adu1_x = %d\n"
                            "adu1_y = %d\n"
                            "adu1_nx = %d\n"
                            "adu1_ny = %d\n"
                            "adu1_prscx = %d\n"
                            "adu1_prscy = %d\n"
                            "adu1_ovscx = %d\n"
                            "adu1_ovscy = %d\n", adu1_x, adu1_y, adu1_nx,
                    adu1_ny, adu1_prscx, adu1_prscy, adu1_ovscx, adu1_ovscy);

            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            llx = adu1_x + adu1_prscx;
            lly = adu1_y + adu1_prscy;
            urx = adu1_x + (adu1_nx - 1) - adu1_ovscx;
            ury = adu1_y + (adu1_ny - 1) - adu1_ovscy;
            image_adu1 = cpl_image_extract(image, llx, lly, urx, ury);

            adu2_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_X);
            adu2_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_Y);
            adu2_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_NX);
            adu2_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_NY);
            adu2_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_PRSCX);
            adu2_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_PRSCY);
            adu2_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_OVSCX);
            adu2_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_OVSCY);
            //adu2_x = SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_X; // !this must be commented when the header is properly created

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO, "ADU sizes: \n"
                            "adu2_x = %d\n"
                            "adu2_y = %d\n"
                            "adu2_nx = %d\n"
                            "adu2_ny = %d\n"
                            "adu2_prscx = %d\n"
                            "adu2_prscy = %d\n"
                            "adu2_ovscx = %d\n"
                            "adu2_ovscy = %d\n", adu2_x, adu2_y, adu2_nx,
                    adu2_ny, adu2_prscx, adu2_prscy, adu2_ovscx, adu2_ovscy);

            llx = adu2_x + adu2_prscx;
            lly = adu2_y + adu2_prscy;
            urx = adu2_x + (adu2_nx - 1) - adu2_ovscx;
            ury = adu2_y + (adu2_ny - 1) - adu2_ovscy;
            image_adu2 = cpl_image_extract(image, llx, lly, urx, ury);

        } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {

            adu1_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_X);
            adu1_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_Y);
            adu1_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_NX);
            adu1_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_NY);
            adu1_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_PRSCX);
            adu1_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_PRSCY);
            adu1_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_OVSCX);
            adu1_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_OVSCY);

            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            llx = adu1_x + adu1_prscx;
            lly = adu1_y + adu1_prscy;
            urx = adu1_x + (adu1_nx - 1) - adu1_ovscx;
            ury = adu1_y + (adu1_ny - 1) - adu1_ovscy;
            image_adu1 = cpl_image_extract(image, llx, lly, urx, ury);

            adu2_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_X);
            adu2_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_Y);
            adu2_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_NX);
            adu2_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_NY);
            adu2_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_PRSCX);
            adu2_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_PRSCY);
            adu2_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_OVSCX);
            adu2_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_OVSCY);
            //adu2_x = SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_X; // this must be commented when the header is properly created
            llx = adu2_x + adu2_prscx;
            lly = adu2_y + adu2_prscy;
            urx = adu2_x + (adu2_nx - 1) - adu2_ovscx;
            ury = adu2_y + (adu2_ny - 1) - adu2_ovscy;
            image_adu2 = cpl_image_extract(image, llx, lly, urx, ury);

        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Wrong camera ID: %d in the header of the fitsfile: %s.",
                    camera_id, cpl_frame_get_filename(curframe));
            return NULL;
        }

        //cpl_image_save( image_adu1, fname+"extract1.fits", CPL_TYPE_FLOAT, pl, 2 )
        //cpl_image_save( image_adu2, fname+"extract2.fits", CPL_TYPE_FLOAT, pl, 2 )
        xsize_adu1 = cpl_image_get_size_x(image_adu1);
        ysize_adu1 = cpl_image_get_size_y(image_adu1);
        xsize_adu2 = cpl_image_get_size_x(image_adu2);
        ysize_adu2 = cpl_image_get_size_y(image_adu2);
        if (xsize_adu1 != xsize_adu2 || ysize_adu1 != ysize_adu2) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Problems with the size of ADUs: \n"
                            "xsize_adu1 = %d\n"
                            "ysize_adu1 = %d\n"
                            "xsize_adu2 = %d\n"
                            "ysize_adu2 = %d\n", xsize_adu1, ysize_adu1,
                    xsize_adu2, ysize_adu2);
            cpl_image_delete(image);
            cpl_image_delete(image_adu1);
            cpl_image_delete(image_adu2);
            return NULL;
        }
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "ADU sizes: \n"
                        "xsize_adu1 = %d\n"
                        "ysize_adu1 = %d\n"
                        "xsize_adu2 = %d\n"
                        "ysize_adu2 = %d\n", xsize_adu1, ysize_adu1, xsize_adu2,
                ysize_adu2);

        image_adus = cpl_image_new(xsize_adu1 + xsize_adu2, ysize_adu1,
                CPL_TYPE_INT);
        cpl_image_copy(image_adus, image_adu1, 1, 1);
        cpl_image_copy(image_adus, image_adu2, xsize_adu1 + 1, 1);
        frame = cpl_frame_new();
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);
        cpl_image_save(image_adus, cpl_frame_get_filename(frame),
                CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE);
        cpl_frameset_insert(results, frame);

        // delete images to free memory
        cpl_image_delete(image);
        cpl_image_delete(image_adu1);
        cpl_image_delete(image_adu2);
        cpl_image_delete(image_adus);

        curframe = cpl_frameset_get_next(exframes);
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }
    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Extract frames from the two cubes of zimpol rawdata
 @return   a pointer to the cpl_frameset with the extracted frames or NULL if
 unsuccessfull.
 @note     A SPH and CPL error is raised in case there was a problem.

 This is a first step in pre-processing of the zimpol rawdata.
 This function extracts frames from the 2 cubes and returns
 cpl_frameset, providing for each frame proper keywords
 for the pre-processing needs:
 - SPH_ZPL_KEYWORD_CAMERA_ID: 1 for camera-1, 2 -- camera-2
 - SPH_ZPL_KEYWORD_PHASE: "ZERO" & "PI" for the phase
 according to the convention
 - SPH_ZPL_KEYWORD_FRAME_ID: integer ID number to identify
 pair of frames ("ZERO", "PI").
 */
/*----------------------------------------------------------------------------*/

static cpl_frameset*
sph_zpl_utils_preproc_extract_frames(const cpl_frame* inframe) {
    int frame_id = SPH_ZPL_KEYWORD_VALUE_FRAME_ID;
    int nplane = 0;
    const int ext_cam1 = 1;
    const int ext_cam2 = 2;
    cpl_frameset* results;
    cpl_image* rawimage_cam1;
    cpl_image* rawimage_cam2;
    cpl_frame* frame;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* plm = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;


    cpl_error_reset();
    if (!inframe)
        return NULL;

    results = cpl_frameset_new();
    while (TRUE) {
        const char* filename = cpl_frame_get_filename(inframe);
        const char* det_read_curname;
        const char* phase;

        if (nplane % 2) {
            phase = SPH_ZPL_KEYWORD_VALUE_PHASE_PI; // odd

        } else {
            phase = SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO; //even (we start by even)
            frame_id = frame_id + 1;
        }
        //read main header into plm
        plm = cpl_propertylist_load_regexp(filename, 0,
                ".*ESO.*", 0);
        det_read_curname = cpl_propertylist_get_string(plm,
                SPH_ZPL_KEYWORD_DET_READ_CURNAME);

        /* camera-1 */
        rawimage_cam1 = cpl_image_load(filename,
                CPL_TYPE_INT, nplane, ext_cam1);
        if (cpl_error_get_code() == 0) {
            frame = sph_filemanager_create_temp_frame(
                    filename, "NONE",
                    CPL_FRAME_GROUP_NONE);
            pl = cpl_propertylist_load_regexp(filename,
                    ext_cam1, ".*ESO.*", 0);

            //append keywords for camera id, phase (0,pi), frame_id (unique frame ID for both phases "0,PI")
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_PHASE, phase);
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FRAME_ID, frame_id);
            if (OVERSCAN) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_OVSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU1_OVSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_OVSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU1_OVSCY);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_OVSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_OVSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_OVSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_OVSCY);
            }
            if (PRESCAN) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_PRSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU1_PRSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_PRSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU1_PRSCY);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_PRSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_PRSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_PRSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_PRSCY);
            }

            //add for new data
            if (ADU_NX) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_NX,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU1_NX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_NX,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_NX);
            }
            if (ADU_NY) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_NY,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU1_NY);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_NY,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_NY);
            }
            if (ADU_X) {
                //only ADU2_X is supposed to be not correct
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_X,
                        SPH_ZPL_KEYWORD_VALUE_CAM1_ADU2_X);
            }

            //In the case of SlowPolarimetry update the following keywords
            if (!strcmp(det_read_curname,
                    SPH_ZPL_KEYWORD_VALUE_DET_READ_CURNAME)) {
                //OVERSCAN Y set to zero
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_OVSCY,
                        0);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_OVSCY,
                        0);
                //PRESCAN Y set to zero
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_PRSCY,
                        0);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_PRSCY,
                        0);
                //_NY = 1024
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_NY,
                        1024);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_NY,
                        1024);
            }

            //add main property list (extension 0) to the current pl
            if (pl && plm) {
                for (int i = 0; i < cpl_propertylist_get_size(plm); i++) {
                    if (!cpl_propertylist_has(
                            pl,
                            cpl_property_get_name(
                                    cpl_propertylist_get(plm, i)))) {
                        rerr = cpl_propertylist_append_property(pl,
                                cpl_propertylist_get(plm, i));
                    } else {
                        rerr = cpl_propertylist_erase(
                                pl,
                                cpl_property_get_name(
                                        cpl_propertylist_get(plm, i)));
                        rerr = cpl_propertylist_append_property(pl,
                                cpl_propertylist_get(plm, i));
                    }
                }
            }

            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Error occured by appending keywords from the main propertylist (ext=0)");
                SPH_RAISE_CPL;
                return NULL;
            }

            cpl_image_save(rawimage_cam1, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE);
            cpl_image_delete(rawimage_cam1);
            cpl_frameset_insert(results, frame);
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "cam1: extracted image saved in filename: %s.\n",
                    cpl_frame_get_filename(frame));

            cpl_propertylist_delete(pl);

        } else {
            if (nplane == 0) {
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "No image can be loaded from raw file: %s.\n"
                                "CPL ERROR CODE: %d",
                        filename, cpl_error_get_code());
                return NULL;
            }
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "%d frames have been extracted from raw file: %s.\n",
                    nplane, filename);
            cpl_error_reset();
            break;
        } //end-if-else-block camera-1
        /* camera-2 */
        rawimage_cam2 = cpl_image_load(filename,
                CPL_TYPE_INT, nplane, ext_cam2);
        if (cpl_error_get_code() == 0) {
            frame = sph_filemanager_create_temp_frame(
                    filename, "NONE",
                    CPL_FRAME_GROUP_NONE);
            pl = cpl_propertylist_load_regexp(filename,
                    ext_cam2, ".*ESO.*", 0);

            //append keywords for camera id, phase (0,pi), frame_id (unique frame ID for both phases "0,PI")
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_PHASE, phase);
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FRAME_ID, frame_id);

            if (OVERSCAN) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_OVSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU1_OVSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_OVSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU1_OVSCY);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_OVSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_OVSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_OVSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_OVSCY);
            }
            if (PRESCAN) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_PRSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU1_PRSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_PRSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU1_PRSCY);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_PRSCX,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_PRSCX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_PRSCY,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_PRSCY);
            }

            //add for new data
            if (ADU_NX) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_NX,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU1_NX);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_NX,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_NX);
            }
            if (ADU_NY) {
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_NY,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU1_NY);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_NY,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_NY);
            }
            if (ADU_X) {
                //only ADU2_X is supposed to be not correct
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_X,
                        SPH_ZPL_KEYWORD_VALUE_CAM2_ADU2_X);
            }
            //In the case of SlowPolarimetry update the following keywords
            if (!strcmp(det_read_curname,
                    SPH_ZPL_KEYWORD_VALUE_DET_READ_CURNAME)) {
                //OVERSCAN Y set to zero
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_OVSCY,
                        0);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_OVSCY,
                        0);
                //PRESCAN Y set to zero
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_PRSCY,
                        0);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_PRSCY,
                        0);
                //_NY = 1024
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_NY,
                        1024);
                cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_NY,
                        1024);
            }

            //add main property list (extension 0) to the current pl
            if (pl && plm) {
                for (int i = 0; i < cpl_propertylist_get_size(plm); i++) {
                    if (!cpl_propertylist_has(
                            pl,
                            cpl_property_get_name(
                                    cpl_propertylist_get(plm, i)))) {
                        rerr = cpl_propertylist_append_property(pl,
                                cpl_propertylist_get(plm, i));
                    } else {
                        rerr = cpl_propertylist_erase(
                                pl,
                                cpl_property_get_name(
                                        cpl_propertylist_get(plm, i)));
                        rerr = cpl_propertylist_append_property(pl,
                                cpl_propertylist_get(plm, i));
                    }
                }
            }
            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Error occured by appending keywords from the main propertylist (ext=0)");
                SPH_RAISE_CPL;
                return NULL;
            }

            cpl_image_save(rawimage_cam2, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE);
            cpl_image_delete(rawimage_cam2);
            cpl_frameset_insert(results, frame);
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "cam2: extracted image saved in filename: %s.\n",
                    cpl_frame_get_filename(frame));
            cpl_propertylist_delete(pl);

        } else {
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "%d frames have been extracted from raw file: %s.\n",
                    nplane, filename);
            sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "Number of frames is not even: %d.",
                    nplane);
            cpl_error_reset();
            break;
        } //end-if-else-block camera-2
        cpl_propertylist_delete(plm);
        nplane = nplane + 1;
    } //end while-loop
    return results;

}

cpl_error_code sph_zpl_qc_add_keys(cpl_propertylist * self,
                                   const cpl_frameset * rawframes,
                                   const cpl_propertylist * propl)
{
    const cpl_errorstate prestate = cpl_errorstate_get();
    const cpl_size    ilast = cpl_frameset_get_size(rawframes) - 1;
    const cpl_frame*  frame = cpl_frameset_get_position_const(rawframes, ilast);
    const char*       file  = cpl_frame_get_filename(frame);
    cpl_propertylist* plist = cpl_propertylist_load_regexp(file, 0,
                                                           SPH_COMMON_KEYWORD_NDIT "$", 0);
    const cpl_propertylist* srcpl = propl ? propl : self;

    double ndit = 0.0;
    double medians = 0.0;
    double leakage = 0.0;

    if (plist != NULL && cpl_propertylist_has(plist, SPH_COMMON_KEYWORD_NDIT)) {
        const int indit = cpl_propertylist_get_int(plist,
                                                   SPH_COMMON_KEYWORD_NDIT);
        if (indit > 0)
            ndit = (double)indit;
    }

    if (ndit <= 0.0 && srcpl != NULL &&
        cpl_propertylist_has(srcpl, SPH_COMMON_KEYWORD_NDIT)) {
        const int indit = cpl_propertylist_get_int(srcpl,
                                                   SPH_COMMON_KEYWORD_NDIT);
        if (indit > 0)
            ndit = (double)indit;
    }

    if (srcpl != NULL && cpl_propertylist_has(srcpl,
                                              SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_MED)) {
        medians = cpl_propertylist_get_double(srcpl,
                                              SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_MED);
    }
    if (srcpl != NULL && cpl_propertylist_has(srcpl,
                                              SPH_ZIMPOL_KEYWORD_QC_MEDIAN_DOUBLEIMAGE_LEAKAGE)) {
        leakage = cpl_propertylist_get_double(srcpl,
                                              SPH_ZIMPOL_KEYWORD_QC_MEDIAN_DOUBLEIMAGE_LEAKAGE);
    }

    cpl_errorstate_set(prestate);

    if (ndit > 0.0 && medians > 0.0) {
        const double isnr = sqrt(ndit * medians);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_IFRAME_SNR,
                                       isnr);
    }
    if (ndit > 0.0 && leakage > 0.0) {
        const double lsnr = sqrt(ndit * leakage);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_LEAKAGE_SNR,
                                       lsnr);
    }

    cpl_propertylist_delete(plist);

    return cpl_error_set_where(cpl_func);
}

cpl_error_code sph_zpl_qc_add_odd_even_keys(cpl_propertylist * self,
                                            const cpl_frameset * rawframes,
                                            const cpl_propertylist * propl)
{
    const cpl_errorstate prestate = cpl_errorstate_get();
    const cpl_size    nframes= cpl_frameset_get_size(rawframes);
    const cpl_size    ilast = nframes - 1;
    const cpl_frame*  frame = cpl_frameset_get_position_const(rawframes, ilast);
    const char*       file  = cpl_frame_get_filename(frame);
    cpl_propertylist* plist = cpl_propertylist_load_regexp(file, 0,
                                                           SPH_COMMON_KEYWORD_NDIT "$", 0);
    const cpl_propertylist* srcpl = propl ? propl : self;

    double ndit = 0.0;
    double ncom = 0.0;
    double zomedians = 0.0;
    double zemedians = 0.0;
    double pomedians = 0.0;
    double pemedians = 0.0;

    if (plist != NULL && cpl_propertylist_has(plist, SPH_COMMON_KEYWORD_NDIT)) {
        const int indit = cpl_propertylist_get_int(plist,
                                                   SPH_COMMON_KEYWORD_NDIT);
        if (indit > 0)
            ndit = (double)indit;
    }

    if (plist != NULL && cpl_propertylist_has(plist, PRO_DATANCOM)) {
        const int dncom = cpl_propertylist_get_int(plist,
                                                   PRO_DATANCOM);
        if (dncom > 0)
            ncom = (double)dncom;
    }

    if (ndit <= 0.0 && srcpl != NULL &&
        cpl_propertylist_has(srcpl, SPH_COMMON_KEYWORD_NDIT)) {
        const int indit = cpl_propertylist_get_int(srcpl,
                                                   SPH_COMMON_KEYWORD_NDIT);
        if (indit > 0)
            ndit = (double)indit;
    }

    if (ncom <= 0.0 && srcpl != NULL && cpl_propertylist_has(srcpl, PRO_DATANCOM)) {
        const int dncom = cpl_propertylist_get_int(srcpl, PRO_DATANCOM);
        if (dncom > 0)
            ncom = (double)dncom;
    }

    if (ncom <= 0.0)
        ncom = (double)nframes;

    if (srcpl != NULL && cpl_propertylist_has(srcpl,
                                              SPH_COMMON_KEYWORD_QC_ZERO_ODD_RAW_MEDIANS_MED)) {
        zomedians = cpl_propertylist_get_double(srcpl,
                                              SPH_COMMON_KEYWORD_QC_ZERO_ODD_RAW_MEDIANS_MED);
    }
    if (srcpl != NULL && cpl_propertylist_has(srcpl,
                                              SPH_COMMON_KEYWORD_QC_ZERO_EVEN_RAW_MEDIANS_MED)) {
        zemedians = cpl_propertylist_get_double(srcpl,
                                              SPH_COMMON_KEYWORD_QC_ZERO_EVEN_RAW_MEDIANS_MED);
    }

    if (srcpl != NULL && cpl_propertylist_has(srcpl,
                                              SPH_COMMON_KEYWORD_QC_PI_ODD_RAW_MEDIANS_MED)) {
        pomedians = cpl_propertylist_get_double(srcpl,
                                              SPH_COMMON_KEYWORD_QC_PI_ODD_RAW_MEDIANS_MED);
    }
    if (srcpl != NULL && cpl_propertylist_has(srcpl,
                                              SPH_COMMON_KEYWORD_QC_PI_EVEN_RAW_MEDIANS_MED)) {
        pemedians = cpl_propertylist_get_double(srcpl,
                                              SPH_COMMON_KEYWORD_QC_PI_EVEN_RAW_MEDIANS_MED);
    }

    cpl_errorstate_set(prestate);

    if (ndit > 0.0 && ncom > 0.0) {
        if (zomedians > 0.0) {
            const double zosnr = sqrt(ncom * ndit * zomedians);
            cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_ZERO_ODD_SNR,
                                           zosnr);
        }
        if (zemedians > 0.0) {
            const double zesnr = sqrt(ncom * ndit * zemedians);
            cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_ZERO_EVEN_SNR,
                                           zesnr);
        }
        if (pomedians > 0.0) {
            const double posnr = sqrt(ncom * ndit * pomedians);
            cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_PI_ODD_SNR,
                                           posnr);
        }
        if (pemedians > 0.0) {
            const double pesnr = sqrt(ncom * ndit * pemedians);
            cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_PI_EVEN_SNR,
                                           pesnr);
        }
    }

    cpl_propertylist_delete(plist);

    return cpl_error_set_where(cpl_func);
}

cpl_error_code sph_zpl_qc_add_plane_keys(cpl_propertylist * self,
                                         const cpl_frameset * rawframes,
                                         cpl_boolean do_counts)
{
    if (self != NULL) {
        const cpl_errorstate prestate = cpl_errorstate_get();
        const cpl_size    ilast = cpl_frameset_get_size(rawframes) - 1;
        const cpl_frame*  frame = cpl_frameset_get_position_const(rawframes,
                                                                  ilast);
        const char*       file  = cpl_frame_get_filename(frame);
        cpl_imagelist * imgcube = cpl_imagelist_load(file, CPL_TYPE_DOUBLE, 1);
        const cpl_size nz = cpl_imagelist_get_size(imgcube);
        const cpl_size nx = cpl_image_get_size_x(cpl_imagelist_get(imgcube, 0));
        const cpl_size ny = cpl_image_get_size_y(cpl_imagelist_get(imgcube, 0));
        const cpl_size nxy = nx * ny;
        double aer = 0.0; /* Average even row */
        double aor = 0.0; /* Average odd row */
        double mean = 0.0;
        double varsum = 0.0;
        double stdev;

        cpl_errorstate_set(prestate);

        for (cpl_size i = 0; i < nz; i++) {
            cpl_image * iimg = cpl_imagelist_get(imgcube, i);
            double * imgdata = cpl_image_get_data_double(iimg);
            cpl_vector * imgvec = cpl_vector_wrap(nxy, imgdata);
            const double aeri = do_counts ?
                cpl_image_get_mean_window(iimg, 60, 238, 260, 238) : 0.0;
            const double aori = do_counts ?
                cpl_image_get_mean_window(iimg, 60, 239, 260, 239) : 0.0;
            /* Window mean must come before median, which permutes the values */
            const double imed = cpl_vector_get_median(imgvec);
            const double delta = imed - mean;

            if (do_counts) {
                aer += (aeri - aer) / (double)(i + 1);
                aor += (aori - aor) / (double)(i + 1);
            }

            varsum += (double)i * delta * (delta / (double)(i + 1));
            mean += delta  / (double)(i + 1);

            (void)cpl_vector_unwrap(imgvec);
        }

        /* Compute the bias-corrected standard deviation */
        stdev = nz > 1 && varsum > 0.0 ? sqrt(varsum / (double)(nz - 1)) : 0.0;

        cpl_propertylist_append_double(self, SPH_COMMON_KEYWORD_QC_PLANE_AVG,
                                       mean);
        cpl_propertylist_append_double(self, SPH_COMMON_KEYWORD_QC_PLANE_SIG,
                                       mean != 0.0 ? stdev / mean : 0.0);

        if (do_counts)
            cpl_propertylist_append_double(self,
                                           SPH_COMMON_KEYWORD_QC_IFRAME_RAW_COUNTS,
                                           CPL_MAX(aer, aor));

        cpl_msg_info(cpl_func, "Cube-stats %d X %d X %d: %g %g (%g<=>%g)",
                     (int)nx, (int)ny, (int)nz, mean, stdev, aer, aor);

        cpl_imagelist_delete(imgcube);
    }

    return cpl_error_set_where(cpl_func);
}

