/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "sph_pixel_description_table.h"
#include "sph_master_frame.h"
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_apertures.h"
#include <math.h>


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the central wavelength for an aperture.
 * @param aps    the aperture list
 * @param n    the index of the aperture to use
 * @param pdt    the pixel description table
 *
 * @return the wavelength
 *
 * This function extracts the wavelength using the
 * given pixel description table to map pixel coordinates of the aperture
 * to wavelength values.
 */
/*----------------------------------------------------------------------------*/
double
sph_apertures_get_lambda( cpl_apertures* aps, int n,
        sph_pixel_description_table* pdt) {
    double dx = 0.0;
    double dy = 0.0;
    double lambda = -1.0;
    sph_pixel_descriptor*    pdescr = NULL;

    dx = cpl_apertures_get_centroid_x(aps,n);
    dy = cpl_apertures_get_centroid_y(aps,n);

    pdescr = sph_pixel_description_table_get_descriptor(pdt,(int)dx,(int)dy);
    if ( pdescr ) {
        if ( pdescr->wavelength >0.0 ) {
            lambda = pdescr->wavelength + pdescr->dlambda * ( dy-(double)((int)dy) );
        }
    }
    return lambda;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the wavelength width for an aperture.
 * @param aps    the aperture list
 * @param n    the index of the aperture to use
 * @param pdt    the pixel description table
 * @param mframe the value map
 *
 * @return the width in wavelength direction
 *
 * This function extracts the width in the y direction (wavelength) using the
 * given pixel description table to map pixel coordinates of the aperture
 * to wavelength values and the provided master frame to provide values
 * for the pixels to determine the width.
 * The width is calculated by taking the detected region around the centroid
 * and then calculating the weighed sum
 * \f[ \sum M(x_0,y_i)*(\lambda_i-\lambda_0)^2 / \sum M(x_0,y_i) \f]
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_apertures_get_width_y_pdt( cpl_apertures* aps, int n,
        sph_pixel_description_table* pdt, sph_master_frame* mframe ) {
    double dx = 0.0;
    double dy = 0.0;
    double value = 0.0;
    double totval = 0.0;
    int min = -1;
    int max = -1;
    int rej = 0;
    double lambda = -1.0;
    double width = 0.0;
    int yy;
    sph_pixel_descriptor*    pdescr = NULL;

    dx = cpl_apertures_get_centroid_x(aps,n);
    dy = cpl_apertures_get_centroid_y(aps,n);

    pdescr = sph_pixel_description_table_get_descriptor(pdt,(int)dx,(int)dy);
    if ( pdescr ) {
        if ( pdescr->wavelength >0.0 ) {
            lambda = pdescr->wavelength + pdescr->dlambda * ( dy-(double)((int)dy) - 0.5 );
        }
    }

    min = cpl_apertures_get_bottom(aps,n);
    max = cpl_apertures_get_top(aps,n);
    for (yy = min; yy <= max; ++yy) {
        pdescr = sph_pixel_description_table_get_descriptor(pdt,(int)dx,yy);
        value = cpl_image_get(mframe->image,(int)dx,yy,&rej);
        if ( pdescr ) {
            if ( pdescr->wavelength > 0.0 ) {
                width += ( lambda-pdescr->wavelength ) *
                        ( lambda-pdescr->wavelength ) * value;
                totval += value;
            }
        }
    }
    if ( totval ) width/=totval;
    width = sqrt(8.0*width*log(2.0)); // convert to FWHM
    return width;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the width in x for an aperture.
 * @param aps    the aperture list
 * @param n    the index of the aperture to use
 * @param mframe the value map
 *
 * @return the width in x direction
 *
 * This function extracts the FWHM width in the x direction from
 * the provided master frame
 * The width is calculated by taking the detected region around the centroid
 * and then calculating the weighed sum
 * \f[ \sum M(x_0,y_i)*(x_i-x_0)^2 / \sum M(x_0,y_i) \f]
 * and converting to FWHM.
 */
/*----------------------------------------------------------------------------*/
double
sph_apertures_get_width_x( cpl_apertures* aps, int n,
        sph_master_frame* mframe ) {
    double dx = 0.0;
    double dy = 0.0;
    double value = 0.0;
    double totval = 0.0;
    int min = -1;
    int max = -1;
    int rej = 0;
    double width = 0.0;
    int xx;

    dx = cpl_apertures_get_centroid_x(aps,n);
    dy = cpl_apertures_get_centroid_y(aps,n);


    min = cpl_apertures_get_left(aps,n);
    max = cpl_apertures_get_right(aps,n);
    for (xx = min; xx <= max; ++xx) {
        value = cpl_image_get(mframe->image,xx,(int)dy,&rej);
        width += ( xx - dx ) *
                ( xx - dx ) * value;
        totval += value;
    }
    if ( totval ) width/=totval;
    width = sqrt(8.0*width*log(2.0)); // convert to FWHM
    return width;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the width in y for an aperture.
 * @param aps    the aperture list
 * @param n    the index of the aperture to use
 * @param mframe the value map
 *
 * @return the width in y direction
 *
 * This function extracts the FWHM width in the xydirection from
 * the provided master frame
 * The width is calculated by taking the detected region around the centroid
 * and then calculating the weighed sum
 * \f[ \sum M(x_0,y_i)*(x_i-x_0)^2 / \sum M(x_0,y_i) \f]
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_apertures_get_width_y( cpl_apertures* aps, int n,
        sph_master_frame* mframe ) {
    double dx = 0.0;
    double dy = 0.0;
    double value = 0.0;
    double totval = 0.0;
    int min = -1;
    int max = -1;
    int rej = 0;
    double width = 0.0;
    int yy;

    dx = cpl_apertures_get_centroid_x(aps,n);
    dy = cpl_apertures_get_centroid_y(aps,n);


    min = cpl_apertures_get_bottom(aps,n);
    max = cpl_apertures_get_top(aps,n);
    for (yy = min; yy <= max; ++yy) {
        value = cpl_image_get(mframe->image,(int)dx,yy,&rej);
        width += ( yy - dy ) *
                ( yy - dy ) * value;
        totval += value;
    }
    if ( totval ) width/=totval;
    width = sqrt(8.0*width*log(2.0)); // convert to FWHM
    return width;
}

/**@}*/
