/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include "sph_error.h"
#include "sph_background.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_fits.h"
#include "sph_filemanager.h"
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/



cpl_frameset*
sph_background_apply_to_raw_cubes(
		cpl_frameset* raw_frames, cpl_frameset* bg_frames) {
	cpl_frameset*		outframes = NULL;
	cpl_frame*			bg_frame_chosen = NULL;
	cpl_frame*			out_frame = NULL;

	int					ff		 = 0;

	SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
	cpl_ensure(raw_frames && bg_frames, CPL_ERROR_NULL_INPUT, NULL);
	outframes = cpl_frameset_new();
	for (ff = 0; ff < cpl_frameset_get_size(raw_frames); ++ff) {
		bg_frame_chosen =
				sph_background_choose_bg_frame_frame(
						bg_frames,
						cpl_frameset_get_position( raw_frames, ff) );
		out_frame = sph_background_subtract_background(
				cpl_frameset_get_position( raw_frames, ff),
				bg_frame_chosen );
		cpl_frameset_insert( outframes, out_frame );
	}
	return outframes;
}

cpl_frame*
sph_background_choose_bg_frame_frame(
		cpl_frameset* bg_frames,
		cpl_frame* raw_frame )
{
	double					dit		= 0.0;
	cpl_frame*				result  = NULL;

	SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
	cpl_ensure( bg_frames && raw_frame, CPL_ERROR_NULL_INPUT, NULL);


	dit = sph_background_get_dit( raw_frame );

	if ( dit > 0.0 ) {
		result = sph_background_choose_bg_frame_dit( bg_frames, dit );
	}
	return result;
}

cpl_frame*
sph_background_subtract_background( cpl_frame* raw_frame, cpl_frame* bg_frame ) {
	cpl_frame*		result = NULL;
	char*			fname = NULL;
	char*			newname = NULL;
	int				pp		= 0;
	int				np = 0;
	cpl_image*		bg = NULL;
	cpl_image*		im = NULL;
	cpl_imagelist* imlist = NULL;
	cpl_propertylist* pl = NULL;

	SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
	cpl_ensure(raw_frame && bg_frame, CPL_ERROR_NULL_INPUT, NULL);
	result = cpl_frame_duplicate(raw_frame);
	fname = sph_filemanager_remove_dir( cpl_frame_get_filename( raw_frame ) );
	newname = sph_filemanager_new_filename_from_base(fname,"bgsubtracted");
	cpl_frame_set_filename( result, newname );

	pl = cpl_propertylist_load( cpl_frame_get_filename( raw_frame), 0 );
  /* Update the header if required */
  sph_utils_update_header(pl);

	bg = cpl_image_load( cpl_frame_get_filename( bg_frame ), CPL_TYPE_INT, 0, 0);
	np = sph_fits_get_nplanes( cpl_frame_get_filename( raw_frame ), 0 );
	SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
			"Subtracting background %s (DIT=%4.3f) from %s (DIT=%4.3f). "
			"Subtraction performed on %d planes.",
			cpl_frame_get_filename( bg_frame),
			sph_background_get_dit( bg_frame),
			cpl_frame_get_filename( raw_frame),
			sph_background_get_dit( raw_frame ),
			np);

	for (pp = 0; pp < np; ++pp) {
		im = cpl_image_load( cpl_frame_get_filename( raw_frame ), CPL_TYPE_INT, pp, 0);
		cpl_image_subtract(im,bg);
		if (pp == 0 && np == 1) {
			cpl_image_save( im, cpl_frame_get_filename( result), CPL_TYPE_INT, pl, CPL_IO_CREATE );
			cpl_image_delete( im ); im = NULL;
		}
		else {
			imlist = cpl_imagelist_new();
			cpl_imagelist_set( imlist, im, 0);
			cpl_ensure( cpl_imagelist_get_size(imlist) > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
			cpl_ensure( cpl_imagelist_is_uniform(imlist) == 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
			if ( pp == 0 ) {
				cpl_imagelist_save( imlist, cpl_frame_get_filename( result), CPL_TYPE_INT, pl, CPL_IO_CREATE );
				SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
			}
			else {
				cpl_imagelist_save( imlist, cpl_frame_get_filename( result), CPL_TYPE_INT, NULL, CPL_IO_APPEND );
				SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
			}
			cpl_imagelist_delete(imlist); imlist = NULL;
		}
	}

	cpl_image_delete( bg );
	cpl_free( fname ); fname = NULL;
	cpl_free( newname ); newname = NULL;
	cpl_propertylist_delete(pl ); pl = NULL;
	SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
	return result;
}

cpl_frame*
sph_background_choose_bg_frame_dit(
		cpl_frameset* bg_frames,
		double dit )
{
	int			ff			= 0;
	int			mind		= -1;
	double		dit_bg		= 0.0;
	double		mindifference		= 1000000000000.0;

	cpl_ensure(bg_frames,CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(cpl_frameset_get_size(bg_frames) > 0,
			CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure( dit > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL);

	for (ff = 0; ff < cpl_frameset_get_size(bg_frames); ++ff) {
		dit_bg = sph_background_get_dit(
						cpl_frameset_get_position(bg_frames,ff) );

		if ( dit_bg > 0.0 ) {
			if ( fabs(dit_bg - dit) < mindifference ) {
				mindifference = fabs(dit_bg - dit);
				mind = ff;
			}
		}
		else {
			SPH_ERROR_RAISE_WARNING(CPL_ERROR_ILLEGAL_INPUT, "Missing keyword.");
		}
	}
	if (mindifference >0){
		cpl_msg_warning(cpl_func,"Could not find identical DITs for background and flat field frames - have to live with a difference of %f s!",mindifference);
		cpl_msg_warning(cpl_func,"Expect a degraded quality of the resulting flat!");
	}
	cpl_ensure(mind >= 0, CPL_ERROR_ILLEGAL_OUTPUT,NULL);
	cpl_ensure(mind < cpl_frameset_get_size(bg_frames),
			CPL_ERROR_ILLEGAL_OUTPUT,NULL);
	return cpl_frameset_get_position(bg_frames,mind);
}

double
sph_background_get_dit( cpl_frame* aframe ) {
	cpl_propertylist* pl		= NULL;
	double		dit		= -1.0;

	pl = sph_keyword_manager_load_properties(
			cpl_frame_get_filename( aframe ), 0 );
	if ( cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_SEQ1DIT) ) {
		dit = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT);
	}
	cpl_propertylist_delete(pl); pl = NULL;
	cpl_msg_info(cpl_func,"Frame %s has DIT %f s!",cpl_frame_get_filename( aframe ),dit);
	return dit;
}
/**@}*/
