/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_error.h"
#include <cpl_error.h>
#include <cpl.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

const int SPH_ERROR_INFO                        = 1;
const int SPH_ERROR_WARNING                     = 2;
const int SPH_ERROR_ERROR                       = 3;

sph_error_code SPH_ERROR_GENERAL            = CPL_ERROR_EOL + 1;
sph_error_code SPH_ERROR_MEMORY_FAIL        = CPL_ERROR_EOL + 2;
sph_error_code SPH_ERROR_INCONSISTENT_INPUT = CPL_ERROR_EOL + 3;
sph_error_code SPH_ERROR_PREVIOUS_OP_FAILED = CPL_ERROR_EOL + 4;
sph_error_code SPH_ERROR_INFO_MESSAGE       = CPL_ERROR_EOL + 5;
sph_error_code SPH_ERROR_CPL_ERROR          = CPL_ERROR_EOL + 6;


#ifdef SPHERE_ERRSYSTEM
static sph_error*   spherr                      = NULL;
#endif



/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_error Sphere Error Handling
 * @par Synopsis:
 * @code
 * typedef struct _sph_error_ {
 *    char**  messages;
 *    int*    errlevels;
 *    int     nerrs;
 * } sph_error;
 * @endcode
 *
 * @par Description:
 * This module provides the functionality to handle errors in the sphere
 * data reduction. The sphere error handling builds upon the CPL error
 * handling and can be regarded as an extension thereof.
 *
 * Error handling in the sphere DRH is described in detail in the SPHERE
 * Data Reduction Library Design documents.
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a new sph_error handler.
 @return   Pointer of the sph_error handler.

 This function creates a new sph_error object. The sph_error object is a
 singleton and can only exist once. If this constructor function is called
 more than once, the sph_error object is simply re-initialized, and the pointer
 to the existing object is returned.

 */
/*----------------------------------------------------------------------------*/
sph_error* sph_error_new(void) {
#ifndef SPHERE_ERRSYSTEM
    return NULL;
#else
    int                 i           = 0;

    if ( spherr == NULL) {
        spherr = cpl_calloc( 1, sizeof(sph_error) );
    }
    else {
        return spherr;
    }

    if ( spherr->errlevels != NULL) {
        for ( i = 0; i < SPH_ERROR_MAX_MESSAGES; i++ ) {
            cpl_free(spherr->messages[i]);
            spherr->messages[i] = NULL;
        }
        cpl_free(spherr->messages);
        spherr->messages = NULL;
        if ( spherr->errlevels ) {
            cpl_free(spherr->errlevels);
        }
        spherr->errlevels = NULL;
    }
    spherr->errlevels = cpl_calloc( SPH_ERROR_MAX_MESSAGES, sizeof(int) );
    spherr->messages = cpl_calloc( SPH_ERROR_MAX_MESSAGES, sizeof(char*));
    for ( i = 0;i < SPH_ERROR_MAX_MESSAGES; i++ ) {
        spherr->messages[i] = cpl_calloc( 2500, sizeof(char) );
    }
    spherr->nerrs = 0;
    spherr->outcpl = 0;
    return spherr;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 @brief  Set the error level that should be reported in the logfile.
 @param  severity the severity above and including which to abort.

 This function sets for which kind of messages an sph_error_raise
 should actually be logged.

 */
/*----------------------------------------------------------------------------*/
void sph_error_set_reportlevel( int lvl ){
#ifndef SPHERE_ERRSYSTEM
    return;
#else
    spherr->outcpl = lvl;
    if ( spherr->outcpl > SPH_ERROR_ERROR ) spherr->outcpl = SPH_ERROR_ERROR;
    if ( spherr->outcpl < 1 ) spherr->outcpl = 0;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 @brief         Abort everything if an error has occured and dump error list.
 @param  severity the severity above and including which to abort.

 @return        0 if no error is in list. If there is code aborts.

 This function aborts execution and dumps the error list if an error is in list
 that has severity level equal or above the given level.

 */
/*----------------------------------------------------------------------------*/

int sph_error_abort( int severity ) {
#ifndef SPHERE_ERRSYSTEM
    return CPL_ERROR_NONE;
#else
    int                     rerr    = CPL_ERROR_NONE;
    int                     i       = 0;
    short                   doabort   = 0;

    if ( spherr == NULL ) {
        return CPL_ERROR_NULL_INPUT;
    }

    for ( i = 0; i < spherr->nerrs; i++ ) {
        if ( spherr->errlevels[i] >= severity ) {
            doabort = 1;
        }
    }
    if ( doabort ) {
        for ( i = 0; i < spherr->nerrs; i++ ) {
            printf( "%s -- LEVEL: %d\n", spherr->messages[i], spherr->errlevels[i]);
        }
        exit( 1 );
    }
    return rerr;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 @brief         Dump everything above a given severity.
 @param  severity the severity above and including which to dump.

 @return        0 if no error occured.

 This function dumps the error list, dumping all messages above severity.
 */
/*----------------------------------------------------------------------------*/

int sph_error_dump( int severity ) {
#ifndef SPHERE_ERRSYSTEM
    return CPL_ERROR_NONE;
#else
    int                     rerr    = CPL_ERROR_NONE;
    int                     i       = 0;
    int                     sevcount[10];
    int                     totcount = 0;
    int                        di         = 0;

    if ( spherr == NULL ) {
        return CPL_ERROR_NULL_INPUT;
    }

    for ( i = 0; i < 10; i++ ) {
        sevcount[i] = 0;
    }
    for ( i = 0; i < spherr->nerrs; i++ ) {
        if ( spherr->errlevels[i] >= spherr->outcpl ) {
            di++;
        }
    }
    if ( di == 0 ) return rerr;

    printf ( "\n"
             "--------------------------------------------------------------\n"
             "-        Messages and Errors in the SPHERE DRH Pipeline       \n"
             "--------------------------------------------------------------\n"
             "- Showing all errors which are above level: %d                \n"
             "- Note that CPL errors are not in all cases written           \n"
             "- Please also see the .logfile (if created) for a             \n"
             "- more complete list of messages.                             \n"
             "--------------------------------------------------------------\n"
             "\n",
             severity );

    for ( i = 0; i < spherr->nerrs; i++ ) {
        if ( spherr->errlevels[i] >= severity ) {
            if ( spherr->errlevels[i] == SPH_ERROR_ERROR ) {
                printf( "SPH ERROR: Level %d ==> %s\n", spherr->errlevels[i],
                    spherr->messages[i]);
            }
            else if ( spherr->errlevels[i] == SPH_ERROR_WARNING ) {
                printf( "SPH WARNING: Level %d ==> %s\n", spherr->errlevels[i],
                    spherr->messages[i]);
            }
            else {
                printf( "SPH INFO: Level %d ==> %s\n", spherr->errlevels[i],
                    spherr->messages[i]);
            }

        }
        sevcount[spherr->errlevels[i]]++;
        totcount++;
    }

    printf ( "\n"
             "--------------------------------------------------------------\n"
             "-        END of Messages and Errors in SPHERE DRH             \n"
             "--------------------------------------------------------------\n"
             "- Total number of messages: %d                                \n"
             "- Of these there were:                                        \n",
             totcount);

    for ( i = 1; i < 4; i++ ) {
             printf ( "-              Level %d   : %d                               \n",
                      i, sevcount[i] );
    }
    printf( "-\n");
    printf( "-\n");
    printf( "- With the levels 1 -> INFO, 2 -> WARNING and 3 -> ERROR        \n" );
    printf( "-------------------------------------------------------------\n" );

    return rerr;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 @brief         Dump last message.
 @param  severity the severity above and including which to dump.

 @return        0 if no error occured.

 This function dumps the last error with a severity above teh given value.
 Nothing is done if there is no such error.
 */
/*----------------------------------------------------------------------------*/
int sph_error_dump_last( int severity ) {
#ifndef SPHERE_ERRSYSTEM
    return CPL_ERROR_NONE;
#else
   int                     rerr    = CPL_ERROR_NONE;
   int                     i       = 0;
   int                     di      = 0;

   if ( spherr == NULL ) {
       return CPL_ERROR_NULL_INPUT;
   }
   for ( i = 0; i < spherr->nerrs; i++ ) {
       if ( spherr->errlevels[i] >= severity ) {
           di = i;
       }
   }

   if ( di ) {
       printf( "%s -- LEVEL: %d\n", spherr->messages[di], spherr->errlevels[di]);
   }
   return rerr;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Unset last error
 *
 * This function unsets (removes) the last message.
 *
 */
/*----------------------------------------------------------------------------*/

void sph_error_unset_last(void) {
#ifndef SPHERE_ERRSYSTEM
    cpl_error_reset();
    return;
#else
    if ( spherr == NULL )
        return;
    if ( spherr->nerrs < 1 )
        return;
    spherr->nerrs--;
    cpl_error_reset();
    return;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 @brief  Return last error code.

 @return        0 if no error occured, otherwise last errorcode.

 This function returns the errorcode of the last error (severity> 2)
 that was raised.
 If no error was raised, 0 is returned.
 */
/*----------------------------------------------------------------------------*/

int sph_error_get_last_code(void) {
#ifndef SPHERE_ERRSYSTEM
    return cpl_error_get_code();
#else
    int                     rerr    = CPL_ERROR_NONE;
    int                     i       = 0;
    int                     di      = 0;
    int                     severity = SPH_ERROR_WARNING;

    if ( spherr == NULL ) {
        return CPL_ERROR_NULL_INPUT;
    }
    if ( spherr->reset == 1 ) {
        return CPL_ERROR_NONE;
    }
    for ( i = 0; i < spherr->nerrs; i++ ) {
        if ( spherr->errlevels[i] > severity ) {
            di = i;
        }
    }

    if ( di ) {
        return di;
    }
    return rerr;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 @brief  Return number of errors.

 @return        0 if no error occured, otherwise number of errors.

 This function returns the number of errors (severity > 2)
 that were raised.
 If no error was raised, 0 is returned.
 */
/*----------------------------------------------------------------------------*/

int sph_error_get_size(void) {

#ifndef SPHERE_ERRSYSTEM
    return 0;
#else
    int                     rerr    = CPL_ERROR_NONE;
    int                     i       = 0;
    int                     di      = 0;
    int                     severity = SPH_ERROR_WARNING;

    if ( spherr == NULL ) {
        return CPL_ERROR_NULL_INPUT;
    }
    for ( i = 0; i < spherr->nerrs; i++ ) {
        if ( spherr->errlevels[i] > severity ) {
            di++;
        }
    }

    if ( di ) {
        return di;
    }
    return rerr;
#endif
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Format the error string
 *
 * @param message The sting to format
 *
 * @return 0 if all ok, otherwise error code
 *
 * This formats the given message nicely (inplace operation)
 *
 */
/*----------------------------------------------------------------------------*/
int sph_error_format( const char* message, char** out ) {
    char*                   temp        = cpl_calloc( 1000, sizeof(char));

#ifndef SPHERE_ERRSYSTEM
    strcat(temp,message);
#else
    int                     length      = 0;
    int                     ii          = 0;
    int                     cc          = 0;
    int                     lc          = 0;
    int                     maxlen      = 80;

    temp[0]= '\0';
    length = strlen(message);
    for (ii = 0; ii < length; ++ii) {
        cc = strlen(temp);
        if ( lc > maxlen && message[ii] == ' ' ) {
            temp[cc] = message[ii];
            temp[cc+1] = '\n';
            temp[cc+2] = '\0';
            strcat( temp, "          ");
            lc = 0;
        }
        else if ( message[ii] == '\n' ) {
            temp[cc] = message[ii];
            temp[cc+1] = '\0';
            strcat( temp, "          ");
            lc = 0;
        }
        else {
            temp[cc] = message[ii];
            temp[cc+1] = '\0';
            lc++;
        }
    }
    cc = strlen(temp);
    temp[cc] = '\n';
    temp[cc+1] = '\0';
#endif
    *out = temp;
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief         Raise an error.
 @param  code     the error code to set
 @param  file     the file the error occured in (can be NULL)
 @param  function the function the error occured in (can be NULL)
 @param  line     the line no the error occured in (can be 0)
 @param  message  the mesaage (must be set)
 @param  severity the severity above and including which to abort.

 @return        error code.

 This function raises the given error and adds it to the list. Note that no
 output is done.

 */
/*----------------------------------------------------------------------------*/

int sph_error_raise( int code, const char* file, const char* func,
                     int line, int severity, const char* message, ...)
{
#ifndef SPHERE_ERRSYSTEM
    if (code != CPL_ERROR_NONE) {
        va_list                 ap;
        char*                   form = NULL;

        va_start(ap, message);
        form = cpl_vsprintf(message, ap);
        va_end(ap);

        if (severity == SPH_ERROR_ERROR ) {
            cpl_error_set_message_macro(func, code, file, line, "%s", form);
        } else if (severity == SPH_ERROR_WARNING ) {
            cpl_msg_warning(func, "%s", form);
        } else if (form) {
            cpl_msg_info(func, "%s", form);
        }

        cpl_free(form);
    }

    return code;
#else
    va_list                 ap;
    char                    temp[1000];
    char                    temp2[1000];
    char*                   form = NULL;

    if ( code == CPL_ERROR_NONE ) return CPL_ERROR_NONE;

    sph_error_format(message,&form);
    
    va_start(ap, form);
    if ( spherr == NULL ) {
      printf( "spherr is NULL. Was going to print message from %s, %s, %d,"
              " that was: %s\n", file, func,line,message);
      exit(1);
      return CPL_ERROR_NULL_INPUT;
    }
    if ( spherr->nerrs == SPH_ERROR_MAX_MESSAGES ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    spherr->reset = 0;
    if ( file && func ) {
        sprintf( spherr->messages[spherr->nerrs], "[SPH: %s, %s, %d]\n          ",
                 file, func, line);
        vsprintf( temp, form, ap);
        strcat( spherr->messages[spherr->nerrs], temp);
    }
    if ( !file && func ) {
        sprintf( spherr->messages[spherr->nerrs], "[SPH: %s]\n          ",
                 func);
        vsprintf( temp, form, ap);
        strcat( spherr->messages[spherr->nerrs], temp);
    }
    if ( file && !func ) {
        sprintf( spherr->messages[spherr->nerrs], "[SPH: %s, %d]\n          ",
                 file, line);
        vsprintf( temp, form, ap);
        strcat( spherr->messages[spherr->nerrs], temp);
    }
    if ( !file && !func ) {
        sprintf( spherr->messages[spherr->nerrs], "[SPH]\n          ");
        vsprintf( temp, form, ap);
        strcat( spherr->messages[spherr->nerrs], temp);
    }

    spherr->errlevels[spherr->nerrs] = severity;
    if ( spherr->outcpl <= severity ) {
        if ( severity == SPH_ERROR_ERROR ) {
            cpl_msg_error( "SPH", "SPH Error Level: %d, Code: %d, Message: %s\n",
                    severity, code, spherr->messages[spherr->nerrs] );
            cpl_error_set_message_macro(func,code,file,line," ");
        }
        else if ( severity == SPH_ERROR_WARNING ) {
            cpl_msg_warning( "SPH", "SPH Warning Level: %d, Code: %d, Message: %s\n",
                    severity, code, spherr->messages[spherr->nerrs] );
        }
        else {
            cpl_msg_info( "SPH", "SPH Info Level: %d, Code: %d, Message: %s\n",
                    severity, code, spherr->messages[spherr->nerrs] );
        }
    }

    spherr->nerrs = spherr->nerrs + 1;

    va_end(ap);
    cpl_free(form);
    return code;
#endif
}

/*----------------------------------------------------------------------------*/
/**
 @brief        Reset the error handler.

 @return        0 if all ok, or 1 otherwise.

 This function resets the error so that sph_error_get_last_code return
 CPL_ERROR_NONE until a new call tp sph_error_raise has been made.

 */
/*----------------------------------------------------------------------------*/

int sph_error_reset(void) {
#ifndef SPHERE_ERRSYSTEM
    cpl_error_reset();
    return CPL_ERROR_NONE;
#else
    if ( spherr == NULL ) {
        return CPL_ERROR_NULL_INPUT;
    }
    spherr->reset = 1;
    spherr->nerrs = 0;
    cpl_error_reset();
    return 0;
#endif
}

/*----------------------------------------------------------------------------*/
/**
 @brief         Delete the error handler.

 @return        0 if all ok, or 1 otherwise.

 This function deletes the new sph_error object. Not that this frees all the
 memory defined inside the sph_error struct, but NOT the memory allocated to the
 sph_error itspherr. That is, when sph_error_new is called after the delete, the
 same pointer is returned as was passed to the delete function -- except the
 error object is now reinitialised.

 */
/*----------------------------------------------------------------------------*/

int sph_error_delete(void) {
#ifndef SPHERE_ERRSYSTEM
    return CPL_ERROR_NONE;
#else
    int                     rerr = CPL_ERROR_NONE;
    int                     i = 0;

    if ( spherr == NULL ) {
        return CPL_ERROR_NULL_INPUT;
    }
    if ( spherr->errlevels != NULL) {
        for ( i = 0; i < SPH_ERROR_MAX_MESSAGES; i++ ) {
            if ( spherr->messages[i] != NULL ) cpl_free(spherr->messages[i]);
            spherr->messages[i] = NULL;
        }
        cpl_free(spherr->messages);
        if ( spherr->errlevels ) {
            cpl_free(spherr->errlevels);
            spherr->errlevels = NULL;
        }
    }
    cpl_free( spherr );
    spherr = NULL;
    return rerr;
#endif
}
/**@}*/
