/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_image_grid.h"
#include "sph_common_keywords.h"
#include <float.h>
#include <string.h>
#include <math.h>
#include <assert.h>

/*----------------------------------------------------------------------------*/
/**
 @brief Create a new image_grid which is empty.

 @param none

 @return the pointer to the newly created image_grid or NULL

 Description: create a new empty image_grid.

 */
/*----------------------------------------------------------------------------*/
sph_image_grid* sph_image_grid_new( void ) {
    sph_image_grid*     result      = NULL;

    result = cpl_calloc( 1, sizeof(sph_image_grid) );
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return an polygon element in the image_grid.

 @param self the grid to get the element from.
 @param x    the x-coordinate
 @param y    the y-coordinate

 @return the element with the given number in the grid or NULL.

 Description: Return the element with the given number in the grid.

 */
/*----------------------------------------------------------------------------*/
sph_polygon* sph_image_grid_get_element( sph_image_grid* self, int x, int y) {
    int                p = -1;
    if ( !self ) return NULL;

    p = self->nx * y + x;
    if ( p<0 || p>=self->nx * self->ny ) return NULL;
    return self->polys[p];
}
/*----------------------------------------------------------------------------*/
/**
 @brief Create a new image_grid from a cpl_image.

 @param image        the cpl_image to create grid from
 @param minx        the left edge in detector coordinates
 @param maxx        the right edge in detector coordinates
 @param miny        the bottom edge in detector coordinates
 @param maxy        the top edge in detector coordinates

 @return pointer to the new image_grid or NULL

 Description: this is the main "constructor" function for a new
 sph_image_grid, based on a cpl_image. The grid is create with
 rectangular polygons, one for each pixel. The coordinates are
 assigned to each polygon so that the bottom left corner of the
 bottom left pixel-polygon has the coordinates (minx,miny) and
 the top right corner of the top right pixel-polygon has the
 coordinates (maxx, maxy).
 Each polygon is associated with the corresponding pixel value.

 */
/*----------------------------------------------------------------------------*/
sph_image_grid* sph_image_grid_new_from_image( cpl_image* image,
                                               double minx,
                                               double maxx,
                                               double miny,
                                               double maxy ) {
    sph_image_grid*     result      = NULL;
    int                 xx          = 0;
    int                 yy          = 0;
    int                 rej         = 0;
    double              dx          = ( maxx - minx );
    double              dy          = ( maxy - miny );
    sph_error_code      rerr        = CPL_ERROR_NONE;

    if ( !image ) {
        return NULL;
    }
    result = sph_image_grid_new();
    if ( !result ) {
        return NULL;
    }
    result->nx = cpl_image_get_size_x( image );
    result->ny = cpl_image_get_size_y( image );

    dx = dx / result->nx;
    dy = dy / result->ny;

    result->polys = cpl_calloc( result->nx * result->ny, sizeof( sph_polygon* ) );
    result->values = cpl_calloc( result->nx * result->ny, sizeof( double ) );
    if ( !result->polys || !result->values ) {
        sph_image_grid_delete(result);
        return NULL;
    }
    for (yy = 0; yy < result->ny; ++yy) {
        for (xx = 0; xx < result->nx; ++xx) {
            result->polys[ yy * result->nx + xx] = sph_polygon_new();
            if ( !result->polys[ yy * result->nx + xx] ) {
                sph_image_grid_delete( result );
                return NULL;
            }
            rerr |= sph_polygon_add_point( result->polys[ yy * result->nx + xx], xx * dx + minx , yy * dy + miny );
            rerr |= sph_polygon_add_point( result->polys[ yy * result->nx + xx], xx * dx + minx + dx , yy * dy + miny );
            rerr |= sph_polygon_add_point( result->polys[ yy * result->nx + xx], xx * dx + minx + dx, yy * dy + miny + dy);
            rerr |= sph_polygon_add_point( result->polys[ yy * result->nx + xx], xx * dx + minx , yy * dy + miny + dy);
            if ( rerr != CPL_ERROR_NONE ) {
                sph_image_grid_delete( result );
                return NULL;
            }
            result->values[ yy * result->nx + xx ] = cpl_image_get( image, xx + 1, yy + 1, &rej );
        }
    }

    if ( result->boundbox ) {
        sph_polygon_delete( result->boundbox );
    }

    result->boundbox = sph_polygon_new();
    sph_polygon_add_point( result->boundbox, minx, miny );
    sph_polygon_add_point( result->boundbox, maxx, miny );
    sph_polygon_add_point( result->boundbox, maxx, maxy );
    sph_polygon_add_point( result->boundbox, minx, maxy );

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Extract the image from a image_grid

 @param self        the image_grid to extract the image from

 @return a pointer to a new cpl_image with the image data or NULL

 Description: this function extracts the values as a cpl_image
 from the image_grid.

 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_image_grid_extract_image( sph_image_grid* self ) {
    cpl_image* result;
    size_t sz;

    if ( !self ) {
        return NULL;
    }
    if ( !self->values ) {
        return NULL;
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        (void)cpl_error_set_where(cpl_func);
        return NULL;
    }

    assert(self->nx > 0);
    assert(self->ny > 0);

    sz = (size_t)(self->nx * self->ny) * sizeof(*(self->values));
    result = cpl_image_wrap_double(self->nx, self->ny, 
                                   memcpy(cpl_malloc(sz), self->values, sz));

    if (result == NULL) (void)cpl_error_set_where(cpl_func);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Rotate and shfit (in that order!) an image_grid

 @param self        the image_grid to shift and rotate
 @param cx            the x-coordinate of the centre of rotation ( detector coord.)
 @param cy            the y-coordinate of the centre of rotation ( detector coord.)
 @param degrees        the degrees (counter clockwise) to rotate by
 @param dx            the shift in x ( detector coord.)
 @param dy            the shift in y ( detector coord.)

 @return error code of the operation

 Description: This function rotates the image_grid by the given angle (in degrees)
 and afterwards shifts by the amount dx and dy in x and y directions respectively.
 Note that the shift is applied after the rotation. If you want to shift before
 rotation, use this function twice, first with rotation angle 0 and then with
 a shift of 0.

 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_image_grid_rotate_and_shift( sph_image_grid* self, double cx,
                                                double cy, double degrees,
                                                double dx, double dy ) {
    int            xx            = 0;
    int            yy            = 0;
    const double   cosa        = cos( degrees * CPL_MATH_RAD_DEG );
    const double   sina        = sin( degrees * CPL_MATH_RAD_DEG );
    sph_error_code rerr        = CPL_ERROR_NONE;

    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            rerr |= sph_polygon_rotate_around( self->polys[ yy * self->nx + xx ],
                                               cx, cy, cosa, sina, dx, dy);
        }
    }
    rerr |= sph_polygon_rotate_around( self->boundbox, cx, cy, cosa, sina, dx, dy);


    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return x and y index of point

 @param self            the target grid to obtain indices from.
 @param x                x coordinate of point to check
 @param y                y coordinate of point to check
 @param pix                pointer to integer to store result
 @param    piy                pointer to integer to store result

 @return error code of the operation

 Description: This obtains the index in x and y of a given point.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_image_grid_get_xy( sph_image_grid* target, double x, double y, int* pix, int* piy ) {
    double         v1x            = 0.0;
    double         v1y            = 0.0;
    double         v2x            = 0.0;
    double         v2y            = 0.0;
    double        sdx            = 0.0;
    double        sdy            = 0.0;
    double        xd            = 0.0;
    double        yd            = 0.0;
    double        ox            = 0.0;
    double        oy            = 0.0;

    if ( !target || !pix ||!piy ) {
        return CPL_ERROR_NULL_INPUT;
    }
    v1x    = target->boundbox->points[1].x - target->boundbox->points[0].x;
    v1y    = target->boundbox->points[1].y - target->boundbox->points[0].y;
    v2x    = target->boundbox->points[3].x - target->boundbox->points[0].x;
    v2y    = target->boundbox->points[3].y - target->boundbox->points[0].y;

    sdx = v1x * v1x + v1y * v1y;
    sdy = v2x * v2x + v2y * v2y;

    ox =  target->boundbox->points[ 0 ].x;
    oy =  target->boundbox->points[ 0 ].y;

    xd = v1x * ( x - ox ) + v1y * ( y - oy );
    yd = v2x * ( x - ox ) + v2y * ( y - oy );
    xd = ( xd ) / sdx * target->nx;
    yd = ( yd ) / sdy * target->ny;
    *pix = (int)xd;
    *piy = (int)yd;
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Maps a polygon to a grid

 @param self            the target grid to map the polygon to
 @param poly            the polygon to map
 @param    val                the value assigned to the polygon

 @return error code of the operation

 Description: This maps a polygon to the target grid. The value assigned is
 not the total value inside the polygon, but rather a value density. This
 means that mapping the polygon onto a smaller polygon that is completely
 inside the bigger polygon would result in the small polygon to also
 get assigend the value val.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_image_grid_map_polygon( sph_image_grid* target, sph_polygon* poly, double value ) {
    double         v1x            = 0.0;
    double         v1y            = 0.0;
    double         v2x            = 0.0;
    double         v2y            = 0.0;
    double        sdx            = 0.0;
    double        sdy            = 0.0;
    double        xd            = 0.0;
    double        yd            = 0.0;
    int            ii            = 0;
    double        minx        = DBL_MAX;
    double        miny        = DBL_MAX;
    double        maxx        = DBL_MIN;
    double        maxy        = DBL_MIN;
    double         overlap        = 0.0;
    double        area        = 0.0;
    int         xmin        = 0;
    int         ymin        = 0;
    int         xmax        = 0;
    int         ymax        = 0;
    int            yy            = 0;
    int            xx            = 0;
    int            tindex        = 0;
    double        ox            = 0.0;
    double        oy            = 0.0;
    sph_polygon*    tpoly    = NULL;
    sph_polygon_workspace*  ws      = NULL;

    if ( !target || !poly ) {
        return CPL_ERROR_NULL_INPUT;
    }
    if ( poly->npoints < 1 ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    ws = sph_polygon_workspace_new();
    v1x    = target->boundbox->points[1].x - target->boundbox->points[0].x;
    v1y    = target->boundbox->points[1].y - target->boundbox->points[0].y;
    v2x    = target->boundbox->points[3].x - target->boundbox->points[0].x;
    v2y    = target->boundbox->points[3].y - target->boundbox->points[0].y;

    sdx = v1x * v1x + v1y * v1y;
    sdy = v2x * v2x + v2y * v2y;

    ox =  target->boundbox->points[ 0 ].x;
    oy =  target->boundbox->points[ 0 ].y;

    for ( ii = 0; ii < poly->npoints; ++ii ) {
        xd = v1x * ( poly->points[ ii ].x - ox ) + v1y * ( poly->points[ ii ].y - oy );
        yd = v2x * ( poly->points[ ii ].x - ox ) + v2y * ( poly->points[ ii ].y - oy );
        if ( xd < minx ) minx = xd;
        if ( yd < miny ) miny = yd;
        if ( xd > maxx ) maxx = xd;
        if ( yd > maxy ) maxy = yd;
    }
    minx = ( minx )/ sdx * target->nx;
    maxx = ( maxx )/ sdx * target->nx;
    maxy = ( maxy )/ sdy * target->ny;
    miny = ( miny )/ sdy * target->ny;
    xmin = (int)minx - 1;
    xmax = (int)maxx + 1;
    ymin = (int)miny - 1;
    ymax = (int)maxy + 1;
    if ( xmin < 0 ) xmin = 0;
    if ( xmax >= target->nx ) xmax = target->nx - 1;
    if ( ymin < 0 ) ymin = 0;
    if ( ymax >= target->ny ) ymax = target->nx - 1;

    for ( yy = ymin; yy < ymax + 1; ++yy ) {
        for ( xx = xmin; xx < xmax + 1; ++xx ) {
            if ( xx >= 0 && yy >= 0 && xx < target->nx && yy < target->ny ) {
                tindex = yy * target->nx + xx;
                tpoly = target->polys [ tindex ];
                overlap = sph_polygon_calculate_overlap( poly, tpoly,ws );
                area = sph_polygon_area( tpoly );
                target->values[ tindex ] += overlap * value / area;
            }
        }
    }
    sph_polygon_workspace_delete(ws);
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Maps a polygon to a grid

 @param self            the target grid to map the polygon to
 @param poly            the polygon to map
 @param    val                the value assigned to the polygon
 @param    mode            mode to use, can be SPH_IMAGE_GRID_TOTAL,
 SPH_IMAGE_GRID_MIN, SPH_IMAGE_GRID_MAX

 @return error code of the operation

 Description: This maps a polygon to the target grid. The value assigned is
 not the total value inside the polygon. Matching is only true of the target
 grids elements centre point is inside the polygon to map.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_image_grid_map_polygon_midpoint( sph_image_grid* target, sph_polygon* poly, double value, sph_image_grid_av_method method ) {
    double         v1x            = 0.0;
    double         v1y            = 0.0;
    double         v2x            = 0.0;
    double         v2y            = 0.0;
    double        sdx            = 0.0;
    double        sdy            = 0.0;
    double        xd            = 0.0;
    double        yd            = 0.0;
    int            ii            = 0;
    double        minx        = DBL_MAX;
    double        miny        = DBL_MAX;
    double        maxx        = DBL_MIN;
    double        maxy        = DBL_MIN;
    double        cx            = 0.0;
    double        cy            = 0.0;
    int         xmin        = 0;
    int         ymin        = 0;
    int         xmax        = 0;
    int         ymax        = 0;
    int            yy            = 0;
    int            xx            = 0;
    int            tindex        = 0;
    double        ox            = 0.0;
    double        oy            = 0.0;
    sph_polygon*    tpoly    = NULL;

    if ( !target || !poly ) {
        return CPL_ERROR_NULL_INPUT;
    }
    if ( poly->npoints < 1 ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    v1x    = target->boundbox->points[1].x - target->boundbox->points[0].x;
    v1y    = target->boundbox->points[1].y - target->boundbox->points[0].y;
    v2x    = target->boundbox->points[3].x - target->boundbox->points[0].x;
    v2y    = target->boundbox->points[3].y - target->boundbox->points[0].y;

    sdx = v1x * v1x + v1y * v1y;
    sdy = v2x * v2x + v2y * v2y;

    ox =  target->boundbox->points[ 0 ].x;
    oy =  target->boundbox->points[ 0 ].y;

    for ( ii = 0; ii < poly->npoints; ++ii ) {
        xd = v1x * ( poly->points[ ii ].x - ox ) + v1y * ( poly->points[ ii ].y - oy );
        yd = v2x * ( poly->points[ ii ].x - ox ) + v2y * ( poly->points[ ii ].y - oy );
        if ( xd < minx ) minx = xd;
        if ( yd < miny ) miny = yd;
        if ( xd > maxx ) maxx = xd;
        if ( yd > maxy ) maxy = yd;
    }
    minx = ( minx )/ sdx * target->nx;
    maxx = ( maxx )/ sdx * target->nx;
    maxy = ( maxy )/ sdy * target->ny;
    miny = ( miny )/ sdy * target->ny;
    xmin = (int)minx - 1;
    xmax = (int)maxx + 1;
    ymin = (int)miny - 1;
    ymax = (int)maxy + 1;
    if ( xmin < 0 ) xmin = 0;
    if ( xmax >= target->nx ) xmax = target->nx - 1;
    if ( ymin < 0 ) ymin = 0;
    if ( ymax >= target->ny ) ymax = target->nx - 1;

    for ( yy = ymin; yy < ymax + 1; ++yy ) {
        for ( xx = xmin; xx < xmax + 1; ++xx ) {
            if ( xx >= 0 && yy >= 0 && xx < target->nx && yy < target->ny ) {
                tindex = yy * target->nx + xx;
                tpoly = target->polys [ tindex ];
                if ( tpoly->npoints ) {
                    cx = 0.0;
                    cy = 0.0;
                    for (ii = 0; ii < tpoly->npoints; ++ii) {
                        cx = cx + tpoly->points[ii].x;
                        cy = cy + tpoly->points[ii].y;
                    }
                    cx = cx / tpoly->npoints;
                    cy = cy / tpoly->npoints;
                    if ( sph_polygon_test_inside( poly, cx, cy) > 0 ) {
                        if ( method == SPH_IMAGE_GRID_TOTAL ) {
                            target->values[ tindex ] += value;
                        }
                        if ( method == SPH_IMAGE_GRID_MIN ) {
                            if ( value < target->values[ tindex ] ) {
                                target->values[ tindex ] = value;
                            }
                        }
                        if ( method == SPH_IMAGE_GRID_MAX ) {
                            if ( value > target->values[ tindex ] ) {
                                target->values[ tindex ] = value;
                            }
                        }
                    }
                }
            }
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Maps a polygon to a grid

 @param self            the target grid to map the polygon to
 @param poly            the polygon to map
 @param    val                the value assigned to the polygon

 @return weighed overlap value

 Description: This maps all polygons overlapping the target from the source grid.
 The value returned is the weighed overlap value. On error, DBL_MAX is returned.

 */
/*----------------------------------------------------------------------------*/
double
sph_image_grid_map_to_polygon( sph_image_grid* source, sph_polygon* targetpoly ) {
    double         v1x            = 0.0;
    double         v1y            = 0.0;
    double         v2x            = 0.0;
    double         v2y            = 0.0;
    double        sdx            = 0.0;
    double        sdy            = 0.0;
    double        xd            = 0.0;
    double        yd            = 0.0;
    int            ii            = 0;
    double        minx        = DBL_MAX;
    double        miny        = DBL_MAX;
    double        maxx        = DBL_MIN;
    double        maxy        = DBL_MIN;
    double         overlap        = 0.0;
    double        area        = 0.0;
    int         xmin        = 0;
    int         ymin        = 0;
    int         xmax        = 0;
    int         ymax        = 0;
    int            yy            = 0;
    int            xx            = 0;
    int            tindex        = 0;
    double        ox            = 0.0;
    double        oy            = 0.0;
    sph_polygon*    tpoly    = NULL;
    double         result        = 0.0;
    double        value        = 0.0;
    sph_polygon_workspace*    ws    = NULL;

    if ( !source || !targetpoly ) {
        return DBL_MAX;
    }
    if ( targetpoly->npoints < 1 ) {
        return DBL_MAX;
    }

    ws = sph_polygon_workspace_new();
    v1x    = source->boundbox->points[1].x - source->boundbox->points[0].x;
    v1y    = source->boundbox->points[1].y - source->boundbox->points[0].y;
    v2x    = source->boundbox->points[3].x - source->boundbox->points[0].x;
    v2y    = source->boundbox->points[3].y - source->boundbox->points[0].y;

    sdx = v1x * v1x + v1y * v1y;
    sdy = v2x * v2x + v2y * v2y;

    ox =  source->boundbox->points[ 0 ].x;
    oy =  source->boundbox->points[ 0 ].y;

    for ( ii = 0; ii < targetpoly->npoints; ++ii ) {
        xd = v1x * ( targetpoly->points[ ii ].x - ox ) + v1y * ( targetpoly->points[ ii ].y - oy );
        yd = v2x * ( targetpoly->points[ ii ].x - ox ) + v2y * ( targetpoly->points[ ii ].y - oy );
        if ( xd < minx ) minx = xd;
        if ( yd < miny ) miny = yd;
        if ( xd > maxx ) maxx = xd;
        if ( yd > maxy ) maxy = yd;
    }
    minx = ( minx )/ sdx * source->nx;
    maxx = ( maxx )/ sdx * source->nx;
    maxy = ( maxy )/ sdy * source->ny;
    miny = ( miny )/ sdy * source->ny;
    xmin = (int)minx - 1;
    xmax = (int)maxx + 1;
    ymin = (int)miny - 1;
    ymax = (int)maxy + 1;
    if ( xmin < 0 ) xmin = 0;
    if ( xmax >= source->nx ) xmax = source->nx - 1;
    if ( ymin < 0 ) ymin = 0;
    if ( ymax >= source->ny ) ymax = source->nx - 1;

    for ( yy = ymin; yy < ymax + 1; ++yy ) {
        for ( xx = xmin; xx < xmax + 1; ++xx ) {
            if ( xx >= 0 && yy >= 0 && xx < source->nx && yy < source->ny ) {
                tindex = yy * source->nx + xx;
                tpoly = source->polys [ tindex ];
                overlap = sph_polygon_calculate_overlap( targetpoly, tpoly,ws );
                area += overlap;
                value = source->values[ tindex ];
                result += overlap * value;
            }
        }
    }
    if ( area ) {
        result = result / area;
    }
    else result = 0.0;
    sph_polygon_workspace_delete(ws);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Mark a polygon at a given point

 @param        self         the image_grid to mark a point in
 @param        x            the x-coordinates to mark a point in
 @param        y            the y-coordinates to mark a point in
 @param        val            the value to mark with

 @return error code of the operation

 Description: MArks a polygon at the specified place with the given value.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_image_grid_mark_point( sph_image_grid* self, double x, double y, double val ) {
    sph_error_code        rerr        = CPL_ERROR_NONE;
    int                    xx            = 0;
    int                    yy            = 0;

    if ( sph_polygon_test_inside( self->boundbox, x, y ) != 1 ) {
        return CPL_ERROR_INCOMPATIBLE_INPUT;
    }

    rerr = sph_image_grid_get_xy( self, x, y, &xx, &yy );
    if ( rerr != CPL_ERROR_NONE ) {
        return rerr;
    }
    self->values[ yy * self->nx + xx ] = val;

    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Map a image_grid to another target image_grid.

 @param self        the image_grid to map
 @param    target        the image_grid to map to

 @return error code of the operation

 Description: Maps the given grid to the target grid. The mapping is performed
 using the GIRMOS mapping.

 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_image_grid_map_to_target( sph_image_grid* self,
                                             sph_image_grid* target ) {
    int                 xx          = 0;
    int                 yy          = 0;
    int                 sindex       = 0;
    sph_polygon*        spoly       = NULL;
    double                result        = 0.0;

    if ( !self || !target ) {
        return CPL_ERROR_NULL_INPUT;
    }

    for (yy = 0; yy < target->ny; ++yy) {
        for (xx = 0; xx < target->nx; ++xx) {
            sindex = yy * target->nx + xx;
            spoly = target->polys[ sindex ];
            if ( !spoly ) {
                return CPL_ERROR_ILLEGAL_INPUT;
            }
            if ( spoly->npoints < 1 ) {
                return CPL_ERROR_ILLEGAL_INPUT;
            }
            result = sph_image_grid_map_to_polygon( self, spoly );
            if ( result == DBL_MAX ) {
                return SPH_ERROR_PREVIOUS_OP_FAILED;
            }
            target->values[ sindex ] = result;
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Delete the image_grid and free resources.

 @param self        the sph_image_grid to delete

 @return nothinh

 Description: Does just that.

 */
/*----------------------------------------------------------------------------*/
void sph_image_grid_delete( sph_image_grid* self) {
    int             ii      = 0;

    if ( self ) {
        if ( self->polys ) {
            for ( ii = 0; ii < self->nx * self->ny; ++ii ) {
                sph_polygon_delete( self->polys[ii] );
            }
        }
        cpl_free( self->polys );
        if ( self->values ) {
            cpl_free( self->values );
        }
        if ( self->boundbox ) {
            sph_polygon_delete( self->boundbox );
        }
        cpl_free( self );
    }
}
