/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*-------------------------------------------------------------------------*/
/**
   @file    sph_iniparser.h
   @author  N. Devillard
   @date    Sep 2007
   @version 3.0
   @brief   Parser for ini files.
*/
/*--------------------------------------------------------------------------*/

/*
 $Id: sph_iniparser.h,v 1.24 2007-11-23 21:38:19 ndevilla Exp $
 $Revision: 1.24 $
*/

#ifndef SPH_INIPARSER_H_
#define SPH_INIPARSER_H_

/*---------------------------------------------------------------------------
           Includes
 ---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*
 * The following #include is necessary on many Unixes but not Linux.
 * It is not needed for Windows platforms.
 * Uncomment it if needed.
 */
/* #include <unistd.h> */

#include "sph_dictionary.h"

/*---------------------------------------------------------------------------
           Macros
 ---------------------------------------------------------------------------*/
/** For backwards compatibility only */
#define sph_iniparser_getstr(d, k)  sph_iniparser_getstring(d, k, NULL)
#define sph_iniparser_setstr        sph_iniparser_setstring

/*-------------------------------------------------------------------------*/
/**
  @brief    Get number of sections in a sph_dictionary
  @param    d   Dictionary to examine
  @return   int Number of sections found in sph_dictionary

  This function returns the number of sections found in a sph_dictionary.
  The test to recognize sections is done on the string stored in the
  sph_dictionary: a section name is given as "section" whereas a key is
  stored as "section:key", thus the test looks for entries that do not
  contain a colon.

  This clearly fails in the case a section name contains a colon, but
  this should simply be avoided.

  This function returns -1 in case of error.
 */
/*--------------------------------------------------------------------------*/

int sph_iniparser_getnsec(sph_dictionary * d);


/*-------------------------------------------------------------------------*/
/**
  @brief    Get name for section n in a sph_dictionary.
  @param    d   Dictionary to examine
  @param    n   Section number (from 0 to nsec-1).
  @return   Pointer to char string

  This function locates the n-th section in a sph_dictionary and returns
  its name as a pointer to a string statically allocated inside the
  sph_dictionary. Do not free or modify the returned string!

  This function returns NULL in case of error.
 */
/*--------------------------------------------------------------------------*/

char * sph_iniparser_getsecname(sph_dictionary * d, int n);


/*-------------------------------------------------------------------------*/
/**
  @brief    Save a sph_dictionary to a loadable ini file
  @param    d   Dictionary to dump
  @param    f   Opened file pointer to dump to
  @return   void

  This function dumps a given sph_dictionary into a loadable ini file.
  It is Ok to specify @c stderr or @c stdout as output files.
 */
/*--------------------------------------------------------------------------*/

void sph_iniparser_dump_ini(sph_dictionary * d, FILE * f);

/*-------------------------------------------------------------------------*/
/**
  @brief    Dump a sph_dictionary to an opened file pointer.
  @param    d   Dictionary to dump.
  @param    f   Opened file pointer to dump to.
  @return   void

  This function prints out the contents of a sph_dictionary, one element by
  line, onto the provided file pointer. It is OK to specify @c stderr
  or @c stdout as output files. This function is meant for debugging
  purposes mostly.
 */
/*--------------------------------------------------------------------------*/
void sph_iniparser_dump(sph_dictionary * d, FILE * f);

/*-------------------------------------------------------------------------*/
/**
  @brief    Get the string associated to a key
  @param    d       Dictionary to search
  @param    key     Key string to look for
  @param    def     Default value to return if key not found.
  @return   pointer to statically allocated character string

  This function queries a sph_dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the pointer passed as 'def' is returned.
  The returned char pointer is pointing to a string allocated in
  the sph_dictionary, do not free or modify it.
 */
/*--------------------------------------------------------------------------*/
const char * sph_iniparser_getstring(const sph_dictionary * d,
                                     const char * key,
                                     const char * def)
    CPL_ATTR_NONNULL;

/*-------------------------------------------------------------------------*/
/**
  @brief    Get the string associated to a key, convert to an int
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   integer

  This function queries a sph_dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.

  Supported values for integers include the usual C notation
  so decimal, octal (starting with 0) and hexadecimal (starting with 0x)
  are supported. Examples:

  - "42"      ->  42
  - "042"     ->  34 (octal -> decimal)
  - "0x42"    ->  66 (hexa  -> decimal)

  Warning: the conversion may overflow in various ways. Conversion is
  totally outsourced to strtol(), see the associated man page for overflow
  handling.

  Credits: Thanks to A. Becker for suggesting strtol()
 */
/*--------------------------------------------------------------------------*/
int sph_iniparser_getint(const sph_dictionary * d,
                         const char * key,
                         int notfound)
    CPL_ATTR_NONNULL;

/*-------------------------------------------------------------------------*/
/**
  @brief    Get the string associated to a key, convert to a double
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   double

  This function queries a sph_dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.
 */
/*--------------------------------------------------------------------------*/
double sph_iniparser_getdouble(const sph_dictionary * d,
                               const char * key,
                               double notfound)
    CPL_ATTR_NONNULL;

/*-------------------------------------------------------------------------*/
/**
  @brief    Get the string associated to a key, convert to a boolean
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   integer

  This function queries a sph_dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.

  A true boolean is found if one of the following is matched:

  - A string starting with 'y'
  - A string starting with 'Y'
  - A string starting with 't'
  - A string starting with 'T'
  - A string starting with '1'

  A false boolean is found if one of the following is matched:

  - A string starting with 'n'
  - A string starting with 'N'
  - A string starting with 'f'
  - A string starting with 'F'
  - A string starting with '0'

  The notfound value returned if no boolean is identified, does not
  necessarily have to be 0 or 1.
 */
/*--------------------------------------------------------------------------*/
int sph_iniparser_getboolean(const sph_dictionary * d,
                             const char * key,
                             int notfound)
    CPL_ATTR_NONNULL;


/*-------------------------------------------------------------------------*/
/**
  @brief    Set an entry in a sph_dictionary.
  @param    ini     Dictionary to modify.
  @param    entry   Entry to modify (entry name)
  @param    val     New value to associate to the entry.
  @return   int 0 if Ok, -1 otherwise.

  If the given entry can be found in the sph_dictionary, it is modified to
  contain the provided value. If it cannot be found, -1 is returned.
  It is Ok to set val to NULL.
 */
/*--------------------------------------------------------------------------*/
int sph_iniparser_setstring(sph_dictionary * ini,
                            const char * entry,
                            char * val)
    CPL_ATTR_NONNULL;


/*-------------------------------------------------------------------------*/
/**
  @brief    Delete an entry in a sph_dictionary
  @param    ini     Dictionary to modify
  @param    entry   Entry to delete (entry name)
  @return   void

  If the given entry can be found, it is deleted from the sph_dictionary.
 */
/*--------------------------------------------------------------------------*/
void sph_iniparser_unset(sph_dictionary * ini, const char * entry)
    CPL_ATTR_NONNULL;

/*-------------------------------------------------------------------------*/
/**
  @brief    Finds out if a given entry exists in a sph_dictionary
  @param    ini     Dictionary to search
  @param    entry   Name of the entry to look for
  @return   integer 1 if entry exists, 0 otherwise

  Finds out if a given entry exists in the sph_dictionary. Since sections
  are stored as keys with NULL associated values, this is the only way
  of querying for the presence of sections in a sph_dictionary.
 */
/*--------------------------------------------------------------------------*/
int sph_iniparser_find_entry(const sph_dictionary * ini,
                             const char * entry)
    CPL_ATTR_NONNULL;

/*-------------------------------------------------------------------------*/
/**
  @brief    Parse an ini file and return an allocated sph_dictionary object
  @param    ininame Name of the ini file to read.
  @return   Pointer to newly allocated sph_dictionary

  This is the parser for ini files. This function is called, providing
  the name of the file to be read. It returns a sph_dictionary object that
  should not be accessed directly, but through accessor functions
  instead.

  The returned sph_dictionary must be freed using sph_iniparser_freedict().
 */
/*--------------------------------------------------------------------------*/
sph_dictionary * sph_iniparser_load(const char * ininame)
    CPL_ATTR_NONNULL;

/*-------------------------------------------------------------------------*/
/**
  @brief    Set an entry in a sph_dictionary.
  @param    ini     Dictionary to modify.
  @param    entry   Entry to modify (entry name)
  @param    val     New value to associate to the entry.
  @return   int 0 if Ok, -1 otherwise.

  If the given entry can be found in the sph_dictionary, it is modified to
  contain the provided value. If it cannot be found, -1 is returned.
  It is Ok to set val to NULL.
 */
/*--------------------------------------------------------------------------*/
int sph_iniparser_set(sph_dictionary * ini,
                      const char * entry,
                      const char * val);
/*-------------------------------------------------------------------------*/
/**
  @brief    Free all memory associated to an ini sph_dictionary
  @param    d Dictionary to free
  @return   void

  Free all memory associated to an ini sph_dictionary.
  It is mandatory to call this function before the sph_dictionary object
  gets out of the current context.
 */
/*--------------------------------------------------------------------------*/
void sph_iniparser_freedict(sph_dictionary * d);

#endif
