/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_ird_instrument_model.h"
#include "sph_error.h"
#include "sph_iniparser.h"
#include "sph_image_grid.h"
#include "sph_filemanager.h"
#include "sph_keyword_manager.h"
#include "sph_common_keywords.h"

#include <strings.h>
#include "sph_dictionary.h"
#include <math.h>
#include <string.h>

const double SPH_IRD_INSTRUMENT_MODEL_DEROT_OFFSETSKY = 0.36;
const double SPH_IRD_INSTRUMENT_MODEL_DEROT_OFFSETELEV = 134.1;

const double SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS     = 18.0;
const double SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_X         = 2048;
const double SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_Y         = 1024;
const double SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_MICRONS     = 18.0 * 2048.0;
const char* SPH_IRD_INSTRUMENT_MODEL_SECNAME                = "ESO DRS IRD MODEL";
const char* SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME            = "ESO DRS IRD PIX SIZE";
const char* SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME            = "ESO DRS IRD DET PIX SIZE X";
const char* SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME            = "ESO DRS IRD DET PIX SIZE Y";
const char* SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_X_NAME    = "ESO DRS IRD ZERO OFF X";
const char* SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_Y_NAME    = "ESO DRS IRD ZERO OFF Y";
const char* SPH_IRD_INSTRUMENT_MODEL_SPLIT_PIXEL_X_NAME    = "ESO DRS IRD SPLIT X";
const char* SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y_NAME    = "ESO DRS IRD ZERO WND POS Y";
const char* SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y_NAME    = "ESO DRS IRD WND SIZE Y";
const char* SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH_NAME    = "ESO DRS IRD SPEC LENGTH";
const char* SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH_NAME    = "ESO DRS IRD SPEC WIDTH";
const char* SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA_NAME    = "ESO DRS IRD WAVEL MAX";
const char* SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA_NAME    = "ESO DRS IRD WAVEL MIN";
const sph_error_code SPH_IRD_INSTRUMENT_MODEL_GENERAL         = SPH_IRD_INSTRUMENT_MODEL_ERR_START + 0;
const sph_error_code SPH_IRD_INSTRUMENT_MODEL_OUTSIDE_ARRAY = SPH_IRD_INSTRUMENT_MODEL_ERR_START + 1;
const double SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_Y = 8.0;
const double SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_X = 2.0;
const int SPH_IRD_INSTRUMENT_MODEL_SPLIT_PIXEL_X    = 1024;
const int SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y    = 0;
const int SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y    = 1024;
const int SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH    = 100;
const int SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH    = 898;
const double SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA    = 0.95;
const double SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA    = 2.05;

/*----------------------------------------------------------------------------*/
/**
 @brief Create a new default IFS lenslet model


 @return the new IFS lens model or NULL in case of error

 Description: This function creates a new IFS lens model in the standard
 default set-up. The parameters for the standard set-up are from the data sheet
 as found in document VLT-TRE-SPH-14690-0201, the IFS overview, version 1.0

 */
/*----------------------------------------------------------------------------*/
sph_ird_instrument_model* sph_ird_instrument_model_new(void) {
    sph_ird_instrument_model*        irdmodel        = NULL;
    irdmodel = cpl_calloc( 1, sizeof(sph_ird_instrument_model) );
    if ( irdmodel ) {
        irdmodel->detsize_microns = SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_MICRONS;
        irdmodel->detsize_pixels_x = SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_X;
        irdmodel->detsize_pixels_y = SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_Y;
        irdmodel->pixsize_microns = SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS;
        irdmodel->zero_offsetx = SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_X;
        irdmodel->zero_offsety = SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_Y;
        irdmodel->split_pixel_x = SPH_IRD_INSTRUMENT_MODEL_SPLIT_PIXEL_X;
        irdmodel->window_size_y = SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y;
        irdmodel->zero_window_start_y = SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y;
        irdmodel->speclength = SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH; // For LRS mode
        irdmodel->specwidth = SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH;
        irdmodel->minlambda = SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA;
        irdmodel->maxlambda = SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA;
        irdmodel->window_mode = 0;
        irdmodel->window_mode_bl = 0;
        irdmodel->window_mode_tr = 1024;
    }
    return irdmodel;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Create a new default IFS lenslet model


 @return the new IFS lens model or NULL in case of error

 Description: This function creates a new IFS lens model in the standard
 default set-up. The parameters for the standard set-up are from the data sheet
 as found in document VLT-TRE-SPH-14690-0201, the IFS overview, version 1.0

 */
/*----------------------------------------------------------------------------*/
sph_ird_instrument_model* sph_ird_instrument_model_new_small(void) {
    sph_ird_instrument_model*        irdmodel        = NULL;
    irdmodel = cpl_calloc( 1, sizeof(sph_ird_instrument_model) );
    if ( irdmodel ) {
        irdmodel->detsize_pixels_x = 256;
        irdmodel->detsize_pixels_y = 128;
        irdmodel->pixsize_microns = SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS;
        irdmodel->detsize_microns = irdmodel->detsize_pixels_x * irdmodel->pixsize_microns;
        irdmodel->zero_offsetx = 0;
        irdmodel->zero_offsety = 0;
        irdmodel->split_pixel_x = irdmodel->detsize_pixels_x / 2;
        irdmodel->window_size_y = irdmodel->split_pixel_x;
        irdmodel->zero_window_start_y = 0;
        irdmodel->speclength = SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH / 8; // For LRS mode
        irdmodel->specwidth = SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH / 8;
        irdmodel->minlambda = SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA;
        irdmodel->maxlambda = SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA;
        irdmodel->window_mode = 0;
        irdmodel->window_mode_bl = 0;
        irdmodel->window_mode_tr = 1024;
    }
    return irdmodel;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Create a new IFS lenslet model from cpl propertylist

 @param the propertylist

 @return pointer to the new lenslet model or NULL in case of error.

 Description: This creates a new lenslet model, getting the data from a cpl
 propertylist. The propertylist must contain all the important parameters:
 <ul>
   <li>SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME             = "PIXEL SIZE"</li>
   <li>SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_NAME             = "DETECTOR PIXEL SIZE"</li>
   <li>SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y_NAME         = "WINDOW SIZE Y"</li>
   <li>SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y_NAME = "ZERO WINDOW START Y"</li>
   <li>SPH_IRD_INSTRUMENT_MODEL_PIXEL_SPLIT_X_NAME       = "SPLIT PIXEL X"</li>
 </ul>

 */
/*----------------------------------------------------------------------------*/
sph_ird_instrument_model* sph_ird_instrument_model_new_from_propertylist( const cpl_propertylist* plist ) {
    sph_ird_instrument_model*        model        = NULL;
    int                   ival      = 0;
    double                dval      = 0.0;

    if ( !plist ) {
        return NULL;
    }
    model = sph_ird_instrument_model_new();
    if ( model ) {
        dval = cpl_propertylist_get_double( plist, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME );
        if ( dval ) model->pixsize_microns = dval;
        ival = cpl_propertylist_get_int( plist, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME );
        if ( ival ) {
            model->detsize_pixels_x = ival;
            model->detsize_microns = ival * model->pixsize_microns;
        }
        ival = cpl_propertylist_get_int( plist, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME );
        if ( ival ) {
            model->detsize_pixels_y = ival;
        }
        dval = cpl_propertylist_get_double( plist, SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_X_NAME );
        if ( dval ) model->zero_offsetx = dval;
        dval = cpl_propertylist_get_double( plist, SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_Y_NAME );
        if ( dval ) model->zero_offsety = dval;
        ival = cpl_propertylist_get_int( plist, SPH_IRD_INSTRUMENT_MODEL_SPLIT_PIXEL_X_NAME );
        if ( ival ) model->split_pixel_x = ival;
        ival = cpl_propertylist_get_int( plist, SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y_NAME );
        if ( ival ) model->window_size_y = ival;
        ival = cpl_propertylist_get_int( plist, SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y_NAME );
        if ( ival ) model->zero_window_start_y = ival;
        dval = cpl_propertylist_get_double( plist, SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA_NAME );
        if ( dval ) model->minlambda = dval;
        dval = cpl_propertylist_get_double( plist, SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA_NAME );
        if ( dval ) model->maxlambda = dval;
        ival = cpl_propertylist_get_int( plist, SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH_NAME );
        if ( ival ) model->speclength = ival;
        ival = cpl_propertylist_get_int( plist, SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH_NAME );
        if ( ival ) model->specwidth = ival;
        cpl_error_reset();
    }
    return model;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Load a new IFS lenslet model from a ASCII ini file

 @param the filename of the ASCII file

 @return pointer to the new lenslet model or NULL in case of error.

 Description: This creates a new lenslet model, loading the data from an ASCII
 file that is in an ini keyword = value type file. The section has to be entitled
 [ LENSLET MODEL ].

 */
/*----------------------------------------------------------------------------*/
sph_ird_instrument_model* sph_ird_instrument_model_load( const char* czFilename ) {
    sph_ird_instrument_model*        model        = NULL;
    sph_dictionary * ini        = NULL;
    char                  keyup[2048];
    int                   ival      = 0;
    double                dval      = 0.0;
    char                    ext[256];
    char                    base[1256];
    cpl_propertylist*       plist = NULL;

    plist = sph_keyword_manager_load_properties(czFilename,0);
    if ( plist ) {
        model = sph_ird_instrument_model_new_from_propertylist(plist);
        cpl_propertylist_delete(plist); plist = NULL;
        if ( ! model ) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
                    "Could not load FITS info "
                    "for IRDIS instrument model from file %s",czFilename);
            return NULL;
        }
        return model;
    }

    cpl_error_reset();
#ifdef SPH_NO_ASCII
    SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
            "Could not load header info "
            "for IRDIS instrument model from file %s",czFilename);
    return NULL;
#endif

    sph_filemanager_split(czFilename,base,ext);

    model = sph_ird_instrument_model_new();
    if ( model ) {
        ini = sph_iniparser_load( czFilename );
        if ( !sph_iniparser_find_entry( ini, SPH_IRD_INSTRUMENT_MODEL_SECNAME ) ) {
            sph_ird_instrument_model_delete( model );
            return NULL;
        }
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->pixsize_microns = dval;

        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) {
            model->detsize_pixels_x = ival;
            model->detsize_microns = ival * model->pixsize_microns;
        }
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) {
            model->detsize_pixels_y = ival;
        }

        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_X_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->zero_offsetx = dval;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_Y_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->zero_offsety = dval;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_SPLIT_PIXEL_X_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) model->split_pixel_x = ival;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) model->zero_window_start_y = ival;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) model->window_size_y = ival;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) model->speclength = ival;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) model->specwidth = ival;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->minlambda = dval;
        sprintf( keyup, "%s:%s",SPH_IRD_INSTRUMENT_MODEL_SECNAME, SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->maxlambda = dval;
        sph_iniparser_freedict(ini);
    }
    return model;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Save the lenslet model to a ASCII file in ini format.

 @param self        the lenslet model to save
 @param czFilename    the filename to save to

 @return error code of the operation

 Description: Saves the lenslet model to a ASCII file in .ini style format.
 The file is created, existing files are overwritten. The file contains
 only one section and under that all the parameters for the model in the
 key = value format.

 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_ird_instrument_model_save( const sph_ird_instrument_model* self,
                                              const char* czFilename ) {
    sph_error_code            rerr        = CPL_ERROR_NONE;
    FILE*                    pFile        = NULL;

    cpl_ensure_code(self       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(czFilename != NULL, CPL_ERROR_NULL_INPUT);

    pFile = fopen( czFilename, "w" );

    cpl_ensure_code(pFile      != NULL, CPL_ERROR_ASSIGNING_STREAM);

    if (fprintf( pFile, "[ %s ]\n", SPH_IRD_INSTRUMENT_MODEL_SECNAME ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME,
                 self->detsize_pixels_x ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME,
                 self->detsize_pixels_y ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME,
                 self->pixsize_microns ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_X_NAME,
                 self->zero_offsetx ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_Y_NAME,
                 self->zero_offsety ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y_NAME,
                 self->window_size_y ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y_NAME,
                 self->zero_window_start_y ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IRD_INSTRUMENT_MODEL_SPLIT_PIXEL_X_NAME,
                 self->split_pixel_x ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH_NAME,
                 self->speclength ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH_NAME,
                 self->specwidth ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA_NAME,
                 self->minlambda ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA_NAME,
                 self->maxlambda ) <= 0) {
        rerr = cpl_error_set(cpl_func, CPL_ERROR_FILE_IO);
    } 

    cpl_ensure_code(fclose( pFile ) == 0, CPL_ERROR_FILE_IO);

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Check if a pixel is inside the illuminated left area
 * @param self
 * @param the pixel in detector coordinates
 * @param the pixel in detector coordinates
 * @return 0 if pixel is outside, 1 if inside, -1 on error.
 *
 */
/*----------------------------------------------------------------------------*/
int sph_ird_instrument_model_pixel_inside_left(sph_ird_instrument_model* self,int xx,int yy) {
    double          centx = 0.0;
    double          centy = 0.0;
    double          rad = 0.0;

    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);
    centx = self->split_pixel_x / 2.0;
    centy = self->window_size_y / 2.0 + self->zero_window_start_y;
    rad = centx * centx;
    if ( ( xx - centx ) * ( xx - centx ) + ( yy -centy ) * ( yy -centy) < rad ) {
        return 1;
    }
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Check if a pixel is inside the illuminated left area
 * @param self
 * @param the pixel in detector coordinates
 * @param the pixel in detector coordinates
 * @return 0 if pixel is outside, 1 if inside, -1 on error.
 *
 */
/*----------------------------------------------------------------------------*/
int sph_ird_instrument_model_pixel_inside_right(sph_ird_instrument_model* self,int xx,int yy) {
    double          centx = 0.0;
    double          centy = 0.0;
    double          rad = 0.0;

    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);
    centx = self->split_pixel_x / 2.0 * 3.0;
    centy = self->window_size_y / 2.0 + self->zero_window_start_y;
    rad = self->split_pixel_x / 2.0;
    rad = rad * rad;
    if ( ( xx - centx ) * ( xx - centx ) + ( yy -centy ) * ( yy -centy) < rad ) {
        return 1;
    }
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the centre pixel coordinate of left FOV.

 @param self            the instrument model
 @param x               the central pixel coordinate (output)
 @param y               the central pixel coordinate (output)
 @return error code of the operation.

 Description: Returns the central pixel coordinates in the variables x and y.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_get_centre_left(
        const sph_ird_instrument_model* self,
        double* x,
        double* y)
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(x,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(y,CPL_ERROR_NULL_INPUT);
    *x = self->split_pixel_x / 2.0;
    *y = self->window_size_y / 2.0;
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 @brief Return the centre pixel coordinate of right FOV.

 @param self            the instrument model
 @param x               the central pixel coordinate (output)
 @param y               the central pixel coordinate (output)
 @return error code of the operation.

 Description: Returns the central pixel coordinates in the variables x and y.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_get_centre_right(
        const sph_ird_instrument_model* self,
        double* x,
        double* y)
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(x,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(y,CPL_ERROR_NULL_INPUT);
    *x = self->split_pixel_x + self->split_pixel_x / 2.0;
    *y = self->window_size_y / 2.0;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the centre pixel coordinate of left FOV in subframe coords.

 @param self            the instrument model
 @param x               the central pixel coordinate (output)
 @param y               the central pixel coordinate (output)
 @return error code of the operation.

 Description: Returns the central pixel coordinates in the variables x and y.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_get_centre_left_subframe_coords(
        const sph_ird_instrument_model* self,
        double* x,
        double* y)
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(x,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(y,CPL_ERROR_NULL_INPUT);

    if ( self->window_mode ) {
        *x = (self->window_mode_tr - self->window_mode_bl) / 2;
        *y = (self->window_mode_tr - self->window_mode_bl) / 2;
    }
    else {
        *x = self->split_pixel_x / 2.0;
        *y = self->window_size_y / 2.0;
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return the centre pixel coordinate of right FOV in subframe coords.

 @param self            the instrument model
 @param x               the central pixel coordinate (output)
 @param y               the central pixel coordinate (output)
 @return error code of the operation.

 Description: Returns the central pixel coordinates in the variables x and y.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_get_centre_right_subframe_coords(
        const sph_ird_instrument_model* self,
        double* x,
        double* y)
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(x,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(y,CPL_ERROR_NULL_INPUT);
    if ( self->window_mode ) {
        *x = (self->window_mode_tr - self->window_mode_bl) / 2;
        *y = (self->window_mode_tr - self->window_mode_bl) / 2;
    }
    else {
        *x = self->split_pixel_x / 2.0;
        *y = self->window_size_y / 2.0;
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the instrument lenslet model as a propertylist.

 @param self            the lenslet model

 @return    the propertylist or NULL in case of error.

 Description: This creates a new propertylist containing all the properties of the
 lenslet model.

 */
/*----------------------------------------------------------------------------*/
cpl_propertylist* sph_ird_instrument_model_get_as_propertylist( sph_ird_instrument_model* self ) {
    cpl_propertylist*        plist        = NULL;
    sph_error_code            rerr        = 0;

    if ( !self ) {
        return NULL;
    }
    plist = cpl_propertylist_new();
    if ( plist ) {
        rerr |= cpl_propertylist_append_int( plist, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME, self->detsize_pixels_x );
        rerr |= cpl_propertylist_append_int( plist, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME, self->detsize_pixels_y );
        rerr |= cpl_propertylist_append_double( plist, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME, self->pixsize_microns );
        rerr |= cpl_propertylist_append_double( plist, SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_X_NAME, self->zero_offsetx );
        rerr |= cpl_propertylist_append_double( plist, SPH_IRD_INSTRUMENT_MODEL_ZERO_OFFSET_Y_NAME, self->zero_offsety );
        rerr |= cpl_propertylist_append_int( plist, SPH_IRD_INSTRUMENT_MODEL_SPLIT_PIXEL_X_NAME, self->split_pixel_x );
        rerr |= cpl_propertylist_append_int( plist, SPH_IRD_INSTRUMENT_MODEL_WINDOW_SIZE_Y_NAME, self->window_size_y );
        rerr |= cpl_propertylist_append_int( plist, SPH_IRD_INSTRUMENT_MODEL_ZERO_WINDOW_START_Y_NAME, self->zero_window_start_y );
        rerr |= cpl_propertylist_append_int( plist, SPH_IRD_INSTRUMENT_MODEL_SPECLENGTH_NAME, self->speclength );
        rerr |= cpl_propertylist_append_int( plist, SPH_IRD_INSTRUMENT_MODEL_SPECWIDTH_NAME, self->specwidth );
        rerr |= cpl_propertylist_append_double( plist, SPH_IRD_INSTRUMENT_MODEL_MIN_LAMBDA_NAME, self->minlambda );
        rerr |= cpl_propertylist_append_double( plist, SPH_IRD_INSTRUMENT_MODEL_MAX_LAMBDA_NAME, self->maxlambda );
    }
    return plist;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get a DBI FOV mask
 * @param irdis model
 * @param border        the border width
 * @return
 *
 * This function returns a DBI FoB mask. The DBI FoV is defined in sub-frame
 * coordinates as the regions inside a circle centered on the _left_ FoV
 * coordinates with a radius equal to half the subframe window size minus the
 * border parameter.
 *
 * The returned mask has the size of a IRDIS sub window.
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
sph_ird_instrument_model_get_mask_dbi_fov( sph_ird_instrument_model* self,
                                           double border ) {
    cpl_mask*       mask = NULL;
    int             xx    = 0;
    int             yy    = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    mask = cpl_mask_new(self->split_pixel_x, self->window_size_y);
    for (yy = 0; yy < self->window_size_y; ++yy) {
        for (xx = 0; xx < self->split_pixel_x; ++xx) {
            if ( sph_ird_instrument_model_check_in_dbi_fov(self, xx, yy, 1,
                                                           border) ) {
                cpl_mask_set(mask,xx+1,yy+1,0);
            }
            else {
                cpl_mask_set(mask,xx+1,yy+1,1);
            }
        }
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return mask;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a mask for the IRDIS field of view right window from model
 * @param the IRDIS model
 *
 * @return a new cpl_mask
 *
 * This creates a new mask that is of the full readout size but masks out the
 * left window.
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
sph_ird_instrument_model_get_mask_rightwin(const sph_ird_instrument_model* self)
{
    cpl_mask* mask = cpl_mask_new(self->detsize_pixels_x,
                                  self->detsize_pixels_y);
    cpl_binary* mptr = cpl_mask_get_data(mask);


    for (int yy = 0; yy < self->detsize_pixels_y;
         ++yy, mptr += self->detsize_pixels_x) {
        (void)memset(mptr, CPL_BINARY_1, self->split_pixel_x);
    }

    return mask;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a mask for the IRDIS field of view left window from model
 * @param the IRDIS model
 *
 * @return a new cpl_mask
 *
 * This creates a new mask that is of the full readout size but masks out the
 * right window.
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
sph_ird_instrument_model_get_mask_leftwin(const sph_ird_instrument_model* self)
{
    cpl_mask* mask = cpl_mask_new(self->detsize_pixels_x,
                                  self->detsize_pixels_y);
    cpl_binary* mptr = cpl_mask_get_data(mask);

    for (int yy = 0; yy < self->detsize_pixels_y;
         ++yy, mptr += self->detsize_pixels_x) {
        (void)memset(mptr + self->split_pixel_x, CPL_BINARY_1,
                     self->detsize_pixels_x - self->split_pixel_x);
    }

    return mask;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the right subwindow of a mask
 * @param self          the instrument model specifying window geometry
 * @param mframe        the mask to extract from
 *
 * @return pointer to newly created cpl_mask of right part only or NULL
 *
 * The left part ( from pixel self->split_pixel_x + 1 to edge ) is extracted
 * from the the mask and returned. The mask must have the size
 * of the full detector as specified in the instrument model.
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
sph_ird_instrument_model_extract_right_mask(
        sph_ird_instrument_model* self, cpl_mask* mask ) {
    cpl_mask*    result    = NULL;
    cpl_mask*    dmask    = NULL;

    if ( !self ) {
        SPH_NO_SELF;
        return NULL;
    }
    if ( !mask ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( cpl_mask_get_size_x( mask ) != self->detsize_pixels_x )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Size of master frame not same as detector "
                "size in model!"
                "The frame must have the x-side equal in size "
                "to the model detector pixel size of %d pixels.",
                self->detsize_pixels_x);
        return NULL;
    }
    if ( cpl_mask_get_size_y( mask ) < self->zero_window_start_y + self->window_size_y )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "The y-side of %d pixels of the input frame "
                "is too small to allow extraction of a"
                " window starting at %d and size %d.",
                (int)cpl_mask_get_size_y( mask ),
                self->zero_window_start_y, self->window_size_y);
        return NULL;
    }
    result = cpl_mask_extract(mask, self->split_pixel_x + 1,
            self->zero_window_start_y + 1, self->detsize_pixels_x,
            self->zero_window_start_y + self->window_size_y);
    if ( !result ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( self->window_mode ) {
        dmask = cpl_mask_extract(result,
                self->window_mode_bl,
                self->window_mode_bl,
                self->window_mode_tr,
                self->window_mode_tr);
        cpl_mask_delete(result);
        result = dmask;
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the left subwindow of a mask
 * @param self          the instrument model specifying window geometry
 * @param mframe        the mask to extract from
 *
 * @return pointer to newly created cpl_mask of left part only or NULL
 *
 * The left part ( from pixel 1 to pixel self->split_pixel_x ) is extracted
 * from the the mask and returned. The mask must have the size
 * of the full detector as specified in the instrument model.
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
sph_ird_instrument_model_extract_left_mask(
        sph_ird_instrument_model* self, cpl_mask* mask ) {
    cpl_mask*    result    = NULL;
    cpl_mask*    dmask    = NULL;

    if ( !self ) {
        SPH_NO_SELF;
        return NULL;
    }
    if ( !mask ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( cpl_mask_get_size_x( mask ) != self->detsize_pixels_x )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Size of master frame not same as detector "
                "size in model!"
                "The frame must have the x-side equal in size "
                "to the model detector pixel size of %d pixels.",
                self->detsize_pixels_x);
        return NULL;
    }
    if ( cpl_mask_get_size_y( mask ) < self->zero_window_start_y + self->window_size_y )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "The y-side of %d pixels of the input frame "
                "is too small to allow extraction of a"
                " window starting at %d and size %d.",
                (int)cpl_mask_get_size_y( mask ),
                self->zero_window_start_y, self->window_size_y);
        return NULL;
    }
    result = cpl_mask_extract(mask,1,self->zero_window_start_y + 1, self->split_pixel_x, self->window_size_y + self->zero_window_start_y);
    if ( !result ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( self->window_mode ) {
        dmask = cpl_mask_extract(result,
                self->window_mode_bl,
                self->window_mode_bl,
                self->window_mode_tr,
                self->window_mode_tr);
        cpl_mask_delete(result);
        result = dmask;
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the left subwindow of a image
 * @param self          the instrument model specifying window geometry
 * @param inim          the image to extract from
 *
 * @return pointer to newly created image of left part only or NULL
 *
 * The left part ( from pixel 1 to pixel self->split_pixel_x ) is extracted
 * from the the image and returned. The image must have the size
 * of the full detector as specified in the instrument model.
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_ird_instrument_model_extract_left_image( sph_ird_instrument_model* self, cpl_image* inim ) {
    cpl_image*    image    = NULL;
    cpl_image*    dimage    = NULL;
    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( inim, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cpl_image_get_size_x( inim ) == self->detsize_pixels_x, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure(cpl_image_get_size_y( inim ) >= self->zero_window_start_y + self->window_size_y, CPL_ERROR_ILLEGAL_INPUT, NULL );
    image = cpl_image_extract( inim, 1, self->zero_window_start_y + 1, self->split_pixel_x, self->zero_window_start_y + self->window_size_y );
    if ( self->window_mode ) {
        dimage = cpl_image_extract(image,
                self->window_mode_bl,
                self->window_mode_bl,
                self->window_mode_tr,
                self->window_mode_tr);
        cpl_image_delete(image);
        image = dimage;
    }
    cpl_image_accept_all(image); //This is necessary since it seems that the cpl_extract
                                         //function does not set the badpixels correctly.
    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the right subwindow of a image
 * @param self          the instrument model specifying window geometry
 * @param inim          the image to extract from
 *
 * @return pointer to newly created image of right part only or NULL
 *
 * The right part ( from pixel self->split_pixel_x to end ) is extracted
 * from the the image and returned. The image must have the size
 * of the full detector as specified in the instrument model.
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_ird_instrument_model_extract_right_image( sph_ird_instrument_model* self, cpl_image* inim ) {
    cpl_image*    image    = NULL;
    cpl_image*    dimage    = NULL;
    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( inim, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cpl_image_get_size_x( inim ) == self->detsize_pixels_x, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure(cpl_image_get_size_y( inim ) >= self->zero_window_start_y + self->window_size_y, CPL_ERROR_ILLEGAL_INPUT, NULL );
    image = cpl_image_extract( inim, self->split_pixel_x + 1,
                self->zero_window_start_y + 1, self->detsize_pixels_x,
                self->zero_window_start_y + self->window_size_y );
    if ( self->window_mode ) {
        dimage = cpl_image_extract(image,
                self->window_mode_bl,
                self->window_mode_bl,
                self->window_mode_tr,
                self->window_mode_tr);
        cpl_image_delete(image);
        image = dimage;
    }

    cpl_image_accept_all(image); //This is necessary since it seems that the cpl_extract
                                         //function does not set the badpixels correctly.
    return image;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the left subwindow of a master frame
 * @param self          the instrument model specifying window geometry
 * @param mframe        the master frame to extract from
 *
 * @return pointer to newly created master_frame of left part only or NULL
 *
 * The left part ( from pixel 1 to pixel self->split_pixel_x ) is extracted
 * from the the master frame and returned. The master frame must have the size
 * of the full detector as specified in the instrument model.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_ird_instrument_model_extract_left_master_frame(const sph_ird_instrument_model* self,
                                                   const sph_master_frame* mframe) {
    sph_master_frame*    result    = NULL;
    cpl_image*           badimage = NULL;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(mframe,CPL_ERROR_NULL_INPUT,NULL);

    if ( cpl_image_get_size_x( mframe->image ) != self->detsize_pixels_x )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Size of master frame not same as detector "
                "size in model!"
                "The frame must have the x-side equal in size "
                "to the model detector pixel size of %d pixels.",
                self->detsize_pixels_x);
        return NULL;
    }
    if ( cpl_image_get_size_y( mframe->image ) < self->zero_window_start_y + self->window_size_y )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "The y-side of %d pixels of the input frame "
                "is too small to allow extraction of a"
                " window starting at %d and size %d.",
                (int)cpl_image_get_size_y( mframe->image ),
                self->zero_window_start_y, self->window_size_y);
        return NULL;
    }
    result = sph_master_frame_new_empty();
    if ( !result ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    result->image = cpl_image_extract( mframe->image, 1, self->zero_window_start_y + 1, self->split_pixel_x, self->zero_window_start_y + self->window_size_y );
    cpl_image_accept_all(result->image); //This is necessary since it seems that the cpl_extract
                                         //function does not set the badpixels correctly.
    badimage = cpl_image_extract( mframe->badpixelmap, 1, self->zero_window_start_y + 1, self->split_pixel_x, self->zero_window_start_y + self->window_size_y );
    result->badpixelmap = cpl_image_duplicate(badimage);
    result->ncombmap = cpl_image_extract( mframe->ncombmap, 1, self->zero_window_start_y + 1, self->split_pixel_x, self->zero_window_start_y + self->window_size_y );
    result->rmsmap = cpl_image_extract( mframe->rmsmap, 1, self->zero_window_start_y + 1, self->split_pixel_x, self->zero_window_start_y + self->window_size_y );
    sph_master_frame_set_bads_from_image(result,badimage);

    if ( mframe->properties ) {
        cpl_propertylist_delete(result->properties); result->properties = NULL;
        result->properties = cpl_propertylist_duplicate( mframe->properties );
    }
    cpl_image_delete(badimage); badimage = NULL;

    if ( self->window_mode ) {
        sph_ird_instrument_model_crop_to_window(self,result);
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the special window mode
 * @param self      the instrument model
 * @param size      the window size
 * @return error code
 *
 * This switches on the special window mode and sets the window size
 * to the given value. The window size must be less than half
 * the sub FOV size or a CPL_ERROR_ILLEGAL_INPUT is returned and
 * the window mode is switched off.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_set_windows(
        sph_ird_instrument_model* self,
        int         window_size)
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);

    self->window_mode = 1;
    self->window_mode_bl = self->split_pixel_x / 2 - window_size / 2 + 1;
    self->window_mode_tr = self->split_pixel_x / 2 + window_size / 2;
    if ( self->window_mode_bl < 1 || self->window_mode_tr > self->split_pixel_x )
    {
        self->window_mode = 0;
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Correct coordinates for window mode
 * @param self      the instrument model
 * @param cxin      the centre x coordinate to correct
 * @param cyin      the centre y coordinate to correct
 * @param cxout     the corrected x coordinate
 * @param cyout     the corrected y coordinate
 *
 * @return error code
 *
 * The input and output coordinates are the internal coordinate
 * system where the left most corner of the left most pixel has
 * coordinates 0.0, 0.0.
 * In case that the window mode is on the output contains the new
 * set of centre coordinates recalculated using the current window
 * size.
 * In case that the window mode is off, the output returned is
 * identical to the input.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_adjust_centre_coords(
        sph_ird_instrument_model* self,
        double cxin,
        double cyin,
        double* cxout,
        double* cyout)
{
    int             full_subframe_size;
    double          cx0 = 0.0;
    double          cy0 = 0.0;
    double          cx1 = 0.0;
    double          cy1 = 0.0;
    double          dx = 0.0;
    double          dy = 0.0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cxout,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cyout,CPL_ERROR_NULL_INPUT);
    if ( self->window_mode == 0 ) {
        *cxout = cxin;
        *cyout = cyin;
        return CPL_ERROR_NONE;
    }

    full_subframe_size = self->window_size_y;
    cx0 = full_subframe_size / 2.0;
    cy0 = full_subframe_size / 2.0;
    cx1 = (self->window_mode_tr - self->window_mode_bl + 1) / 2.0;
    cy1 = (self->window_mode_tr - self->window_mode_bl + 1) / 2.0;
    dx = cxin - cx0;
    dy = cyin - cy0;

    *cxout = dx + cx1;
    *cyout = dy + cy1;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Correct coordinates for window mode
 * @param self      the instrument model
 * @param cxin      the centre x coordinate to correct
 * @param cyin      the centre y coordinate to correct
 * @param cxout     the corrected x coordinate
 * @param cyout     the corrected y coordinate
 *
 * @return error code
 *
 * The input and output coordinates are the internal coordinate
 * system where the left most corner of the left most pixel has
 * coordinates 0.0, 0.0.
 * In case that the window mode is on the output contains the new
 * set of centre coordinates recalculated using the current window
 * size.
 * In case that the window mode is off, the output returned is
 * identical to the input.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_adjust_centre_coords_v(
        sph_ird_instrument_model* self,
        cpl_vector* cxin,
        cpl_vector* cyin,
        cpl_vector** cxout,
        cpl_vector** cyout)
{
    int             ii = 0;
    double          cxn = 0.0;
    double          cyn = 0.0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cxout,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cyout,CPL_ERROR_NULL_INPUT);
    *cxout = cpl_vector_duplicate(cxin);
    *cyout = cpl_vector_duplicate(cyin);
    if ( self->window_mode == 0 ) {
        return CPL_ERROR_NONE;
    }

    for (ii = 0; ii < cpl_vector_get_size(cxin); ++ii) {
        sph_ird_instrument_model_adjust_centre_coords(
                self,
                cpl_vector_get(cxin,ii),
                cpl_vector_get(cyin,ii),
                &cxn,
                &cyn);
        cpl_vector_set(*cxout,ii,cxn);
        cpl_vector_set(*cyout,ii,cyn);
    }

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Crops master frame to smaller IRDIS windows
 * @param self      the IRDIS instrument model
 * @param result    the master frame (input/output)
 *
 * @return  error code
 *
 * This function modifies the master frame in place, cropping
 * it to the small window if the special window mode is set.
 * If not, nothing happens.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ird_instrument_model_crop_to_window(
        const sph_ird_instrument_model* self,
        sph_master_frame* result) {
    cpl_image* dimage = NULL;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    if ( self->window_mode == 0 ) return CPL_ERROR_NONE;
    cpl_ensure_code(result,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    dimage = cpl_image_extract(result->image,self->window_mode_bl,self->window_mode_bl,self->window_mode_tr,self->window_mode_tr);
    cpl_image_delete(result->image);
    result->image = dimage;
    dimage = cpl_image_extract(result->badpixelmap,self->window_mode_bl,self->window_mode_bl,self->window_mode_tr,self->window_mode_tr);
    cpl_image_delete(result->badpixelmap);
    result->badpixelmap = dimage;
    dimage = cpl_image_extract(result->ncombmap,self->window_mode_bl,self->window_mode_bl,self->window_mode_tr,self->window_mode_tr);
    cpl_image_delete(result->ncombmap);
    result->ncombmap = dimage;
    dimage = cpl_image_extract(result->rmsmap,self->window_mode_bl,self->window_mode_bl,self->window_mode_tr,self->window_mode_tr);
    cpl_image_delete(result->rmsmap);
    result->rmsmap = dimage;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the right subwindow of a master frame
 * @param self          the instrument model specifying window geometry
 * @param mframe        the master frame to extract from
 *
 * @return pointer to newly created master_frame of left part only or NULL
 *
 * The right part ( from pixel self->split_pixel_x + 1 to end) is extracted
 * from the the master frame and returned. The master frame must have the size
 * of the full detector as specified in the instrument model.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_ird_instrument_model_extract_right_master_frame( const sph_ird_instrument_model* self,
                                                     const sph_master_frame* mframe ) {
    sph_master_frame*    result    = NULL;
    cpl_image* 		     badimage = NULL;
    if ( !self ) {
        SPH_NO_SELF;
        return NULL;
    }
    if ( !mframe ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( cpl_image_get_size_x( mframe->image ) != self->detsize_pixels_x )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Size of master frame not same as detector "
                "size in model!"
                "The frame must have the x-side equal in size "
                "to the model detector pixel size of %d pixels.",
                self->detsize_pixels_x);
        return NULL;
    }
    if ( cpl_image_get_size_y( mframe->image ) < self->zero_window_start_y + self->window_size_y )
    {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "The y-side of %d pixels of the input frame "
                "is too small to allow extraction of a"
                " window starting at %d and size %d.",
                (int)cpl_image_get_size_y( mframe->image ),
                self->zero_window_start_y, self->window_size_y);
        return NULL;
    }
    result = sph_master_frame_new_empty();
    if ( !result ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    result->image = cpl_image_extract( mframe->image, self->split_pixel_x + 1,
            self->zero_window_start_y + 1, self->detsize_pixels_x,
            self->zero_window_start_y + self->window_size_y );
    cpl_image_accept_all(result->image); //This is necessary since it seems that the cpl_extract
                                         //function does not set the badpixels correctly.
    badimage = cpl_image_extract( mframe->badpixelmap,
            self->split_pixel_x + 1, self->zero_window_start_y + 1,
            self->detsize_pixels_x, self->zero_window_start_y + self->window_size_y );
    result->badpixelmap = cpl_image_duplicate(badimage);
    result->ncombmap = cpl_image_extract( mframe->ncombmap,
            self->split_pixel_x + 1, self->zero_window_start_y + 1,
            self->detsize_pixels_x, self->zero_window_start_y + self->window_size_y );
    result->rmsmap = cpl_image_extract( mframe->rmsmap,
            self->split_pixel_x + 1, self->zero_window_start_y + 1,
            self->detsize_pixels_x, self->zero_window_start_y + self->window_size_y );
    sph_master_frame_set_bads_from_image(result,badimage);
    if ( mframe->properties ) {
        cpl_propertylist_delete(result->properties); result->properties = NULL;
        result->properties = cpl_propertylist_duplicate( mframe->properties );
    }
    cpl_image_delete(badimage); badimage = NULL;

    if ( self->window_mode ) {
        sph_ird_instrument_model_crop_to_window(self,result);
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Assemble a new full image from left and right parts
 * @param self IRDIS instrument model
 * @param left left image
 * @param right right image
 *
 * @return new image containing left and right together.
 *
 * This creates a new full size image that has the same size as the
 * original detector readout window in the raw frames (as specified in the
 * instrument model by the X and Y detector size parameters) and contains
 * the input left and right images "glued" into the right places.
 *
 *
 */
/*----------------------------------------------------------------------------*/

cpl_image*
sph_ird_instrument_model_assemble_image(
        sph_ird_instrument_model* self,
        cpl_image* left,
        cpl_image* right)
{
    cpl_image*   result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(left, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(right, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->detsize_pixels_x ==
            cpl_image_get_size_x(left) +
            cpl_image_get_size_x(right),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(self->window_size_y ==
            cpl_image_get_size_y(left),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(self->window_size_y ==
            cpl_image_get_size_y(right),
            CPL_ERROR_ILLEGAL_INPUT, NULL);

    result = cpl_image_new(self->detsize_pixels_x,self->detsize_pixels_y,cpl_image_get_type(left));
    cpl_image_copy(result,left,1,self->zero_window_start_y+1);
    cpl_image_copy(result,right,self->split_pixel_x + 1, self->zero_window_start_y+1);
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Assemble a new full frame from left and right parts
 * @param self IRDIS instrument model
 * @param left left sph_master_frame
 * @param right right sph_master_frame
 *
 * @return new master frame containing left and right together.
 *
 * This creates a new full size master_frame that has the same size as the
 * original detector readout window in the raw frames (as specified in the
 * instrument model by the X and Y detector size parameters) and contains
 * the input left and right frames "glued" into the right places.
 *
 *
 */
/*----------------------------------------------------------------------------*/

sph_master_frame*
sph_ird_instrument_model_assemble(
        sph_ird_instrument_model* self,
        sph_master_frame* left,
        sph_master_frame* right)
{
    sph_master_frame*   result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(left, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(right, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->detsize_pixels_x ==
            cpl_image_get_size_x(left->image) +
            cpl_image_get_size_x(right->image),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(self->window_size_y ==
            cpl_image_get_size_y(left->image),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(self->window_size_y ==
            cpl_image_get_size_y(right->image),
            CPL_ERROR_ILLEGAL_INPUT, NULL);

    result = sph_master_frame_new(self->detsize_pixels_x,self->detsize_pixels_y);
    cpl_image_copy(result->image,left->image,1,self->zero_window_start_y+1);
    cpl_image_copy(result->badpixelmap,left->badpixelmap,1,self->zero_window_start_y+1);
    cpl_image_copy(result->ncombmap,left->ncombmap,1,self->zero_window_start_y+1);
    cpl_image_copy(result->rmsmap,left->rmsmap,1,self->zero_window_start_y+1);
    cpl_image_copy(result->image,right->image,self->split_pixel_x + 1, self->zero_window_start_y+1);
    cpl_image_copy(result->badpixelmap,right->badpixelmap,self->split_pixel_x + 1,self->zero_window_start_y+1);
    cpl_image_copy(result->ncombmap,right->ncombmap,self->split_pixel_x + 1,self->zero_window_start_y+1);
    cpl_image_copy(result->rmsmap,right->rmsmap,self->split_pixel_x + 1,self->zero_window_start_y+1);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Check that a specific pixel of sub-image is inside the DBI FOV
 * @param irdis instrument model
 * @param xx  pixel xx
 * @param yy  pixel yy
 * @param ok  input flag, if set to 0, this func always returns 0
 * @param border    the width of the border to leave
 *
 * @return 1 if yes, 0 if not, -1 on error
 *
 * This function checks if a pixel in an IRDIS sub-image is inside the
 * definition of the IRDIS DBI field of view. This FoV is defined in
 * subframe coordinates, as a circle around the center of
 * the _left_ FoV center with a radius given by the window size minus the border.
 *
 *
 */
/*----------------------------------------------------------------------------*/
int
sph_ird_instrument_model_check_in_dbi_fov(const sph_ird_instrument_model* model,
                                          int xx, int yy, int ok, double border)
{
    if (ok) {
        double centimx = 0.0;
        double centimy = 0.0;

        sph_ird_instrument_model_get_centre_left_subframe_coords(model,
                                                                 &centimx,
                                                                 &centimy);

        if((xx - centimx) * (xx - centimx) + (yy - centimy) * (yy - centimy) >
           (centimx - border) * (centimx - border))
            {
                ok = 0;
            }
    }

    return ok;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Delete the lenslet model.

 @param self     pointer to the lenslet model to delete

 @return nothing

 Description: Deletes the lenslet model, freeing up the space.

 */
/*----------------------------------------------------------------------------*/
void
sph_ird_instrument_model_delete( sph_ird_instrument_model* self ) {
    if ( self ) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief    Add QC counts parameters to QC propertylist
 @param    self        The QC propertylist to update
 @param    rawframes   The list of raw frames (only last one used)
 @param    left_mask   The mask of left channel pixels to use
 @param    right_mask  The mask of right channel pixels to use
 @param    do_extra    Whether to do additional QC stats (stdev and max)
 @return   Nothing

 */
/*----------------------------------------------------------------------------*/
void sph_ird_instrument_raw_add_qc_counts(cpl_propertylist* self,
                                          const cpl_frameset* rawframes,
                                          const cpl_mask* left_mask,
                                          const cpl_mask* right_mask,
                                          cpl_boolean do_extra)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const cpl_size nraw = cpl_frameset_get_size(rawframes);
    const cpl_frame* rawframe = cpl_frameset_get_position_const(rawframes,
                                                                 nraw - 1);
    const char* filename = cpl_frame_get_filename(rawframe);

    /* Load only the last image */
    cpl_propertylist * plnaxis =
        cpl_propertylist_load_regexp(filename, 0, "^NAXIS$|^NAXIS3$",
                                     CPL_FALSE);
    const int nimg = cpl_propertylist_get_int(plnaxis, "NAXIS") == 2 ?
        1 : cpl_propertylist_get_int(plnaxis, "NAXIS3");
    cpl_image *lastimg =
        cpl_image_load(filename, CPL_TYPE_UNSPECIFIED, nimg - 1, 0);

    double right_med = 0.0;
    double right_max = 0.0;
    double right_stdev = 0.0;

    double left_med = 0.0;
    double left_max = 0.0;
    double left_stdev = 0.0;

    const cpl_error_code rcode = cpl_image_reject_from_mask(lastimg,
                                                            right_mask);
    cpl_error_code code;
    cpl_error_code lcode;

    if (do_extra) {
        cpl_stats * mystats = cpl_stats_new_from_image(lastimg, CPL_STATS_MAX |
                                                       CPL_STATS_MEDIAN |
                                                       CPL_STATS_STDEV);
        right_med = cpl_stats_get_median(mystats);
        right_max = cpl_stats_get_max(mystats);
        right_stdev = cpl_stats_get_stdev(mystats);
        cpl_stats_delete(mystats);
    } else {
        right_med = cpl_image_get_median(lastimg);
    }

    code = cpl_image_accept_all(lastimg);
    lcode = cpl_image_reject_from_mask(lastimg, left_mask);

    if (do_extra) {
        cpl_stats * mystats = cpl_stats_new_from_image(lastimg, CPL_STATS_MAX |
                                                       CPL_STATS_MEDIAN |
                                                       CPL_STATS_STDEV);
        left_med = cpl_stats_get_median(mystats);
        left_max = cpl_stats_get_max(mystats);
        left_stdev = cpl_stats_get_stdev(mystats);
        cpl_stats_delete(mystats);
    } else {
        left_med = cpl_image_get_median(lastimg);
    }

    cpl_image_delete(lastimg);
    cpl_propertylist_delete(plnaxis);

    if (lcode || code || rcode || !cpl_errorstate_is_equal(prestate) ||
        (do_extra &&
         cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_MAX_RIGHT,
                                        right_max)) ||
        (do_extra &&
         cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_STDEV_RIGHT,
                                        right_stdev)) ||
        (do_extra &&
         cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_MAX_LEFT,
                                        left_max)) ||
        (do_extra &&
         cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_STDEV_LEFT,
                                        left_stdev)) ||
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_COUNTS_LEFT,
                                       left_med) ||
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_QC_COUNTS_RIGHT,
                                       right_med)) {
        cpl_msg_warning(cpl_func, "Could not compute "
                        SPH_COMMON_KEYWORD_QC_COUNTS_LEFT " and "
                        SPH_COMMON_KEYWORD_QC_COUNTS_RIGHT " from last "
                        "of %d image(s) in last of %d frame(s) in %s (%g) (%g)",
                        (int)nimg, (int)nraw, filename, left_med, right_med);
        cpl_errorstate_dump(prestate, CPL_FALSE,
                            cpl_errorstate_dump_one_warning);
        cpl_errorstate_set(prestate);
    }
}
