/*
This file is part of ``kdtree'', a library for working with kd-trees.
Copyright (C) 2007-2011 John Tsiombikas <nuclear@member.fsf.org>

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#ifndef _KDTREE_H_
#define _KDTREE_H_

#include <cpl.h>

typedef struct kdtree sph_kdtree;
typedef struct kdres sph_kdres;
struct kdhyperrect {
    int dim;
    double *min, *max;              /* minimum/maximum coords */
};

struct kdnode {
    double *pos;
    int dir;
    void *data;

    struct kdnode *left, *right;    /* negative/positive side */
};

struct res_node {
    struct kdnode *item;
    double dist_sq;
    struct res_node *next;
};

struct kdtree {
    int dim;
    struct kdnode *root;
    struct kdhyperrect *rect;
    void (*destr)(void*);
};

struct kdres {
    struct kdtree *tree;
    struct res_node *rlist, *riter;
    int size;
};



/* create a kd-tree for "k"-dimensional data */
struct kdtree *sph_kd_create(int k)
     CPL_ATTR_ALLOC;

/* free the struct kdtree */
void sph_kd_free(struct kdtree *tree);

/* remove all the elements from the tree */
void sph_kd_clear(struct kdtree *tree);

/* if called with non-null 2nd argument, the function provided
 * will be called on data pointers (see sph_kd_insert) when nodes
 * are to be removed from the tree.
 */
void sph_kd_data_destructor(struct kdtree *tree, void (*destr)(void*));

/* insert a node, specifying its position, and optional data */
int sph_kd_insert(struct kdtree *tree, const double *pos, void *data);

/* Find the nearest node from a given point.
 *
 * This function returns a pointer to a result set with at most one element.
 */
struct kdres *sph_kd_nearest(struct kdtree *tree, const double *pos,
			     double mindist)
     CPL_ATTR_ALLOC;

/* Find the N nearest nodes from a given point.
 *
 * This function returns a pointer to a result set, with at most N elements,
 * which can be manipulated with the sph_kd_res_* functions.
 * The returned pointer can be null as an indication of an error. Otherwise
 * a valid result set is always returned which may contain 0 or more elements.
 * The result set must be deallocated with sph_kd_res_free after use.
 */
/*
struct kdres *sph_kd_nearest_n(struct kdtree *tree, const double *pos, int num);
struct kdres *sph_kd_nearest_nf(struct kdtree *tree, const float *pos, int num);
struct kdres *sph_kd_nearest_n3(struct kdtree *tree, double x, double y, double z);
struct kdres *sph_kd_nearest_n3f(struct kdtree *tree, float x, float y, float z);
*/

/* Find any nearest nodes from a given point within a range.
 *
 * This function returns a pointer to a result set, which can be manipulated
 * by the sph_kd_res_* functions.
 * The returned pointer can be null as an indication of an error. Otherwise
 * a valid result set is always returned which may contain 0 or more elements.
 * The result set must be deallocated with sph_kd_res_free after use.
 */
struct kdres *sph_kd_nearest_range(struct kdtree *tree,
				   const double *pos,
				   double range)
     CPL_ATTR_ALLOC;

/* frees a result set returned by sph_kd_nearest_range() */
void sph_kd_res_free(struct kdres *set);

/* returns the size of the result set (in elements) */
int sph_kd_res_size(struct kdres *set);

/* rewinds the result set iterator */
void sph_kd_res_rewind(struct kdres *set);

/* returns non-zero if the set iterator reached the end after the last element */
int sph_kd_res_end(struct kdres *set);

/* advances the result set iterator, returns non-zero on success, zero if
 * there are no more elements in the result set.
 */
int sph_kd_res_next(struct kdres *set);


#endif	/* _KDTREE_H_ */
