
/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * sph_keywordnode.h
 *
 *  Created on: Apr 2, 2009
 *      Author: pavlov
 *      Re-written in pure C (& further developed for sphere needs) from an original c++ code
 *      (lcfglCFG) of  Matthias Heininger.
 */

#ifndef SPH_KEYWORDNODE_H_
#define SPH_KEYWORDNODE_H_

#include <cpl_frame.h>
#include <cpl_frameset.h>
#include "sph_master_frame.h"
#include "sph_double_image.h"

#include <stdio.h>

/**
 * @def keywordMAX_CFG_NAME_LEN
 * This constant defines the maximum numbers of characters for an identifier (including trailing null byte).
 */
#define keywordMAX_CFG_NAME_LEN    256

/**
 * This enumeration defines all different data types which can be stored and accessed in a configuration file.
 * keyword_GROUP has the special meaning that no values are present in the current node but this node has
 * child nodes.
 */
typedef enum {keyword_GROUP, keyword_CHAR, keyword_INT, keyword_DOUBLE, keyword_STRING,
	keyword_CPLFRAME, keyword_CPLFRAMESET, 	keyword_SPHMASTERFRAME, keyword_SPHDOUBLEIMAGE} keyword_TYPE;


/**
 *
 * The structure keeps a hierarchical structure (tree) which represents
 * an external configuration file of keywords. These configuration files are plain ASCII files
 * with a very simple syntax:
 *
 * - comments start with an \# character and go till the end of a line
 * - each valid line must have the following parts:
 *   - Qualified name
 *   - Equal sign
 *   - Value enclosed in Quotes
 *   - An optional comment started by a hash
 * - a qualified name must conform to the pattern \<identifier\> { "." \<identifier\> }
 * - an identifier must conform to the pattern \<alpha\> { \<alpha\> } { \<digit\> }
 *
 * The syntax for configuration files is:
@verbatim
<file>          := { <line> }
<line>          := <commentLine> | <valueLine>
<commentLine>   := "#" { <letter> }
<valueLine>     := <qualifiedName> "=" "\"" <value> "\"" [ "#" <letter> ]
<qualifiedName> := <identifier> { "." <identifier> }
<identifier>    := <alpha> { <alpha> } { <digit> }
<value>         := { <letter> }
@endverbatim
 *
 * During loading of a configuration file a tree structure is created where the name
 * and number components of the qualified names are used to define a path from the root
 * of the tree to the leaf where the value is stored.
 *
 * For example if a file contains the following entries:
 *
 *
@verbatim
PIPELINE.VERSION = "0.9" \# current version of the PIPELINE
PIPELINE.ZIMPOL.CAM1.CALNAME = "DARK"
PIPELINE.ZIMPOL.CAM1.DIT = "30.5"
PIPELINE.ZIMPOL.CAM1.NDIT = "20"
PIPELINE.ZIMPOL.CAM1.CALNAME = "DARK"
PIPELINE.ZIMPOL.CAM1.DIT = "15.25"
PIPELINE.ZIMPOL.CAM1.NDIT = "40"
@endverbatim
 *
 * the following tree is created:
 *
@verbatim
ROOT
 +-- PIPELINE
      +-- VERSION = 0.9 (double)
      +-- CAM
           +-- 1
           |   +-- CALNAME = DARK (string, i.e. char*)
           |   +-- DIT = 30.5 (double)
           |   +-- NDIT = 20 (int)
           +-- 2
              +-- CALNAME = DARK (string, i.e. char*)
           |   +-- DIT = 15.25 (double)
           |   +-- NDIT = 40 (int)

@endverbatim
 */

typedef struct _sph_keywordnode_
{
    char            name[keywordMAX_CFG_NAME_LEN]; ///< base name or empty string if the keyword node represents the number part of an identifier.
    char            value[256]; ///a value storage.
    int             nr;     ///< number or -1 if name
    keyword_TYPE   type;   ///< value type or keyword_GROUP
    struct _sph_keywordnode_       *parent; ///< Pointer to parent of this keyword node or NULL if root.
    struct _sph_keywordnode_       *next;   ///< Next keyword node at the same level and with the same keyword parent.
    struct _sph_keywordnode_       *prev;   ///< Previous keyword node at the same level and with the same keyword parent.
    struct _sph_keywordnode_       *first;  ///< First keyword child of the current node or NULL if no childs.
    struct _sph_keywordnode_       *last;   ///< Last keyword child or equal to first keyworld child (only one child) or NULL.
    char            				cVal;   ///< Character value of the node if type is equal to keyword_CHAR.
    int             				iVal;   ///< Integer value of the node if type is equal to keyword_INT.
    double          				dVal;   ///< Double value of the node if type is equal to keyword_DOUBLE
    char           					*sVal;   ///< String value of the node if type is equal to keyword_STRING.
    size_t          				sSize;  ///< Number of characters in the string value.
    cpl_frame	    				*frame; // < Pointer to the cpl_frame structure if type is equal to keyword_FRAME
    cpl_frameset    				*frames; // < Pointer to the cpl_frameset structure if type is equal to keyword_FRAMESET
    sph_master_frame				*masterframe;// < Pointer to the sph_master_frame structure if type is equal to keyword_MASTERFRAME
    sph_double_image				*doubleimage;// < Pointer to the sph_double_image structure if type is equal to keyword_SPHDOUBLEIMAGE
    //sph_cube						*cube;// < Pointer to the sph_cube structure if type is equal to keyword_SPHCUBE (not yet implemented)
} sph_keywordnode;


/**
 * Create a new keywordnode (child) with no name,  with a given name, with a specific numbering
 * A-la constructors
 */
sph_keywordnode *sph_keywordnode_new(sph_keywordnode *parent) CPL_ATTR_ALLOC;
sph_keywordnode *sph_keywordnode_new_name(sph_keywordnode *parent, const char *name)
    CPL_ATTR_ALLOC;
sph_keywordnode *sph_keywordnode_new_nr(sph_keywordnode *parent, int nr)
    CPL_ATTR_ALLOC;
void sph_keywordnode_destroy(sph_keywordnode *pthis);

/**
 * Save a keyword configuration into an opened file (given by pointer)
 * @param  pthis This defines a pointer to the given keywordnode
 * @param fp This argument defines a pointer to the opened file
 */
int sph_keywordnode_save(const sph_keywordnode* pthis, FILE *fp);

/**
 * A specific node relative to the current node can be located and/or created by
 * using a relative path. This path uses the syntax for qualified names. If the
 * trailing number is equal to 0 a new node with an automatic generated number is created.
 * @param qalName This argument defines a relative path from the current node.
 * If this argument is NULL, the current node is used.
 * @param force If the destination node is not found a new node is created (including
 * the intermediate nodes) if this flag is 1.
 */
sph_keywordnode *sph_keywordnode_LocateByName(sph_keywordnode *pthis,
                                              const char *qalName,
                                              int force); // = 0);

/*
 * Adapter function to the sph_keywordnode_LocateByName function with force = 0.
 * A specific node relative to the current node is found by
 * using a relative path. This path uses the syntax for qualified names.
 * @param pthis This defines a pointer to the given keywordnode
 * @param qalName This argument defines a relative path from the current node.
 */
sph_keywordnode *sph_keywordnode_get_keywordnode_by_name(sph_keywordnode *pthis, const char *qualName);

/**
 * A specific child keyword node is located by using a number. If nr is 0 and force is equal to 1, a new
 * node with a new number is generated (automatic numbering).
 * @param nr This argument defines a number (child).
 * @param force If the destination node is not found and force is 1, a new node is created.
 */
sph_keywordnode *sph_keywordnode_LocateByNr(sph_keywordnode *pthis, int nr, int force); // = 0);

/** Access to the first child keyword node of the current node. */
sph_keywordnode *sph_keywordnode_GetFirstChild(sph_keywordnode* pthis);

/** Access to the next keyword node at the same level. */
sph_keywordnode *sph_keywordnode_GetNext(sph_keywordnode* pthis);

sph_keywordnode *sph_keywordnode_GetParent(sph_keywordnode* pthis);


/** Build a qualified name in order to save the n-tree keywords into configuration files in terms of
  * of brunch.leaf */
int   sph_keywordnode_BuildQualifiedName(const sph_keywordnode *pthis,
                                         char *qualName,
                                         int size);

/** Build a qualified name as above and keywordlist of one given brunch. Keywordlist contains
 * keyword-value pairs: keywords are named nodes whereas values are their numbered childs (if exists). */
int sph_keywordnode_BuildQualifiedNameList(sph_keywordnode *pthis, char *qualName, int size, cpl_propertylist* keywordlist);


/** @brief Access to the name of the current keyword node, empty string if the current node has e number. */
char *sph_keywordnode_get_name(sph_keywordnode* pthis);

/** @brief Access to the number of the current keyword node, -1 if the current  node has a name. */
int   sph_keywordnode_GetNr(const sph_keywordnode* pthis);

/** @brief Access to the data type of the current keyword node. */
keyword_TYPE sph_keywordnode_GetType(const sph_keywordnode* pthis);


/**
 * Set a value (char, int, double, string) of an existing keyword node or create a new node with the given value.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The character value.
 * @return 1 if the value could be set, otherwise 0.
 */
int   sph_keywordnode_SetChar(sph_keywordnode* pthis, const char *qalName, char value);
int   sph_keywordnode_SetInt(sph_keywordnode* pthis, const char *qalName, int value);
int   sph_keywordnode_SetDouble(sph_keywordnode* pthis, const char *qalName, double value);
int   sph_keywordnode_SetString(sph_keywordnode* pthis, const char *qalName, const char *value);
int   sph_keywordnode_SetSingleFrame(sph_keywordnode *pthis, const char *qualName, const cpl_frame *frame);
int   sph_keywordnode_SetFrames(sph_keywordnode* pthis, const char *qalName, const cpl_frameset *frameset);
int   sph_keywordnode_SetMasterFrame(sph_keywordnode* pthis, const char *qalName, sph_master_frame *masterframe);
int   sph_keywordnode_SetDoubleImage(sph_keywordnode* pthis, const char *qualName, sph_double_image *doubleimage);

/**
 * Get the value  (char, int, double, string) of an existing keyword node.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The returned character value.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
char  	 			sph_keywordnode_get_char(sph_keywordnode* pthis );
int   				sph_keywordnode_get_int(sph_keywordnode* pthis );
double   			sph_keywordnode_get_double(sph_keywordnode* pthis );
const char* 				sph_keywordnode_get_string(sph_keywordnode* pthis );

cpl_frame* 			sph_keywordnode_GetSingleFrame(sph_keywordnode* pthis, const char *qualName);
cpl_frameset*		sph_keywordnode_GetFrames(sph_keywordnode* pthis, const char *qualName);
sph_master_frame*	sph_keywordnode_GetMasterFrame(sph_keywordnode* pthis, const char *qualName);
sph_double_image* 	sph_keywordnode_GetDoubleImage(sph_keywordnode* pthis, const char *qualName);
int 				sph_keywordnode_GetChar(sph_keywordnode *pthis, const char *qualName, char *value);
int 				sph_keywordnode_GetInt(sph_keywordnode *pthis, const char *qualName, int *value);
int 				sph_keywordnode_GetDouble(sph_keywordnode *pthis, const char *qualName, double *value);
int 				sph_keywordnode_GetString(sph_keywordnode *pthis, const char *qualName, char *value, int size);


/*
 * Load a keyword configuration from a given file. cpl_frameset is not included
 * @param This argument defines filename with a saved keyword configurations
 * @return Pointer to the keyword root, or NULL.
 */
sph_keywordnode *sph_keywordnode_load_configuration( const char *cfgName)
  CPL_ATTR_ALLOC;

/*
 * Save a keyword configuration from a given file.
 * @param This argument defines filename with a saved keyword configurations
 * @return Pointer to the keyword root, or NULL.
 */
int sph_keywordnode_SaveConfiguration(const sph_keywordnode* pthis,
                                      const char *cfgName);


#endif /* SPH_KEYWORDNODE_H_ */
