/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_line.h"
#include "sph_error.h"

#include <cpl.h>
#include <math.h>
#include <assert.h>

/*----------------------------------------------------------------------------*/
/**
 @brief Create a new line

 @param x1        the x-coordinate of the start point
 @param y1        the y-coordinate of the start point
 @param x2        the x-coordinate of the end point
 @param xy2        the y-coordinate of the end point

 @return the new line or NULL

 Description: create a new line with the given endpoints.

 */
/*----------------------------------------------------------------------------*/
sph_line* sph_line_new( double x1, double y1, double x2, double y2 ) {

    sph_line* self = cpl_malloc(sizeof(*self));

    assert (self !=NULL);

    self->p1 = sph_point_new( x1, y1 );
    self->p2 = sph_point_new( x2, y2 );

    return self;
}



/*----------------------------------------------------------------------------*/
/**
 @brief Test if two lines intersect

 @param lineA        the first line
 @param lineB        the second line

 @return 1 if lines intersect otherwise 0

 Description: checks if two lines intersect. Note that lines are
 considered infinite for this test. Also, lines that overlap are
 considered to be intersecting.

 */
/*----------------------------------------------------------------------------*/
short sph_line_intersect_test( const sph_line* lineA, const sph_line* lineB ) {

    const double dx = lineB->p1->x - lineA->p1->x;
    const double dy = lineB->p1->y - lineA->p1->y;

    if ( dx * dx + dy * dy < DBL_MIN ) {
        /* lineA->p1 lies on LineB(p1) */
        return 1;
    } else {

        const double dx2 = lineB->p2->x - lineA->p1->x;
        const double dy2 = lineB->p2->y - lineA->p1->y;

        if ( dx2 * dx2 + dy2 * dy2 < DBL_MIN ) {
            /* lineA->p1 lies on LineB(p2) */
            return 1;
        } else {

            const double fdx = fabs(dx);
            const double fdy = fabs(dy);

            const double adx = fabs ( lineA->p2->x - lineA->p1->x );
            const double ady = fabs ( lineA->p2->y - lineA->p1->y );
            const double bdx = fabs ( lineB->p2->x - lineB->p1->x );
            const double bdy = fabs ( lineB->p2->y - lineB->p1->y );
            const double dd = adx * fdx + ady * fdy;
            const double ds = sqrt( (adx * adx + ady * ady) *
                                    (bdx * bdx + bdy * bdy) );

            if ( fabs( dd - ds ) < DBL_MIN ) return 1;
        }
    }
    return
        !sph_point_same_side(lineB->p1, lineB->p2, lineA->p1, lineA->p2) &&
        !sph_point_same_side(lineA->p1, lineA->p2, lineB->p1, lineB->p2);
}
/*----------------------------------------------------------------------------*/
/**
 @brief Find intersection point of two lines.
 @param self         the intersection point or (0,0) on error
 @param lineA        the first line
 @param lineB        the second line
 @return CPL_ERROR_NONE or CPL_ERROR_DIVISION_BY_ZERO on zero determinant
 @note No check for NULL pointer input!

 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_line_intersect_point(sph_point*      self,
                                        const sph_line* lineA,
                                        const sph_line* lineB )
{
    const double x1 = lineA->p1->x;
    const double x2 = lineA->p2->x;
    const double y1 = lineA->p1->y;
    const double y2 = lineA->p2->y;
    const double x3 = lineB->p1->x;
    const double x4 = lineB->p2->x;
    const double y3 = lineB->p1->y;
    const double y4 = lineB->p2->y;

    const double x =
        ( x1 * y2 - y1 * x2 ) * (x3 - x4) -
        ( x1 - x2 ) * ( x3 * y4 - y3 * x4 );

    const double y =
        ( x1 * y2 - y1 * x2 ) * (y3 - y4) -
        ( y1 - y2 ) * ( x3 * y4 - y3 * x4 );

    const double determ =
        ( x1 - x2 ) * ( y3 - y4 ) - ( y1 - y2 ) * ( x3 - x4 );

    if (determ != 0.0) {
        self->x = x / determ;
        self->y = y / determ;
    } else {
        self->x = 0.0;
        self->y = 0.0;
        return cpl_error_set(cpl_func, CPL_ERROR_DIVISION_BY_ZERO);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Delete the line and free space

 @param self    the line to delete

 @return nothing

 Description: Does just that.

 */
/*----------------------------------------------------------------------------*/
void sph_line_delete( sph_line* self ) {
    if ( self ) {
        sph_point_delete( self->p1 );
        sph_point_delete( self->p2 );
        cpl_free(self);
    }
}
