/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_master_frame_H_
#define SPH_master_frame_H_
#define SPH_MASTER_FRAME_PACKAGE "sph_master_frame"
#define SPH_MASTER_FRAME_VERSION "0.1"

#define SPH_COLL_ALG_NAME  "SPH_COLL_ALG_NAME"

#include "sph_error.h"
#include "sph_polygon.h"
#include <cpl.h>

#define SPH_MASTER_FRAME_BAD_RMS 1e10

/*-----------------------------------------------------------------------------
                                   New types
 -----------------------------------------------------------------------------*/

#ifdef SPH_MASTER_FRAME_C
const int SPH_MASTER_FRAME_IMAGE_EXT;
const int SPH_MASTER_FRAME_BPIX_EXT;
const int SPH_MASTER_FRAME_NC_EXT ;
const int SPH_MASTER_FRAME_RMS_EXT;
#else
extern const int SPH_MASTER_FRAME_IMAGE_EXT;
extern const int SPH_MASTER_FRAME_BPIX_EXT;
extern const int SPH_MASTER_FRAME_NC_EXT ;
extern const int SPH_MASTER_FRAME_RMS_EXT;
#endif
typedef struct _sph_master_frame_
{
    cpl_image* image;
    cpl_image* badpixelmap;
    cpl_image* ncombmap;
    cpl_image* rmsmap;
    const cpl_parameterlist* alg_parlist;
    cpl_propertylist* properties;
    cpl_propertylist* qclist;
} sph_master_frame;

/*-----------------------------------------------------------------------------
 Function prototypes
 -----------------------------------------------------------------------------*/
extern sph_error_code SPH_MASTER_FRAME_GENERAL;
extern sph_error_code SPH_MASTER_FRAME_BAD_ALGORITHM;
extern sph_error_code SPH_MASTER_FRAME_NO_ALGORITHM;

sph_master_frame* sph_master_frame_new( int nx, int ny )
    CPL_ATTR_ALLOC;
sph_master_frame* sph_master_frame_new_empty(void)
    CPL_ATTR_ALLOC;
sph_master_frame* sph_master_frame_new_extract( const sph_master_frame* mframe,
        int ll_x, int ll_y, int ur_x, int ur_y )
    CPL_ATTR_ALLOC;
sph_master_frame* sph_master_frame_duplicate( const sph_master_frame* self )
    CPL_ATTR_ALLOC;

void sph_master_frame_delete(sph_master_frame* self);

sph_error_code
sph_master_frame_accept_all( sph_master_frame* self );

cpl_image* sph_master_frame_get_weights( const sph_master_frame* self)
    CPL_ATTR_ALLOC;

sph_error_code
sph_master_frame_take_log10( sph_master_frame* self );

sph_error_code sph_master_frame_sqrt( sph_master_frame* self );
sph_error_code sph_master_frame_mask_tolerance( sph_master_frame* self,
                                                double lowtol,
                                                double uptol );
cpl_image* sph_master_frame_get_badpixelmap( const sph_master_frame* self)
    CPL_ATTR_ALLOC;

sph_error_code sph_master_frame_mask_sigma( sph_master_frame* self,
                                            double sigma );
sph_error_code sph_master_frame_quality_check( sph_master_frame* self );
sph_master_frame*
sph_master_frame_load( const char* czFilename, int plane )
    CPL_ATTR_ALLOC;

sph_master_frame* sph_master_frame_load_( const cpl_frame*, int)
    CPL_ATTR_ALLOC;

sph_error_code
sph_master_frame_subtract_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe );

sph_error_code
sph_master_frame_multiply_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe );
sph_error_code
sph_master_frame_divide_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe );


sph_error_code sph_master_frame_set_rms_poisson( sph_master_frame* self,
        double rms_zeros, int bad_zeros);
sph_error_code
sph_master_frame_add_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe );

cpl_image*
sph_master_frame_get_badpixelmap_float( const sph_master_frame* self)
    CPL_ATTR_ALLOC;

int sph_master_frame_get_nbads( const sph_master_frame* self);
sph_master_frame*
sph_master_frame_inverse( const sph_master_frame* mframe )
    CPL_ATTR_ALLOC;

sph_error_code sph_master_frame_set_rms_from_weightmap(
        sph_master_frame* self,
        const cpl_image* wim,
        int bad_zeros);
sph_error_code
sph_master_frame_set_bads_from_image( sph_master_frame* self,
                                      const cpl_image* mask );

cpl_mask* sph_master_frame_get_badpixelmask( const sph_master_frame* self)
    CPL_ATTR_ALLOC;

cpl_image* sph_master_frame_extract_image( const sph_master_frame* self,
                                           int rej_flag)
    CPL_ATTR_ALLOC;

sph_error_code sph_master_frame_multiply_double( sph_master_frame* self,
                                                 double value );
sph_error_code sph_master_frame_divide_double( sph_master_frame* self,
                                               double value );
sph_master_frame**
sph_master_frame_zpl_split( const sph_master_frame* self )
    CPL_ATTR_ALLOC;
sph_master_frame* sph_master_frame_new_from_cpl_image( const cpl_image* image );

sph_master_frame*
sph_master_frame_add_master_frame_create( const sph_master_frame* in,
        const sph_master_frame* mframe )
    CPL_ATTR_ALLOC;

sph_master_frame*
sph_master_frame_subtract_master_frame_create( const sph_master_frame* in,
        const sph_master_frame* mframe )
    CPL_ATTR_ALLOC;

sph_error_code
sph_master_frame_set_bads_from_mask( sph_master_frame* self,
                                     const cpl_mask* mask );

sph_error_code sph_master_frame_save( const sph_master_frame* self,
                                      const char* czFilename,
                                      const cpl_propertylist* pl );

sph_error_code
sph_master_frame_save_extend( const sph_master_frame* self,
        const char* czFilename, const cpl_propertylist* pli );

int sph_master_frame_set_bads( sph_master_frame* self, double val );
sph_error_code
sph_master_frame_save_dfs(
        const sph_master_frame* self,
        const char* outfilename,
        cpl_frameset* allframes,
        const cpl_frame* template_frame,
        const cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        const cpl_propertylist* plist);
sph_error_code sph_master_frame_fit_gauss_hist( const sph_master_frame* self,
                                                cpl_vector* binpos,
                                                cpl_vector* binvals,
                                                double  minbin,
                                                double  maxbin,
                                                double* mean,
                                                double* rms,
                                                double* redchi );

sph_error_code
sph_master_frame_smooth( sph_master_frame* self, double fwhm);

sph_error_code
sph_master_frame_interpolate_bpix( sph_master_frame* self);

cpl_image* sph_master_frame_sn_map( const sph_master_frame* self)
    CPL_ATTR_ALLOC;
cpl_image* sph_master_frame_get_rms( const sph_master_frame* self)
    CPL_ATTR_ALLOC;
cpl_image* sph_master_frame_get_variance( const sph_master_frame* self)
    CPL_ATTR_ALLOC;

double
sph_master_frame_get_absflux( const sph_master_frame* self, double* pstdev );
double
sph_master_frame_get_mean( const sph_master_frame* self, double* pstdev );

double
sph_master_frame_get_median( const sph_master_frame* self, double* pstdev );

double
sph_master_frame_get_mean_variance( const sph_master_frame* self,
                                    double* pstdev );
double
sph_master_frame_get_mean_rms( const sph_master_frame* self, double* pstdev );
double
sph_master_frame_get_median_variance( const sph_master_frame* self,
                                      double* pstdev );

cpl_error_code sph_master_frame_filter_median(sph_master_frame*, cpl_size);

#endif /*TECHCAL_master_frame_H_*/
