/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "sph_point_pattern.h"

#include "sph_fitting.h"

#include "irplib_utils.h"

#include <string.h>
#include <strings.h>
#include <math.h>
#include <limits.h>
#include <assert.h>
#include <stdlib.h>

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define SPH_POINT_PATTERN_COL1_NAME "POINT_ID"
#define SPH_POINT_PATTERN_COL2_NAME "POINT_X"
#define SPH_POINT_PATTERN_COL3_NAME "POINT_Y"
#define SPH_POINT_PATTERN_COL4_NAME "POINT_DX"
#define SPH_POINT_PATTERN_COL5_NAME "POINT_DY"



/*-----------------------------------------------------------------------------
                                   New types
 -----------------------------------------------------------------------------*/

struct sph_point_pattern_ {
    double * points;
    size_t alloc;
    size_t use;
};

/*-----------------------------------------------------------------------------
                           Private Function prototypes
 -----------------------------------------------------------------------------*/

inline static void sph_point_pattern_set_x(sph_point_pattern*,
                                           cpl_size, double) CPL_ATTR_NONNULL;

inline static void sph_point_pattern_set_y(sph_point_pattern*,
                                           cpl_size, double) CPL_ATTR_NONNULL;

static int sph_point_pattern_add_gauss_to_image__(cpl_image*,
                                                  double,
                                                  double,
                                                  double,
                                                  double);

static double sph_point_pattern_get_fwhm(const cpl_image * pp_image,
                                         double xcen, double ycen)
    CPL_ATTR_NONNULL;

/*-----------------------------------------------------------------------------
                              Function definitions
 -----------------------------------------------------------------------------*/

/* Read one line from standard input, */
/* copying it to line array (but no more than max chars). */
/* Does not place terminating \n in line array. */
/* Returns line length, or 0 for empty line, or EOF for end-of-file. */
static int sph_point_pattern_getline__(FILE* f,char line[], int max)
{
    int nch = 0;
    int c;
    max = max - 1;          /* leave room for '\0' */

    while((c = fgetc(f)) != EOF)
    {
        if(c == '\n')
            break;

        if(nch < max)
        {
            line[nch] = c;
            nch = nch + 1;
        }
    }

    if(c == EOF && nch == 0)
        return EOF;

    line[nch] = '\0';
    return nch;
}
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new point pattern object with specified preallocated space
 * @param size The number of elements with preallocated space
 * @return The new object
 */
/*----------------------------------------------------------------------------*/
sph_point_pattern* sph_point_pattern_new_(size_t size) {
    sph_point_pattern* self = (sph_point_pattern*)cpl_malloc(sizeof(*self));

    self->alloc = size;
    self->use   = 0;

    self->points = (double*)cpl_malloc(2 * self->alloc * sizeof(double));

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new point pattern object
 * @return The new object
 */
/*----------------------------------------------------------------------------*/
sph_point_pattern* sph_point_pattern_new(void) {

    return sph_point_pattern_new_(1024);
}

sph_point_pattern* sph_point_pattern_duplicate(const sph_point_pattern* self)
{
    sph_point_pattern* other = NULL;

    if (self != NULL) {
        other = (sph_point_pattern*)memcpy(cpl_malloc(sizeof(*self)), self,
                                           sizeof(*self));
        other->points = (double*)memcpy(cpl_malloc(2 * self->alloc * sizeof(double)),
                                        self->points,
                                        2 * self->use * sizeof(double));
    }

    return other;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the size of a point pattern object
 * @param  self The object to process
 * @return The size of the object
 */
/*----------------------------------------------------------------------------*/
inline int sph_point_pattern_get_size(const sph_point_pattern* self)
{
    return self ? (int)self->use : -1;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the X-component of the specified point pattern object element
 * @param  self The object to access
 * @param  i    The element to access
 * @return The X-component
 */
/*----------------------------------------------------------------------------*/
inline double sph_point_pattern_get_x(const sph_point_pattern* self,
                                      cpl_size i)
{
    return self->points[2 * i];
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the Y-component of the specified point pattern object element
 * @param  self The object to access
 * @param  i    The element to access
 * @return The Y-component
 */
/*----------------------------------------------------------------------------*/
inline double sph_point_pattern_get_y(const sph_point_pattern* self,
                                      cpl_size i)
{
    return self->points[2 * i + 1];
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Get a pointer to the specified point pattern object element
 * @param  self The object to access
 * @param  i    The element to access
 * @return The array of two double numbers
 */
/*----------------------------------------------------------------------------*/
inline const double* sph_point_pattern_get(const sph_point_pattern* self,
                                           cpl_size i)
{
    return self->points + 2 * i;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Set the X-component of the specified point pattern object element
 * @param  self The object to access
 * @param  i    The element to access
 * @param  val  The value to set
 */
/*----------------------------------------------------------------------------*/
inline static void sph_point_pattern_set_x(sph_point_pattern* self,
                                           cpl_size i, double val)
{
    self->points[2 * i] = val;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the Y-component of the specified point pattern object element
 * @param  self The object to access
 * @param  i    The element to access
 * @param  val  The value to set
 */
/*----------------------------------------------------------------------------*/
inline static void sph_point_pattern_set_y(sph_point_pattern* self,
                                           cpl_size i, double val)
{
    self->points[2 * i + 1] = val;
}


sph_error_code
sph_point_pattern_save_dfs(
        sph_point_pattern* self,
        const char* outfilename,
        cpl_frameset* allframes,
        cpl_frame* template_frame,
        cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        cpl_propertylist* plist )
{
    sph_error_code        rerr = CPL_ERROR_NONE;
    int np = sph_point_pattern_get_size(self);
    int ii = 0;
    sph_point* p = NULL;
    cpl_table*    tab = cpl_table_new(np);
    cpl_propertylist*       alist = NULL;

    alist = cpl_propertylist_new();
    if ( plist ) {
        cpl_propertylist_append(alist,plist);
    }
    cpl_table_new_column(tab, "POINT_ID", CPL_TYPE_INT);
    cpl_table_new_column(tab, "POINT_X", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab, "POINT_Y", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab, "POINT_DX", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab, "POINT_DY", CPL_TYPE_DOUBLE);
    for (ii = 0; ii < np; ++ii) {
        cpl_table_set(tab,"POINT_ID",ii,ii + 1);
        p = sph_point_pattern_get_point(self,ii);
        if ( p ) {
            cpl_table_set(tab,"POINT_X",ii,p->x);
            cpl_table_set(tab,"POINT_Y",ii,p->y);
            cpl_table_set(tab,"POINT_DX",ii,0.0);
            cpl_table_set(tab,"POINT_DY",ii,0.0);
            sph_point_delete(p);
        }
    }
    cpl_propertylist_append_string(alist,CPL_DFS_PRO_CATG,tag);
    cpl_dfs_save_table(allframes,
            NULL,
            params,
            allframes,
            template_frame,
            tab,
            plist,
            recipe,
            alist,
            NULL,
            pipename,outfilename);
    cpl_table_delete(tab);
    cpl_propertylist_delete(alist);
    return rerr;
}
sph_error_code
sph_point_pattern_save( sph_point_pattern* self, const char* filename ) {
    sph_error_code        rerr = CPL_ERROR_NONE;
    int np = sph_point_pattern_get_size(self);
    int ii = 0;
    sph_point* p = NULL;
    cpl_table*    tab = cpl_table_new(np);

    cpl_table_new_column(tab, "POINT_ID", CPL_TYPE_INT);
    cpl_table_new_column(tab, "POINT_X", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab, "POINT_Y", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab, "POINT_DX", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab, "POINT_DY", CPL_TYPE_DOUBLE);
    for (ii = 0; ii < np; ++ii) {
        cpl_table_set(tab,"POINT_ID",ii,ii + 1);
        p = sph_point_pattern_get_point(self,ii);
        if ( p ) {
            cpl_table_set(tab,"POINT_X",ii,p->x);
            cpl_table_set(tab,"POINT_Y",ii,p->y);
            cpl_table_set(tab,"POINT_DX",ii,0.0);
            cpl_table_set(tab,"POINT_DY",ii,0.0);
            sph_point_delete(p);
        }
    }
    cpl_table_save(tab,NULL,NULL,filename,CPL_IO_DEFAULT);
    cpl_table_delete(tab);
    return rerr;
}

sph_error_code
sph_point_pattern_save_ascii(
        sph_point_pattern* self,
        const char* filename ) {
    sph_error_code        rerr = CPL_ERROR_NONE;
    int np = sph_point_pattern_get_size(self);
    int ii = 0;
    sph_point* p = NULL;
    FILE*       f = NULL;

    cpl_ensure_code(filename,CPL_ERROR_NULL_INPUT);
    f = fopen(filename,"w");
    cpl_ensure_code(f,CPL_ERROR_FILE_IO);
    for (ii = 0; ii < np; ++ii) {
        p = sph_point_pattern_get_point(self,ii);
        if ( p ) {
            fprintf(f,"     %10.4f %10.4f\n",p->x, p->y);
            sph_point_delete(p);
        }
    }
    fclose(f); f = NULL;
    return rerr;
}

sph_point_pattern*
sph_point_pattern_load_ascii(const char* filename) {
    FILE*       f = NULL;
    char       line[256];
    double      clx = 0.0;
    double      cly = 0.0;
    int         nmatches = 0;
    int         linesize = 0;
    sph_point_pattern*    self = NULL;

    cpl_ensure(filename,CPL_ERROR_NULL_INPUT,NULL);
    f = fopen(filename,"r");
    cpl_ensure(f,CPL_ERROR_FILE_NOT_FOUND,NULL);

    self = sph_point_pattern_new();

    while ( (linesize = sph_point_pattern_getline__(f,line,256)) != EOF ) {
        if ( line[0]== '#' || linesize == 0 ) {
            ;
        }
        else {
            nmatches = sscanf(line,"%lf %lf",
                    &clx,&cly);
            if ( nmatches == 2 ) {
                sph_point_pattern_add_point(self,clx,cly);
            }
            else {
                SPH_ERROR_ENSURE_GOTO_EXIT(0,CPL_ERROR_BAD_FILE_FORMAT);
            }
        }
    }
    fclose(f); f = NULL;

    return self;

EXIT:
    sph_point_pattern_delete(self); self = NULL;
    return NULL;
}


sph_point_pattern* sph_point_pattern_new_from_table(const cpl_frame* framein,
                                                    int ext)
{
    const char* filename = framein ? cpl_frame_get_filename(framein) : NULL;
    sph_point_pattern* result = NULL;
    cpl_table*    tab = NULL;
    cpl_size nval;

    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT,    NULL);
    cpl_ensure(ext > 0,          CPL_ERROR_ILLEGAL_INPUT, NULL);

    tab = cpl_table_load(filename, ext, 0);
    if (tab == NULL) {
        (void)cpl_error_set_where(cpl_func);
        return NULL;
    }

    if (cpl_table_get_column_type(tab, SPH_POINT_PATTERN_COL1_NAME)
        != CPL_TYPE_INT) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                    "Point Pattern table %s extension %d has "
                                    "no integer-column %s", filename, ext,
                                    SPH_POINT_PATTERN_COL1_NAME);
        return NULL;
    }
    if (cpl_table_get_column_type(tab, SPH_POINT_PATTERN_COL2_NAME)
        != CPL_TYPE_DOUBLE) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                    "Point Pattern table %s extension %d has "
                                    "no double-column %s", filename, ext,
                                    SPH_POINT_PATTERN_COL2_NAME);
        return NULL;
    }
    if (cpl_table_get_column_type(tab, SPH_POINT_PATTERN_COL3_NAME)
        != CPL_TYPE_DOUBLE) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                    "Point Pattern table %s extension %d has "
                                    "no double-column %s", filename, ext,
                                    SPH_POINT_PATTERN_COL3_NAME);
        return NULL;
    }
    if (cpl_table_get_column_type(tab, SPH_POINT_PATTERN_COL4_NAME)
        != CPL_TYPE_DOUBLE) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                    "Point Pattern table %s extension %d has "
                                    "no double-column %s", filename, ext,
                                    SPH_POINT_PATTERN_COL4_NAME);
        return NULL;
    }
    if (cpl_table_get_column_type(tab, SPH_POINT_PATTERN_COL5_NAME)
        != CPL_TYPE_DOUBLE) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                    "Point Pattern table %s extension %d has "
                                    "no double-column %s", filename, ext,
                                    SPH_POINT_PATTERN_COL5_NAME);
        return NULL;
    }

    if (cpl_table_get_ncol(tab) != 5) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                    "Point Pattern table %s extension %d has "
                                    "%d != 5 columns", filename, ext,
                                    (int)cpl_table_get_ncol(tab));
        return NULL;
    }

    nval = cpl_table_get_nrow(tab);

    result = sph_point_pattern_new_(nval);

    for (cpl_size ii = 0; ii < nval; ii++) {
        int bv = 0;
        const double x =
            cpl_table_get_double(tab, SPH_POINT_PATTERN_COL2_NAME, ii, &bv);
        const double y =
            cpl_table_get_double(tab, SPH_POINT_PATTERN_COL3_NAME, ii, &bv);

        if (bv == 0) {
            sph_point_pattern_add_point(result,x,y);
        }
    }

    cpl_msg_info(cpl_func, "From %s loaded %d-point pattern from (%g,%g) over "
                 "center (%g,%g) to (%g,%g)", filename, (int)nval,
                 cpl_table_get_column_min(tab, SPH_POINT_PATTERN_COL2_NAME),
                 cpl_table_get_column_min(tab, SPH_POINT_PATTERN_COL3_NAME),
                 cpl_table_get_column_mean(tab, SPH_POINT_PATTERN_COL2_NAME),
                 cpl_table_get_column_mean(tab, SPH_POINT_PATTERN_COL3_NAME),
                 cpl_table_get_column_max(tab, SPH_POINT_PATTERN_COL2_NAME),
                 cpl_table_get_column_max(tab, SPH_POINT_PATTERN_COL3_NAME));

    assert(sph_point_pattern_get_size(result) == nval);

    cpl_table_delete(tab);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a point pattern from an image
 * @param image              The image to process
 * @param errim              The associated error image
 * @param sigma              The sigma threshold for the detection of each point
 * @param point_width_estimate The point width estimate
 * @param boxfactor          The boxfactor
 * @param boxborder          The boxborder
 * @param refsz              The number of reference points to ideally detect
 * @param pfwhm              Iff non-NULL, the median FWHM of the points
 * @return A new point pattern or NULL on error
 *
 */
/*----------------------------------------------------------------------------*/
sph_point_pattern* sph_point_pattern_new_from_image(
        const cpl_image* image,
        const cpl_image* errim,
        double sigma,
        double point_width_estimate,
        double boxfactor,
        double boxborder,
        cpl_size refsz,
        double * pfwhm)
{
    cpl_apertures    * aps = NULL;
    sph_point_pattern* result = NULL;
    double             mean_sz_aperture = 0.0;
    int                ii;
    const double       x_cent  = 0.5 * (1 + cpl_image_get_size_x(image));
    const double       y_cent  = 0.5 * (1 + cpl_image_get_size_y(image));
    cpl_image*         internal_errim = NULL;
    cpl_image*         internal_image = NULL;
    const cpl_image*   use_errim = errim;
    cpl_vector *       fwhm = NULL;
    cpl_size           nfwhm = 0;
    double             med_fwhm = 0.0;
    cpl_array        * out_pars;
    double           * out_parsd;
    double             mean_parsd[7] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    cpl_array        * do_pars;
    cpl_size           aps_sz;
    cpl_size           ngauss = 0;
    cpl_size           ncent  = 0;

    const cpl_type  type   = cpl_image_get_type(image);
    const cpl_size  nx     = cpl_image_get_size_x(image);
    const cpl_size  ny     = cpl_image_get_size_y(image);
    const void    * imdata = cpl_image_get_data_const(image);
    const cpl_size  nbad0  = cpl_image_count_rejected(image);
    cpl_size        nbad1;
    cpl_mask      * bpm;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    bpm = cpl_mask_new(nx, ny);

    /* Will not modify pixel buffer - but set a bpm with it */
    CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
    internal_image = cpl_image_wrap(nx, ny, type, (void*)imdata);
    CPL_DIAG_PRAGMA_POP;

    /* Reject anything non-positive (already bad ones remain bad) */
    cpl_mask_threshold_image(bpm, image, 0.0, DBL_MAX, CPL_BINARY_0);

    nbad1 = cpl_mask_count(bpm);

    if (nbad1 > 0) {
        const cpl_mask* oldbpm = cpl_image_set_bpm(internal_image, bpm);
        assert(oldbpm == NULL);
    } else {
        assert(nbad0 == 0);
        cpl_mask_delete(bpm);
    }

    if (use_errim == NULL) {
        internal_errim = cpl_image_abs_create(internal_image);
        cpl_image_power(internal_errim, 0.5);
        use_errim = internal_errim;
    } else {
        /* Need to check for non-positive errors */
        cpl_mask * ebpm = cpl_mask_new(nx, ny);

        cpl_mask_threshold_image(ebpm, use_errim, 0.0, DBL_MAX, CPL_BINARY_0);

        if (cpl_mask_is_empty(ebpm)) {
            cpl_mask_delete(ebpm);
        } else {
            internal_errim = cpl_image_duplicate(use_errim);
            cpl_image_set_bpm(internal_errim, ebpm);
            use_errim = internal_errim;
        }
    }

    if (nbad1 > nbad0) {
        cpl_msg_warning(cpl_func, "Rejecting %d non-positive pixels, total "
                        "rejected: %d/%d", (int)(nbad1 - nbad0), (int)nbad1,
                        (int)(nx*ny));
    } else {
        cpl_msg_info(cpl_func, "Image has %d/%d bad pixel(s)", (int)nbad0,
                        (int)(nx*ny));
    }

    aps = cpl_apertures_extract_sigma(image, sigma);
    if (aps == NULL) {
        SPH_RAISE_CPL;
        sph_error_raise( CPL_ERROR_DATA_NOT_FOUND,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_WARNING,
                "Could not find any points, "
                "using a sigma of %f", sigma);
        return NULL;
    }
    aps_sz = cpl_apertures_get_size(aps);
    if (aps_sz < 1) {
        sph_error_raise( CPL_ERROR_DATA_NOT_FOUND,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_WARNING,
                "Could not find any points, using a sigma of %f. "
                "Try lowering the sigma.", sigma);
        return NULL;
    }

    for (ii = 0; ii < aps_sz; ++ii) {
        mean_sz_aperture += cpl_apertures_get_npix( aps, ii + 1 );
    }

    mean_sz_aperture /= (double)aps_sz;

    /* Elements not set are flagged as invalid and not fitted */
    do_pars  = cpl_array_new(7, CPL_TYPE_INT);
    out_pars = cpl_array_new(7, CPL_TYPE_DOUBLE);
    cpl_array_set(do_pars, 2, 0);    /* Correlation is frozen */

    out_parsd = cpl_array_get_data_double(out_pars);

    /* Clear the invalid flag */
    cpl_array_set_double(out_pars, 2, 0.0);
    cpl_array_set_double(out_pars, 3, 0.0);
    cpl_array_set_double(out_pars, 4, 0.0);
    /* First first guess */
    cpl_array_set_double(out_pars, 5, point_width_estimate);
    cpl_array_set_double(out_pars, 6, point_width_estimate);

    result = sph_point_pattern_new_(aps_sz);
    if (pfwhm != NULL) {
        *pfwhm = 0.0;
        fwhm = cpl_vector_new(aps_sz);
    }

    for (ii = 0; ii < aps_sz; ++ii) {
        const cpl_size npix = cpl_apertures_get_npix(aps, ii+1);

        if (fabs((double)npix - mean_sz_aperture) < 40000.0) {
            cpl_errorstate prestate = cpl_errorstate_get();

            /* First guess parameters for Gaussian fit */
            /* Two-try approach w. mean of previous fits as 1st guess */
            const double   xpos0   = cpl_apertures_get_centroid_x(aps, ii + 1);
            const double   ypos0   = cpl_apertures_get_centroid_y(aps, ii + 1);
            const cpl_size ngauss0 = ngauss;


            out_parsd[3] = xpos0;
            out_parsd[4] = ypos0;

            if (ngauss > 0) {
                out_parsd[1] = mean_parsd[1]; /* Norm */
                out_parsd[5] = mean_parsd[5]; /* X-sigma */
                out_parsd[5] = mean_parsd[6]; /* Y-sigma */
            }

            for (int itry = 0; itry < 2; itry++) {

                const int code = sph_fitting_fit_gauss2D_all(image,
                                                             use_errim,
                                                             out_pars, do_pars,
                                                             boxfactor,
                                                             boxborder);

                if (!code) {
                    ngauss++; /* Fit succeeded */
                } else if (ngauss > 0) {
                    const double dist =
                        sqrt((xpos0 - x_cent) * (xpos0 - x_cent) +
                             (ypos0 - y_cent) * (ypos0 - y_cent));

                    /* Retry with default width as 1st guess */
                    cpl_msg_warning(cpl_func, "Re-trying fit %d/%d at distance "
                                    "from center %g w. default width at: "
                                    "(%g, %g)", 1+(int)ii, (int)aps_sz, dist,
                                    xpos0, ypos0);
                    if (!cpl_errorstate_is_equal(prestate)) {
                        cpl_errorstate_dump(prestate, CPL_FALSE, 
                                            cpl_errorstate_dump_one_warning);
                        cpl_errorstate_set(prestate);
                    }

                    /* Reinitialize, as outside loop */
                    cpl_array_fill_window_double(out_pars, 0, 3, 0.0);
                    cpl_array_fill_window_invalid(out_pars, 0, 2);

                    cpl_array_set_double(out_pars, 3, xpos0);
                    cpl_array_set_double(out_pars, 4, ypos0);
                    cpl_array_set_double(out_pars, 5, point_width_estimate);
                    cpl_array_set_double(out_pars, 6, point_width_estimate);

                    continue; /* Second try */
                } else if (!cpl_errorstate_is_equal(prestate)) {
                    cpl_errorstate_dump(prestate, CPL_FALSE, 
                                        cpl_errorstate_dump_one_warning);
                    cpl_errorstate_set(prestate);
                }
                break; /* Done, whether fitted or not */
            }

            if (ngauss > ngauss0) {
                /* Update the mean for a more robust first guess */
                for (cpl_size i = 0; i < 7; i++) {
                    mean_parsd[i] += (out_parsd[i] - mean_parsd[i])
                        / (double)(1 + ngauss);
                }
            } else {
                cpl_msg_warning(cpl_func, "Using centroid, not gaussian at "
                                "(%g, %g)", xpos0, ypos0);
                if (!cpl_errorstate_is_equal(prestate)) {
                    cpl_errorstate_dump(prestate, CPL_FALSE, 
                                        cpl_errorstate_dump_one_warning);
                    cpl_errorstate_set(prestate);
                }
                out_parsd[3] = xpos0;
                out_parsd[4] = ypos0;
                ncent++;
            }
            sph_point_pattern_add_point(result, out_parsd[3], out_parsd[4]);
            if (pfwhm != NULL) {
                const double fwhmi = sph_point_pattern_get_fwhm(image,
                                                                out_parsd[3],
                                                                out_parsd[4]);
                if (fwhmi > 0.0)
                    cpl_vector_set(fwhm, nfwhm++, fwhmi);
            }
        } else {
            cpl_msg_warning(cpl_func, "Ignoring aperture %d/%d with outlier "
                            "size: |%d - %g| >= %g", 1+ii, (int)aps_sz,
                            (int)npix, mean_sz_aperture, 40000.0);
        }
    }

    if (refsz > 0) {
        cpl_msg_info(cpl_func, "From %d apertures using sigma of %f and out of "
                     "%d reference points found %d point(s), %d Gaussian "
                     "fitted, %d centroid(s)", (int)aps_sz, sigma, (int)refsz,
                     sph_point_pattern_get_size(result),
                     (int)ngauss, (int)ncent);

    } else {
        cpl_msg_info(cpl_func, "From %d apertures using sigma of %f found %d "
                     "point(s), %d Gaussian fitted, %d centroid(s)",
                     (int)aps_sz, sigma, sph_point_pattern_get_size(result),
                     (int)ngauss, (int)ncent);
    }

    cpl_msg_info(cpl_func, "The mean of the %d fitted gaussians: %g, %g, "
                 "(%g,%g), (%g,%g)", (int)ngauss, mean_parsd[0], mean_parsd[1],
                 mean_parsd[3], mean_parsd[4], mean_parsd[5], mean_parsd[6]);

    if (pfwhm != NULL && nfwhm > 0) {
        cpl_vector * median = cpl_vector_wrap(nfwhm, cpl_vector_get_data(fwhm));
        double fmin = cpl_vector_get_min(median);
        double fmax = cpl_vector_get_max(median);
        med_fwhm = cpl_vector_get_median(median);
        cpl_vector_unwrap(median);

        cpl_msg_info(cpl_func, "The min < median < max FWHM of the %d point"
                     "(s): %g < %g < %g", (int)nfwhm, fmin, med_fwhm, fmax);
        *pfwhm = med_fwhm;
    }

    cpl_image_delete(internal_errim);
    (void)cpl_image_unwrap(internal_image); /* If present, also deletes bpm */

    cpl_apertures_delete(aps);
    cpl_array_delete(out_pars);
    cpl_array_delete(do_pars);
    cpl_vector_delete(fwhm);

    return result;
}

static double
sph_point_pattern_get_fwhm(const cpl_image * pp_image,
                           double xcen, double ycen)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    double xfwhm = 0.0;
    double yfwhm = 0.0;
    double fwhm = 0.0;

    if (cpl_image_get_fwhm(pp_image, (cpl_size)(0.5 + xcen),
                           (cpl_size)(0.5 + ycen), &xfwhm, &yfwhm)) {
        cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
        cpl_errorstate_set(prestate);
    } else if (xfwhm > 0.0 && yfwhm > 0.0) {
        fwhm = (xfwhm + yfwhm)/2.0;
    } else if (xfwhm > 0.0) {
        fwhm = xfwhm;
    } else if (yfwhm > 0.0) {
        fwhm = yfwhm;
    }

    return fwhm;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Append a point to the specified point pattern object
 * @param  self The object to access
 * @param  x    The X-component of the element to append
 * @param  y    The Y-component of the element to append
 * @return CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_point_pattern_add_point(sph_point_pattern* self,
                                           double x,
                                           double y)
{

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    if (self->use == self->alloc) {
        self->alloc *= 2;
        self->points = (double*)cpl_realloc((void*)self->points,
                                            2 * self->alloc * sizeof(double));
    }

    self->points[2 * self->use    ] = x;
    self->points[2 * self->use + 1] = y;

    self->use++;

    return CPL_ERROR_NONE;
}

sph_error_code
sph_point_pattern_rotate(
        sph_point_pattern* self,
        double cx, double cy,
        double angle )
{
    int vs = 0;
    int rerr = CPL_ERROR_NONE;
    int vv = 0;
    sph_point* apoint = NULL;
    double      sinang = sin(angle * CPL_MATH_RAD_DEG);
    double      cosang = cos(angle * CPL_MATH_RAD_DEG);

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    vs = sph_point_pattern_get_size(self);

    apoint = sph_point_new(0.0,0.0);
    cpl_ensure_code( apoint, CPL_ERROR_ILLEGAL_OUTPUT );

    for (vv = 0; vv < vs; ++vv) {
        apoint->x = sph_point_pattern_get_x( self, vv);
        apoint->y = sph_point_pattern_get_y( self, vv);
        sph_point_rotate_around( apoint, cx,cy,
                cosang, sinang, 0.0, 0.0);
        sph_point_pattern_set_x( self, vv, apoint->x);
        sph_point_pattern_set_y( self, vv, apoint->y);
    }
    sph_point_delete(apoint);
    return rerr;
}

sph_error_code
sph_point_pattern_scale_around(
        sph_point_pattern* self,
        double cx, double cy,
        double scale )
{
    int vs = 0;
    int rerr = CPL_ERROR_NONE;
    int vv = 0;
    sph_point* apoint = NULL;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    vs = sph_point_pattern_get_size(self);

    apoint = sph_point_new(0.0,0.0);
    cpl_ensure_code( apoint, CPL_ERROR_ILLEGAL_OUTPUT );

    for (vv = 0; vv < vs; ++vv) {
        apoint->x = sph_point_pattern_get_x( self, vv);
        apoint->y = sph_point_pattern_get_y( self, vv);

        apoint->x = ( apoint->x - cx ) * scale + cx;
        apoint->y = ( apoint->y - cy ) * scale + cy;

        sph_point_pattern_set_x( self, vv, apoint->x);
        sph_point_pattern_set_y( self, vv, apoint->y);
    }
    sph_point_delete(apoint);
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Shift the center of a point pattern to its center point
 * @param self   The point pattern to process
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_point_pattern_centralise(sph_point_pattern* self)
{
    double x0 = 0.0;
    double y0 = 0.0;

    if (sph_point_pattern_get_centre(self, &x0, &y0)) {
        return cpl_error_set_where(cpl_func);
    }

    if (sph_point_pattern_add_offset(self, -x0, -y0)) {
        return cpl_error_set_where(cpl_func);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Find the point closest to the Center of Origin (0,0) of a point pattern
 * @param self   The point pattern to process
 * @param px     Iff non-NULL and on success, pointer to the X-coordinate
 * @param py     Iff non-NULL and on success, pointer to the Y-coordinate
 * @return on success the index (zero for first), or negative on error
 *
 */
/*----------------------------------------------------------------------------*/
int sph_point_pattern_get_central_point(const sph_point_pattern* self,
                                        double* px, double* py)
{
    double x, x0 = 0.0;
    double y, y0 = 0.0;
    const int nsize = sph_point_pattern_get_size(self);
    double mindist = DBL_MAX;
    int minind = 0;

    if (sph_point_pattern_get_centre(self, &x0, &y0)) {
        (void)cpl_error_set_where(cpl_func);
        return -1;
    }

    for (int ii = 0; ii < nsize; ii++) {
        const double xx = sph_point_pattern_get_x(self, ii);
        const double yy = sph_point_pattern_get_y(self, ii);
        const double l = ( xx - x0 ) * ( xx - x0 ) + ( yy - y0 ) * ( yy - y0 );

        if ( l < mindist ) {
            mindist = l;
            minind = ii;
        }
    }

    x = sph_point_pattern_get_x(self, minind);
    y = sph_point_pattern_get_y(self, minind);

    cpl_msg_info(cpl_func, "Center point %d/%d at (%g, %g) is %g pixel(s) "
                 "from points center (%g, %g)", minind+1, nsize, x, y,
                 sqrt(mindist), x0, y0);

    if (px != NULL) *px = x;
    if (py != NULL) *py = y;

    return minind;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute the center of a point pattern
 * @param self   The point pattern to process
 * @param x      On success, pointer to the X-coordinate
 * @param y      On success, pointer to the Y-coordinate
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_point_pattern_get_centre(const sph_point_pattern* self,
                                            double* x, double* y )
{
    int nsize;

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(x    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(y    != NULL, CPL_ERROR_NULL_INPUT);

    nsize = sph_point_pattern_get_size(self);

    if (nsize > 0) {
        double x0 = 0.0;
        double y0 = 0.0;

        for (int ii = 0; ii < nsize; ++ii) {
            x0 += sph_point_pattern_get_x(self, ii);
            y0 += sph_point_pattern_get_y(self, ii);
        }

        *x = x0 / (double)nsize;
        *y = y0 / (double)nsize;
    } else {
        return cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add a point pattern to another
 * @param self   The point pattern to modify
 * @param other  The point pattern to add
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_point_pattern_add(sph_point_pattern* self,
                                     const sph_point_pattern* other)
{
    const int nsize = sph_point_pattern_get_size(self);

    cpl_ensure_code(self  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(other != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sph_point_pattern_get_size(other) == nsize,
                    CPL_ERROR_INCOMPATIBLE_INPUT);

    for (cpl_size i = 0; i < nsize; i++) {
        const double* xyi = sph_point_pattern_get(self,  i);
        const double* xyj = sph_point_pattern_get(other, i);

        sph_point_pattern_set_x(self, i, xyi[0] + xyj[0]);
        sph_point_pattern_set_y(self, i, xyi[1] + xyj[1]);
    }

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Add an offset to a point pattern
 * @param self   The point pattern to modify
 * @param x      The X-coordinate to add
 * @param y      The Y-coordinate to add
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_point_pattern_add_offset(sph_point_pattern* self, double x0, double y0)
{
    const int nsize = sph_point_pattern_get_size(self);

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    for (int ii = 0; ii < nsize; ii++) {
        const double xx = sph_point_pattern_get_x(self, ii);
        const double yy = sph_point_pattern_get_y(self, ii);
        sph_point_pattern_set_x(self, ii, xx + x0);
        sph_point_pattern_set_y(self, ii, yy + y0);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Shift the center of a point pattern to its center
 * @param self   The point pattern to process
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_point_pattern_centralise_wrt_central_point(sph_point_pattern* self)
{
    double x0 = 0.0;
    double y0 = 0.0;

    if (sph_point_pattern_get_central_point(self, &x0, &y0) < 0) {
        return cpl_error_set_where(cpl_func);
    }

    if (sph_point_pattern_add_offset(self, -x0, -y0)) {
        return cpl_error_set_where(cpl_func);
    }

    return CPL_ERROR_NONE;
}

sph_point*
sph_point_pattern_get_point(const sph_point_pattern* self, int n)
{
    sph_point* result = NULL;

    if (self != NULL) {

        if (n < 0 ||
            n > sph_point_pattern_get_size(self) ||
            n > sph_point_pattern_get_size(self)) {
            return NULL;
        }
        result = sph_point_new(sph_point_pattern_get_x(self, n),
                               sph_point_pattern_get_y(self, n));
    }
    return result;
}

cpl_size sph_point_pattern_find_closest(const sph_point_pattern* self,
                                        const double xy[2])
{
    const cpl_size np = sph_point_pattern_get_size(self);
    double mindist = DBL_MAX;
    cpl_size best_ind = -1;

    for (cpl_size i = 0; i < np; ++i) {
        const double* xyi = sph_point_pattern_get(self, i);
        const double dist = (xy[0] - xyi[0]) * (xy[0] - xyi[0])
                          + (xy[1] - xyi[1]) * (xy[1] - xyi[1]);

        if (dist < mindist) {
            mindist = dist;
            best_ind = i;
        }
    }

    if (best_ind < 0) {
        (void)cpl_error_set(cpl_func, self == NULL
                            ? CPL_ERROR_NULL_INPUT
                            : CPL_ERROR_DATA_NOT_FOUND);
        return -1;
    }

    return best_ind;
}

//cpl_image*
//sph_point_pattern_create_image( sph_point_pattern* self, int nx, int ny, double psize ) {
//    cpl_image* image = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE );
//    cpl_image* smallim = NULL;
//    cpl_image* extractim = NULL;
//    int size0 = (int)psize * 20;
//    int size = size0;
//    int ddx = 0;
//    int ddy = 0;
//    sph_error_code rerr = CPL_ERROR_NONE;
//    double xpos = 0.0;
//    double ypos = 0.0;
//    double xpos0 = 0.0;
//    double ypos0 = 0.0;
//    int ii = 0;
//    for ( ii = 0; ii < sph_point_pattern_get_size(self); ++ii) {
//        xpos = sph_point_pattern_get_x(self, ii);
//        ypos = sph_point_pattern_get_y(self, ii);
//        size = size0;
//        xpos0 = xpos - size * 0.5;
//        ypos0 = ypos - size * 0.5;
//        if ( xpos0 + size < 0 || ypos0 + size < 0 ) {
//            continue;
//        }
//        if ( xpos0 >= nx || ypos0 >= ny ) {
//            continue;
//        }
//        if ((int)xpos0 + size >= nx) {
//            size = nx - (int)xpos0;
//        }
//        if ((int)ypos0 + size >= ny)
//            size = ny - (int)ypos0;
//        if ((int)xpos0 < 0)
//            xpos0 = 0;
//        if ((int)ypos0 < 0)
//            ypos0 = 0;
//        ddx = xpos - (int)(xpos0);
//        ddy = ypos - (int)(ypos0);
//        smallim = cpl_image_new(size,size,CPL_TYPE_DOUBLE);
//        rerr = cpl_image_fill_gaussian( smallim, ddx + 1.0,
//                ddy + 1.0,
//                1000.0,
//                psize, psize );
//        extractim = cpl_image_extract(image,(int)(xpos0) + 1, (int)(ypos0) + 1,
//                (int)(xpos0) + size, (int)(ypos0) + size);
//        rerr = cpl_image_add(extractim,smallim);
//        rerr = cpl_image_copy( image, extractim, (int)(xpos0) + 1, (int)(ypos0) + 1);
//        cpl_image_delete(smallim);smallim = NULL;
//        cpl_image_delete(extractim);extractim = NULL;
//    }
//    return image;
//}

cpl_image*
sph_point_pattern_create_image( sph_point_pattern* self, int nx, int ny, double psize )
{
    cpl_image* image = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE );
    int         ii      = 0;

    for (ii = 0; ii < sph_point_pattern_get_size(self); ++ii) {
        sph_point_pattern_add_gauss_to_image__(
                image,
                sph_point_pattern_get_x(self, ii),
                sph_point_pattern_get_y(self, ii),
                psize, 1000.0
                );
    }
    return image;
}

int
sph_point_pattern_find_closest_neighbour( sph_point_pattern* self, int n, double* pdist ) {
    double d = 0.0;
    double xx = 0.0;
    double yy = 0.0;
    double x = 0.0;
    double y = 0.0;
    double mind = -1.0;
    int best_ind = -1;
    int ii;
    if (!self )
        return -1;

    if ( n > sph_point_pattern_get_size(self) ||
            n > sph_point_pattern_get_size(self) ||
            n < 0 ) {
        return -1;
    }
    x = sph_point_pattern_get_x(self, n);
    y = sph_point_pattern_get_y(self, n);
    for (ii = 0; ii < sph_point_pattern_get_size(self); ++ii) {
        if ( ii != n ) {
            xx = sph_point_pattern_get_x(self, ii);
            yy = sph_point_pattern_get_y(self, ii);
            d = (x - xx) * (x - xx) + (y - yy) * (y - yy);
            if (mind < -0.5 || d < mind) {
                mind = d;
                best_ind = ii;
            }
        }
    }
    if ( best_ind < 0 ||
         best_ind > sph_point_pattern_get_size(self) ||
         best_ind > sph_point_pattern_get_size(self) ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not find a point.");
        return -1;
    }
    if ( pdist ) {
        *pdist = sqrt(mind);
    }
    return best_ind;
}

sph_point_pattern*
sph_point_pattern_find_square( const sph_point_pattern* self ) {
    sph_point_pattern* result = NULL;
    cpl_vector*        pairs_a    = NULL;
    cpl_vector*        pairs_b    = NULL;
    cpl_vector*        distances    = NULL;
    cpl_vector*        dotprods    = NULL;
    int             npairs = 0;
    double            score = 1.0e10;
    double             best_score = -1.0;
    int                point_index[4];
    int                best_index[4] = {0, 0, 0, 0};
    int                ii,jj;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);

    cpl_ensure( sph_point_pattern_make_pairs(self,&pairs_a,&pairs_b,&distances,&dotprods) == CPL_ERROR_NONE,
            cpl_error_get_code(), NULL);
    npairs = cpl_vector_get_size(pairs_a);

    if ( npairs > 0 ) {
        result = sph_point_pattern_new_(4);
        for ( ii = 0; ii < npairs; ++ii ) {
            for ( jj = ii + 1; jj < npairs; ++jj ) {
                // Only look at pairs that have no point in common
                point_index[0] = cpl_vector_get(pairs_a,ii);
                point_index[1] = cpl_vector_get(pairs_b,ii);
                point_index[2] = cpl_vector_get(pairs_a,jj);
                point_index[3] = cpl_vector_get(pairs_b,jj);
                if ( point_index[0] != point_index[2] &&
                     point_index[0] != point_index[3] &&
                     point_index[1] != point_index[2] &&
                     point_index[1] != point_index[3] )
                {
                    score = sph_point_pattern_score_square(self,point_index);
                    if ( score < best_score || best_score < 0.0 ) {
                        best_index[0] = point_index[0];
                        best_index[1] = point_index[1];
                        best_index[2] = point_index[2];
                        best_index[3] = point_index[3];
                        best_score = score;
                    }
                }
            }
        }
    }
    if ( best_score > -1.0 ) {
        for ( ii = 0; ii < 4; ++ii ) {
            sph_point_pattern_add_point(result,
                    sph_point_pattern_get_x(self, 
                            best_index[ii]),
                    sph_point_pattern_get_y(self, 
                            best_index[ii]));
        }
    }
    cpl_vector_delete(pairs_a);
    cpl_vector_delete(pairs_b);
    cpl_vector_delete(distances);
    cpl_vector_delete(dotprods);
    return result;
}

double sph_point_pattern_dotprod( const sph_point_pattern* self, int pA, int pB, int pC) {
    double xAB = sph_point_pattern_get_x(self, pA) - sph_point_pattern_get_x(self, pB);
    double xCB = sph_point_pattern_get_x(self, pC) - sph_point_pattern_get_x(self, pB);
    double yAB = sph_point_pattern_get_y(self, pA) - sph_point_pattern_get_y(self, pB);
    double yCB = sph_point_pattern_get_y(self, pC) - sph_point_pattern_get_y(self, pB);
    double dotprodA = (xAB * xCB + yAB * yCB) / ( xAB * xAB + yAB * yAB);
    double xAC = sph_point_pattern_get_x(self, pA) - sph_point_pattern_get_x(self, pC);
    double xBC = sph_point_pattern_get_x(self, pB) - sph_point_pattern_get_x(self, pC);
    double yAC = sph_point_pattern_get_y(self, pA) - sph_point_pattern_get_y(self, pC);
    double yBC = sph_point_pattern_get_y(self, pB) - sph_point_pattern_get_y(self, pC);
    double dotprodB = (xAC * xBC + yAC * yBC) / ( xAC * xAC + yAC * yAC);
    return dotprodA < dotprodB ? dotprodA : dotprodB;
}

sph_error_code
sph_point_pattern_make_pairs( const sph_point_pattern* self,
        cpl_vector** ppairs_a, cpl_vector** ppairs_b,
        cpl_vector** pdistances,
        cpl_vector** pdotprods)
{
    int             npoints = 0;
    int             ii         = 0;
    int                jj         = 0;
    cpl_vector*        pairs_a    = NULL;
    cpl_vector*        pairs_b    = NULL;
    cpl_vector*        distances    = NULL;
    cpl_vector*        dotprods    = NULL;
    double            dx = 0.0;
    double             dy = 0.0;
    int                count = 0;

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ppairs_a,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ppairs_b,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pdistances,CPL_ERROR_NULL_INPUT);

    npoints = sph_point_pattern_get_size(self);
    cpl_ensure_code(npoints > 1, CPL_ERROR_ILLEGAL_INPUT);

    pairs_a = *ppairs_a = cpl_vector_new((npoints * (npoints - 1 )) / 2);
    pairs_b = *ppairs_b = cpl_vector_new((npoints * (npoints - 1 )) / 2);
    distances = *pdistances = cpl_vector_new((npoints * (npoints - 1)) / 2);
    dotprods = *pdotprods = cpl_vector_new((npoints * (npoints - 1)) / 2);
    count = 0;
    for ( ii = 0; ii < npoints; ++ii ) {
        for ( jj = ii + 1; jj < npoints; ++jj ) {
            cpl_vector_set( pairs_a, count, ii);
            cpl_vector_set( pairs_b, count, jj);
            dx = sph_point_pattern_get_x(self, jj) - sph_point_pattern_get_x(self, ii);
            dy = sph_point_pattern_get_y(self, jj) - sph_point_pattern_get_y(self, ii);
            cpl_vector_set( distances, count, sqrt( dx * dx + dy * dy) );
            if ( dx * dx < dy * dy ) { // Pick either the most veritcal...
                cpl_vector_set( dotprods, count, dx * dx / sqrt( dx * dx + dy * dy) );
            }
            else { // or horizontal pair
                cpl_vector_set( dotprods, count, dy * dy / sqrt( dx * dx + dy * dy) );
            }
            ++count;
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

double
sph_point_pattern_score_square( const sph_point_pattern* self,
        const int* p)
{
    double dx;
    double dy;
    double sqdistance[4][4];
    double score = 0.0;
    int i,j,ii,jj;
    int oldsize = 0;
    cpl_vector* dists;
    cpl_vector* dotprods;

    for ( i = 0; i < 4; ++i ) {
        for ( j = i + 1; j < 4; ++j ) {
            dx = sph_point_pattern_get_x(self, p[i]) - sph_point_pattern_get_x(self, p[j]);
            dy = sph_point_pattern_get_y(self, p[i]) - sph_point_pattern_get_y(self, p[j]);
            sqdistance[i][j] = dx * dx + dy * dy;
        }
    }
    dotprods = cpl_vector_new(3);
    cpl_vector_set( dotprods, 0, sph_point_pattern_dotprod( self, p[0],p[1],p[2] ) );
    cpl_vector_set( dotprods, 1, sph_point_pattern_dotprod( self, p[0],p[1],p[3] ) );
    cpl_vector_set( dotprods, 2, sph_point_pattern_dotprod( self, p[1],p[2],p[3] ) );

    cpl_vector_power(dotprods,2);

    cpl_vector_sort(dotprods,CPL_SORT_ASCENDING);
    for ( i = 0; i < 2; ++i ) {
        score += cpl_vector_get(dotprods,i);
    }
    dists = cpl_vector_new(1);
    for ( i = 0; i < 4; ++i ) {
        for ( j = i + 1; j < 4; ++j ) {
            for ( ii = 0; ii < 4; ++ii ) {
                for ( jj = ii + 1; jj < 4; ++jj ) {
                    oldsize = cpl_vector_get_size(dists);
                    cpl_vector_set(dists,oldsize-1,
                            (sqdistance[i][j] - sqdistance[ii][jj]) * (sqdistance[i][j] - sqdistance[ii][jj]) /
                            ((sqdistance[i][j] + sqdistance[ii][jj]) * (sqdistance[i][j] + sqdistance[ii][jj])));
                    cpl_vector_set_size(dists,oldsize+1);
                }
            }
        }
    }
    cpl_vector_set_size(dists,oldsize);
    cpl_vector_sort(dists,CPL_SORT_ASCENDING);
    for ( i = 0; i < 20; ++i ) {
        score += cpl_vector_get(dists,i);
    }
    cpl_vector_delete(dists);
    cpl_vector_delete(dotprods);
    return score;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete a point pattern object
 * @param self    The object to delete
 */
/*----------------------------------------------------------------------------*/
void sph_point_pattern_delete(sph_point_pattern* self) {
    if ( self != NULL) {
        cpl_msg_info(cpl_func, "Deleting PPT: %d <= %d",
                     (int)self->use, (int)self->alloc);
        cpl_free(self->points);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Draw lines between the point pattern 
 * @param image   The image to draw onto
 * @param self    The point pattern to draw from
 * @param radius  The non-negative drawing radius
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 * @see sph_point_pattern_draw_save()
 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code sph_point_pattern_draw_lines(cpl_image* image,
                                            const sph_point_pattern* self,
                                            double radius)
{
    const cpl_size np = (int)sph_point_pattern_get_size(self);
    const cpl_size nx = (int)cpl_image_get_size_x(image);
    const cpl_size ny = (int)cpl_image_get_size_y(image);
    const double imax = cpl_image_get_max(image);
    double* xvals = (double*)cpl_malloc(2*(size_t)np*sizeof(*xvals));
    double* yvals = xvals + (size_t)np;
    double xprev, yprev;
    cpl_vector* vwrap;
    cpl_size nr = 0, nc = 0;

    for (cpl_size i = 0; i < np; i++) {
        const double* pxy = sph_point_pattern_get(self, i);
        xvals[i] = pxy[0];
        yvals[i] = pxy[1];
    }

    vwrap = cpl_vector_wrap(np, xvals);
    (void)cpl_vector_sort(vwrap, CPL_SORT_ASCENDING);
    (void)cpl_vector_unwrap(vwrap);

    vwrap = cpl_vector_wrap(np, yvals);
    (void)cpl_vector_sort(vwrap, CPL_SORT_ASCENDING);
    (void)cpl_vector_unwrap(vwrap);

    xprev = yprev = 1.0;
    for (cpl_size k = 0; k < np; k++) {
        if (xvals[k] >= xprev + radius) {
            /* Draw a line here */
            const cpl_size i = (cpl_size)(0.5 * (1.0 + xvals[k] + xprev));

            if (1 <= i && i <= nx) {
                nr++;
                for (cpl_size j = 1; j <= ny; j++) {
                    (void)cpl_image_set(image, i, j, imax);
                }
            }
        }
        if (yvals[k] >= yprev + radius) {
            /* Draw a line here */
            const cpl_size j = (cpl_size)(0.5 * (1.0 + yvals[k] + yprev));

            if (1 <= j && j <= ny) {
                nc++;
                for (cpl_size i = 1; i <= nx; i++) {
                    (void)cpl_image_set(image, i, j, imax);
                }
            }
        }

        xprev = xvals[k];
        yprev = yvals[k];
    }

    cpl_msg_info(cpl_func, "Drew %d horizontal and %d vertical line(s)"
                 " onto %d X %d image", (int)nr, (int)nc, (int)nx, (int)ny);

    cpl_free(xvals);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Draw circles onto an image around each point in a pattern
 * @param image   The image to draw onto
 * @param self    The point pattern to draw
 * @param radius  The non-negative drawing radius
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 * @see sph_point_pattern_draw_save()
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code sph_point_pattern_draw(cpl_image* image,
                                             const sph_point_pattern* self,
                                             double radius)
{

    const int np = sph_point_pattern_get_size(self);
    const int nx = (int)cpl_image_get_size_x(image);
    const int ny = (int)cpl_image_get_size_y(image);
    const double imax = cpl_image_get_max(image);
    double xmin = 1e9, xmax = -1e9;
    double ymin = 1e9, ymax = -1e9;
    int mp = 0;


    cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(radius >= 0.0, CPL_ERROR_ILLEGAL_INPUT);

    for (int k = 0; k < np; k++) {
        const double* pxy = sph_point_pattern_get(self, k);
        cpl_boolean ok = CPL_FALSE;

        if (pxy[0] < xmin) {
            xmin = pxy[0];
        }
        if (pxy[0] > xmax) {
            xmax = pxy[0];
        }
        if (pxy[1] < ymin) {
            ymin = pxy[0];
        }
        if (pxy[1] > ymax) {
            ymax = pxy[1];
        }

        for (int i = CPL_MAX(1, pxy[0] + floor(-radius));
             i <= CPL_MIN(nx, pxy[0] + ceil(radius)); i++) {

            for (int j = CPL_MAX(1, pxy[1] + floor(-radius));
                 j <= CPL_MIN(ny, pxy[1] + ceil(radius)); j++) {
                const double dist = sqrt((i - pxy[0]) * (i - pxy[0]) +
                                         (j - pxy[1]) * (j - pxy[1]));
                if (fabs(dist - radius) < 0.5) {
                    (void)cpl_image_set(image, i, j, imax);
                    ok = CPL_TRUE;
                }
            }
        }
        if (ok) {
            mp++;
        } else {
            cpl_msg_warning(cpl_func, "Could not draw point %d/%d at (%g,%g) "
                            "onto %d X %d image", k+1, np, pxy[0], pxy[1],
                            nx, ny);
        }
    }

    cpl_msg_info(cpl_func, "Drew %d/%d point(s) ranging from (%g,%g) to (%g,%g)"
                 " onto %d X %d image", mp, np, xmin, ymin, xmax, ymax, nx, ny);

    sph_point_pattern_draw_lines(image,self, radius);


    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extend FITS file with Draw extensions
 * @param pp_detect1  The 1st point pattern of the detected objects
 * @param imgdetect1  The 1st background image with detected objects
 * @param pp_detect2  The 2nd point pattern of the detected objects, or NULL
 * @param imgdetect2  The 2nd background image with detected objects, or NULL
 * @param pp_correct  The reference point pattern, or NULL
 * @param imgcorrect1 The 1st distortion corrected image, or NULL
 * @param optaxis1_x  The X-coordinate of the optical axis in the 1st image
 * @param optaxis1_y  The Y-coordinate of the optical axis in the 1st image
 * @param imgcorrect2 The 2nd distortion corrected image, or NULL
 * @param optaxis2_x  The X-coordinate of the optical axis in the 2nd image
 * @param optaxis2_y  The Y-coordinate of the optical axis in the 2nd image
 * @param radius     The non-negative drawing radius
 * @param filename   The FITS file to extend
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 * @see sph_point_pattern_draw()
 * @note All non-NULL images must have identical type and sizes
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_point_pattern_draw_save(const sph_point_pattern* pp_detect1,
                                           const cpl_image* imgdetect1,
                                           const sph_point_pattern* pp_detect2,
                                           const cpl_image* imgdetect2,
                                           const sph_point_pattern* pp_correct,
                                           const cpl_image* imgcorrect1,
                                           double optaxis1_x,
                                           double optaxis1_y,
                                           const cpl_image* imgcorrect2,
                                           double optaxis2_x,
                                           double optaxis2_y,
                                           double radius,
                                           const char* filename)
{
    cpl_image* img_draw;
    cpl_propertylist * extlist = cpl_propertylist_new();
    const char* lab_detect  = imgdetect2  ? "Detected left" : "Detected image";


    cpl_propertylist_append_string(extlist, "EXTNAME", lab_detect);
    img_draw = cpl_image_duplicate(imgdetect1);
    sph_point_pattern_draw(img_draw, pp_detect1, radius);
    skip_if(cpl_image_save(img_draw, filename, CPL_TYPE_UNSPECIFIED, extlist,
                           CPL_IO_EXTEND));

    if (imgdetect2 != NULL && pp_detect2 != NULL) {

        cpl_propertylist_set_string(extlist, "EXTNAME", "Detected right");
        cpl_image_copy(img_draw, imgdetect2, 1, 1);
        sph_point_pattern_draw(img_draw, pp_detect2, radius);
        skip_if(cpl_image_save(img_draw, filename, CPL_TYPE_UNSPECIFIED,
                               extlist, CPL_IO_EXTEND));
    }

    if (pp_correct != NULL) {
        const char* lab_correct = imgcorrect2
            ? "Corrected left"
            : "Corrected image";

        if (imgcorrect1 != NULL) {
            sph_point_pattern* ref_pp = sph_point_pattern_duplicate(pp_correct);

            sph_point_pattern_add_offset(ref_pp, optaxis1_x, optaxis1_y);

            cpl_propertylist_set_string(extlist, "EXTNAME", lab_correct);
            cpl_image_copy(img_draw, imgcorrect1, 1, 1);
            sph_point_pattern_draw(img_draw, ref_pp, radius);
            skip_if(cpl_image_save(img_draw, filename, CPL_TYPE_UNSPECIFIED,
                                   extlist, CPL_IO_EXTEND));
            sph_point_pattern_delete(ref_pp);
        }

        if (imgcorrect2 != NULL) {
            sph_point_pattern* ref_pp = sph_point_pattern_duplicate(pp_correct);

            sph_point_pattern_add_offset(ref_pp, optaxis2_x, optaxis2_y);

            cpl_propertylist_set_string(extlist, "EXTNAME", "Corrected right");
            cpl_image_copy(img_draw, imgcorrect2, 1, 1);
            sph_point_pattern_draw(img_draw, ref_pp, radius);
            skip_if(cpl_image_save(img_draw, filename, CPL_TYPE_UNSPECIFIED,
                                   extlist, CPL_IO_EXTEND));

            if (imgcorrect1 != NULL) {
                const double fact1 = cpl_image_get_flux(imgcorrect1);
                const double fact2 = cpl_image_get_flux(imgcorrect2);

                assert( fact1 != 0.0);

                cpl_image_copy(img_draw, imgcorrect1, 1, 1);
                cpl_image_multiply_scalar(img_draw, fact2 / fact1);

                cpl_image_subtract(img_draw, imgcorrect2);
                cpl_propertylist_set_string(extlist, "EXTNAME",
                                            "Corrected scaled left - right");
                sph_point_pattern_draw(img_draw, ref_pp, radius);
                skip_if(cpl_image_save(img_draw, filename, CPL_TYPE_UNSPECIFIED,
                                       extlist, CPL_IO_EXTEND));
            }
            sph_point_pattern_delete(ref_pp);
        }
    }

    end_skip;

    cpl_image_delete(img_draw);
    cpl_propertylist_delete(extlist);

    return CPL_ERROR_NONE;

}


/*----------------------------------------------------------------------------*/
/**
 * @brief Sort the elements according to their distance from the center
 * @param self      The point pattern to sort
 * @return Nothing
 */
/*----------------------------------------------------------------------------*/
static int sph_point_pattern_compare_dist(const void * p1, const void * p2)
{

    if (*(const double*)p1 < *(const double*)p2) {
        return -1;
    } else if (*(const double*)p1 > *(const double*)p2) {
        return 1;
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Sort the elements according to their distance from the center
 * @param self      The point pattern to sort
 * @return Nothing
 */
/*----------------------------------------------------------------------------*/
void sph_point_pattern_sort(sph_point_pattern* self)
{
    const cpl_size n = sph_point_pattern_get_size(self);

    if (n > 1) {
        double * tricol = (double*)cpl_malloc((size_t)n * 3 * sizeof(tricol));

        for (cpl_size i = 0; i < n; i++) {
            const double sqdist = self->points[2 * i] * self->points[2 * i]
                + self->points[2 * i + 1] * self->points[2 * i + 1];

            tricol[3 * i    ] = sqdist;
            tricol[3 * i + 1] = self->points[2 * i];
            tricol[3 * i + 2] = self->points[2 * i + 1];
        }

        qsort(tricol, n, 3 * sizeof(double), sph_point_pattern_compare_dist);

        for (cpl_size i = 0; i < n; i++) {
            self->points[2 * i    ] = tricol[3 * i + 1];
            self->points[2 * i + 1] = tricol[3 * i + 2];
        }

        cpl_free(tricol);
    }
}

/**@}*/

static
int
sph_point_pattern_add_gauss_to_image__( cpl_image* inimage,
        double dxpos,
        double dypos,
        double psize,
        double value )
{
    int size0 = (int)psize * 20;
    int size;
    int xpos = (int)dxpos;
    int ypos = (int)dypos;
    int ypos0;
    int xpos0;
    int nx = cpl_image_get_size_x(inimage);
    int ny = cpl_image_get_size_y(inimage);
    double ddx;
    double ddy;
    sph_error_code        rerr = CPL_ERROR_NONE;
    cpl_image*    smallim = NULL;
    cpl_image*  extractim = NULL;

    cpl_ensure_code(inimage,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    if ( size0 < 2 ) {
        SPH_ERR("Too small gaussian to add to image.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    size = size0;
    xpos0 = xpos - size * 0.5;
    ypos0 = ypos - size * 0.5;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    if ((int)xpos0 < 0)
        xpos0 = 0;
    if ((int)ypos0 < 0)
        ypos0 = 0;

    if ((int)xpos0 + size >= nx) {
        size = nx - (int)xpos0;
    }
    if ((int)ypos0 + size >= ny)
        size = ny - (int)ypos0;
    if ( size  < 1 ) return CPL_ERROR_NONE;
    ddx = dxpos - (int)(xpos0);
    ddy = dypos - (int)(ypos0);
    smallim = cpl_image_new(size,size,CPL_TYPE_DOUBLE);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    //SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"ddx=%f,ddy=%f",ddx,ddy);
    rerr = cpl_image_fill_gaussian( smallim, ddx,
                             ddy,
                             value,
                             psize, psize );
    SPH_RAISE_CPL_RESET
    extractim = cpl_image_extract(inimage,(int)(xpos0) + 1, (int)(ypos0) + 1,
                                  (int)(xpos0) + size, (int)(ypos0) + size);
    if ( extractim == NULL ) {
        SPH_RAISE_CPL;
        SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,"Could not extract image size %d from %d,%d.",size,(int)xpos0,(int)ypos0);
        return SPH_ERROR_GENERAL;
    }
    rerr = cpl_image_add(extractim,smallim);
    SPH_RAISE_CPL_RESET
    rerr = cpl_image_copy( inimage, extractim, (int)(xpos0) + 1, (int)(ypos0) + 1);
    cpl_image_delete(smallim);smallim = NULL;
    cpl_image_delete(extractim);extractim = NULL;
    SPH_RAISE_CPL_RESET;
    return rerr;
}
