/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_pixel_polyfit_table.h"
#include "sph_cube.h"
#include "sph_framelist.h"
#include "sph_fitting.h"
#include "sph_smart_imagelist.h"
#include "sph_ptc.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Calculate the means on a set of frames (cubes)
 * @param rawframes the input set of cubes
 * @param collalg the collapse algo
 * @param params the collapse parameters
 * @param mask an optional bad pixel mask (may be NULL)
 * @param stdev optional output vector of errors on means (may be NULL)
 * @param errs optional output vecotr of mean of rms (may be NULL)
 *
 * @return vector of means
 *
 * This function calculates all the means and optionally errors on means
 * and the mean RMS values for the set of input cubes. The Input cubes
 * are collapsed with the given collapse algorithm.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_vector* sph_ptc_calculate_means(
        cpl_frameset* rawframes,
        sph_collapse_algorithm collalg,
        cpl_parameterlist* params,
        cpl_mask*   mask,
        cpl_vector** stdev,
        cpl_vector** errs)
{
    cpl_vector*         means   = NULL;
    int                 ff      = 0;
    double              mean    = 0.0;
    double              mean_stdev   = 0.0;
    sph_master_frame*   temp_master = NULL;
    cpl_frame*          frame = NULL;
    int                 plane = 0;
    double              rms = 0.0;
    double              rms_stdev = 0.0;
    cpl_frameset*       single_frameset = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    means = cpl_vector_new(cpl_frameset_get_size(rawframes));
    if ( errs ) {
        *errs = cpl_vector_new(cpl_frameset_get_size(rawframes));
    }
    if ( stdev ) {
        *stdev = cpl_vector_new(cpl_frameset_get_size(rawframes));
    }
    for (ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
        frame = cpl_frameset_get_position(rawframes,ff);
        single_frameset = cpl_frameset_new();
        cpl_frameset_insert(single_frameset,cpl_frame_duplicate(frame));
        temp_master = sph_framecombination_master_frame_from_cpl_frameset(single_frameset,collalg,params);
        if ( temp_master )
        {
            if ( mask ) {
                sph_master_frame_set_bads_from_mask(temp_master,mask);
            }
            mean = sph_master_frame_get_mean(temp_master,&mean_stdev);
            cpl_vector_set(means, ff, mean);
            if ( stdev ) {
                cpl_vector_set(*stdev, ff, mean_stdev);
            }
            if (errs) {
                rms = sph_master_frame_get_mean_variance(temp_master,&rms_stdev);
                cpl_vector_set(*errs, ff, rms);
            }
            sph_master_frame_delete(temp_master);temp_master = NULL;
        }
        else {
            SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
                    "Could not create/load flat frame from file %s and plane %d",
                    cpl_frame_get_filename(frame),
                    plane);
            return NULL;
        }
        cpl_frameset_delete(single_frameset); single_frameset = NULL;
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return means;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Calculate the medians on a set of frames (cubes)
 * @param collapsed_frames the input set of collapsed cubes
 * @param mask an optional bad pixel mask (may be NULL)
 * @param stdev optional output vector of errors on medians (may be NULL)
 * @param errs optional output vecotr of mean of rms (may be NULL)
 *
 * @return vector of means
 *
 * This function calculates all the medians and optionally errors on medians
 * and the mean RMS values for the set of input cubes. The input frames
 * should be master frames that were obtained by collapsing the cubes.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_vector* sph_ptc_calculate_medians(
        cpl_frameset* collapsed_frames,
        cpl_mask*   mask,
        cpl_vector** stdev,
        cpl_vector** errs)
{
    cpl_vector*         means   = NULL;
    int                 ff      = 0;
    double              mean    = 0.0;
    double              mean_stdev   = 0.0;
    sph_master_frame*   temp_master = NULL;
    cpl_frame*          frame = NULL;
    int                 plane = 0;
    double              rms = 0.0;
    double              rms_stdev = 0.0;
    //cpl_frameset*       single_frameset = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    means = cpl_vector_new(cpl_frameset_get_size(collapsed_frames));
    if ( errs ) {
        *errs = cpl_vector_new(cpl_frameset_get_size(collapsed_frames));
    }
    if ( stdev ) {
        *stdev = cpl_vector_new(cpl_frameset_get_size(collapsed_frames));
    }
    for (ff = 0; ff < cpl_frameset_get_size(collapsed_frames); ++ff) {
        frame = cpl_frameset_get_position(collapsed_frames,ff);
        cpl_msg_info(cpl_func,"Working on frame %s",cpl_frame_get_filename(frame));
        temp_master = sph_master_frame_load_(frame,0);
        if ( temp_master )
        {
            if ( mask ) {
                sph_master_frame_set_bads_from_mask(temp_master,mask);
            }
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
            mean = sph_master_frame_get_median(temp_master,&mean_stdev);
            cpl_msg_info(cpl_func,"Determined median %f +/- %f!	",mean,mean_stdev);
            cpl_vector_set(means, ff, mean);
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
            if ( stdev ) {
                cpl_vector_set(*stdev, ff, mean_stdev);
            }
            if (errs) {
                rms = sph_master_frame_get_median_variance(temp_master,&rms_stdev);
                cpl_msg_info(cpl_func,"Determined rms of %f!",rms );
                cpl_vector_set(*errs, ff, rms);
            }
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
            sph_master_frame_delete(temp_master);temp_master = NULL;
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        }
        else {
            SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
                    "Could not create/load flat frame from file %s and plane %d",
                    cpl_frame_get_filename(frame),
                    plane);
            goto EXIT;
        }
        //cpl_frameset_delete(single_frameset); single_frameset = NULL;
    }
    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
    return means;
EXIT:
    if ( errs ) {
    	cpl_vector_delete(*errs); *errs = NULL;
    }
    if ( stdev ) {
    	cpl_vector_delete(*stdev); *stdev = NULL;
    }
    cpl_vector_delete(means); means = NULL;
    return NULL;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fit single pixels using two image lists
 * @param fittab    the pixel fit table to fill with results
 * @param ox        the x value in output table of first pixel
 * @param oy        the y value in output table of first pixel
 * @param valimlist the imagelist of the values to fit (x-axis)
 * @param errimlist the imagelist of the values to fit (y-axis)
 * @param order     the fitting order
 *
 * @return error code
 *
 * The input polyfit table fittab is filled with the fit results for each
 * pixel in the two imagelists. For each pixel in the two imagelists,
 * two vectors are constructed, one for each imagelist, giving the x and
 * y values of the polynomial fit. The fit is performed and the resulting
 * polynomial saved in the position of the fittab. To position of the fittab
 * is taken as the pixel position in the imagelist plus the offset ox and oy.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ptc_fit( sph_pixel_polyfit_table* fittab,
        int ox,
        int oy,
        cpl_imagelist* valimlist,
        cpl_imagelist* errimlist, int order ) {
    cpl_vector* means = NULL;
    cpl_vector* vals = NULL;
    cpl_vector* errs = NULL;
    int     xx  = 0;
    int     yy  = 0;
    int     nx  = 0;
    int     ny = 0;
    int     np = 0;
    int     ii = 0;
    int     bp = 0;
    double  val = 0.0;
    double  err = 0.0;
    double  redchi = 0.0;
    cpl_polynomial* poly = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    nx = cpl_image_get_size_x(cpl_imagelist_get(valimlist,0));
    ny = cpl_image_get_size_y(cpl_imagelist_get(valimlist,0));
    np = cpl_imagelist_get_size(valimlist);
    vals = cpl_vector_new(np);
    means = cpl_vector_new(np);
    errs = cpl_vector_new(np);
    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            for (ii = 0; ii < np; ++ii) {
                val = cpl_image_get(cpl_imagelist_get(valimlist,ii),xx + 1,yy + 1,&bp);
                if ( bp == 0 )
                    err = cpl_image_get(cpl_imagelist_get(errimlist,ii),xx + 1,yy + 1,&bp);
                else err = SPH_MASTER_FRAME_BAD_RMS;
                cpl_vector_set(means,ii,val);
                cpl_vector_set(vals,ii,err);
                if ( val > 0.0 ) cpl_vector_set(errs,ii,sqrt(val));
                else cpl_vector_set(errs,ii,1.0);
            }
            poly = sph_fitting_fit_poly1d(means,vals,errs,0,
                    order,0,1.0,&redchi);
            if ( poly ) {
                sph_pixel_polyfit_table_set(fittab,xx + ox, yy + oy, redchi, poly);
                cpl_polynomial_delete(poly); poly = NULL;
            }
            else {
                if ( cpl_error_get_code() == CPL_ERROR_CONTINUE ) {
                    cpl_error_reset();
                }
                if ( cpl_error_get_code() == CPL_ERROR_SINGULAR_MATRIX ) {
                    cpl_error_reset();
                    sph_pixel_polyfit_table_set_invalid(fittab,xx+ox,yy+oy);
                }
            }
        }
    }


    cpl_vector_delete(vals); vals = NULL;
    cpl_vector_delete(means); means = NULL;
    cpl_vector_delete(errs); errs = NULL;


    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create the PTC for all pixels
 * @param inframes      set of collapsed frames
 * @param order         the fitting order
 *
 * @return the PTC as a pixel polyfit table
 *
 * This calculates the pixel by pixel PTC using a set of input
 * frames (master frames). A fit is
 * performed for each pixel of rms^2 against signal and the
 * polynomial fit for each pixel is saved in the resulting
 * pixel polyfit table. The gain is then for example the inverse
 * of the first coefficient.
 *
 *
 */
/*----------------------------------------------------------------------------*/

sph_pixel_polyfit_table*
sph_ptc_create_full(
        cpl_frameset* inframes,
        int order)
{
    sph_pixel_polyfit_table*            fittab                 = NULL;
    cpl_imagelist*      imlist = NULL;
    cpl_imagelist*      errims = NULL;
    cpl_imagelist*      valims = NULL;
    sph_smart_imagelist*        smartlist = NULL;
    sph_master_frame* aplane = NULL;
    cpl_image*          tmpim = NULL;
    int                 ii = 0;
    cpl_frame*          frame = NULL;
    int                 set = 0;
    int     nx  = 0;
    int     ny = 0;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    tmpim = cpl_image_load(cpl_frame_get_filename(cpl_frameset_get_first(inframes)),CPL_TYPE_INT,0,0);
    cpl_ensure(tmpim,CPL_ERROR_FILE_IO,NULL);

    nx = cpl_image_get_size_x(tmpim);
    ny = cpl_image_get_size_y(tmpim);
    cpl_image_delete(tmpim);tmpim = NULL;



    fittab = sph_pixel_polyfit_table_new(nx,ny,order);
    smartlist = sph_smart_imagelist_create(inframes,0);

    for (set = 0; set < smartlist->nregions; ++set) {
        errims = cpl_imagelist_new();
        valims = cpl_imagelist_new();
        for (ii = 0; ii < cpl_frameset_get_size(inframes); ++ii) {
            frame = cpl_frameset_get_position(inframes,ii);
            aplane = sph_master_frame_load_(frame,0);
            cpl_imagelist_set(valims,cpl_image_duplicate(aplane->image),ii);
            cpl_imagelist_set(errims,sph_master_frame_get_variance(aplane),ii);
            sph_master_frame_delete(aplane);aplane = NULL;
            cpl_imagelist_delete(imlist); imlist = NULL;
        }

        sph_ptc_fit(fittab,
                smartlist->regions[set].ll_x,
                smartlist->regions[set].ll_y,
                valims,errims,order);
        cpl_imagelist_delete(errims); errims = NULL;
        cpl_imagelist_delete(valims); valims = NULL;
    }
    sph_smart_imagelist_delete(smartlist);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return fittab;
}

/**@}*/
