/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_filemanager.h"
#include "sph_keyword_manager.h"
#include "sph_fft.h"
#include "sph_fits.h"
#include "sph_smart_imagelist.h"
#include "sph_common_science.h"
#include "sph_time.h"
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include "sph_fctable.h"
#include "sph_cube.h"
#include "sph_transform.h"
#include "sph_simple_adi.h"

static
sph_error_code
sph_simple_adi_clear_process_sets__( sph_simple_adi* self) ;
static
sph_error_code
sph_simple_adi_append_imlist__(
        cpl_imagelist* inlist,
        cpl_imagelist* applist);
static
sph_error_code
sph_simple_adi_append_angles__(
        cpl_vector* inangles,
        cpl_vector* appangles);
static
sph_error_code
sph_simple_adi_calculate_speckle_frame_imlist__(
        sph_simple_adi* self);
static
sph_error_code
sph_simple_adi_calculate_speckle_frame_frameset__(
        sph_simple_adi* self);
static
sph_error_code
sph_simple_adi_store_plane__(
        sph_simple_adi* self,
        sph_master_frame* mframe,
        int plane);
static
sph_master_frame*
sph_simple_adi_get_first__( sph_simple_adi* self );

static
sph_master_frame*
sph_simple_adi_get_next__( sph_simple_adi* self );
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new ADI processing strucuture
 * @param transform         the transform object to use
 *
 * @return new pointer or NULLL
 *
 * The transform must not be deleted until after sph_simple_adi_delete
 * is called.
 */
/*----------------------------------------------------------------------------*/
sph_simple_adi*
sph_simple_adi_new(
        sph_transform*  transform)
{
    sph_simple_adi*   self = NULL;
    cpl_ensure(transform,CPL_ERROR_NULL_INPUT, NULL);
    self = cpl_calloc(1,sizeof(sph_simple_adi));
    self->transform = transform;
    self->currentplane = 0;
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the ADI structure
 * @param self          pointer to ADI structure
 *
 */
/*----------------------------------------------------------------------------*/
void
sph_simple_adi_delete( sph_simple_adi* self) {
    if ( self ) {
        sph_simple_adi_clear_process_sets__(self);
        if ( self->current_imlist)
            cpl_imagelist_delete(self->current_imlist);
        if ( self->current_bmaplist)
            cpl_imagelist_delete(self->current_bmaplist);
        if ( self->current_wmaplist)
            cpl_imagelist_delete(self->current_wmaplist);
        if ( self->current_cube_angles)
            cpl_vector_delete(self->current_cube_angles);
        if ( self->current_frameset)
            cpl_frameset_delete(self->current_frameset);
        if ( self->result_cube )
            sph_cube_delete(self->result_cube);
        if ( self->result_frameset)
            cpl_frameset_delete(self->result_frameset);
        if ( self->result_imlist)
            cpl_imagelist_delete(self->result_imlist);
        if ( self->result_bmaplist)
            cpl_imagelist_delete(self->result_bmaplist);
        if ( self->result_wmaplist)
            cpl_imagelist_delete(self->result_wmaplist);
        if ( self->speckleframe)
            sph_master_frame_delete(self->speckleframe);

        cpl_free( self );
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the current IRDIS model
 * @param self          the ADI structure
 *
 * @return pointer to the IRDIS model or NULL
 *
 * The returned pointer must not be deallocated (using a call to
 * ..._delete) until the ADI structure is
 * not used anymore and has been deleted itself.
 *
 */
/*----------------------------------------------------------------------------*/
sph_ird_instrument_model*
sph_simple_adi_get_ird_model( sph_simple_adi* self) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    return self->irdmodel;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief set the IRDIS model
 * @param self          the ADI structure
 * @param model         the IRDIS model
 *
 * @return error code
 *
 * Set the IRDIS model
 * Do not delete the model after calling this
 * function until the ADI struct is not used anymore and
 * has been deleted.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_set_ird_model( sph_simple_adi* self,
        sph_ird_instrument_model* irdmodel) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( irdmodel, CPL_ERROR_NULL_INPUT );
    self->irdmodel = irdmodel;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the current transform used
 * @param self          the ADI structure
 *
 * @return pointer to the transform or NULL
 *
 * The returned pointer must not be deallocated (using a call to
 * ..._delete) until the ADI structure is
 * not used anymore and has been deleted itself.
 *
 */
/*----------------------------------------------------------------------------*/
sph_transform*
sph_simple_adi_get_transform( sph_simple_adi* self) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    return self->transform;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief set the transform structure to use
 * @param self          the ADI structure
 * @param model         the transform struct to use
 *
 * @return error code
 *
 * Set the transform structure to use for the geometrical
 * image transformations that need to be done during ADI
 * processing.
 * Do not delete the transform struct after calling this
 * function until the ADI struct is not used anymore and
 * has been deleted.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_set_transform( sph_simple_adi* self,
        sph_transform* transform) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( transform, CPL_ERROR_NULL_INPUT );
    self->transform = transform;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the current angles
 * @param self          the ADI structure
 *
 * @return pointer to the rotation angles or NULL
 *
 * The returned pointer must not be deallocated (using a call to
 * ..._delete) until the ADI structure is
 * not used anymore and has been deleted itself.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_vector*
sph_simple_adi_get_angles( sph_simple_adi* self) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    return self->current_cube_angles;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief set the rotation angles
 * @param self          the ADI structure
 * @param model         the rotation angles to set
 *
 * @return error code
 *
 * This function allows the setting of the rotation angles
 * by hand. In the most common use of the ADI module this
 * will not be necessary, but this function may be used to override
 * the angles if needed.
 * The input vector of anlges must have the same length as there
 * are images to be processed by ADI.
 * Angles are measured in degrees, clockwise from x=0 axis.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_set_angles( sph_simple_adi* self,
        cpl_vector* angles)
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(angles, CPL_ERROR_NULL_INPUT);

    if ( self->current_imlist ) {
        cpl_ensure_code(
                cpl_imagelist_get_size(self->current_imlist) ==
                cpl_vector_get_size(angles),
                CPL_ERROR_ILLEGAL_INPUT );
    }
    else if ( self->current_frameset ) {
        cpl_ensure_code(
            sph_common_science_get_nraws(
                    self->current_frameset,0) ==
                cpl_vector_get_size(angles),
                CPL_ERROR_ILLEGAL_INPUT);
    }
    else {
        cpl_ensure_code(0, CPL_ERROR_DATA_NOT_FOUND );
    }
    if ( self->current_cube_angles ) {
        cpl_vector_delete(self->current_cube_angles);
        self->current_cube_angles = NULL;
    }
    self->current_cube_angles = cpl_vector_duplicate(angles);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the current speckle frame
 * @param self          the ADI structure
 *
 * @return pointer to the speckle frame or NULL
 *
 * The returned pointer must not be deallocated (using a call to
 * ..._delete) until the ADI structure is
 * not used anymore and has been deleted itself.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_simple_adi_get_speckle_frame( sph_simple_adi* self) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    return self->speckleframe;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief set the speckle frame
 * @param self          the ADI structure
 * @param model         the speckle frame to set
 *
 * @return error code
 *
 * This function allows the setting of the speckle frame
 * by hand. In the most common use of the ADI module this
 * will not be necessary, but this function may be used to override
 * the automatically calculated speckle frame if needed.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_set_speckle_frame( sph_simple_adi* self,
        sph_master_frame* speckle) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( speckle, CPL_ERROR_NULL_INPUT);

    if ( self->speckleframe ) {
        sph_master_frame_delete(self->speckleframe);
        self->speckleframe = NULL;
    }
    self->speckleframe = sph_master_frame_duplicate(
           speckle);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set ADI processing set from a cube
 * @param self          the ADI structure
 * @param cube          A cpl_frame of a data cube
 * @param ext_im        extension for image
 * @param ext_bp        badpixel extension (or -1 if none)
 * @param ext_wpm       weightmap extension (or -1 if none)
 * @return error code
 *
 * This functions sets the images to use for the processing
 * from a single cube.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_set_current_cube_frame( sph_simple_adi* self,
        cpl_frame* cube_frame,
        int ext_im,
        int ext_bp,
        int ext_wm) {

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cube_frame,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ext_im >=0,CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(sph_simple_adi_clear_process_sets__( self ) ==
            CPL_ERROR_NONE,
            cpl_error_get_code());
    self->current_frameset = cpl_frameset_new();
    cpl_frameset_insert( self->current_frameset,
            cpl_frame_duplicate(cube_frame));
    self->current_cube_angles = cpl_vector_new(
            sph_fits_get_nplanes(cpl_frame_get_filename(cube_frame),ext_im)
            );
    self->ext_bp = ext_bp;
    self->ext_wm = ext_wm;
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set ADI processing set from frameset
 * @param self          the ADI structure
 * @param ext_im        extension for image
 * @param ext_bp        badpixel extension (or -1 if none)
 * @param ext_wpm       weightmap extension (or -1 if none)
 * @param frameset      A cpl_frameset of a NAXIS3=1 frames
 * @return error code
 *
 * This functions sets the images to use for the processing
 * from a set of frames that have single planes only
 * (NAXIS=2 or NAXIS3=1).
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_set_current_frameset( sph_simple_adi* self,
        cpl_frameset* no_cubes_frameset,
        int ext_im,
        int ext_bp,
        int ext_wm) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(no_cubes_frameset,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ext_im >=0,CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(
            sph_simple_adi_clear_process_sets__( self ) == CPL_ERROR_NONE,
            cpl_error_get_code());
    self->current_frameset = cpl_frameset_duplicate(no_cubes_frameset);
    self->current_cube_angles = cpl_vector_new(
            cpl_frameset_get_size(no_cubes_frameset)
            );
    self->ext_bp = ext_bp;
    self->ext_wm = ext_wm;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Append an image to the set of process images
 * @param self      the ADI structure
 * @param image     the image to append
 * @param bpix      the corresponding bpix image (maybe NULL)
 * @param weight    the corresponding weight image (maybe NULL)
 * @param angle     the corresponding angle in degree (CW)
 * @return error code
 *
 * Appends an image (makes an internal duplicate)
 * and optionally some bpixmap and weight image.
 * This will also automatically purge any cube or frameset that
 * has been set previously.
 * All appended images are duplicated and the duplicate is added,
 * so it is safe to delete the pointers to the original images
 * after calling this function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_append_image( sph_simple_adi* self,
        cpl_image* image,
        cpl_image* bpix,
        cpl_image* weight,
        double angle) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( image, CPL_ERROR_NULL_INPUT );

    if ( self->current_imlist == NULL ) {
        cpl_ensure_code(
                sph_simple_adi_clear_process_sets__( self )
                == CPL_ERROR_NONE,
            cpl_error_get_code());
        self->current_imlist = cpl_imagelist_new();
        if ( bpix ) self->current_bmaplist = cpl_imagelist_new();
        if ( weight ) self->current_wmaplist = cpl_imagelist_new();
        self->current_cube_angles = cpl_vector_new(1);
    }
    cpl_ensure_code( self->current_cube_angles,
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(
            (bpix && self->current_bmaplist) ||
            (bpix == NULL && self->current_bmaplist == NULL),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(
            (weight && self->current_wmaplist) ||
            (weight == NULL && self->current_wmaplist == NULL),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(
            cpl_imagelist_set( self->current_imlist,
            cpl_image_duplicate(image),
            cpl_imagelist_get_size(
                    self->current_imlist))
                    == CPL_ERROR_NONE,
            cpl_error_get_code());
    if ( bpix ) {
        cpl_ensure_code(
                cpl_imagelist_set( self->current_bmaplist,
                        cpl_image_duplicate(bpix),
                cpl_imagelist_get_size(
                        self->current_bmaplist))
                        == CPL_ERROR_NONE,
                cpl_error_get_code());
    }
    if ( weight ) {
        cpl_ensure_code(
                cpl_imagelist_set( self->current_wmaplist,
                        cpl_image_duplicate(weight),
                cpl_imagelist_get_size(
                        self->current_wmaplist))
                        == CPL_ERROR_NONE,
                cpl_error_get_code());
    }
    cpl_vector_set_size(self->current_cube_angles,
            cpl_imagelist_get_size(self->current_imlist));
    cpl_vector_set(self->current_cube_angles,
            cpl_imagelist_get_size(self->current_imlist) - 1,
            angle);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Append an imagelist to the set of process images
 * @param self      the ADI structure
 * @param image     the imagelist to append
 * @param bpix      the corresponding bpixs image (maybe NULL)
 * @param weight    the corresponding weights image (maybe NULL)
 * @param angle     the corresponding angles in degree (CW)
 * @return error code
 *
 * Appends an imagelist and optionally some list if bpixmap
 * and weight image list.
 * This will also automatically purge and cube or frameset that
 * has been set previously.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_append_imagelist( sph_simple_adi* self,
        cpl_imagelist* images,
        cpl_imagelist* bpixs,
        cpl_imagelist* weights,
        cpl_vector* angles) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( images, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( angles, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( cpl_imagelist_get_size(images) ==
            cpl_vector_get_size(angles),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    if ( bpixs )
        cpl_ensure_code( cpl_imagelist_get_size(images) ==
                cpl_imagelist_get_size(bpixs),
                CPL_ERROR_INCOMPATIBLE_INPUT);
    if ( weights )
        cpl_ensure_code( cpl_imagelist_get_size(images) ==
                cpl_imagelist_get_size(weights),
                CPL_ERROR_INCOMPATIBLE_INPUT);
    if ( self->current_imlist == NULL ) {
        cpl_ensure_code(
                sph_simple_adi_clear_process_sets__( self )
                == CPL_ERROR_NONE,
            cpl_error_get_code());
        self->current_imlist = cpl_imagelist_duplicate(images);
        if ( bpixs )
            self->current_bmaplist = cpl_imagelist_duplicate(bpixs);
        if ( weights )
            self->current_wmaplist = cpl_imagelist_duplicate(weights);
        self->current_cube_angles = cpl_vector_duplicate(angles);
        return CPL_ERROR_NONE;
    }
    cpl_ensure_code( self->current_cube_angles,
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(
            (bpixs && self->current_bmaplist) ||
            (bpixs == NULL && self->current_bmaplist == NULL),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(
            (weights && self->current_wmaplist) ||
            (weights == NULL && self->current_wmaplist == NULL),
            CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_ensure_code(
            sph_simple_adi_append_imlist__(
                    self->current_imlist,
                    images)
                    == CPL_ERROR_NONE,
            cpl_error_get_code());
    if ( bpixs ) {
        cpl_ensure_code(
                sph_simple_adi_append_imlist__(
                        self->current_bmaplist,
                        bpixs)
                        == CPL_ERROR_NONE,
                cpl_error_get_code());
    }
    if ( weights ) {
        cpl_ensure_code(
                sph_simple_adi_append_imlist__(
                        self->current_wmaplist,
                        weights)
                        == CPL_ERROR_NONE,
                cpl_error_get_code());
    }
    cpl_ensure_code(
            sph_simple_adi_append_angles__(
                    self->current_cube_angles,
                    angles)
                    == CPL_ERROR_NONE,
            cpl_error_get_code());
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create the speckle frame
 * @param self          the ADI struct
 *
 * @return error code
 *
 * This function creates the speckle frame by collapsing
 * all the input frames
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_calculate_speckle_frame( sph_simple_adi* self ) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( self->current_frameset ||
            self->current_imlist,
            CPL_ERROR_ILLEGAL_INPUT );
    if ( self->speckleframe ) {
        sph_master_frame_delete(self->speckleframe);
        self->speckleframe = NULL;
    }
    if ( self->current_frameset ) {
        return (sph_simple_adi_calculate_speckle_frame_frameset__(self));
    }
    else if ( self->current_imlist ) {
        return (sph_simple_adi_calculate_speckle_frame_imlist__(self));
    }
    return CPL_ERROR_UNSUPPORTED_MODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Main ADI processing
 * @param self          pointer to ADI
 *
 * @return error code
 *
 * This does the main ADI processing.
 * For each image in the set, it subtracts the speckle image and
 * then rotates/transforms the image, storing the result.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_simple_adi_process(sph_simple_adi* self ) {
    int         pp          = 0;

    sph_master_frame*   mframe = NULL;
    double      xc = 0.0;
    double      yc = 0.0;

    mframe = sph_simple_adi_get_first__(self);
    cpl_ensure_code(mframe, cpl_error_get_code());
    xc = cpl_image_get_size_x(mframe->image) / 2.0;
    yc = cpl_image_get_size_y(mframe->image) / 2.0;

    pp = 0;
    while ( mframe ) {
        cpl_ensure_code(mframe->image,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        cpl_ensure_code(mframe->badpixelmap,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        cpl_ensure_code(mframe->ncombmap,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        cpl_ensure_code(mframe->rmsmap,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        if ( self->speckleframe ) {
            sph_master_frame_subtract_master_frame(mframe,
                    self->speckleframe);
        }
        sph_transform_apply(self->transform,
                mframe,NULL,xc,yc,
                cpl_vector_get(
                        self->current_cube_angles,
                        pp++),
                1.0,
                self->irdmodel);
        cpl_ensure_code(mframe->image,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        cpl_ensure_code(mframe->badpixelmap,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        cpl_ensure_code(mframe->ncombmap,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        cpl_ensure_code(mframe->rmsmap,CPL_ERROR_ACCESS_OUT_OF_RANGE);
        sph_simple_adi_store_plane__(self,mframe,pp);
        sph_master_frame_delete(mframe); mframe = NULL;
        mframe = sph_simple_adi_get_next__(self);
    }
    sph_master_frame_delete(mframe); mframe = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Main ADI processing for imagelist
 * @param self          pointer to ADI
 *
 * @return pointer to collapsed master_frame or NULL
 *
 * This does the main ADI processing.
 * For each image in the set, it subtracts the speckle image and
 * then rotates/transforms the image. The resulting lists are
 * collapsed and a master frame returned
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_simple_adi_process_imlists(
        sph_simple_adi* self,
        sph_collapse_algorithm coll_alg )
{
    sph_master_frame*  result      = NULL;
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->current_imlist, CPL_ERROR_ILLEGAL_INPUT, NULL);
    if (
    		sph_simple_adi_process(self) == CPL_ERROR_NONE )
    {
        sph_common_science_combine_imlists(
                coll_alg,
                self->result_imlist,
                self->result_bmaplist,
                self->result_wmaplist,
                &result);
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Perform simple ADI on a frameset of cubes
 * @param input_frames the frameset of the cubes
 * @param origframes   the original frameset of the RAW science frames
 * @param fc_frames    frameset of field centre frames containing angle info
 * @param irdmodel     the iridis instrument model
 *
 * @return a new frameset of the processed cubes or NULL on error
 *
 * This function performs ADI processing, that is: speckle subtraction and
 * rotation, on the input cubes. A new frameset is created that is a duplicate
 * of the input frames (same tag, same group) but with a different filenames:
 * a string "_adi" is supplemented before the ".fits" part of the filename
 * (so "myfile.fits" becomes "myfile_adi.fits").
 * The speckle frame that is subtracted is calculated on a cube by cube
 * basis -- for each cube one speckle image is calculated which is then
 * subtracted from all images in that cube.
 * The angles are taken from the field centre table and
 * should be the angles to rotate by when measured CW going
 * from the +x axis as 0.
 * The irdmodel is needed to determine the centre of rotation, which is
 * specified as the centre coordinate of the FOV subwindows (frames
 * are assumed to be centered correctly already!).
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_simple_adi_process_cubes( cpl_frameset* incube_frames,
        cpl_frameset* origframes,
        cpl_vector* angles,
        sph_transform* transform,
        sph_ird_instrument_model* irdmodel,
        int ext_im,
                int ext_bp,
                int ext_wm,
                int full_frameset_collapse)
{
    cpl_frameset*       result = NULL;
    int                 ff      = 0;
    int                 nf      = 0;
    cpl_frame*          in_frame  = NULL;
    cpl_frame*          orig_frame  = NULL;
    cpl_frame*          out_frame  = NULL;
    cpl_frameset*       tmpframeset = NULL;
    sph_master_frame*   speckles = NULL;
    cpl_vector*         tmp_angles = NULL;
    int                 np          = 0;
    int                 totp        = 0;
    int                 vv          = 0;
    cpl_ensure(ext_im >=0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(incube_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(angles, CPL_ERROR_NULL_INPUT, NULL);
    nf = cpl_frameset_get_size(incube_frames);
    cpl_ensure(nf > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    result = cpl_frameset_new();

    if ( full_frameset_collapse ) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Creating full speckle frame...");
        speckles = sph_common_science_combine(
                incube_frames,
                SPH_COLL_ALG_MEDIAN,
                ext_im,ext_bp,ext_wm);
        if ( speckles == NULL ) {
            SPH_ERROR_RAISE_ERR( cpl_error_get_code(),
                    "Could not create the speckle frame for the frameset");
        }
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Created full speckle frame");
    }

    for (ff = 0; ff < nf; ++ff) {
        tmpframeset = cpl_frameset_new();
        in_frame = cpl_frameset_get_position(incube_frames,ff);
        np = sph_fits_get_nplanes(
                cpl_frame_get_filename(in_frame), ext_im);
        if ( !full_frameset_collapse ) {
            orig_frame = cpl_frameset_get_position(origframes,ff);
            cpl_frameset_insert(tmpframeset,cpl_frame_duplicate(in_frame));
            speckles = sph_common_science_combine(tmpframeset,SPH_COLL_ALG_MEDIAN,ext_im,ext_bp,ext_wm);
            if ( speckles == NULL ) {
                SPH_ERROR_RAISE_ERR( cpl_error_get_code(),
                        "Could not create the speckle frame for frame %s",
                        cpl_frame_get_filename(in_frame));
                break;
            }
        }
        tmp_angles = cpl_vector_new(np);
        for (vv = totp; vv < np + totp; ++vv) {
            cpl_vector_set(tmp_angles,
                    vv - totp,
                    -1.0 * cpl_vector_get(angles,vv));
        }
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Starting ADI processing...");
        out_frame = sph_simple_adi_process_single_cube(
                in_frame,
                speckles,
                tmp_angles,
                transform,
                irdmodel,
                ext_im,ext_bp,ext_wm);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Done ADI processing.");
        cpl_frameset_insert(result,out_frame);
        cpl_frameset_delete(tmpframeset); tmpframeset = NULL;
        if ( !full_frameset_collapse ) {
            sph_master_frame_delete(speckles); speckles = NULL;
        }
        cpl_vector_delete(tmp_angles); tmp_angles = NULL;
        totp = totp + np;
    }
    sph_master_frame_delete(speckles); speckles = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Perform simple ADI on a single cube
 * @param input_frame the frame of the cube
 * @param speckleframe      the speckle image (as sph_master_frame)
 * @param angles            a vector of rotation angles, size of Nplanes
 * @param irdmodel          the iridis instrument model
 *
 * @return a new frame of the processed cube or NULL on error
 *
 * This function performs ADI processing, that is: speckle subtraction and
 * rotation, on the input cube. A new frame is created that is a duplicate
 * of the input frame (same tag, same group) but with a different filename:
 * a string "_adi" is supplemented before the ".fits" part of the filename
 * (so "myfile.fits" becomes "myfile_adi.fits").
 * The speckle frame that is subtracted is optional, so that this
 * function can also be used for simple de-rotation of a cube.
 * The angles should be the angles to rotate by when measured CW going
 * from the +x axis as 0.
 * The irdmodel is needed to determine the centre of rotation, which is
 * specified as the centre coordinate of the FOV subwindows (frames
 * are assumed to be centered correctly already!).
 */
/*----------------------------------------------------------------------------*/
cpl_frame*
sph_simple_adi_process_single_cube( cpl_frame* incube_frame,
        sph_master_frame* speckleframe,
        cpl_vector* angles,
        sph_transform* transform,
        sph_ird_instrument_model* irdmodel,
        int ext_im,
        int ext_bp,
        int ext_wm) {
    int         nx          = 0;
    int         ny          = 0;
    cpl_frame*  result      = NULL;
    cpl_image*  dummyim     = NULL;
    sph_simple_adi*     adi = NULL;

    cpl_ensure(ext_im >=0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(incube_frame, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(transform, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(angles, CPL_ERROR_NULL_INPUT, NULL);
    dummyim = cpl_image_load(cpl_frame_get_filename(incube_frame),
            CPL_TYPE_INT,0,0);
    cpl_ensure(dummyim,CPL_ERROR_FILE_IO,NULL);
    nx = cpl_image_get_size_x(dummyim);
    ny = cpl_image_get_size_y(dummyim);
    cpl_image_delete(dummyim); dummyim = NULL;
    adi = sph_simple_adi_new(transform);
    sph_simple_adi_set_current_cube_frame(
            adi,
            incube_frame,
            ext_im,ext_bp,ext_wm);
    sph_simple_adi_set_angles(adi,angles);
    if ( irdmodel )
        sph_simple_adi_set_ird_model(adi,irdmodel);
    if ( speckleframe ) {
        cpl_ensure( cpl_image_get_size_x(speckleframe->image) == nx,
                CPL_ERROR_ILLEGAL_INPUT, NULL);
        cpl_ensure( cpl_image_get_size_y(speckleframe->image) == ny,
                CPL_ERROR_ILLEGAL_INPUT, NULL);
        sph_simple_adi_set_speckle_frame(adi,speckleframe);
    }
    else {
        cpl_ensure(
                sph_simple_adi_calculate_speckle_frame(adi) ==
                        CPL_ERROR_NONE,
                cpl_error_get_code(),NULL);
    }

    if ( sph_simple_adi_process(adi) == CPL_ERROR_NONE )  {
        result = cpl_frame_duplicate(
                cpl_frameset_get_first(
                        adi->result_frameset) );
    }
    sph_simple_adi_delete(adi); adi = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}


static
sph_error_code
sph_simple_adi_clear_process_sets__( sph_simple_adi* self)
{
    cpl_ensure_code(self,CPL_ERROR_NONE);
    if ( self->current_bmaplist ) {
        cpl_imagelist_delete(self->current_bmaplist);
        self->current_bmaplist = NULL;
    }
    if ( self->current_imlist ) {
        cpl_imagelist_delete(self->current_imlist);
        self->current_imlist = NULL;
    }
    if ( self->current_wmaplist ) {
        cpl_imagelist_delete(self->current_wmaplist);
        self->current_wmaplist = NULL;
    }
    if ( self->current_frameset ) {
        if ( self->result_cube ) {
            sph_cube_finalise_file(self->result_cube->filename);
        }
        cpl_frameset_delete( self->current_frameset );
        self->current_frameset = NULL;
    }
    if ( self->current_cube_angles ) {
        cpl_vector_delete( self->current_cube_angles );
        self->current_cube_angles = NULL;
    }
    self->currentplane = 0;
    return CPL_ERROR_NONE;
}


static
sph_error_code
sph_simple_adi_append_angles__(
        cpl_vector* inangles,
        cpl_vector* appangles)
{
    int         ntoadd          = 0;
    int         orig_size       = 0;
    int         vv               = 0;
    cpl_ensure_code( inangles, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( appangles, CPL_ERROR_NULL_INPUT );

    ntoadd = cpl_vector_get_size(appangles);
    orig_size = cpl_vector_get_size(inangles);
    cpl_vector_set_size(inangles,ntoadd + orig_size);
    for (vv = 0; vv < ntoadd; ++vv) {
        cpl_vector_set( inangles,
                cpl_vector_get(appangles,vv),
                vv + orig_size);
    }
    return CPL_ERROR_NONE;
}

static
sph_error_code
sph_simple_adi_append_imlist__(
        cpl_imagelist* inlist,
        cpl_imagelist* applist)
{
    int         ntoadd          = 0;
    int         orig_size       = 0;
    int         vv               = 0;
    cpl_ensure_code( inlist, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( applist, CPL_ERROR_NULL_INPUT );

    ntoadd = cpl_imagelist_get_size(applist);
    orig_size = cpl_imagelist_get_size(inlist);
    for (vv = 0; vv < ntoadd; ++vv) {
        cpl_imagelist_set( inlist,
                cpl_image_duplicate(
                        cpl_imagelist_get(applist,vv)
                        ),
                vv + orig_size);
    }
    return CPL_ERROR_NONE;
}

static
sph_error_code
sph_simple_adi_calculate_speckle_frame_frameset__(
        sph_simple_adi* self)
{
    int                 nf  = 0;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( self->current_frameset, CPL_ERROR_NULL_INPUT );

    nf = cpl_frameset_get_size(self->current_frameset);
    cpl_ensure_code(nf > 0, CPL_ERROR_ILLEGAL_INPUT);
    self->speckleframe = sph_common_science_combine(
            self->current_frameset,
            SPH_COLL_ALG_MEDIAN,
            self->ext_im,
            self->ext_bp,
            self->ext_wm);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_error_code
sph_simple_adi_calculate_speckle_frame_imlist__(
        sph_simple_adi* self)
{
    int                 nf  = 0;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( self->current_imlist, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( self->current_frameset == NULL,
            CPL_ERROR_ILLEGAL_INPUT );

    nf = cpl_imagelist_get_size( self->current_imlist);
    cpl_ensure_code(nf > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(sph_common_science_combine_imlists(
            SPH_COLL_ALG_MEDIAN,
            self->current_imlist,
            self->current_bmaplist,
            self->current_wmaplist,
            &self->speckleframe) == CPL_ERROR_NONE,
            cpl_error_get_code());
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_error_code
sph_simple_adi_store_plane__(
        sph_simple_adi* self,
        sph_master_frame* mframe,
        int plane)
{
    cpl_frame*          aframe = NULL;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mframe,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(plane >= 0,CPL_ERROR_ILLEGAL_INPUT);

    if ( self->current_frameset ) {
        if ( cpl_frameset_get_size(self->current_frameset) == 1) {
            if ( self->result_cube == NULL ) {
                cpl_ensure_code(self->result_frameset == NULL,
                        CPL_ERROR_UNSPECIFIED);
                self->result_frameset = cpl_frameset_new();
                aframe = sph_filemanager_get_unique_duplicate_frame(
                        cpl_frameset_get_first(self->current_frameset) );
                sph_filemanager_add_tmp_file(cpl_frame_get_filename(aframe));
                self->result_cube = sph_cube_new(
                        cpl_frame_get_filename(aframe));
                cpl_frameset_insert(self->result_frameset,aframe);
            }
            sph_cube_append_master(self->result_cube,mframe,plane);
        }
        else {
            if ( self->result_frameset == NULL )
                self->result_frameset = cpl_frameset_new();
            aframe = sph_filemanager_get_unique_duplicate_frame(
                    cpl_frameset_get_position(self->current_frameset,plane) );
            sph_filemanager_add_tmp_file(cpl_frame_get_filename(aframe));
            sph_master_frame_save(
                    mframe,
                    cpl_frame_get_filename(aframe),
                    NULL);
            cpl_frameset_insert(self->result_frameset,aframe);
        }
    }
    else {
        if ( self->current_imlist ) {
            if ( self->result_imlist == NULL)
                self->result_imlist = cpl_imagelist_new();
            cpl_imagelist_set( self->result_imlist,
                    cpl_image_duplicate(mframe->image),
                    cpl_imagelist_get_size(self->result_imlist) );
        }
        if ( self->current_bmaplist ) {
            if ( self->result_bmaplist == NULL)
                self->result_bmaplist = cpl_imagelist_new();
            cpl_imagelist_set( self->result_bmaplist,
                    cpl_image_duplicate(mframe->badpixelmap),
                    cpl_imagelist_get_size(self->result_bmaplist) );
        }
        if ( self->current_wmaplist ) {
            if ( self->result_wmaplist == NULL)
                self->result_wmaplist = cpl_imagelist_new();
            cpl_imagelist_set( self->result_wmaplist,
                    cpl_image_duplicate(mframe->ncombmap),
                    cpl_imagelist_get_size(self->result_wmaplist) );
        }
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_master_frame*
sph_simple_adi_get_first__( sph_simple_adi* self ) {
    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL );
    return sph_simple_adi_get_next__(self);
}

static
sph_master_frame*
sph_simple_adi_get_next__( sph_simple_adi* self ) {
    sph_master_frame*       result = NULL;

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL );
    result = sph_master_frame_new_empty();
    if ( self->current_frameset ) {
        self->currentplane = sph_common_science_get_current_plane();
        result->image = sph_common_science_get_next_raw(
                self->current_frameset,
                self->ext_im);
        if ( result->image == NULL ) {
            if ( self->result_cube ) {
                sph_cube_finalise_file(
                        cpl_frame_get_filename(
                                cpl_frameset_get_first(
                                        self->result_frameset) ) );
            }
            sph_master_frame_delete(result); return NULL;
        }
        if ( self->ext_bp >= 0 ) {
            result->badpixelmap = cpl_image_load(
                    cpl_frame_get_filename(
                            sph_common_science_get_current_raw_frame()),
                            CPL_TYPE_DOUBLE,
                            self->currentplane,
                            self->ext_bp);
        }
        else {
            result->badpixelmap = cpl_image_new(
                    cpl_image_get_size_x(result->image),
                    cpl_image_get_size_y(result->image),
                    CPL_TYPE_INT
                    );
        }
        if ( self->ext_wm >=0 ) {
            result->ncombmap = cpl_image_load(
                    cpl_frame_get_filename(
                            sph_common_science_get_current_raw_frame()),
                            CPL_TYPE_DOUBLE,
                            self->currentplane,
                            self->ext_wm);
        }
        else {
            result->ncombmap = cpl_image_multiply_scalar_create(
                    result->image,0.0);
        }
        result->rmsmap = cpl_image_multiply_scalar_create(
                result->image,0.0);
    }
    else {
        if ( self->currentplane >= cpl_imagelist_get_size(self->current_imlist)) {
            sph_master_frame_delete(result);
            result = NULL;
            return NULL;
        }
        result->image = cpl_image_duplicate(
                cpl_imagelist_get(self->current_imlist,
                self->currentplane) );
        cpl_ensure(result->image,cpl_error_get_code(),NULL);
        if ( self->current_bmaplist ) {
            result->badpixelmap = cpl_image_duplicate(cpl_imagelist_get(
                    self->current_bmaplist,
                    self->currentplane));
        }
        else {
            result->badpixelmap = cpl_image_new(
                    cpl_image_get_size_x(result->image),
                    cpl_image_get_size_y(result->image),
                    CPL_TYPE_INT
                    );

        }
        if ( self->current_wmaplist ) {
            result->ncombmap = cpl_image_duplicate(
                    cpl_imagelist_get(
                    self->current_wmaplist,
                    self->currentplane) );
        }
        else {
            result->ncombmap = cpl_image_multiply_scalar_create(
                    result->image,0.0);
        }
        result->rmsmap = cpl_image_multiply_scalar_create(
                result->ncombmap,0.0);
        self->currentplane++;
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}
