/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_STREHL_H_
#define SPH_STREHL_H_

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

#ifndef SPH_STREHL_PSCALE
#define SPH_STREHL_PSCALE "PIXSCAL"
#endif


#ifndef SPH_STREHL_PSCALE_IRDIS
#define SPH_STREHL_PSCALE_IRDIS 0.01227
#endif

#ifndef SPH_STREHL_PSCALE_ZIMPOL
#define SPH_STREHL_PSCALE_ZIMPOL 0.0034
#endif

#ifndef SPH_STREHL_FILTER_KEY_IRD1
/* The broad band filter of IRDIS */
#define SPH_STREHL_FILTER_KEY_IRD1 "INS1 FILT"
#endif

#ifndef SPH_STREHL_FILTER_KEY_IRD2
/* The narrow band filter of IRDIS, may be open */
#define SPH_STREHL_FILTER_KEY_IRD2 "INS1 OPTI2"
#endif

#ifndef SPH_STREHL_FILTER_KEY_ZPL0
/* ZIMPOL common filter */
#define SPH_STREHL_FILTER_KEY_ZPL0 "INS3 OPTI2"
#endif

#ifndef SPH_STREHL_FILTER_KEY_ZPL1
/* ZIMPOL filter in arm 1 */
#define SPH_STREHL_FILTER_KEY_ZPL1 "INS3 OPTI5"
#endif

#ifndef SPH_STREHL_FILTER_KEY_ZPL2
/* ZIMPOL filter in arm 2 */
#define SPH_STREHL_FILTER_KEY_ZPL2 "INS3 OPTI6"
#endif

#define SPH_STREHL_FILTER_IRD1 "ESO " SPH_STREHL_FILTER_KEY_IRD1 " NAME"
#define SPH_STREHL_FILTER_IRD2 "ESO " SPH_STREHL_FILTER_KEY_IRD2 " NAME"

#define SPH_STREHL_FILTER_ZPL0 "ESO " SPH_STREHL_FILTER_KEY_ZPL0 " NAME"
#define SPH_STREHL_FILTER_ZPL1 "ESO " SPH_STREHL_FILTER_KEY_ZPL1 " NAME"
#define SPH_STREHL_FILTER_ZPL2 "ESO " SPH_STREHL_FILTER_KEY_ZPL2 " NAME"


/*-----------------------------------------------------------------------------
                                   Structs declarations
 -----------------------------------------------------------------------------*/

typedef struct {
	double strehl;
	double strehl_error;
	double star_x;
	double star_y;
	double sigma;
	double star_flux;
	double star_peak;
	double background;
	double background_noise;
} sph_strehl_qc_pars;


typedef enum {
    SPH_ARM_UNKNOWN,
    SPH_ARM_ZIMPOL1,
    SPH_ARM_ZIMPOL2,
    SPH_ARM_IRDIS
} sph_arm;

/*-----------------------------------------------------------------------------
                                   Function declarations
 -----------------------------------------------------------------------------*/

cpl_error_code
sph_strehl_irdis_left(const cpl_image         * left_image,
                      const cpl_frame         * rawframe,
                      const cpl_frame         * filterframe,
                      const char              * recipe,
                      const cpl_parameterlist * parlist,
				      sph_strehl_qc_pars      * qc_out_pars);

cpl_error_code
sph_strehl_irdis_right(const cpl_image         * right_image,
                       const cpl_frame         * rawframe,
                       const cpl_frame         * filterframe,
                       const char              * recipe,
                       const cpl_parameterlist * parlist,
				       sph_strehl_qc_pars      * qc_out_pars);


cpl_error_code
sph_strehl_zimpol(const cpl_image           * image,
                    const cpl_frame         * rawframe,
                    const cpl_frame         * filterframe,
                    const char              * recipe,
                    const cpl_parameterlist * parlist,
					const cpl_boolean		  is_arm1,
					sph_strehl_qc_pars      * qc_out_pars);

cpl_error_code
sph_strehl_disabled_ao_flux_zimpol(const cpl_image         * image_cam,
                                   const cpl_frame         * rawframe,
                                   const cpl_parameterlist * parlist,
                                   const char              * recipe,
                                   const cpl_boolean         is_arm1,
                                   double                  * flux,
                                   double                  * background);

cpl_error_code
sph_strehl_disabled_ao_flux_irdis(const cpl_image         * image_cam,
                                   const cpl_frame         * rawframe,
                                   const cpl_parameterlist * parlist,
                                   const char              * recipe,
                                   double                  * flux,
                                   double                  * background);


cpl_error_code
sph_strehl_fill_qc_pars_irdis_left(cpl_propertylist * self,
							  	   const sph_strehl_qc_pars * qc_pars);

cpl_error_code
sph_strehl_fill_qc_pars_irdis_right(cpl_propertylist * self,
							  	    const sph_strehl_qc_pars * qc_pars);

cpl_error_code
sph_strehl_fill_qc_pars_zimpol(cpl_propertylist * self,
							   const sph_strehl_qc_pars * qc_pars);

cpl_error_code
sph_strehl_fill_qc_pars_zimpol_ao_disabled(cpl_propertylist * self,
							   const double flux,
							   const double background);


cpl_error_code
sph_strehl_irdis_and_append(cpl_propertylist  * plist_left,
							cpl_propertylist  * plist_right,
							const cpl_image   * img_left,
							const cpl_image   * img_right,
							const cpl_frame   * raw_frame,
                            const cpl_frame   * filter_frame,
                            const char        * recipe,
							cpl_parameterlist * parlist,
							double            * flux_left,
							double            * flux_right);


cpl_error_code sph_strehl_read_filter_and_scale(const cpl_propertylist * self,
												sph_arm                  arm,
												double                 * ppscale,
												const char            ** pkey,
												const char            ** pfilter);

#endif /* SPH_STREHL_H_ */
