/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_transform.h"
#include <math.h>
#include <assert.h>

#define SPH_TRANSFORM_VAL_BADISBAD 0.000001
#define SPH_TRANSFORM_MAX_BADPIX 10000000
#define SPH_TRANSFORM_OK_BADPIX 100000


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/


static
sph_transform* sph_transform_new_(void) CPL_ATTR_ALLOC;
static
sph_error_code sph_transform_apply_filter_( sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap);
static
sph_error_code
sph_transform_distort_(const sph_distortion_model* distortionmap,
                       cpl_image* image,
                       cpl_image* ncombmap,
                       cpl_image* rmsmap);
static
sph_error_code
sph_transform_rotate_(
        sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        double angle);
static
sph_error_code
sph_transform_scale_(
        sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        double scale);
static
sph_error_code
sph_transform_shift_(
        sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        double dx,
        double dy);
static
sph_error_code
sph_transform_apply_to_images_fft_(
        sph_transform* self,
        cpl_image** image,
        cpl_image** badpix,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        sph_distortion_model* distortionmap,
        double centx,
        double centy,
        double angle,
        double scale,
        sph_ird_instrument_model* irdmodel);
static
sph_error_code
sph_transform_apply_to_images_cpl_warp_(
        sph_transform* self,
        cpl_image** image,
        cpl_image** badpix,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        sph_distortion_model* distortionmap,
        double centx,
        double centy,
        double angle,
        double scale,
        sph_ird_instrument_model* irdmodel);
static
sph_error_code
sph_transform_apply_displacements_cpl_(
        sph_transform* self,
        cpl_image** image,
        cpl_image** rmsmap,
        cpl_image** ncombmap,
        cpl_image* displacement_x,
        cpl_image* displacement_y) ;
/*----------------------------------------------------------------------------*/
/**
 * @brief Create new default transformation
 *
 * @return new default transformation structure
 *
 *----------------------------------------------------------------------------*/
sph_transform* sph_transform_new_default(void) {
    sph_transform*      result = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result = sph_transform_new_();
    result->method = SPH_TRANSFORM_METHOD_FFT;
    result->filt_method = SPH_FFT_FILTER_METHOD_NONE;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create new cpl_warp transformation
 *
 * @return new transformation structure
 *
 *----------------------------------------------------------------------------*/
sph_transform* sph_transform_new_cpl_warp(void) {
    sph_transform*      result = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result = sph_transform_new_();
    result->method = SPH_TRANSFORM_METHOD_CPL_WARP;
    result->kerneltype = CPL_KERNEL_DEFAULT;
    result->radius = CPL_KERNEL_DEF_WIDTH;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create new FFT transformation
 * @param filtmehthod       the FFT filter method to use
 * @param param1            first parameter of filter
 * @param param2            second parameter of filter
 *
 * @return new transformation structure
 *
 * The parameters are interpreted depending on the FFT filter method. For
 * the top hat filter, the first parameter is the freq. radius,
 * and the second is ignored.
 * For the Fermi filter, the first is the freq. radius and the second the
 * temperature.
 * For the Butterworth filter, the first is the pass frequency and the second
 * the stop frequency.
 * All frequencies should be expressed as fractions of the total frequency
 * domain, and so should all be ranging for 0 (no frequencies filtered) to
 * 1 (all frequencies fitlered out).
 *
 *----------------------------------------------------------------------------*/
sph_transform* sph_transform_new_fft(sph_fft_filter_method filtmethod,
        double param1,
        double param2) {
    sph_transform*      result = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result = sph_transform_new_();
    result->method = SPH_TRANSFORM_METHOD_FFT;
    result->filt_method = filtmethod;
    if ( filtmethod == SPH_FFT_FILTER_METHOD_TH ) {
        result->fft_filter_radius = param1;
    }
    if ( filtmethod == SPH_FFT_FILTER_METHOD_FERMI ) {
        result->fft_filter_radius = param1;
        result->fft_fermi_temp = param2;
    }
    if ( filtmethod == SPH_FFT_FILTER_METHOD_BUTTER ) {
        result->fft_butter_pass= param1;
        result->fft_butter_stop = param2;
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Center (shift), dedistort and rotate master frame
 * @param mframe        input master frame (change on output)
 * @param distortionmap the distortionmap to apply (maybe NULL)
 * @param centx         the centre in x (pixel coords)
 * @param centy         the centre in y (pixel coords)
 * @param angle         the rotation angle to derotate from (degrees ccw x-axis)
 * @param scale         the scaling (1.0 no change,<1 shrink,>1 enlarge)
 * @param filter_radius frequency filter radius (see sph_fft_freq_filter)
 * @param irdis_model    IRDIS model, set to NULL for all other instruments.
 * @return error code
 *
 * This routine will first recenter the master frame so that the
 * image centre corresponds to the given centre coordinates. For
 * even sized images, the image centre is between the 4 central pixels
 * for odd sized images it is the centre of the central pixel.
 * For example, if the image size is 20 x 20 pixels, giving centx and centy
 * as 10.0 and 10.0 will not result in a shift, but giving 8.5 and 11.5
 * will result in a shift of 1.5 pixels in x and -1.5 pixels in y.
 *
 * After centering, the distortion map is applied. This is skipped if
 * the distortionmap pointer is NULL.
 *
 * If the angle provided is not 0 the master frame is now rotated around the
 * image centre. If the scaling is not 1 the image is now scaled.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_transform_apply(
        sph_transform* self,
        sph_master_frame* mframe,
        sph_distortion_model* distortionmap,
        double centx,
        double centy,
        double angle,
        double scale,
        sph_ird_instrument_model* irdmodel)
{
    cpl_image*      bpix = NULL;
    cpl_ensure_code(mframe,CPL_ERROR_NULL_INPUT);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    bpix = sph_master_frame_get_badpixelmap(mframe);

    sph_master_frame_interpolate_bpix(mframe);

    if ( cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND ) {
        SPH_INFO_MSG( "All pixels were bad in an image - so am not applying transformation");
        cpl_error_reset();
    }

    cpl_ensure_code( sph_transform_apply_to_images(
            self,
            &mframe->image,
            &bpix,
            &mframe->ncombmap,
            &mframe->rmsmap,
            distortionmap,
            centx,centy,angle,scale,irdmodel) == CPL_ERROR_NONE,
            cpl_error_get_code() );
    sph_master_frame_set_bads_from_image(mframe,bpix);
    cpl_image_delete(bpix); bpix = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Center (shift), dedistort and rotate double image (DOUBLE MODE)
 * @param doubleimage   input double image (change on output)
 * @param distortionmap the distortionmap to apply (maybe NULL)
 * @param centlx         the centre in x (pixel coords)
 * @param cently         the centre in y (pixel coords)
 * @param centrx         the centre in x (pixel coords)
 * @param centry         the centre in y (pixel coords)
 * @param angle         the rotation angle to derotate from (degrees ccw x-axis)
 * @param scale         the scaling (1.0 no change,<1 shrink,>1 enlarge)
 * @param filter_radius frequency filter radius (see sph_fft_freq_filter)
  * @return error code
 *
 * This routine is a wrapper for  sph_common_science_apply_transforms functions.
 * It will first be applied to the intensity master frame (iframe) of the given
 * double image and then to the polarization master frame (pframe) of the given
 * double image. See sph_common_science_apply_transforms for the detailed
 * transformation description.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_transform_apply_to_doubleimage_doublemode(
        sph_transform* self,
        sph_double_image* doubleimage,
        sph_distortion_model* distortionmap,
        double centlx,
        double cently,
        double centrx,
        double centry,
        double angle,
        double scale)
{
    cpl_ensure_code(doubleimage->iframe,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(doubleimage->pframe,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    sph_transform_apply(self,doubleimage->iframe, distortionmap,
            centlx, cently, angle, scale, NULL );
    sph_transform_apply(self,doubleimage->pframe, distortionmap,
            centrx, centry, angle, scale, NULL );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Center (shift), dedistort and rotate double image (SINGLE MODE)
 * @param doubleimage   input double image (change on output)
 * @param distortionmap the distortionmap to apply (maybe NULL)
 * @param centlx         the centre in x (pixel coords)
 * @param cently         the centre in y (pixel coords)
 * @param angle         the rotation angle to derotate from (degrees ccw x-axis)
 * @param scale         the scaling (1.0 no change,<1 shrink,>1 enlarge)
 * @param filter_radius frequency filter radius (see sph_fft_freq_filter)
  * @return error code
 *
 * This routine is a wrapper for  sph_common_science_apply_transforms functions.
 * It will first be applied to the intensity master frame (iframe) of the given
 * double image and then to the polarization master frame (pframe) of the given
 * double image. See sph_common_science_apply_transforms for the detailed
 * transformation description.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_transform_apply_to_doubleimage(
        sph_transform* self,
        sph_double_image* doubleimage,
        sph_distortion_model* distortionmap,
        double centx,
        double centy,
        double angle,
        double scale)
{
    cpl_ensure_code(doubleimage->iframe,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(doubleimage->pframe,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    sph_transform_apply(self,doubleimage->iframe, distortionmap,
            centx, centy, angle, scale, NULL );
    sph_transform_apply(self,doubleimage->pframe, distortionmap,
            centx, centy, angle, scale, NULL );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Center (shift), dedistort and rotate master frame
 * @param image         input image (change on output, maybe NULL)
 * @param badpixel      input image (change on output, maybe NULL)
 * @param weights       input image (change on output, maybe NULL)
 * @param rms           input image (change on output, maybe NULL)
 * @param distortionmap the distortionmap to apply (maybe NULL)
 * @param centx         the centre in x (pixel coords)
 * @param centy         the centre in y (pixel coords)
 * @param angle         the rotation angle (degrees ccw x-axis)
 * @param scale         the scaling (1.0 no change,<1 shrink,>1 enlarge)
 * @param irdis_model    IRDIS model, set to NULL for all other instruments.
 * @return error code
 *
 * This routine will first recenter the images so that the
 * image centre corresponds to the given centre coordinates. For
 * even sized images, the image centre is between the 4 central pixels
 * for odd sized images it is the centre of the central pixel.
 * For example, if the image size is 20 x 20 pixels, giving centx and centy
 * as 10.0 and 10.0 will not result in a shift, but giving 8.5 and 11.5
 * will result in a shift of 1.5 pixels in x and -1.5 pixels in y.
 *
 * After centering, the distortion map is applied. This is skipped if
 * the distortionmap pointer is NULL.
 *
 * If the angle provided is not 0 the images are now rotated around the
 * image centre. If the scaling is not 1 the images is now scaled.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_transform_apply_to_images(
        sph_transform* self,
        cpl_image** image,
        cpl_image** badpix,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        sph_distortion_model* distortionmap,
        double centx,
        double centy,
        double angle,
        double scale,
        sph_ird_instrument_model* irdmodel)
{

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image || badpix || ncombmap || rmsmap,CPL_ERROR_NULL_INPUT);

    cpl_msg_info(cpl_func, "Transforming image-triplet using %d <=> %d",
                 (int)self->method, (int)SPH_TRANSFORM_METHOD_CPL_WARP);

    if ( self->method == SPH_TRANSFORM_METHOD_FFT ) {
        sph_transform_apply_to_images_fft_(self, image, badpix, ncombmap,
                                            rmsmap, distortionmap, centx, centy,
                                            angle, scale, irdmodel);
    }
    else if ( self->method == SPH_TRANSFORM_METHOD_CPL_WARP ) {
        sph_transform_apply_to_images_cpl_warp_(self, image, badpix, ncombmap,
                                                rmsmap, distortionmap, centx,
                                                centy, angle, scale, irdmodel);
    }
    else {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_UNSUPPORTED_MODE,
                "The transform method is not supported");
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
cpl_image* sph_transform_create_array_of_badpixel_positions(
        const cpl_image* inbadpix,
        const sph_ird_instrument_model* irdmodel,
        int** pbadout, size_t** pbadpos_x, size_t** pbadpos_y, size_t* y,
        int* bpcount)
{
    cpl_image* imout;
    const int*   pbadin_int = NULL;
    const double* pbadin_double = NULL;
    const size_t nx     = cpl_image_get_size_x(inbadpix);
    const size_t ny     = cpl_image_get_size_y(inbadpix);
    size_t       yy;
    size_t       xx;

    if ( cpl_image_get_type(inbadpix) == CPL_TYPE_INT ) {
        pbadin_int = cpl_image_get_data_int_const(inbadpix);
        cpl_ensure(pbadin_int, CPL_ERROR_NULL_INPUT, NULL);
    }
    else { // badpixelmap type must be int or double
        assert( cpl_image_get_type(inbadpix) == CPL_TYPE_DOUBLE );
        pbadin_double = cpl_image_get_data_double_const(inbadpix);
        cpl_ensure(pbadin_double, CPL_ERROR_NULL_INPUT, NULL);
    }


    imout = cpl_image_new(nx, ny, CPL_TYPE_INT); /* Start with no bad pixels */
    *pbadout = cpl_image_get_data_int(imout);
    *pbadpos_x = cpl_malloc(SPH_TRANSFORM_MAX_BADPIX * sizeof(**pbadpos_x));
    *pbadpos_y = cpl_malloc(SPH_TRANSFORM_MAX_BADPIX * sizeof(**pbadpos_y));
    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            if ((pbadin_int && !pbadin_int[xx + yy * nx]) || (pbadin_double && !pbadin_double[xx + yy * nx]))
            {
                /* Good pixel locations already reset */
            } else if (irdmodel == NULL
                    || sph_ird_instrument_model_check_in_dbi_fov(irdmodel, xx,
                            yy, 1, 10.0)) {
                (*pbadpos_x)[*bpcount] = xx;
                (*pbadpos_y)[*bpcount] = yy;

                if (++(*bpcount) == SPH_TRANSFORM_MAX_BADPIX)
                    break;

            } else {

                /*
                 Dont do anything here -- pixels that are outside the
                 IRDIS FOV are not changed in their badpixel properties
                 and not rotated.
                 */
                if ( pbadin_int ) {
                    (*pbadout)[xx + yy * nx] = pbadin_int[xx + yy * nx];
                }
                else {
                    (*pbadout)[xx + yy * nx] = pbadin_double[xx + yy * nx];
                }
            }
        }
        if (xx < nx)
            break;
    }
    *y = yy;
    return imout;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Remap the badpixels to rotated, shifted and distortion coords.
 * @param inbadbpix  the original badpixelmap
 * @param angle     the rotation angle [degree]
 * @param centx     rotation centre in pixel coordinates
 * @param centy     rotation centre in pixel coordinates
 * @param dx        shift in x
 * @param dy        shift in y
 * @param distmap   the polynomial distortion model to apply (maybe NULL)
 * @param irdis_model    IRDIS model, set to NULL for all other instruments.
 *
 * @return the new badpixel image or NULL on error.
 *
 * This function creates a new badpixel image of the same type as the input
 * image (usually CPL_TYPE_INT). It maps all pixels with values >= 1 to
 * a new image using the transformation specified (shift, rotation and,
 * if pointer to a model is given, distortion).
 * The transformation is purely geometric -- the pixel is transformed to
 * a polygon (rectangle) that is rotated and mapped onto the output image.
 * In the output image, all pixels that overlap the original bad pixel are
 * set to the value 1.
 *
 * The special input parameter "irdis_model" is should be set
 * if this function is to rotate a badpixel map from IRDIS in DBI or DPI mode,
 * as it correctly handles the badpixels at the edge of the images due to
 * the circular aperture. Set to NULL for any other instrument.
 *
 * Note that the function will not work correctly for
 * extreme distortions (with maximal distortions of more than a few pixels).
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_transform_remap_badpix( sph_transform* self,
                            const cpl_image* inbadpix,
                            double angle, double centx, double centy,
                            double dx0, double dy0,
                            sph_distortion_model* distmap,
                            double scale,
                            const sph_ird_instrument_model* irdmodel)
{
    const size_t nx     = cpl_image_get_size_x(inbadpix);
    const size_t ny     = cpl_image_get_size_y(inbadpix);
    cpl_image*   imout;
    int*         pbadout;
    size_t*      pbadpos_x = NULL;
    size_t*      pbadpos_y = NULL;
    size_t       yy;
    int          bpcount = 0;

    static int   already_warned_ = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    cpl_ensure(self,   CPL_ERROR_NULL_INPUT, NULL);

    imout = sph_transform_create_array_of_badpixel_positions(inbadpix, irdmodel,
                                                             &pbadout,
                                                             &pbadpos_x,
                                                             &pbadpos_y, &yy,
                                                             &bpcount);

    if (bpcount > 0) {
        sph_polygon*        poly  = NULL;
        sph_polygon*        poly2 = NULL;
        sph_polygon_workspace* ws = NULL;
        int                 bpix = 0;
        const double        cosangle = cos(angle * CPL_MATH_RAD_DEG);
        const double        sinangle = sin(angle * CPL_MATH_RAD_DEG);
        sph_point           midpoint;
        double              dx = dx0;
        double              dy = dy0;
        int                 bpactual = 0;
        int                 jj;


        if (yy < ny) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                                "The number of bad pixels is at least: "
                                CPL_STRINGIFY(SPH_TRANSFORM_MAX_BADPIX)
                                "This recipe can not cope with such bad quality"
                                " data. Please check your dark and flat quality"
                                " or recompile with a higher value of "
                                CPL_XSTRINGIFY(SPH_TRANSFORM_MAX_BADPIX));
            cpl_image_delete(imout);
            cpl_free(pbadpos_x);
            cpl_free(pbadpos_y);
            return NULL;
        }

        if (bpcount >= SPH_TRANSFORM_OK_BADPIX && already_warned_ == 0) {
            SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,
                                    "The total number pf bad pixels is %d >= "
                                    CPL_STRINGIFY(SPH_TRANSFORM_OK_BADPIX)
                                    ". This is so large that the performance is"
                                    " suffering. I will continue, but please "
                                    "check your dark and flat quality",bpcount);
#           ifdef _OPENMP
#           pragma omp atomic
#           endif
            already_warned_ = 1;
        }

        poly = sph_polygon_new();
        /* FIXME: Why this ? */
        sph_polygon_add_point(poly, 0.0, 0.0);
        sph_polygon_add_point(poly, 0.0, 1.0);
        sph_polygon_add_point(poly, 1.0, 1.0);
        sph_polygon_add_point(poly, 1.0, 0.0);

        poly2 = sph_polygon_new();
        sph_polygon_add_point(poly2, 0.0, 0.0);
        sph_polygon_add_point(poly2, 0.0, 1.0);
        sph_polygon_add_point(poly2, 1.0, 1.0);
        sph_polygon_add_point(poly2, 1.0, 0.0);

        ws = sph_polygon_workspace_new();

        for (bpix = 0; bpix < bpcount; bpix++) {
            const double xbad = (const double)pbadpos_x[bpix];
            const double ybad = (const double)pbadpos_y[bpix];
            const double delta = 0.00000000001;

            poly->points[0].x = xbad;
            poly->points[0].y = ybad;

            poly->points[1].x = xbad;
            poly->points[1].y = ybad + 1 - delta;

            poly->points[2].x = xbad + 1 - delta;
            poly->points[2].y = ybad + 1 - delta;

            poly->points[3].x = xbad + 1 - delta;
            poly->points[3].y = ybad;

            sph_polygon_get_mid_xy(&midpoint, poly);

            if ( distmap ) {
                sph_polygon_shift(poly, -centx, -centy);
                sph_distortion_model_apply_poly(distmap, poly);
                sph_polygon_shift(poly, centx, centy);
            }
            if ( scale != 1.0 && scale != 0.0 ) {
                dx = dx0 + (midpoint.x - centx) * ( scale - 1.0);
                dy = dy0 + (midpoint.y - centy) * ( scale - 1.0);
            }

            sph_polygon_rotate_around(poly, centx, centy, cosangle, sinangle,
                                      dx, dy);

            sph_polygon_get_mid_xy(&midpoint, poly);

            for (jj = (int)midpoint.y - 1; jj <= (int)midpoint.y + 1; jj++) {
                int ii;

                if (jj < 0) continue;
                if (jj >= (int)ny) continue;

                for (ii = (int)midpoint.x - 1; ii <= (int)midpoint.x + 1; ii++) {
                    if (0 <= ii && ii < (int)nx) {
                        double overlap;

                        poly2->points[0].x = ii;
                        poly2->points[0].y = jj;

                        poly2->points[1].x = ii;
                        poly2->points[1].y = jj + 1.0;

                        poly2->points[2].x = ii + 1.0;
                        poly2->points[2].y = jj + 1.0;

                        poly2->points[3].x = ii + 1.0;
                        poly2->points[3].y = jj;

                        overlap = sph_polygon_calculate_overlap(poly, poly2, ws);

                        if (overlap > SPH_TRANSFORM_VAL_BADISBAD) {
                            bpactual++;
                            pbadout[ii + jj * nx] = 1;
                        }
                    }
                }
            }
        }
        sph_polygon_workspace_delete(ws);
        sph_polygon_delete(poly);
        sph_polygon_delete(poly2);
        sph_polygon_free_all();
    }

    cpl_free(pbadpos_x);
    cpl_free(pbadpos_y);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return imout;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the transform object and free up space.
 * @param self      the transform structure to delete
 */
/*----------------------------------------------------------------------------*/
void sph_transform_delete(sph_transform* self)
{
    cpl_free(self);
}

/*----------------------------------------------------------------------------*/
/**
 *  Static functons
 */
/*----------------------------------------------------------------------------*/
static
sph_transform* sph_transform_new_(void) {
    sph_transform*      result = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    result = cpl_calloc(1,sizeof(sph_transform));
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

static
sph_error_code
sph_transform_apply_to_images_cpl_warp_(
        sph_transform* self,
        cpl_image** image,
        cpl_image** badpix,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        sph_distortion_model* distortionmap,
        double centx,
        double centy,
        double angle,
        double scale,
        sph_ird_instrument_model* irdmodel)
{
    cpl_image*          displacement_x = NULL;
    cpl_image*          displacement_y = NULL;
    double              imcentx = 0.0;
    double              imcenty = 0.0;
    double              dx = 0.0;
    double              dy = 0.0;
    cpl_image*          tmpim = NULL;
    int                 nx = 0;
    int                 ny = 0;
    int                 xx  = 0;
    int                 yy  = 0;
    sph_point*           pix_cent = NULL;
    sph_point*           new_cent = NULL;
    double              cosangle = 0.0;
    double              sinangle = 0.0;
    double              v = 0.0;

    cpl_ensure_code(image || badpix || ncombmap || rmsmap,CPL_ERROR_NULL_INPUT);

    if ( image ) {
        nx = cpl_image_get_size_x(*image);
        ny = cpl_image_get_size_y(*image);
    }
    else if ( badpix ) {
        nx = cpl_image_get_size_x(*badpix);
        ny = cpl_image_get_size_y(*badpix);
    }
    else if ( ncombmap ) {
        nx = cpl_image_get_size_x(*ncombmap);
        ny = cpl_image_get_size_y(*ncombmap);
    }
    else if ( rmsmap ) {
        nx = cpl_image_get_size_x(*rmsmap);
        ny = cpl_image_get_size_y(*rmsmap);
    }

    imcentx = 0.5 * (double)nx;
    imcenty = 0.5 * (double)ny;
    dx = imcentx - centx;
    dy = imcenty - centy;


    if ( distortionmap ) {
        sph_transform_distort_(distortionmap, *image, *ncombmap, *rmsmap);
    }
    displacement_x = cpl_image_new( nx, ny, CPL_TYPE_DOUBLE);
    displacement_y = cpl_image_new( nx, ny, CPL_TYPE_DOUBLE);
    if ( dx != 0.0 || dy != 0.0 ) {
        cpl_image_add_scalar(displacement_x,dx);
        cpl_image_add_scalar(displacement_y,dy);
        sph_transform_apply_displacements_cpl_(self,image,rmsmap,ncombmap,displacement_x,displacement_y);
        cpl_image_multiply_scalar(displacement_x,0.0);
        cpl_image_multiply_scalar(displacement_y,0.0);
    }

    if ( angle != 0.0 || scale != 1.0 ) {
        pix_cent = sph_point_new(0.0,0.0);
        new_cent = sph_point_new(0.0,0.0);
        cosangle = cos(1.0 * angle * CPL_MATH_RAD_DEG);
        sinangle = sin(1.0 * angle * CPL_MATH_RAD_DEG);
        for (yy = 0; yy < ny; ++yy) {
            for (xx = 0; xx < nx; ++xx) {
                new_cent->x = pix_cent->x = (double)xx + 0.5;
                new_cent->y = pix_cent->y = (double)yy + 0.5;
                sph_point_rotate_around(new_cent,imcentx,imcenty,cosangle,sinangle,0.0,0.0);
                new_cent->x = (new_cent->x - imcentx) / scale + imcentx;
                new_cent->y = (new_cent->y - imcenty) / scale + imcenty;
                v =  pix_cent->x - new_cent->x;
                cpl_image_set(displacement_x, xx + 1, yy + 1, v);
                v = pix_cent->y - new_cent->y;
                cpl_image_set(displacement_y, xx + 1, yy + 1, v);
            }
        }
        sph_point_delete(pix_cent);
        sph_point_delete(new_cent);
        sph_transform_apply_displacements_cpl_(self,image,rmsmap,ncombmap,displacement_x,displacement_y);
        cpl_image_multiply_scalar(displacement_x,0.0);
        cpl_image_multiply_scalar(displacement_y,0.0);
    }


    if ( badpix ) {

        tmpim = sph_transform_remap_badpix(self,*badpix,
                -1.0*angle,centx,centy,imcentx-centx,imcenty-centy,
                distortionmap,scale,irdmodel);

        cpl_image_delete(*badpix); *badpix = NULL;

        *badpix = tmpim;  tmpim = NULL;
    }

    cpl_image_delete(displacement_x); displacement_x = NULL;
    cpl_image_delete(displacement_y); displacement_y = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}



static
sph_error_code
sph_transform_apply_to_images_fft_(
        sph_transform* self,
        cpl_image** image,
        cpl_image** badpix,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        sph_distortion_model* distortionmap,
        double centx,
        double centy,
        double angle,
        double scale,
        sph_ird_instrument_model* irdmodel)
{
    cpl_image*          tmpim = NULL;
    double              imcentx = 0.0;
    double              imcenty = 0.0;
    sph_fft*            fft = NULL;

    cpl_ensure_code(image || badpix || ncombmap || rmsmap,CPL_ERROR_NULL_INPUT);

    if ( image ) {
        imcentx = (double)cpl_image_get_size_x(*image);
        imcenty = (double)cpl_image_get_size_y(*image);
    }
    else if ( badpix ) {
        imcentx = (double)cpl_image_get_size_x(*badpix);
        imcenty = (double)cpl_image_get_size_y(*badpix);
    }
    else if ( ncombmap ) {
        imcentx = (double)cpl_image_get_size_x(*ncombmap);
        imcenty = (double)cpl_image_get_size_y(*ncombmap);
    }
    else if ( rmsmap ) {
        imcentx = (double)cpl_image_get_size_x(*rmsmap);
        imcenty = (double)cpl_image_get_size_y(*rmsmap);
    }

    imcentx /= 2.0;
    imcenty /= 2.0;

    fft = sph_fft_new(SPH_FFT_FFTW3_DP);

    sph_transform_apply_filter_(self, fft, image, ncombmap, rmsmap);

    if ( fabs(imcentx - centx) > 0.000001 ||
            fabs(imcenty - centy) > 0.000001 ) {
        sph_transform_shift_(self,
                fft,image,
                ncombmap,rmsmap,
                imcentx-centx,imcenty-centy);
    }
    if ( distortionmap ) {
        sph_transform_distort_(distortionmap,
                *image,
                *ncombmap,*rmsmap);
    }

    sph_transform_rotate_(self,fft,image,ncombmap,rmsmap,angle);

    if (scale != 1.0 ) {
        sph_transform_scale_(self,fft,image,ncombmap,rmsmap,scale);
    }

    if ( badpix ) {

        tmpim = sph_transform_remap_badpix(self,*badpix,
                -1.0*angle,centx,centy,imcentx-centx,imcenty-centy,
                distortionmap,scale,irdmodel);

        cpl_image_delete(*badpix); *badpix = NULL;

        *badpix = tmpim;  tmpim = NULL;
    }

    sph_fft_delete(fft); fft = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_error_code
sph_transform_apply_displacements_cpl_(
        sph_transform* self,
        cpl_image** image,
        cpl_image** rmsmap,
        cpl_image** ncombmap,
        cpl_image* displacement_x,
        cpl_image* displacement_y)
{
    cpl_vector*         profile = NULL;
    cpl_image*          tmpim = NULL;
    cpl_image*          correction_map = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    profile = cpl_vector_new( 2 * (int)self->radius);
    cpl_vector_fill_kernel_profile(profile,self->kerneltype,self->radius);

    if ( image ) {
        tmpim = cpl_image_new( cpl_image_get_size_x( *image ),
                cpl_image_get_size_y(*image), cpl_image_get_type(*image));
        cpl_image_warp(tmpim,*image,displacement_x, displacement_y, profile, self->radius, profile, self->radius);
        cpl_image_delete(*image);*image = NULL;
        correction_map = cpl_image_new(cpl_image_get_size_x(tmpim),
                                        cpl_image_get_size_y(tmpim),
                                        cpl_image_get_type(tmpim));
        cpl_image_fill_jacobian(correction_map, displacement_x, displacement_y);
        *image = cpl_image_multiply_create(tmpim, correction_map);
        cpl_image_delete(tmpim); tmpim = NULL;
        cpl_image_delete(correction_map); correction_map = NULL;
    }
    if ( rmsmap ) {
        tmpim = cpl_image_new( cpl_image_get_size_x( *rmsmap ),
                cpl_image_get_size_y(*rmsmap), cpl_image_get_type(*rmsmap));
        cpl_image_warp(tmpim,*rmsmap,displacement_x, displacement_y, profile, self->radius, profile, self->radius);
        cpl_image_delete(*rmsmap);*rmsmap = NULL;
        *rmsmap = tmpim; tmpim = NULL;
    }
    if ( ncombmap ) {
        tmpim = cpl_image_new( cpl_image_get_size_x( *ncombmap ),
                cpl_image_get_size_y(*ncombmap), cpl_image_get_type(*ncombmap));
        cpl_image_warp(tmpim,*ncombmap,displacement_x, displacement_y, profile, self->radius, profile, self->radius);
        cpl_image_delete(*ncombmap);*ncombmap = NULL;
        *ncombmap = tmpim; tmpim = NULL;
    }
    cpl_vector_delete(profile); profile = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}




static
sph_error_code sph_transform_apply_filter_( sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap)
{
    cpl_image*          tmpim = NULL;

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fft,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    if ( self->filt_method == SPH_FFT_FILTER_METHOD_TH ) {
        tmpim = sph_fft_freq_filter(fft,*image,SPH_FFT_FILTER_METHOD_TH,
                                    self->fft_filter_radius,0.0,0.0);
        cpl_image_delete(*image);*image = NULL;
        *image = tmpim; tmpim = NULL;
        if ( *rmsmap ) {
            tmpim = sph_fft_freq_filter(fft,*rmsmap,SPH_FFT_FILTER_METHOD_TH,
                                        self->fft_filter_radius,0.0,0.0);
            cpl_image_delete(*rmsmap);*rmsmap = NULL;
            *rmsmap = tmpim; tmpim = NULL;
        }
        tmpim = sph_fft_freq_filter(fft,*ncombmap,SPH_FFT_FILTER_METHOD_TH,
                                    self->fft_filter_radius,0.0,0.0);
        cpl_image_delete(*ncombmap);*ncombmap = NULL;
        *ncombmap = tmpim; tmpim = NULL;
    }
    if ( self->filt_method == SPH_FFT_FILTER_METHOD_FERMI ) {
        tmpim = sph_fft_freq_filter(fft,*image,SPH_FFT_FILTER_METHOD_TH,
                self->fft_filter_radius,self->fft_fermi_temp,0.0);
        cpl_image_delete(*image);*image = NULL;
        *image = tmpim; tmpim = NULL;
        if ( *rmsmap ) {
            tmpim = sph_fft_freq_filter(fft,*rmsmap,SPH_FFT_FILTER_METHOD_TH,
                    self->fft_filter_radius,self->fft_fermi_temp,0.0);
            cpl_image_delete(*rmsmap);*rmsmap = NULL;
            *rmsmap = tmpim; tmpim = NULL;
        }
        tmpim = sph_fft_freq_filter(fft,*ncombmap,SPH_FFT_FILTER_METHOD_TH,
                self->fft_filter_radius,self->fft_fermi_temp,0.0);
        cpl_image_delete(*ncombmap);*ncombmap = NULL;
        *ncombmap = tmpim; tmpim = NULL;
    }
    if ( self->filt_method == SPH_FFT_FILTER_METHOD_BUTTER ) {
        tmpim = sph_fft_freq_filter(fft,*image,SPH_FFT_FILTER_METHOD_TH,
                0.0,self->fft_butter_pass,self->fft_butter_stop);
        cpl_image_delete(*image);*image = NULL;
        *image = tmpim; tmpim = NULL;
        if ( *rmsmap ) {
            tmpim = sph_fft_freq_filter(fft,*rmsmap,SPH_FFT_FILTER_METHOD_TH,
                    0.0,self->fft_butter_pass,self->fft_butter_stop);
            cpl_image_delete(*rmsmap);*rmsmap = NULL;
            *rmsmap = tmpim; tmpim = NULL;
        }
        tmpim = sph_fft_freq_filter(fft,*ncombmap,SPH_FFT_FILTER_METHOD_TH,
                0.0,self->fft_butter_pass,self->fft_butter_stop);
        cpl_image_delete(*ncombmap);*ncombmap = NULL;
        *ncombmap = tmpim; tmpim = NULL;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_error_code
sph_transform_shift_(
        sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        double dx,
        double dy)
{
    cpl_image*          tmpim = NULL;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fft,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_msg_debug(cpl_func, "FFT-shift(%p:%p:%p): dx=%g. dy=%g (%u <=> %u)",
                 (const void*)image, (const void*)ncombmap, (const void*)rmsmap,
                 dx, dy, fft->method, SPH_FFT_FFTW3_DP);

    if (fft->method == SPH_FFT_FFTW3_DP && image && rmsmap && ncombmap &&
        *image && *rmsmap && *ncombmap) {

        fft->nx = fft->ny = cpl_image_get_size_x(*image);
        if (sph_fft_shift_triplet(fft, *image, *rmsmap, *ncombmap, dx, dy)) {
            return cpl_error_set_where(cpl_func);
        }
    } else {
        if ( image ) {
            tmpim = sph_fft_shift_image(fft,*image,dx,dy);
            cpl_image_delete(*image);*image = NULL;
            *image = tmpim; tmpim = NULL;
        }
        if ( ncombmap) {
            tmpim = sph_fft_shift_image(fft,*ncombmap,dx,dy);
            cpl_image_delete(*ncombmap);*ncombmap = NULL;
            *ncombmap = tmpim; tmpim = NULL;
        }
        if ( rmsmap ) {
            tmpim = sph_fft_shift_image(fft,*rmsmap,dx,dy);
            cpl_image_delete(*rmsmap);*rmsmap = NULL;
            *rmsmap = tmpim; tmpim = NULL;
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


static
sph_error_code
sph_transform_scale_(
        sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        double scale)
{
    cpl_image*          tmpim = NULL;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fft,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_msg_debug(cpl_func, "FFT-scale(%p:%p:%p): scale=%g",
                 (const void*)image, (const void*)ncombmap, (const void*)rmsmap,
                  scale);

    if ( image ) {
        tmpim = sph_fft_scale(fft,*image,scale);
        cpl_image_delete(*image);*image = NULL;
        *image = tmpim; tmpim = NULL;
    }
    if ( ncombmap ) {
        tmpim = sph_fft_scale(fft,*ncombmap,scale);
        cpl_image_delete(*ncombmap);*ncombmap = NULL;
        *ncombmap = tmpim; tmpim = NULL;
    }
    if ( rmsmap ) {
        tmpim = sph_fft_scale(fft,*rmsmap,scale);
        cpl_image_delete(*rmsmap);*rmsmap = NULL;
        *rmsmap = tmpim; tmpim = NULL;
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_error_code
sph_transform_rotate_(
        sph_transform* self,
        sph_fft* fft,
        cpl_image** image,
        cpl_image** ncombmap,
        cpl_image** rmsmap,
        double angle)
{
    cpl_image*          tmpim = NULL;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fft,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_msg_info(cpl_func, "FFT-rotate(%p:%p:%p): angle=%g (%d:%d)",
                 (const void*)image, (const void*)ncombmap, (const void*)rmsmap,
                 angle, fft->method, SPH_FFT_FFTW3_DP);

    if ( image ) {
        tmpim = sph_fft_rotate_image(fft,*image,-1.0*angle);
        cpl_image_delete(*image);*image = NULL;
        *image = tmpim; tmpim = NULL;
    }
    if ( ncombmap ) {
        tmpim = sph_fft_rotate_image(fft,*ncombmap,-1.0*angle);
        cpl_image_delete(*ncombmap);*ncombmap = NULL;
        *ncombmap = tmpim; tmpim = NULL;
    }
    if ( rmsmap ) {
        tmpim = sph_fft_rotate_image(fft,*rmsmap,-1.0 * angle);
        cpl_image_delete(*rmsmap);*rmsmap = NULL;
        *rmsmap = tmpim; tmpim = NULL;
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_error_code
sph_transform_distort_(const sph_distortion_model* distortionmap,
        cpl_image* image,
        cpl_image* ncombmap,
        cpl_image* rmsmap)
{
    cpl_ensure_code(distortionmap,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_msg_info(cpl_func, "Applying distortion correction: %d <=> %d",
                 (int)cpl_polynomial_get_degree(distortionmap->polyx),
                 (int)cpl_polynomial_get_degree(distortionmap->polyy));

    if (image != NULL) {
        cpl_image* copy = cpl_image_duplicate(image);

        sph_distortion_model_apply_1(image, distortionmap,
                                     copy,
                                     CPL_KERNEL_DEFAULT,
                                     CPL_KERNEL_DEF_WIDTH,
                                     NULL, NULL, NULL);
        cpl_image_delete(copy);
    }
    if (ncombmap != NULL) {
        cpl_image* copy = cpl_image_duplicate(ncombmap);
        sph_distortion_model_apply_1(ncombmap, distortionmap,
                                     copy,
                                     CPL_KERNEL_DEFAULT,
                                     CPL_KERNEL_DEF_WIDTH,
                                     NULL, NULL, NULL);
        cpl_image_delete(copy);
    }
    if (rmsmap != NULL) {
        cpl_image* copy = cpl_image_duplicate(rmsmap);
        sph_distortion_model_apply_1(rmsmap, distortionmap,
                                     copy,
                                     CPL_KERNEL_DEFAULT,
                                     CPL_KERNEL_DEF_WIDTH,
                                     NULL, NULL, NULL);
        cpl_image_delete(copy);
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/**@}*/
