/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_utils.h"

#include "sph_time.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_double_image.h"
#include "sph_fits.h"
#include "sph_keyword_manager.h"
#include "sph_master_frame.h"
#include "sph_cube.h"
#include "sph_filemanager.h"

#include <cpl.h>
#include <fitsio.h>

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <assert.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/

/* Disable incorrect WCS for now */
#ifdef SPH_USE_WCS
#define SPH_USE_RADEC 1
#else
#define SPH_USE_RADEC 0
#endif

#define DIT_EXPTIME_ACCURACY 0.001

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_utils  Sphere utility functions.
 *
 * @par Descirption:
 * This module contains routines to support the functionality in other
 * sphere DRH modules. Several utility functions are included, many of them
 * extended the functionality of the CPL.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
sph_error_code SPH_UTILS_GENERAL            = SPH_UTILS_ERR_START + 0;
sph_error_code SPH_UTILS_FRAMES_MISSING     = SPH_UTILS_ERR_START + 1;
sph_error_code SPH_UTILS_TOO_FINE_DITHER    = SPH_UTILS_ERR_START + 2;


/*----------------------------------------------------------------------------*/
/**
 @brief    Return the license

 @return   a character pointer to the license text.

 Returns the license text for the SPHERE data reduction library.
 */
/*----------------------------------------------------------------------------*/
const char* sph_get_license( void ) {
    char* text = NULL;

    return text;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Initialise all sphere and CPL systems.

 @return   error code

 Does some standard debug initialisation of error reporting systems for
 the Sphere API and CPL -- and initialised CPL as well. If there is an error
 returns SPH_ERROR_GENERAL. In particular the routine does the following:
 <ol>
     <li> call cpl_init() </li>
     <li> set component on </li>
     <li> set domain to Testing and on </li>
     <li> set CPL messages to console to OFF </li>
     <li> set CPL messages to file to CPL_MSG_DEBUG </li>
     <li> call sph_error_new <li>
 </ol>
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_init_erex(void) {

    sph_error_new();
    sph_error_set_reportlevel( 0 );
    sph_keyword_manager_new();


    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Initialise all sphere and CPL systems.

 @return   error code

 Does some standard debug initialisation of error reporting systems for
 the Sphere API and CPL -- and initialised CPL as well. If there is an error
 returns SPH_ERROR_GENERAL. In particular the routine does the following:
 <ol>
     <li> call cpl_init() </li>
     <li> set component on </li>
     <li> set domain to Testing and on </li>
     <li> set CPL messages to console to OFF </li>
     <li> set CPL messages to file to CPL_MSG_DEBUG </li>
     <li> call sph_error_new <li>
 </ol>
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_init(void) {

#if defined CPL_VERSION_CODE && CPL_VERSION_CODE >= CPL_VERSION(4, 0, 0)
    cpl_init(CPL_INIT_DEFAULT);
#else
    cpl_init();
#endif

    cpl_msg_set_time_on();
    cpl_msg_set_component_on();
    cpl_msg_set_domain("Testing");
    cpl_msg_set_domain_on();
    cpl_msg_set_level( CPL_MSG_OFF );
    cpl_msg_set_log_level( CPL_MSG_ERROR );

    sph_error_new();
    sph_error_set_reportlevel( 3 );
    sph_keyword_manager_new();


    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Append a string to propertyname in list
 * @param proplist          pointer to the propertylist
 * @param postfix           the string to append
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_utils_append_to_propertynames( cpl_propertylist* proplist,
        const char* postfix)
{
    int ii;
    cpl_property* property = NULL;
    const char* name = NULL;
    char newname[256];

    for (ii = 0; ii < cpl_propertylist_get_size(proplist); ++ii ) {
        property = cpl_propertylist_get(proplist,ii);
        if ( property ) {
            name = cpl_property_get_name(property);
            snprintf(newname,256,"%s %s",name,postfix);
            cpl_property_set_name(property,newname);
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Update header prior to saving
 * @param  plist               property list to update
 *
 * This function updates the given cpl_propertylist by translating
 * RADECSYS to RADESYS and removing HDRVER.
 */
/*----------------------------------------------------------------------------*/
void sph_utils_update_header(cpl_propertylist* plist)
{
    const char *radecsys_hdr = "RADECSYS";
    const char *radesys_hdr = "RADESYS";
    const char *hdrver_hdr = "HDRVER";
    const char *func = "update_header";
    char *radec_str = NULL;
    char *comment_str = NULL;
    cpl_error_code save_code = cpl_error_get_code();

    if (save_code != CPL_ERROR_NONE) {
      cpl_msg_warning(func, "Error on entry to function: %s", cpl_error_get_message());
      return;
    }

    if (plist) {
        // Remove HDRVER
        cpl_propertylist_erase(plist, hdrver_hdr);
        // Translate RADECSYS to RADESYS.
        radec_str = cpl_strdup(cpl_propertylist_get_string(plist, radecsys_hdr));
        comment_str = cpl_strdup(cpl_propertylist_get_comment(plist, radecsys_hdr));
        cpl_propertylist_erase(plist, radecsys_hdr);
        cpl_propertylist_update_string(plist, radesys_hdr, radec_str);
        cpl_propertylist_set_comment(plist, radesys_hdr, comment_str);
        if (cpl_error_get_code() != CPL_ERROR_NONE) {
          /* Only give a warning if the header exists but there was a problem
           * in translating it */
          if (radec_str) {
              cpl_msg_warning(func, "Error in updating header: %s",
                  cpl_error_get_message());
          }
          if (save_code == CPL_ERROR_NONE) {
            // Don't want this error to propagate up from here
            // But any old errors need to persist
            cpl_error_reset();
          }
        }
        cpl_free(radec_str);
        cpl_free(comment_str);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief copy singular keys from one propertylist to another
 * @param proplist_in          pointer to the input propertylist
 * @param proplist_out         pointer to the output propertylist
 * @return error code
 *
 * This function simply copies all singular (i.e. non-HIERARCH) header keys
 * from the input to the output
 *
 * This is a workaround because certain singular keys are not copied by
 * cpl_save_dfs by default!
 *
 * Overwrites only if the target property is not already present in the
 * destination property list!
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_utils_simple_copy_singular(const cpl_propertylist* pin,
                                              cpl_propertylist* pout)
{

    if (pin == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (pout == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        cpl_propertylist* ptmp = cpl_propertylist_new();
        /* FIXME: The regexp will deselect no HIERARCH keys */
        cpl_error_code code =
            cpl_propertylist_copy_property_regexp(ptmp, pin, "^ESO", CPL_TRUE);
        const cpl_size nsin = code ? 0 : cpl_propertylist_get_size(ptmp);

        for (cpl_size pos = 0; pos < nsin; pos++) {
            const char* pname =
                cpl_property_get_name(cpl_propertylist_get(ptmp, pos));

            if (pname != NULL && !cpl_propertylist_has(pout, pname)) {
                cpl_propertylist_copy_property (pout, ptmp, pname);
            }
        }

        cpl_propertylist_delete(ptmp);
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief determine EXPTIME from raw frames and put into a provided property list
 * @param rawframes            pointer to raw framset
 * @param proplist_out         pointer to the output propertylist
 * @return error code
 *
 * Currently this function simply takes the EXPTIME keyword from the first
 * * raw frame and copies it into the output plist
 *
 * Overwrites only if the target property is not already present in the
 * destination property list!
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_utils_get_exptime_from_raw( cpl_frameset* rawframes, cpl_propertylist* pout)
{


    cpl_propertylist*  ptmp = NULL;
    const char*        ppty = NULL;
    //
    //
    ptmp = cpl_propertylist_load(cpl_frame_get_filename(cpl_frameset_get_first(rawframes)),
                                        0);
    ppty = cpl_property_get_name (cpl_propertylist_get_property(ptmp, SPH_COMMON_KEYWORD_EXPTIME2));
    if (ppty){
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Copying EXPTIME of %f seconds to result header!", cpl_propertylist_get_float(ptmp,"EXPTIME"));
        cpl_propertylist_copy_property (pout,ptmp,ppty);
    }
    ppty = NULL;
    cpl_propertylist_delete(ptmp); ptmp=NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Initialise all sphere and CPL systems verbosely.

 @return   error code

 Does some standard debug initialisation of error reporting systems for
 the Sphere API and CPL -- and initialised CPL as well. If there is an error
 returns SPH_ERROR_GENERAL. In particular the routine does the following:
 <ol>
     <li> call cpl_init() </li>
     <li> set component on </li>
     <li> set domain to Testing and on </li>
     <li> set CPL messages to console to OFF </li>
     <li> set CPL messages to file to CPL_MSG_DEBUG </li>
     <li> call sph_error_new <li>
 </ol>
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_init_debug_verbose(void) {

    cpl_init(CPL_INIT_DEFAULT);

    cpl_msg_set_time_on();
    cpl_msg_set_component_on();
    cpl_msg_set_domain("Testing");
    cpl_msg_set_domain_on();
    cpl_msg_set_level( CPL_MSG_DEBUG );
    cpl_msg_set_log_level( CPL_MSG_DEBUG );

    sph_error_new();
    sph_error_set_reportlevel( 0 );
    sph_keyword_manager_new();

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Initialise all sphere and CPL systems.

 @return   error code

 Does some standard debug initialisation of error reporting systems for
 the Sphere API and CPL -- and initialised CPL as well. If there is an error
 returns SPH_ERROR_GENERAL. In particular the routine does the following:
 <ol>
     <li> call cpl_init() </li>
     <li> set component on </li>
     <li> set domain to Testing and on </li>
     <li> set CPL messages to console to OFF </li>
     <li> set CPL messages to file to CPL_MSG_DEBUG </li>
     <li> call sph_error_new <li>
 </ol>
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_init_debug(void) {

#if defined CPL_VERSION_CODE && CPL_VERSION_CODE >= CPL_VERSION(4, 0, 0)
    cpl_init(CPL_INIT_DEFAULT);
#else
    cpl_init();
#endif

    cpl_msg_set_time_on();
    cpl_msg_set_component_on();
    cpl_msg_set_domain("Testing");
    cpl_msg_set_domain_on();
    cpl_msg_set_level( CPL_MSG_OFF );
    cpl_msg_set_log_level( CPL_MSG_DEBUG );

    sph_error_new();
    sph_error_set_reportlevel( 0 );
    sph_keyword_manager_new();


    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    End all sphere and CPL systems.

 @return   error code

 Closes the Sphere API and CPL system.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_end_erex(void)
{
  sph_filemanager_delete(0);
  sph_keyword_manager_delete();
  sph_error_delete();
  sph_polygon_free_all();
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    End all sphere and CPL systems.

 @return   error code

 Closes the Sphere API and CPL system.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_end(void)
{
  sph_keyword_manager_delete();
  sph_error_delete();
  sph_polygon_free_all();
  cpl_end();
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete files from the hard disk
 @param    frameset   The input frameset which files to be deleted from
 @return   error code.

Delete files from the hard disk
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_utils_frames_unlink( const cpl_frameset* self ) {
    const cpl_size n = cpl_frameset_get_size( self );

    cpl_ensure_code( self != NULL, CPL_ERROR_NULL_INPUT);

    for (cpl_size ii = 0; ii < n ; ++ii) {
        const cpl_frame* iframe = cpl_frameset_get_position_const(self, ii);
        const char* filename = cpl_frame_get_filename(iframe);

        if (filename != NULL) {
            const int ret = remove(filename);
            if (ret)
                cpl_msg_warning(cpl_func, "RM(%d): %s", ret, filename);
        }
    }

    return CPL_ERROR_NONE;
}



/*----------------------------------------------------------------------------*/
/**
 @brief    Extract  frames with a specific keyword setting from a frameset
 @param    frames   The input frames
 @param    tag       A string identifying the  tag of the frames to extract

 @return   new cpl_frameset containing the extracted frames, or NULL on error

 Extract a set of frames from an input frameset which share a common tag.
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_utils_extract_frames( const cpl_frameset* frameset, const char* tag ) {
    cpl_frameset*   RawTagList     = NULL;
    const cpl_frame*aRawFound;

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
         sph_error_raise( SPH_UTILS_GENERAL,
                          __FILE__, __func__, __LINE__ ,
                          SPH_ERROR_WARNING,
                          "Attention cpl error has already been set!  The CPL message: %s\n",
                          cpl_error_get_message() );
         cpl_error_reset();
     }


    if ( !frameset ) {
        SPH_NO_SELF;
        return NULL;
    }
    if ( ! tag ) {
        SPH_NULL_ERROR;
        return NULL;
    }

    aRawFound = cpl_frameset_find_const(frameset, tag);
    if (aRawFound == NULL) {
        const int nframes = (int)cpl_frameset_get_size(frameset);
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                    "Frameset of %d frame(s) has no "
                                    "frame with tag: %s", nframes, tag);
        return NULL;
    }

    RawTagList = cpl_frameset_new();

    do {
        cpl_frame* aRawTag = cpl_frame_duplicate( aRawFound );
        cpl_frameset_insert( RawTagList, aRawTag );
    } while ((aRawFound = cpl_frameset_find_const( frameset, NULL )) != NULL);

    assert( cpl_frameset_get_size(RawTagList) > 0);

    return RawTagList;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Extract  frames with a specific group setting from a frameset
 @param    frames   The input frames
 @param    tag       A string identifying the  group of the frames to extract

 @return   new cpl_frameset containing the extracted frames.

 Extract a set of frames from an input frameset which share a common group.
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_utils_extract_frames_group( cpl_frameset* frameset, cpl_frame_group group ) {
    cpl_frameset*   RawFlatList     = NULL;
    cpl_frame*      aRawFlat        = NULL;
    int             rerr            = 0;

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
         sph_error_raise( SPH_UTILS_GENERAL,
                          __FILE__, __func__, __LINE__ ,
                          SPH_ERROR_WARNING,
                          "Attention cpl error has already been set!  The CPL message: %s\n",
                          cpl_error_get_message() );
         cpl_error_reset();
     }


    if ( !frameset ) {
        SPH_NO_SELF;
        return NULL;
    }

    RawFlatList = cpl_frameset_new();
    aRawFlat = cpl_frame_duplicate( cpl_frameset_get_first(frameset ) );
    if ( ! aRawFlat )
    {
        if ( RawFlatList ) {
            cpl_frameset_delete( RawFlatList );
        }
        cpl_error_reset();
        return NULL;
    }

    rerr = cpl_error_get_code();
    while ( aRawFlat )
    {
        if ( cpl_frame_get_group(aRawFlat) == group ) {
            cpl_frameset_insert( RawFlatList, aRawFlat );
        }
        else {
            cpl_frame_delete( aRawFlat ); aRawFlat = NULL;
        }
        aRawFlat = cpl_frame_duplicate( cpl_frameset_get_next( frameset) );
    }
    rerr = cpl_error_get_code();
    if ( rerr == CPL_ERROR_NULL_INPUT ) {
        cpl_error_reset();
    }
    return RawFlatList;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Extract  frame with a specific keyword setting from a frameset
 @param    frames   The input frames
 @param    tag       A string identifying the  tag of the frame to extract

 @return   new cpl_frame containing of the extracted frame.

 Extract a frame from an input frameset which share a common tag.
 */
/*----------------------------------------------------------------------------*/
cpl_frame*
sph_utils_extract_frame( const cpl_frameset* frameset, const char* tag ) {
    const cpl_frame* find_frame;

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
         sph_error_raise( SPH_UTILS_GENERAL,
                          __FILE__, __func__, __LINE__ ,
                          SPH_ERROR_WARNING,
                          "Attention cpl error has already been set!  The CPL message: %s\n",
                          cpl_error_get_message() );
     }

    if ( !frameset ) {
        SPH_NO_SELF;
        return NULL;
    }
    if ( ! tag ) {
        SPH_NULL_ERROR;
        return NULL;
    }

    find_frame = cpl_frameset_find_const(frameset, tag);

    if ( ! find_frame )
    {
        /*sph_error_raise( SPH_UTILS_FRAMES_MISSING,
                         __FILE__, __func__, __LINE__ ,
                         SPH_ERROR_WARNING,
                         "No %s frames found!\n"
                         "Check that they have the %s tag.\n",
                         tag, tag );
                         */
        return NULL;
    }

    return cpl_frame_duplicate(find_frame);
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a standard calibration product propertylist

 @return   new cpl_propertylist or NULL on error.

 Create a new standard calibration product propertylist, with IFS specific
 properties.

 */
/*----------------------------------------------------------------------------*/
cpl_propertylist*
sph_utils_create_calib_propertylist( void ) {
    cpl_propertylist*       proplist            = NULL;
    int                     timeref             = 0;
    int                     status              = 0;
    char                    timestamp[256];

    proplist = cpl_propertylist_new();

    fits_get_system_time( timestamp, &timeref, &status );

    cpl_propertylist_append_string( proplist, SPH_COMMON_KEYWORD_PRO_REDLEVEL, SPH_COMMON_KEYWORD_VALUE_PRO_REDLEVEL_FULL );
    cpl_propertylist_append_string( proplist, SPH_COMMON_KEYWORD_PRO_REC1_START, timestamp );
    cpl_propertylist_append_string( proplist, SPH_COMMON_KEYWORD_PRO_REC1_STOP, "UNDEFINED" );

    return proplist;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a standard calibration product frame

 @param     outframename     The name of the new frame
 @param     frametype        The type of the new frame
 @param     framelevel       The reduction level of the new frame
 @param     inframes         The frameset of the input frames used to create
                             the new frame
 @param     recid            The name of the recipe creating this frame
 @param     pipelineid       The name/id of the pipeline
 @param     dicid            The ID of the sph_dictionary used

 @return   new cpl_frame or NULL on error.

 Create a new standard calibration product propertylist, with IFS specific
 properties.

 */
/*----------------------------------------------------------------------------*/
cpl_frame*
sph_utils_create_calib_frame( const char* outframename, cpl_frame_type frametype,
                              cpl_frame_level framelevel, cpl_frameset* inframes,
                              const cpl_parameterlist* parlist, const char* prod_tag,
                              const char* recid, const char* pipelineid, cpl_frame* inherit_frame )
{
    cpl_frame*              result      = NULL;
    cpl_propertylist*       proplist    = NULL;
    cpl_frameset*           fset        = NULL;
    int                     ii          = 0;
    cpl_error_code          rerr        = CPL_ERROR_NONE;

    result = cpl_frame_new();
    rerr |= cpl_frame_set_filename( result, outframename );
    rerr |= cpl_frame_set_group( result, CPL_FRAME_GROUP_PRODUCT );
    rerr |= cpl_frame_set_level(result, framelevel );
    rerr |= cpl_frame_set_type( result, frametype );
    rerr |= cpl_frame_set_tag( result, prod_tag );

    proplist = sph_utils_create_calib_propertylist( );

    if ( inherit_frame ){
        rerr |= cpl_dfs_setup_product_header( proplist, result,
                                      inframes, parlist,
                                      recid, pipelineid, SPH_COMMON_KEYWORD_VALUE_DIC_ID, inherit_frame );
    } else {
        rerr |= cpl_dfs_setup_product_header( proplist, result,
                                  inframes, parlist,
                                  recid, pipelineid, SPH_COMMON_KEYWORD_VALUE_DIC_ID, cpl_frameset_get_first( inframes ) );
    }
    if ( rerr == CPL_ERROR_DATA_NOT_FOUND ) {
        cpl_error_reset();                          /* It seems as if cpl_dfs_setup_
                                                        product header always sets
                                                        the data not found error
                                                        even if the inframes list
                                                        is not null.
                                                        Need to check this with Lars Lundin
                                                    */



        rerr = CPL_ERROR_NONE;
    }
    rerr |= sph_fits_update_checksum( outframename );
    rerr |= cpl_propertylist_append_string( proplist, SPH_COMMON_KEYWORD_PRO_CATG, prod_tag);

    /* Update the headers if need be */
    sph_utils_update_header(proplist);

    for (ii = 0; ii < cpl_propertylist_get_size(proplist); ++ii) {
        rerr |= sph_fits_update_property( outframename, cpl_propertylist_get( proplist, ii), 0);
    }

    fset = cpl_frameset_new( );
    rerr |= cpl_frameset_insert( fset, cpl_frame_duplicate( result ) );
    rerr |= cpl_dfs_update_product_header( fset );
    cpl_frameset_delete( fset );
    if ( rerr != CPL_ERROR_NONE ) {
        SPH_RAISE_CPL
    }
    cpl_frameset_insert(inframes,result);
    cpl_propertylist_delete( proplist );
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Create a standard list of valid DITs as a FITS table

 @param     outframename     The name of the new frame

 @return   new cpl_frame or NULL on error.

 Create a new standard calibration valid dit list.

 */
/*----------------------------------------------------------------------------*/
cpl_frame*
sph_utils_create_default_dit_list( const char* fname ) {
    cpl_frame*      result      = NULL;
    int             ii          = 0;
    int             jj          = 0;
    int             numnreads   = 10;
    int             numexps     = 500;
    int             numrows     = numnreads * numexps;
    float           exptime     = 1.3;
    int             nrow        = 0;
    cpl_table*        dit_table    = NULL;

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
         sph_error_raise( SPH_UTILS_GENERAL,
                          __FILE__, __func__, __LINE__ ,
                          SPH_ERROR_WARNING,
                          "Attention cpl error has already been set!  The CPL message: %s\n",
                          cpl_error_get_message() );
         cpl_error_reset();
     }


    dit_table   = cpl_table_new(numrows);


    if (cpl_table_new_column(dit_table, "NREADOUTS", CPL_TYPE_INT)
            != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: Error adding column: %s", cpl_func,
                __LINE__, cpl_error_get_where());
        return NULL;

    }
    if (cpl_table_new_column(dit_table, "DIT", CPL_TYPE_DOUBLE)
            != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: Error adding column: %s", cpl_func,
                __LINE__, cpl_error_get_where());
        return NULL;
    }
    for ( ii = 0; ii < numnreads; ii++ ) {
        for ( jj = 0; jj < numexps; jj++) {
            cpl_table_set(dit_table, "NREADOUTS", nrow, ii+1);
            cpl_table_set(dit_table, "DIT", nrow, (jj+1) * exptime);
            nrow++;
        }
    }
    cpl_table_save(dit_table, NULL, NULL, fname, CPL_IO_DEFAULT);

    result = cpl_frame_new();
    cpl_frame_set_filename( result, fname );
    cpl_frame_set_group( result, CPL_FRAME_GROUP_CALIB );
    cpl_frame_set_level( result, CPL_FRAME_LEVEL_TEMPORARY );
    cpl_frame_set_type( result, CPL_FRAME_TYPE_TABLE );
    cpl_table_delete( dit_table );
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 *    @brief Creates a new frameset with master_frames from the input frameset.
 *
 *    @param    inframes        the input frames to duplicate
 *
 *  @return    the new cpl_frameset or NULL in case of error
 *  @note This function may set a CPL error and return a non-NULL pointer
 *
 *  This function creates a copy of all input frames, where each frame in the
 *  new framelist points to a file which contains either a sph_master_frame
 *  or a sph_cube. The input frameset can contain frames of these types:
 *  <ul>
 *  <li>a raw frame with a single image (one extension, one plane)</li>
 *  <li>a sph_master_frame with 4 extensions and one plane each</li>
 *  <li>a sph_cube with 4 extensions and N planes each.</li>
 *  </ul>
 *  The function automatically transforms any raw frame inputs into a
 *  sph_master_frame by filling the image extension with the image and
 *  leaving the other extensions with filled with 0.
 *  The sph_master_frame and sph_cube inputs are copied to a new file unchanged.
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_utils_create_master_frameset( const cpl_frameset* inframes ) {
    const cpl_frame*     curframe = cpl_frameset_get_first_const( inframes );
    cpl_frameset*        result;

    if (inframes == NULL) {
        (void)cpl_error_set_where(cpl_func);
        return NULL;
    }

    if (sph_keyword_manager_new() == NULL) return NULL;

    result = cpl_frameset_new();

    while ( curframe ) {
        const char* czFrame    = cpl_frame_get_filename( curframe );
        const char* czType     =
            sph_keyword_manager_get_string( SPH_COMMON_KEYWORD_SPH_TYPE);
        cpl_frame*  dupframe   = cpl_frame_duplicate( curframe );
        char*       czFrameNew = cpl_sprintf("%s.tmp", czFrame);

        sph_keyword_manager_load_keywords( czFrame, 0 );

        if (!czType ||
            (strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_CUBE ) &&
             strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_MASTER_FRAME)))
            {
                cpl_propertylist* pl =
                    sph_keyword_manager_load_properties( czFrame, 0 );
                sph_master_frame* mframe = sph_master_frame_new_empty();

                if ( pl != NULL && mframe != NULL) {
                    mframe->image = cpl_image_load(czFrame, CPL_TYPE_DOUBLE, 0, 0 );
                    if ( mframe->image ) {
                        const cpl_size nx = cpl_image_get_size_x(mframe->image);
                        const cpl_size ny = cpl_image_get_size_y(mframe->image);

                        mframe->badpixelmap = cpl_image_new(nx, ny, CPL_TYPE_INT);
                        mframe->ncombmap    = cpl_image_new(nx, ny, CPL_TYPE_INT);
                        mframe->rmsmap      = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

                        sph_master_frame_save( mframe, czFrameNew, pl );
                    }
                }
                cpl_propertylist_delete(pl); pl = NULL;
                sph_master_frame_delete( mframe ); mframe = NULL;
            }

        cpl_frame_set_filename( dupframe, czFrameNew );
        cpl_frameset_insert( result, dupframe );
        cpl_free(czFrameNew);
        curframe = cpl_frameset_get_next_const( inframes );
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Creates temporary copies of input frames for further processing.
 *
 * @param    inframes        the input frames to copy
 * @param    framesout        a frameset to which the temporary frames will be added
 *
 * return        error code of the operation
 *
 * This creates a temporary copy of the file pointer to by each frame and add
 * it to the output frameslist framesout. This function cann handle all RAW
 * input frames, and frames of type SPH_MASTER_FRAME and SPH_DOUBLE_IMAGE but
 * not of type SPH_CUBE. A frame of that type in the input causes a
 * CPL_ERROR_INCOMPATIBLE_INPUT to be returned.
 * @deprecated
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_utils_copy_frames( const cpl_frameset* inframes, cpl_frameset* framesout )
{
    sph_error_code             rerr         = CPL_ERROR_NONE;
    const cpl_frame*                curframe    = NULL;
    cpl_frame*                frameA        = NULL;
    cpl_propertylist*        pl            = NULL;
    sph_master_frame*        mframe        = NULL;
    sph_double_image*        dimage        = NULL;
    sph_keyword_manager*    keym        = NULL;
    const char*                    czFrame        = NULL;
    char                    czFrameA[2056];
    const char*                    czType        = NULL;
    short                    sphformat    = 0;
    cpl_image*                image        = NULL;

    keym = sph_keyword_manager_new();
    if ( !keym ) {
        return CPL_ERROR_NULL_INPUT;
    }

    curframe = cpl_frameset_get_first_const( inframes );

    while ( curframe ) {
        czFrame = cpl_frame_get_filename( curframe );
        sph_keyword_manager_load_keywords( czFrame, 0 );
        czType = sph_keyword_manager_get_string( SPH_COMMON_KEYWORD_SPH_TYPE );
        if ( !czType ) {
            cpl_error_reset();
        }
        sphformat = 0;
        strncpy( czFrameA, czFrame, 2056 );
        strcat( czFrameA, ".tmp");
        if ( czType ) {
            if ( !strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_CUBE ) )
            {
                return CPL_ERROR_INCOMPATIBLE_INPUT;
            }
            if ( !strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_MASTER_FRAME ) )
            {
                pl = sph_keyword_manager_load_properties( czFrame, 0 );
                mframe = sph_master_frame_load_(curframe, 0);
                sph_master_frame_save( mframe, czFrameA, pl );
                sph_master_frame_delete( mframe );
                sphformat = 1;
            }
            if ( !strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE ) )
            {
                pl = sph_keyword_manager_load_properties( czFrame, 0 );
                dimage = sph_double_image_load( czFrame, 0 );
                sph_double_image_save( dimage, czFrameA, pl ,CPL_IO_CREATE);
                sph_double_image_delete( dimage );
                sphformat = 1;
            }
        }
        if ( !sphformat ) {
            image = cpl_image_load(czFrame,
                    CPL_TYPE_DOUBLE, 0, 0 );
            mframe = sph_master_frame_new_from_cpl_image( image );
            cpl_image_delete( image ); image = NULL;
            if ( mframe->properties ) {
                cpl_propertylist_delete( mframe->properties );
                mframe->properties = NULL;
            }
            mframe->properties = sph_keyword_manager_load_properties( czFrame, 0 );
            sph_master_frame_save( mframe, czFrameA, pl );
            sph_master_frame_delete( mframe );mframe = NULL;
        }
        frameA = cpl_frame_duplicate( curframe );
        cpl_frame_set_filename( frameA, czFrameA );
        cpl_frameset_insert( framesout, frameA );
        curframe = cpl_frameset_get_next_const( inframes );
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Check frame against a DIT list and return the DIT ID for the frame

 @param     inframes        The frame to check for DIT ID
 @param     dit_list        The valid dit list to use for the check

 @return   dit id, or -1 on error.

 Check the validitiy of the EXPTIME and DET READOUT NO keywords and return the
 associated DIT ID.
 In case the frame has an invalid DIT value an SPH_INVALID_DIT
 error code is returned. The routine assumes the EXPTIME and DET READOUT NO is
 stored for the entire frame in the primary HDU.

 */
/*----------------------------------------------------------------------------*/
int
sph_utils_get_dit_id_from_cpl_table( const cpl_frame* inframe, const cpl_table* dit_table )
{
    cpl_propertylist*   proplist        = NULL;
    const char*               filename        = NULL;
    double              exptime         = 0.0;
    int                 readno          = 0;
    int                 ii              = 0;
    int                 found           = 0;
    int                 temp_nreads     = 0;
    int                 badpix          = 0;
    int                 dit_id          = -1;
    double              temp_exptime    = 0;

    filename        = cpl_frame_get_filename( inframe );
    proplist        = sph_keyword_manager_load_properties( filename, 0 );

    exptime = cpl_propertylist_get_double( proplist, SPH_COMMON_KEYWORD_EXPTIME );
    readno  = cpl_propertylist_get_int( proplist, SPH_COMMON_KEYWORD_NREADS );
    sph_error_raise( SPH_ERROR_INFO, __FILE__, __func__, __LINE__, SPH_ERROR_INFO, "Assing FITS dit from extime = %g and ROM = %d", exptime, readno );
    for ( ii = 0; ii < cpl_table_get_nrow(dit_table); ii++ )
    {
        temp_exptime    = cpl_table_get_double( dit_table,
                                                "DIT", ii, &badpix );
        temp_nreads     = cpl_table_get_int( dit_table,
                                                "NREADOUTS", ii, &badpix );
        if ( temp_nreads == readno && fabs( exptime - temp_exptime ) < DIT_EXPTIME_ACCURACY ) {
            found = found + 1;
            dit_id = ii + 1;
            sph_error_raise( SPH_ERROR_INFO, __FILE__, __func__, __LINE__, SPH_ERROR_INFO, "Found dit id %d", dit_id );
        }
    }
    cpl_propertylist_delete( proplist );
    if ( dit_id < 1 ) {
        sph_error_raise( SPH_ERROR_GENERAL,
                         __FILE__, __func__,
                         __LINE__,
                         SPH_ERROR_WARNING,
                         "Could not find a DIT id for the combination of "
                         "DIT = %g and NREADOUTS = %d "
                         "in the provided (or default) valid DIT table.",
                         exptime, readno );
    }
    return dit_id;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Check frame against a DIT list and return the DIT ID for the frame

 @param     inframes        The frame to check for DIT ID
 @param     dit_list        The frame of the valid dit list to use for the check

 @return   dit id, or -1 on error.

 Check the validitiy of the EXPTIME and DET READOUT NO keywords and return the
 associated DIT ID.
 In case the frame has an invalid DIT value an SPH_INVALID_DIT
 error code is returned. The routine assumes the EXPTIME and DET READOUT NO is
 stored for the entire frame in the primary HDU.

 */
/*----------------------------------------------------------------------------*/
int
sph_utils_get_dit_id( const cpl_frame* inframe, const cpl_frame* dit_list )
{
    const char*         dit_filename    = NULL;
    cpl_table*          dit_table       = NULL;
    int                    dit_id            = -1;

    dit_filename    = cpl_frame_get_filename( dit_list );

    dit_table = cpl_table_load( dit_filename, 1, 0 );
    if ( dit_table ) {
        dit_id = sph_utils_get_dit_id_from_cpl_table( inframe, dit_table );

        cpl_table_delete( dit_table );
    }
    return dit_id;
}


double
sph_utils_get_exposure_time( const cpl_frame* inframe ) {
    cpl_propertylist* pl = NULL;
    double exptime = 1.0;

    cpl_ensure(inframe, CPL_ERROR_ILLEGAL_INPUT, 1.0);

    pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(inframe),0);
    if (  cpl_propertylist_has(pl,SPH_COMMON_KEYWORD_EXPTIME) )
        exptime = cpl_propertylist_get_double(pl,SPH_COMMON_KEYWORD_EXPTIME);
    cpl_propertylist_delete(pl);pl = NULL;
    return exptime;
}

int
sph_utils_check_exposure_times_are_same( const cpl_frameset* inframes ) {
    const cpl_frame* frame = NULL;
    double exptime = 1.0;

    cpl_ensure(inframes, CPL_ERROR_ILLEGAL_INPUT, 0);

    frame = cpl_frameset_get_first_const( inframes );
    exptime = sph_utils_get_exposure_time(frame);
    while ( frame ) {
        if ( fabs(sph_utils_get_exposure_time(frame) - exptime ) > 0.000001 ) {
            return 0;
        }
        frame = cpl_frameset_get_next_const(inframes);
    }
    return 1;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Return the DITHER ID for the frame

 @param     inframe         The frame to return the dither id for
 @param     res             The dither resolution allowed

 @return   dither id, or -1 on error.

 This function returns a dither "ID" which is constructed from the x and y
 position of the detector. The ID allows for a given dither resolution --
 positional information below this resolution is lost.

 */
/*----------------------------------------------------------------------------*/
int
sph_utils_get_dither_id( const cpl_frame* inframe, float res,
                         const char* keyx,
                         const char* keyy )
{
    cpl_propertylist*   proplist        = NULL;
    const char*         filename        = NULL;
    long int            dit_id          = -1;
    double              dDitherx        = 0.0;
    double              dDithery        = 0.0;
    double              dID             = 0.0;

    filename        = cpl_frame_get_filename( inframe );
    proplist        = sph_keyword_manager_load_properties( filename, 0);

    dDitherx = cpl_propertylist_get_double( proplist, keyx );
    dDithery = cpl_propertylist_get_double( proplist, keyy );

    dID = ( dDitherx / res ) * ( dDithery / res );
    if ( dID > __INT_MAX__ ) {
        sph_error_raise( SPH_UTILS_TOO_FINE_DITHER,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "The dither resolution given "
                         "was too fine ! Decrease "
                         "it by at least a factor "
                         "of %lf.",
                         dID / (double)__INT_MAX__);
        return -1;
    }
    dit_id = (int)dID;
    cpl_propertylist_delete( proplist ); proplist = NULL;
    return dit_id;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Attach a propertylist to a frame
 *
 * @param plist        the propertylist to attach
 * @param frame        the frame to attach propertylist to
 * @param extension the extension to write propertylist to
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_utils_attach_plist( cpl_propertylist* plist, cpl_frame* frame, int ext ) {
    int            pcount        = 0;
    sph_error_code    rerr    = CPL_ERROR_NONE;

    if ( !plist || !frame || ext < 0 ) return CPL_ERROR_NULL_INPUT;

    for (pcount = 0; pcount < cpl_propertylist_get_size(plist); ++pcount) {
        rerr |= sph_fits_update_property( cpl_frame_get_filename( frame ), cpl_propertylist_get(plist, pcount), ext );
    }
    return rerr;
}


sph_error_code
sph_utils_cpl_image_fill_double( cpl_image * image, double value ){
    const cpl_size nx = cpl_image_get_size_x(image);
    const cpl_size ny = cpl_image_get_size_y(image);
    sph_error_code rerr     = CPL_ERROR_NONE;
    int            xx        = 0;
    int               yy        = 0;

    if ( !image ){
        SPH_NO_SELF
        return sph_error_get_last_code();
    }
    if ( cpl_image_get_type( image ) != CPL_TYPE_DOUBLE ){
        SPH_ERR("wrong CPL_TYPE")
        return sph_error_get_last_code();
    }

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            cpl_image_set( image, xx + 1, yy + 1, value );
        }
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE ){
        SPH_RAISE_CPL;
        rerr = sph_error_get_last_code();
    }

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief     Remove all files from the given frames
 * @param    inframes    a pointer to the cpl_frameset
 * @return     sphere error code
 *
 * Deletes all files on the disk from the given frames
 *
 */
/*----------------------------------------------------------------------------*/

sph_error_code sph_utils_delete_files( cpl_frameset* inframes )
{
    cpl_frame*        curframe    = NULL;
    cpl_ensure_code( inframes, CPL_ERROR_NULL_INPUT );

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    curframe = cpl_frameset_get_first( inframes );
    while ( curframe ){
        unlink( cpl_frame_get_filename( curframe ));
        curframe = cpl_frameset_get_next( inframes );
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*
 * A simple converting function from uppercase to lowcase
 */
void sph_utils_lowercase(char string[]){
    int i = 0;

       while ( string[i] )
       {
          string[i] = tolower(string[i]);
          i++;
       }

       return;
}

/*
 * A simple converting function from lowcase to uppercase
 */
void sph_utils_uppercase(char string[]){
    int i = 0;

       while ( string[i] )
       {
          string[i] = toupper(string[i]);
          i++;
       }

       return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create property list of the WCS keywords
 * @param ra            RA (deg)
 * @param dec           DEC (deg)
 * @param xpix          X coordinate of reference pixel
 * @param ypix          Y coordinate of reference pixel
 * @param xscale        X pixel scale (mas)
 * @param yscale        Y pixel scale (mas)
 *
 *
 * @return pl, cpl_propertylist
 *
 * This puts the WCS keywords in the header. Reference RA and DEC are taken from the
 * propertylist (target coordinates). Projection is always RA---TAN/DEC--TAN.
  */
/*----------------------------------------------------------------------------*/
cpl_propertylist* sph_utils_astrometry_create_wcs_pl(double ra, double dec,
                                                     double xpix, double ypix,
                                                     double xscale, double yscale)
{
    cpl_propertylist* pl;
    char comment_string[256];
    char time_string[256];

    cpl_ensure( ra >= 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( ra <= 360.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( dec >= -90.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( dec <= 90.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( xpix > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( ypix > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( xscale > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    cpl_ensure( yscale > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL );

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        (void)cpl_error_set_where(cpl_func);
        return NULL;
    }

    pl = cpl_propertylist_new();

    if (SPH_USE_RADEC) {

        //ra
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL, ra);
        sph_radec_deg_to_iso8601string(ra, time_string, 1);
        sprintf(comment_string,"%s, RA at ref pixel", time_string);
        cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                     comment_string);

        //dec
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL, dec);
        sph_radec_deg_to_iso8601string(dec, time_string, 0);
        sprintf(comment_string,"%s, DEC at ref pixel", time_string);
        cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                     comment_string);

        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CRPIXX,xpix);
        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CRPIXY,ypix);

        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE_VAL);
        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE_VAL);

        /* unit degree */
        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CD11,
                                       -xscale/1000.0/3600.0);
        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CD22,
                                       yscale/1000.0/3600.0);

        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT_VAL);
        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT_VAL);
    } else {
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                       xpix);
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                       ypix);

        cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                     "Reference pixel (X)");
        cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                     "Reference pixel (Y)");

        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CRPIXX, xpix);
        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CRPIXY, ypix);

        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                       "PIXEL");
        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                       "PIXEL");

        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CD11, 1.0);
        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CD22, 1.0);

        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                       "pixel");
        cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                       "pixel");
    }

    cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CD12, 0.0);
    cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_WCS_CD21, 0.0);

    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXX,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXX_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXY,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXY_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXTYPE_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXTYPE_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CD11,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CD12,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CD21,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CD22,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXUNIT_COMMENT);
    cpl_propertylist_set_comment(pl, SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXUNIT_COMMENT);


    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        (void)cpl_error_set_where(cpl_func);
        cpl_propertylist_delete(pl);
        pl = NULL;
    }

    return pl;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Remove any 3rd dimension WCS keywords from the propertylist
 * @param self The propertylist to modify in place
 * @return CPL_ERROR_NONE on success, otherwise the relevant CPL error
  */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_utils_remove_wcs_3d(cpl_propertylist* self)
{
    cpl_error_code code = CPL_ERROR_NONE;

    if (self == NULL) {
        code = cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const int ndel3 = cpl_propertylist_erase_regexp(self, "^(CRPIX3|CRVAL3|"
                                                        "CTYPE3|CD._3|CD3_.)$",
                                                        0);
        if (ndel3 > 0) {
            cpl_msg_info(cpl_func, "Removing %d WCS NAXIS3-cards from "
                         "propertylist", ndel3);
        }
    }

    return code;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Reset any 1st-2nd-dimension WCS keyword (to 1) in the propertylist
 * @param self The propertylist to modify in place
 * @return CPL_ERROR_NONE on success, otherwise the relevant CPL error
  */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_utils_reset_wcs_12d(cpl_propertylist* self)
{
    cpl_error_code code = CPL_ERROR_NONE;

    if (self == NULL) {
        code = cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (cpl_propertylist_has(self, SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE)) {
        cpl_error_code r = CPL_ERROR_NONE;
        r |= cpl_propertylist_update_string(self,
                                               SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                            "PIXEL");
        r |= cpl_propertylist_update_string(self,
                                            SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                            "PIXEL");

        r |= cpl_propertylist_update_string(self,
                                            SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                            "pixel");
        r |= cpl_propertylist_update_string(self,
                                            SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                            "pixel");

        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CRPIXX, 1);
        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CRPIXY, 1);
        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                            1);
        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                            1);

        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CD11, 1.0);
        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CD12, 0.0);
        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CD21, 0.0);
        r |= cpl_propertylist_update_double(self,
                                            SPH_COMMON_KEYWORD_WCS_CD22, 1.0);
        if (r) code = cpl_error_set_where(cpl_func);
    }

    return code;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Determine if a frame is from a single exposure or not
 @return   TRUE if this frame is from a single exposure
           FALSE if not
 @note     This function checks the header defined as SPH_ZPL_KEYWORD_SINGLE 

 */
/*----------------------------------------------------------------------------*/
int sph_utils_is_single_frame(cpl_propertylist* pl)
{
  int result = FALSE;
  if (pl &&
      cpl_propertylist_has(pl, SPH_ZPL_KEYWORD_SINGLE) &&
      cpl_propertylist_get_bool(pl, SPH_ZPL_KEYWORD_SINGLE)) {
    result = TRUE;
  }
  return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Determine if a frame is from a single exposure or not
 @return   TRUE if this frame is from a single exposure
           FALSE if not
 @note     This function checks the header defined as SPH_ZPL_KEYWORD_SINGLE 

 */
/*----------------------------------------------------------------------------*/
int sph_utils_set_single_frame(cpl_propertylist* pl)
{
  int result = FALSE;
  if (pl) {
    if (cpl_propertylist_has(pl, SPH_ZPL_KEYWORD_SINGLE)) {
      result = cpl_propertylist_set_bool(pl, SPH_ZPL_KEYWORD_SINGLE, TRUE);
    } else {
      result = cpl_propertylist_append_bool(pl, SPH_ZPL_KEYWORD_SINGLE, TRUE);
    }
  }
  return result;
}
