/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_zpl_detector_time_model.h"
#include "sph_error.h"
#include <math.h>

const double SPH_ZPL_DETECTOR_TIME_MODEL_START_TIME_DELAY_STD_IMAGING = 0.826;
const double SPH_ZPL_DETECTOR_TIME_MODEL_DIT_GAP_STD_IMAGING = 0.06581;
const double SPH_ZPL_PARALACTIC_ANGLE_OFFSET_STD_IMAGING = 135.0; //Imaging mode is used in the pupil stabilized mode

const double SPH_ZPL_DETECTOR_TIME_MODEL_START_TIME_DELAY_FAST_POL = 0.510;
const double SPH_ZPL_DETECTOR_TIME_MODEL_DIT_GAP_FAST_POL = 0.066;
const double SPH_ZPL_PARALACTIC_ANGLE_OFFSET_FAST_POL = 0.0;

const double SPH_ZPL_DETECTOR_TIME_MODEL_START_TIME_DELAY_SLOW_POL = 0.820;
const double SPH_ZPL_DETECTOR_TIME_MODEL_DIT_GAP_SLOW_POL = 0.074;
const double SPH_ZPL_PARALACTIC_ANGLE_OFFSET_SLOW_POL = 0.0;

const int    SPH_ZPL_DETECTOR_MODE_ID_FASTPOL = 1;
const int    SPH_ZPL_DETECTOR_MODE_ID_FASTWINPOL = 2;
const int    SPH_ZPL_DETECTOR_MODE_ID_SLOWPOL = 3;
const int    SPH_ZPL_DETECTOR_MODE_ID_STDIMG  = 4;
const int    SPH_ZPL_DETECTOR_MODE_ID_SNAPSHOTIMG = 5;


sph_zpl_detector_time_model* sph_zpl_detector_time_model_new_days(const int detmode_id){
    return sph_zpl_detector_time_model_new( detmode_id, CPL_TRUE);
}

sph_zpl_detector_time_model* sph_zpl_detector_time_model_new(const int detmode_id, cpl_boolean set_days) {
    sph_zpl_detector_time_model*    zpltmodel   = NULL;

    cpl_ensure(detmode_id > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    if ( detmode_id == SPH_ZPL_DETECTOR_MODE_ID_FASTPOL ) {
        SPH_INFO_MSG("Fast Polarization detector read out mode found.\n");
        zpltmodel = sph_zpl_detector_time_model_new_fastpol();
    } else if ( detmode_id ==  SPH_ZPL_DETECTOR_MODE_ID_SLOWPOL ) {
        SPH_INFO_MSG("Slow Polarization detector read out mode found.\n");
        zpltmodel = sph_zpl_detector_time_model_new_slowpol();
    } else if ( detmode_id == SPH_ZPL_DETECTOR_MODE_ID_STDIMG ) {
        SPH_INFO_MSG("Standard Imaging detector read out mode found.\n")
        zpltmodel = sph_zpl_detector_time_model_new_stdimg();
    } else {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL, "Illegal detmode_id = %d (current supported 1,3,4)", detmode_id);
        return NULL;
    }

    if ( set_days ) sph_zpl_detector_time_model_set_days( zpltmodel );

    return zpltmodel;
}

sph_zpl_detector_time_model* sph_zpl_detector_time_model_new_fastpol(void) {
    sph_zpl_detector_time_model*    zpltmodel   = NULL;
    zpltmodel = cpl_calloc( 1, sizeof( sph_zpl_detector_time_model) );
    if ( zpltmodel ){
        zpltmodel->start_time_delay = SPH_ZPL_DETECTOR_TIME_MODEL_START_TIME_DELAY_FAST_POL;
        zpltmodel->dit_gap = SPH_ZPL_DETECTOR_TIME_MODEL_DIT_GAP_FAST_POL;
        zpltmodel->detmod_id= SPH_ZPL_DETECTOR_MODE_ID_FASTPOL;
        zpltmodel->days = CPL_FALSE;
    }
    return zpltmodel;
}

sph_zpl_detector_time_model* sph_zpl_detector_time_model_new_slowpol(void) {
    sph_zpl_detector_time_model*    zpltmodel   = NULL;
    zpltmodel = cpl_calloc( 1, sizeof( sph_zpl_detector_time_model) );
    if ( zpltmodel ){
        zpltmodel->start_time_delay = SPH_ZPL_DETECTOR_TIME_MODEL_START_TIME_DELAY_SLOW_POL;
        zpltmodel->dit_gap = SPH_ZPL_DETECTOR_TIME_MODEL_DIT_GAP_SLOW_POL;
        zpltmodel->detmod_id= SPH_ZPL_DETECTOR_MODE_ID_SLOWPOL;
        zpltmodel->days = CPL_FALSE;
    }
    return zpltmodel;
}

sph_zpl_detector_time_model* sph_zpl_detector_time_model_new_stdimg(void) {
    sph_zpl_detector_time_model*    zpltmodel   = NULL;
    zpltmodel = cpl_calloc( 1, sizeof( sph_zpl_detector_time_model) );
    if ( zpltmodel ){
        zpltmodel->start_time_delay = SPH_ZPL_DETECTOR_TIME_MODEL_START_TIME_DELAY_STD_IMAGING;
        zpltmodel->dit_gap = SPH_ZPL_DETECTOR_TIME_MODEL_DIT_GAP_STD_IMAGING;
        zpltmodel->detmod_id= SPH_ZPL_DETECTOR_MODE_ID_STDIMG;
        zpltmodel->days = CPL_FALSE;
    }
    return zpltmodel;
}

sph_error_code sph_zpl_detector_time_model_set_days( sph_zpl_detector_time_model* self){
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);

    self->start_time_delay = self->start_time_delay/60./60./24.;
    self->dit_gap = self->dit_gap/60./60./24.;
    self->days = CPL_TRUE;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    return CPL_ERROR_NONE;
}

sph_error_code sph_zpl_detector_time_model_delete( sph_zpl_detector_time_model* self ){
    if ( self ) {
        cpl_free( self );

        return CPL_ERROR_NONE;
    }
    else {
        return CPL_ERROR_NULL_INPUT;
    }
}
