/*
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* for mkstemp */
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE 500
#endif

#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_filemanager.h"
#include <stdlib.h>
#include <math.h>

sph_test_pupilimage_creator*
sph_test_pupilimage_creator_new(void) {
    sph_test_pupilimage_creator* result = NULL;

    result = cpl_calloc(1, sizeof(sph_test_pupilimage_creator));
    if (!result)
        return NULL;
    result->coronagraph_in = 0;
    result->nx = 256;
    result->ny = 256;
    result->frameset = cpl_frameset_new();
    return result;
}

static void sph_test_pupilimage_creator_apply_coronagraph(
        sph_test_pupilimage_creator* self, cpl_image* im) {
    if (!self || !im)
        return;
    if (!self->coronagraph_in)
        return;
    return;
}

void sph_test_pupilimage_creator_create_const_image(
        sph_test_pupilimage_creator* self, double value, int nframes) {

    for (int ii = 0; ii < nframes; ++ii) {
        cpl_frame* frame = NULL;
        cpl_image* im = NULL;
        int fp = 0;
        char template[256];

        sprintf(template, "pupilimage.fits.XXXXXX");
        frame = cpl_frame_new();
        cpl_frame_set_tag(frame, SPH_TEST_PUPILIMAGE_TAG);
        fp = mkstemp(template);
        close(fp);
        unlink(template);
        cpl_frame_set_filename(frame, template);
        im = sph_test_image_tools_create_flat_image_double(self->nx, self->ny,
                value);
        if (self->coronagraph_in)
            sph_test_pupilimage_creator_apply_coronagraph(self, im);
        cpl_image_save(im, template, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
        cpl_image_delete(im);
        im = NULL;
        cpl_frameset_insert(self->frameset, frame);
    }
    return;
}

sph_error_code sph_test_pupilimage_creator_create_frames(
        sph_test_pupilimage_creator* self, int nframes) {

    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->base_image, CPL_ERROR_NULL_INPUT);

    for (int ii = 0; ii < nframes; ++ii) {
        cpl_frame* frame = NULL;
        cpl_image* im = NULL;
        char template[256];
        int fp = 0;

        sprintf(template, "pupilimage.fits.XXXXXX");
        frame = cpl_frame_new();
        cpl_frame_set_tag(frame, SPH_TEST_PUPILIMAGE_TAG);
        fp = mkstemp(template);
        close(fp);
        unlink(template);
        cpl_frame_set_filename(frame, template);
        im = cpl_image_duplicate(self->base_image);
        if (self->coronagraph_in)
            sph_test_pupilimage_creator_apply_coronagraph(self, im);
        cpl_image_save(im, template, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
        cpl_image_delete(im);
        im = NULL;
        cpl_frameset_insert(self->frameset, frame);
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

cpl_image*
sph_test_pupilimage_creator_create_binary_star_image(
        sph_test_pupilimage_creator* self, double dist, double angle,
        double counts, double contrast, double psf_size) {
    cpl_image* im = NULL;
    cpl_image* starim = NULL;
    cpl_image* compim = NULL;
    double dx;
    double dy;
    im = sph_test_image_tools_create_flat_image_double(self->nx, self->ny, 0.0);
    compim = cpl_image_duplicate(im);
    starim = cpl_image_duplicate(im);
    cpl_image_fill_gaussian(starim, cpl_image_get_size_x(im) / 2.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5,
            counts * CPL_MATH_2PI * psf_size * psf_size, psf_size, psf_size);
    dx = dist * cos(angle * CPL_MATH_RAD_DEG);
    dy = dist * sin(angle * CPL_MATH_RAD_DEG);
    cpl_image_fill_gaussian(compim, cpl_image_get_size_x(im) / 2.0 + dx + 0.5,
            cpl_image_get_size_y(im) / 2.0 + dy + 0.5,
            counts * CPL_MATH_2PI * psf_size * psf_size / contrast, psf_size,
            psf_size);
    cpl_image_add(im, starim);
    cpl_image_add(im, compim);
    cpl_image_delete(compim);
    compim = NULL;
    cpl_image_delete(starim);
    starim = NULL;

    return im;
}

void sph_test_pupilimage_creator_create_binary_star_frames(
        sph_test_pupilimage_creator* self, double dist, double angle0,
        double dangle, double counts, double contrast, double psf_size,
        int nframes) {
    double angle = angle0;
    cpl_propertylist* pl = NULL;

    for (int ii = 0; ii < nframes; ++ii) {
        cpl_frame* frame = cpl_frame_new();
        char*   template = sph_filemanager_get_numbered_filename
            ("pupilimage.fits", ii);
        cpl_image* im = sph_test_pupilimage_creator_create_binary_star_image
            (self, dist, angle, counts, contrast, psf_size);

        if (pl == NULL) {
            pl = cpl_propertylist_new();
            cpl_propertylist_append_double(pl,
                                           SPH_TEST_PUPILIMAGE_KEYWORD_POSANG0,
                                           angle0);
            cpl_propertylist_append_double(pl, SPH_TEST_PUPILIMAGE_KEYWORD_POSANG,
                                           angle);
        } else {
            cpl_propertylist_update_double(pl, SPH_TEST_PUPILIMAGE_KEYWORD_POSANG,
                                           angle);
        }

        cpl_image_save(im, template, CPL_TYPE_DOUBLE, pl, CPL_IO_CREATE);

        cpl_frame_set_tag(frame, SPH_TEST_PUPILIMAGE_TAG);
        cpl_frame_set_filename(frame, template);
        cpl_frameset_insert(self->frameset, frame);
        angle += dangle;

        cpl_free(template);
        cpl_image_delete(im);
    }
    cpl_propertylist_delete(pl);
    return;
}
void sph_test_pupilimage_creator_add_speckles(sph_test_pupilimage_creator* self,
        int nspeckles, double value, gsl_rng* pRNG) {
    int nframes;
    double* xpos = NULL;
    double* ypos = NULL;
    double* svalue = NULL;
    double* size = NULL;
    double size0 = 0.0;
    double xpos0 = 0.0;
    double ypos0 = 0.0;


    if (!self)
        return;
    if (!self->frameset)
        return;

    nframes = cpl_frameset_get_size(self->frameset);

    xpos = cpl_calloc(nspeckles, sizeof(double));
    ypos = cpl_calloc(nspeckles, sizeof(double));
    svalue = cpl_calloc(nspeckles, sizeof(double));
    size = cpl_calloc(nspeckles, sizeof(double));

    for (int jj = 0; jj < nspeckles; ++jj) {
        svalue[jj] = value + gsl_ran_gaussian(pRNG, value * 0.1);
        size[jj] = fabs(gsl_ran_gaussian(pRNG, 0.0001)) + 0.001;
        gsl_ran_bivariate_gaussian(pRNG, 0.1, 0.1, 0.0, &xpos[jj], &ypos[jj]);
    }
    for (int ii = 0; ii < nframes; ++ii) {
        const cpl_frame* frame = cpl_frameset_get_position(self->frameset, ii);
        const char* filename = cpl_frame_get_filename(frame);
        cpl_image* im = cpl_image_load(filename, CPL_TYPE_DOUBLE, 0, 0);
        cpl_propertylist* pl = cpl_propertylist_load(filename, 0);
        sph_error_code rerr;
        
        for (int jj = 0; jj < nspeckles; ++jj) {
            size0 =
                    size[jj]
                            * sqrt(
                                    (double) cpl_image_get_size_x(im)
                                            * (double) cpl_image_get_size_x(im)
                                            + (double) cpl_image_get_size_y(im)
                                                    * (double) cpl_image_get_size_y(
                                                            im));
            xpos0 = (xpos[jj] + 0.5) * cpl_image_get_size_x(im);
            ypos0 = (ypos[jj] + 0.5) * cpl_image_get_size_y(im);
            sph_test_image_tools_add_speckle(im, xpos0, ypos0, size0,
                    svalue[jj]);
        }
        if (self->coronagraph_in)
            sph_test_pupilimage_creator_apply_coronagraph(self, im);
        rerr = cpl_image_save(im, filename,
                CPL_TYPE_DOUBLE, pl, CPL_IO_DEFAULT);
        cpl_image_delete(im);
        cpl_propertylist_delete(pl);
        if (rerr) cpl_error_set_where(cpl_func);
    }

    cpl_free(xpos);
    cpl_free(ypos);
    cpl_free(size);
    cpl_free(svalue);
    return;
}

cpl_frameset* sph_test_pupilimage_creator_get_frameset(
        sph_test_pupilimage_creator* self) {
    return self->frameset;
}
cpl_frame* sph_test_pupilimage_creator_get_first_frame(
        sph_test_pupilimage_creator* self) {
    return cpl_frameset_get_first(self->frameset);
}
int sph_test_pupilimage_creator_get_nx(sph_test_pupilimage_creator* self) {
    return self->nx;
}

int sph_test_pupilimage_creator_get_ny(sph_test_pupilimage_creator* self) {
    return self->ny;
}

void sph_test_pupilimage_creator_set_size(sph_test_pupilimage_creator* self,
        int nx, int ny) {
    self->nx = nx;
    self->ny = ny;
}
void sph_test_pupilimage_creator_set_image(sph_test_pupilimage_creator* self,
        cpl_image* im) {
    self->base_image = im;
}

void sph_test_pupilimage_creator_delete(sph_test_pupilimage_creator* self) {
    if (self) {
        if (self->base_image) {
            cpl_image_delete(self->base_image);
        }
        cpl_frameset_delete(self->frameset);
        cpl_free(self);
    }
}
