/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_master_frame.h"
#include "sph_ird_tags.h"
#include "sph_fits.h"
#include "sph_cube.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

#include <math.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

static
int cutest_init_test_ngc_simulator_testsuite(void);

static
int cutest_clean_test_ngc_simulator_testsuite(void);

static
void
cutest_test_ngc_simulator_raw_cube_from_frame(void);

static
void
cutest_test_ngc_simulator_new(void);
static
void cutest_test_ngc_simulator_raw_cube_from_frame_many_planes(void);

static
int cutest_init_test_ngc_simulator_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_test_ngc_simulator_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_ngc_simulator_new(void) {
    sph_test_ngc_ir_simulator* fc = NULL;

    fc = sph_test_ngc_ir_simulator_new(256, 256);
    cpl_test_assert( fc);

    sph_test_ngc_ir_simulator_delete(fc);
    fc = NULL;

    return;
}
/*----------------------------------------------------------------------------*/
/**
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_ngc_simulator_raw_cube_from_frame(void) {
    sph_test_ngc_ir_simulator* fc = NULL;
    cpl_frame* frame = NULL;
    cpl_frame* inframe = NULL;
    cpl_image* image = NULL;
    int nx = 256;
    int ny = 256;

    image = sph_test_image_tools_create_flat_image_double(nx, ny, 1.0);
    inframe = sph_test_frame_image(image, CPL_BPP_32_SIGNED,
            SPH_TEST_PUPILIMAGE_TAG, CPL_FRAME_GROUP_RAW);
    cpl_test_nonnull( inframe );
    cpl_test_nonnull( image );

    fc = sph_test_ngc_ir_simulator_new(256, 256);
    cpl_test_assert( fc);
    frame = sph_test_ngc_ir_simulator_raw_cube_from_frame(fc, inframe, 10,
            "SPHERE_IRDIFS_DARK.fits", 0, NULL, SPH_IRD_TAG_DARK_RAW);
    cpl_test_assert( frame);

    cpl_test_eq(sph_fits_get_nplanes(cpl_frame_get_filename(frame),0), 10);
    sph_test_ngc_ir_simulator_delete(fc);
    fc = NULL;
    cpl_frame_delete(inframe);
    cpl_frame_delete(frame);
    cpl_image_delete(image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_ngc_simulator_raw_cube_from_frame_many_planes(void) {
    sph_test_ngc_ir_simulator* fc = NULL;
    cpl_frame* frame = NULL;
    cpl_frame* inframe = NULL;
    cpl_image* image = NULL;
    int nx = 256;
    int ny = 256;
    int pp = 0;

    image = sph_test_image_tools_create_flat_image_double(nx, ny, 1.0);
    inframe = sph_test_frame_image(image, CPL_BPP_32_SIGNED,
            SPH_TEST_PUPILIMAGE_TAG, CPL_FRAME_GROUP_RAW);
    cpl_test_nonnull( inframe );
    cpl_test_nonnull( image );
    cpl_image_delete(image);
    image = NULL;

    for (pp = 1; pp < 10; ++pp) {
        image = sph_test_image_tools_create_flat_image_int(nx, ny, pp);
        sph_cube_append_image(cpl_frame_get_filename(inframe), image, NULL, 0);
        cpl_image_delete(image);
        image = NULL;
    }
    sph_cube_finalise_file(cpl_frame_get_filename(inframe));
    cpl_test_eq(sph_fits_get_nplanes(cpl_frame_get_filename(inframe), 0),
            10);

    fc = sph_test_ngc_ir_simulator_new(256, 256);
    cpl_test_assert( fc);
    frame = sph_test_ngc_ir_simulator_raw_cube_from_frame(fc, inframe, 10,
            "SPHERE_IRDIFS_DARK.fits", 0, NULL, SPH_IRD_TAG_DARK_RAW);
    cpl_test_assert( frame);

    cpl_test_eq(sph_fits_get_nplanes(cpl_frame_get_filename(frame),0), 10);
    sph_test_ngc_ir_simulator_delete(fc);
    fc = NULL;
    cpl_frame_delete(inframe);
    cpl_frame_delete(frame);
    cpl_image_delete(image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 *   Unit test of the NGC simulator
 *
 *
 *
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_test_famecreator_irdis",
            cutest_init_test_ngc_simulator_testsuite,
            cutest_clean_test_ngc_simulator_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_test_ngc_ir_simulator_new",
                    cutest_test_ngc_simulator_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_test_ngc_ir_simulator_raw_cube_frame_frame",
                    cutest_test_ngc_simulator_raw_cube_from_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_test_ngc_ir_simulator_raw_cube_frame_frame_many",
                    cutest_test_ngc_simulator_raw_cube_from_frame_many_planes)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
