/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "validations/sph_test_validations.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_validation_framestructure(void) {
    cpl_frame* inframe = NULL;
    cpl_image* im1 = NULL;
    cpl_image* im2 = NULL;
    sph_test_validation_result res = SPH_TEST_VALIDATION_RESULT_ERROR;

    res = sph_test_validation_framestructure(inframe, 1, 1, 2, 2, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_ERROR);

    im1 = sph_test_image_tools_create_flat_image_int(256, 256, 10);
    im2 = sph_test_image_tools_create_flat_image_int(256, 256, 10);
    inframe = sph_test_frame_image(im1, CPL_TYPE_FLOAT, "NONE",
            CPL_FRAME_GROUP_RAW);
    cpl_image_save(im1, cpl_frame_get_filename(inframe), CPL_TYPE_FLOAT, NULL,
            CPL_IO_EXTEND);
    cpl_test_nonnull( inframe );
    res = sph_test_validation_framestructure(inframe, 1, 1, 1, 1, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_VALID);
    res = sph_test_validation_framestructure(inframe, 1, -1, 1, 1, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_VALID);
    res = sph_test_validation_framestructure(inframe, 1, -1, 1, -1, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_VALID);
    res = sph_test_validation_framestructure(inframe, 1, 2, 1, 3, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_VALID);
    res = sph_test_validation_framestructure(inframe, 1, 0, 1, 3, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_INVALID);
    res = sph_test_validation_framestructure(inframe, 1, 2, 1, -1, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_VALID);
    res = sph_test_validation_framestructure(inframe, 1, 2, 2, 2, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_INVALID);
    res = sph_test_validation_framestructure(inframe, 2, 2, 3, 3, 0);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_INVALID);
    res = sph_test_validation_framestructure(inframe, 1, 2, 1, 3, 1);
    cpl_test_eq(res, SPH_TEST_VALIDATION_RESULT_INVALID);

    cpl_frame_delete(inframe);
    cpl_image_delete(im1);
    cpl_image_delete(im2);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing the validators", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "validation_framesetructure",
                    cutest_test_validation_framestructure)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
