/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_ifs_master_detector_flat.h"
#include "sph_ifs_tags.h"
#include "sph_common_keywords.h"
#include "sph_test.h"

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup
 *
 * This test verifies that when the template feeds in a complete set of
 * rawframes (all lamps) the recipe does what it should.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
int cutest_init_ifs_master_dff_all_inputs_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_ifs_master_dff_all_inputs_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Common setup for the unit tests
 *
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_dff_all_inputs_common_setup(cpl_frameset** frames,
        cpl_parameterlist** params) {
    sph_error_code rerr = CPL_ERROR_NONE;

    cpl_test_nonnull( frames );
    cpl_test_nonnull( params );

    *frames = cpl_frameset_new();

    rerr = sph_test_create_ifs_raw_dff_frameset_broad_band(*frames, 3, 1);
    rerr = sph_test_create_ifs_raw_dff_frameset_broad_band(*frames, 3, 2);
    rerr = sph_test_create_ifs_raw_dff_frameset_broad_band(*frames, 3, 3);
    rerr = sph_test_create_ifs_raw_dff_frameset_broad_band(*frames, 3, 4);
    rerr = sph_test_create_ifs_raw_dff_frameset_broad_band(*frames, 3, 5);
    rerr = sph_test_create_ifs_raw_dff_frameset_broad_band(*frames, 3, 6);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    *params = sph_ifs_master_detector_flat_create_paramlist();

    cpl_test_nonnull( *frames );
    cpl_test_nonnull( *params );
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Common teardown for the unit tests
 *
 * @param   frames      pointer to the input frames to clean up
 * @param   params      pointer to the input parameterlist to clean up
 *
 * This cleans up the frameset and parameterlist
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_dff_all_inputs_common_teardown(cpl_frameset* frames,
        cpl_parameterlist* params) {

    cpl_test_nonnull( frames );
    cpl_test_nonnull( params );

    cpl_parameterlist_delete(params);
    sph_test_frameset_delete(frames);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the s[h_ifs_master_dark main function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_dff_all_inputs_check_frames(void) {
    sph_ifs_master_detector_flat* ifs_master_detector_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_dff_all_inputs_common_setup(&inframes, &inparams);

    ifs_master_detector_flat = sph_ifs_master_detector_flat_new(inframes,
            inparams);

    cpl_test_nonnull( ifs_master_detector_flat );
    cpl_frameset_delete(ifs_master_detector_flat->rawframes);
    cpl_frameset_delete(ifs_master_detector_flat->current_raw_frameset);
    /* Run code... */
    rerr = sph_ifs_master_detector_flat_check_frames(ifs_master_detector_flat);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( ifs_master_detector_flat->inframes );

    cpl_test_eq( cpl_frameset_get_size( ifs_master_detector_flat->inframes),
            3 * 6);

    cpl_test_error(CPL_ERROR_NONE);

    sph_ifs_master_detector_flat_delete(ifs_master_detector_flat);
    cutest_ifs_master_dff_all_inputs_common_teardown(inframes, inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the s[h_ifs_master_dark main function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_dff_all_inputs_run(void) {
    sph_ifs_master_detector_flat* ifs_master_detector_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frameset* result_frames = NULL;

    /* Setup */
    cutest_ifs_master_dff_all_inputs_common_setup(&inframes, &inparams);

    ifs_master_detector_flat = sph_ifs_master_detector_flat_new(inframes,
            inparams);

    /* Run code... */
    rerr = sph_ifs_master_detector_flat_run(ifs_master_detector_flat);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);

    result_frames = sph_utils_extract_frames(ifs_master_detector_flat->inframes,
            SPH_IFS_TAG_DFF_LONG1_CALIB);

    cpl_test_error(CPL_ERROR_NONE);

    sph_ifs_master_detector_flat_delete(ifs_master_detector_flat);
    cutest_ifs_master_dff_all_inputs_common_teardown(inframes, inparams);

    cpl_test_nonnull( result_frames );
    cpl_test_noneq( cpl_frameset_get_size(result_frames), 0);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_frameset_delete(result_frames);
    result_frames = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_detector_flat
 recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("master_dff_all_inputs_test",
            cutest_init_ifs_master_dff_all_inputs_testsuite,
            cutest_clean_ifs_master_dff_all_inputs_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "check_frames",
                    cutest_ifs_master_dff_all_inputs_check_frames)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "run", cutest_ifs_master_dff_all_inputs_run)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
