/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_ifs_science.h"
#include "sph_ifs_test_fixtures_calib_products.h"
#include "sph_ldt.h"
#include "sph_framecombination.h"
#include "sph_utils.h"

#include <cpl.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/*-                            Declarations of private functions              */
/*----------------------------------------------------------------------------*/

static
void cutest_ifs_science_collapse_ldts(void);

static
void cutest_ifs_science_combine_ldts(void);

static cpl_frameset* cutest_ifs_science_collapse_ldts_fixture(int n);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_ifs_science_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_ifs_science_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_science_collapse_ldts(void) {
    cpl_frameset* inframes = NULL;
    cpl_frameset* allframes = NULL;
    cpl_frameset* collapsed = NULL;
    sph_master_frame* result = NULL;
    double median = 0.0;
    inframes = cutest_ifs_science_collapse_ldts_fixture(1);
    allframes = cpl_frameset_duplicate(inframes);
    cpl_msg_info(__func__, "allframes in cutest_ifs_science_collapse_ldts:");
    cpl_frameset_dump(allframes, stdout);
    cpl_msg_info(__func__, "inframes in cutest_ifs_science_collapse_ldts:");
    cpl_frameset_dump(inframes, stdout);
    collapsed = sph_ifs_science_collapse_ldts(inframes, allframes, NULL,
            "SOMERECIPE", "SOMETAG");
    cpl_test_nonnull( collapsed );
    cpl_test_eq(cpl_frameset_get_size(collapsed), 1);
    result = sph_master_frame_load_(cpl_frameset_get_first_const(collapsed), 0);
    cpl_test_nonnull( result );
    median = sph_master_frame_get_median(result, NULL);
    sph_master_frame_divide_double(result, median);
    sph_master_frame_mask_tolerance(result, 0.000001, 1000000.0);
    //cpl_test_abs( sph_master_frame_get_mean(result,NULL), 1.0, 0.01);

    // The accuracy for the check below is so bad, because
    // the lenslet model slits are wider than the fixture model slits
    cpl_test_abs( cpl_image_get_max(result->image), 1.0, 0.5);
    cpl_test_abs( cpl_image_get_median(result->image), 1.0, 0.01);
    sph_master_frame_save(result, "result.fits", NULL);
    sph_master_frame_delete(result);
    cpl_frameset_delete(collapsed);
    cpl_frameset_delete(inframes);
    cpl_frameset_delete(allframes);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_science_combine_ldts(void) {
    cpl_frameset* inframes = NULL;
    cpl_frameset* allframes = NULL;
    sph_ldt* collapsed = NULL;
    sph_master_frame* result = NULL;
    double median = 0.0;
    inframes = cutest_ifs_science_collapse_ldts_fixture(1);
    allframes = cpl_frameset_duplicate(inframes);
    collapsed = sph_ifs_science_combine_ldts(inframes, SPH_COLL_ALG_MEDIAN);
    cpl_test_nonnull( collapsed );
    result = sph_ldt_collapse_interpolate(collapsed, 0.0, 2.0);
    cpl_test_nonnull( result );

    median = sph_master_frame_get_median(result, NULL);
    sph_test_noneq_abs( median, 0.0, 0.00001);
    sph_master_frame_divide_double(result, median);
    sph_master_frame_mask_tolerance(result, 0.000001, 1000000.0);
    cpl_test_abs( sph_master_frame_get_mean(result, NULL), 1.0,
                                0.1);

    // The accuracy for the check below is so bad, because
    // the lenslet model slits are wider than the fixture model slits
    cpl_test_abs( cpl_image_get_max(result->image), 1.0, 0.5);
    cpl_test_abs( cpl_image_get_median(result->image), 1.0, 0.1);
    sph_master_frame_save(result, "result_combined.fits", NULL);
    sph_master_frame_delete(result);
    sph_ldt_delete(collapsed);
    cpl_frameset_delete(inframes);
    cpl_frameset_delete(allframes);
    return;
}

static 
cpl_frameset* cutest_ifs_science_collapse_ldts_fixture(int n) {
    sph_ldt* ldt = NULL;
    sph_test_ngc_ir_simulator* ngc = NULL;
    sph_test_ifs_instrument_model* ifsmodel = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* aframe = NULL;
    int ii = 0;

    ngc = sph_test_ngc_ir_simulator_new(1024, 1024);
    cpl_test_nonnull( ngc );
    ifsmodel = sph_test_ifs_instrument_model_new(ngc->det_size_x,
            SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_MICRONS,
            SPH_IFS_LENSLET_MODEL_LENSLETS_SIDE);
    cpl_test_nonnull( ifsmodel );
    inframes = cpl_frameset_new();
    for (ii = 0; ii < n; ++ii) {
        ldt = sph_ifs_test_fixtures_calib_products_simple_ldt(ngc, ifsmodel);

        cpl_test_nonnull( ldt );
        aframe = sph_filemanager_create_temp_frame("LDT_collapse", "TESTTAG",
                CPL_FRAME_GROUP_CALIB);

        sph_ldt_save_cube(ldt, cpl_frame_get_filename(aframe), NULL, NULL, NULL, NULL,
                NULL, NULL, NULL, 0.00746,0.0,NULL);

        cpl_frameset_insert(inframes, aframe);

        sph_ldt_delete(ldt);
    }

    sph_test_ngc_ir_simulator_delete(ngc);
    sph_test_ifs_instrument_model_delete(ifsmodel);
    return inframes;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A testsuite", cutest_ifs_science_init_testsuite,
            cutest_ifs_science_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing coilapse LDTs",
                    cutest_ifs_science_collapse_ldts)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing combine LDTs",
                    cutest_ifs_science_combine_ldts)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
