/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_ird_common_science.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_irdis_model.h"
#include "sph_common_science.h"
#include "sph_time.h"
#include "sph_fctable.h"
#include "sph_ird_tags.h"
#include "sph_fits.h"
#include "sph_test_ird_fixtures.h"
#include "sph_common_keywords.h"


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
cpl_frame*
sph_test_ird_fixtures_create_corresponding_fc_frame(cpl_frame* input_cube_frame,
        int txtflag) {
    cpl_frame* result = cpl_frame_duplicate(input_cube_frame);
    char* fname = NULL;
    cpl_frame_set_tag(result, SPH_IRD_TAG_STAR_CENTER_CALIB);
    fname = sph_fctable_construct_filename(input_cube_frame, txtflag);
    cpl_frame_set_filename(result, fname);
    cpl_free(fname);
    fname = NULL;
    return result;
}
cpl_frame*
sph_test_ird_fixtures_create_dark(sph_ird_instrument_model* model, double val,
        double rms) {
    cpl_frame* dark_frame = NULL;
    cpl_image* im = NULL;
    sph_master_frame* mframe = NULL;
    cpl_ensure(model, CPL_ERROR_NULL_INPUT, NULL);
    dark_frame = sph_filemanager_create_temp_frame("ird_dark",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    im = sph_test_image_tools_create_flat_image_double(model->detsize_pixels_x,
            model->window_size_y, val);
    if (rms)
        sph_test_image_tools_add_noise(im, rms, NULL);
    mframe = sph_master_frame_new_from_cpl_image(im);
    sph_master_frame_save(mframe, cpl_frame_get_filename(dark_frame), NULL);
    cpl_image_delete(im);
    im = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
    return dark_frame;
}
cpl_frame*
sph_test_ird_fixtures_create_flat(sph_ird_instrument_model* model, double rms) {
    cpl_frame* flat_frame = NULL;
    cpl_image* im = NULL;
    sph_master_frame* mframe = NULL;
    cpl_propertylist* pl = NULL;
    cpl_ensure(model, CPL_ERROR_NULL_INPUT, NULL);
    flat_frame = sph_filemanager_create_temp_frame("ird_flat",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    im = sph_test_image_tools_create_flat_image_double(model->detsize_pixels_x,
            model->window_size_y, 1.0);
    if (rms)
        sph_test_image_tools_add_noise(im, rms, NULL);
    mframe = sph_master_frame_new_from_cpl_image(im);
    pl = sph_ird_instrument_model_get_as_propertylist(model);
    sph_master_frame_save(mframe, cpl_frame_get_filename(flat_frame), pl);
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
    return flat_frame;
}

cpl_frame*
sph_test_ird_fixtures_create_science_frame_stable_fc(int np,
        sph_ird_instrument_model* model, double fc_offset_x, double fc_offset_y,
        cpl_frame* actual_dark, cpl_frame* actual_flat,
        cpl_frame* actual_distortion) {
    return sph_test_ird_fixtures_create_science_frame_stable_fc_rot(np, model,
            fc_offset_x, fc_offset_y, actual_dark, actual_flat,
            actual_distortion, 0.0, 0.0, 1.0);
}

cpl_frame*
sph_test_ird_fixtures_create_science_frame_stable_fc_rot(int np,
        sph_ird_instrument_model* model, double fc_offset_x, double fc_offset_y,
        cpl_frame* actual_dark, cpl_frame* actual_flat,
        cpl_frame* actual_distortion, double angle_offset, double dangle,
        double relscale) {
    cpl_frame* result = NULL;
    sph_test_ngc_ir_simulator* ngc = NULL;
    sph_test_irdis_model* testmodel = NULL;
    cpl_image* im = NULL;
    cpl_imagelist* imlist = NULL;
    cpl_image* im2 = NULL;
    cpl_image* imleft = NULL;
    cpl_image* imright = NULL;
    int pp = 0;
    sph_master_frame* adark = NULL;
    sph_master_frame* aflat = NULL;
    sph_distortion_model* distmap_left = NULL;
    sph_distortion_model* distmap_right = NULL;
    sph_point* p = NULL;
    double angle = 0;

    cpl_test_error(CPL_ERROR_NONE);

    if (actual_dark) {
        adark = sph_master_frame_load_(actual_dark, 0);
    }

    if (actual_flat) {
        aflat = sph_master_frame_load_(actual_flat, 0);
    }
    if (actual_distortion) {
        distmap_left = sph_distortion_model_load(
                cpl_frame_get_filename(actual_distortion), 0,
                                                 SPH_COMMON_KEYWORD_DISTMAP_COEFFX,
                                                 SPH_COMMON_KEYWORD_DISTMAP_COEFFY);
        distmap_right = sph_distortion_model_load(
                cpl_frame_get_filename(actual_distortion), 4,
                                                 SPH_COMMON_KEYWORD_DISTMAP_COEFFX,
                                                 SPH_COMMON_KEYWORD_DISTMAP_COEFFY);
    }

    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_lt(0, np);

    ngc = sph_test_ngc_ir_simulator_new(model->detsize_pixels_x,
            model->detsize_pixels_y);

    ngc->mean_bias = ngc->mean_dark = ngc->sigma_bias = ngc->sigma_dark =
            ngc->ron = 0.0;

    testmodel = sph_test_irdis_model_new(ngc);

    result = sph_filemanager_create_temp_frame("SCIENCETEST",
            SPH_IRD_TAG_SCIENCE_IMAGING_RAW, CPL_FRAME_GROUP_RAW);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);
    p = sph_point_new(0.0, 0.0);
    angle = angle_offset;
    for (pp = 0; pp < np; ++pp) {

        im = sph_test_image_tools_create_flat_image_double(testmodel->nx,
                testmodel->ny, 0.0);
        cpl_test_error(CPL_ERROR_NONE);

        p->x = 30.0;
        p->y = 0.0;
        sph_point_rotate_around(p, 0.0, 0.0, cos(angle * CPL_MATH_RAD_DEG),
                sin(angle * CPL_MATH_RAD_DEG), 0.0, 0.0);
        sph_test_image_tools_add_gauss(im,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_x,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_y, 2.0,
                1000000.0);
        cpl_test_error(CPL_ERROR_NONE);

        sph_test_image_tools_add_gauss(im,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_x + p->x,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_y + p->y, 2.0,
                10000.0);
        cpl_test_error(CPL_ERROR_NONE);

        if (distmap_left) {
            imleft = sph_distortion_model_apply(distmap_left, im,
                                                CPL_KERNEL_DEFAULT,
                                                CPL_KERNEL_DEF_WIDTH,
                                                NULL, NULL, NULL);
        } else
            imleft = cpl_image_duplicate(im);

        cpl_image_multiply_scalar(im, 0.0);
        sph_test_image_tools_add_gauss(im,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_x,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_y, 2.0,
                1000000.0);
        sph_test_image_tools_add_gauss(
                im,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_x
                        + p->x * relscale,
                model->detsize_pixels_x / 4 + 0.5 + fc_offset_y
                        + p->y * relscale, 2.0, 10000.0);

        if (distmap_right) {
            imleft = sph_distortion_model_apply(distmap_right, im,
                                                CPL_KERNEL_DEFAULT,
                                                CPL_KERNEL_DEF_WIDTH,
                                                NULL, NULL, NULL);
        } else
            imright = cpl_image_duplicate(im);
        cpl_image_delete(im);
        im = NULL;

        cpl_test_error(CPL_ERROR_NONE);
        im2 = sph_test_irdis_model_process_image_left_right(testmodel, imleft,
                imright);
        cpl_image_delete(imleft);
        imleft = NULL;
        cpl_image_delete(imright);
        imright = NULL;
        im = sph_test_ngc_ir_simulator_create_raw_image(ngc, im2, 0.0, 0.0);
        cpl_image_delete(im2);
        im2 = NULL;

        if (aflat) {
            cpl_image_multiply(im, aflat->image);
        }
        if (adark) {
            cpl_image_add(im, adark->image);
        }

        sph_test_image_tools_apply_poisson_noise(im, ngc->pRNG);
        if (pp == 0) {
            if (np == 1) {
                cpl_image_save(im, cpl_frame_get_filename(result),
                        CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
            } else {
                imlist = cpl_imagelist_new();
                cpl_imagelist_set(imlist, im, 0);
                cpl_imagelist_save(imlist, cpl_frame_get_filename(result),
                        CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
                cpl_imagelist_unset(imlist, 0);
                cpl_imagelist_delete(imlist);
                imlist = NULL;
            }
        } else {
            im2 = cpl_image_cast(im, CPL_TYPE_INT);
            sph_cube_append_image(cpl_frame_get_filename(result), im2, NULL, 0);
            cpl_image_delete(im2);
        }
        cpl_image_delete(im);
        im = NULL;
        cpl_test_error(CPL_ERROR_NONE);
        angle += dangle;
    }
    sph_cube_finalise_file(cpl_frame_get_filename(result));
    sph_test_ngc_ir_simulator_delete(ngc);
    ngc = NULL;
    sph_test_irdis_model_delete(testmodel);
    testmodel = NULL;
    sph_master_frame_delete(adark);
    adark = NULL;
    sph_master_frame_delete(aflat);
    aflat = NULL;
    sph_point_delete(p);
    p = NULL;
    return result;
}

/*
 void
 sph_test_ird_fixtures_create_centre_vectors_single_frame(
 cpl_frame* aframe,
 sph_ird_instrument_model* model,
 cpl_vector** cent_lx,
 cpl_vector** cent_ly,
 cpl_vector** cent_rx,
 cpl_vector** cent_ry)
 {
 double          xcl = 0.0;
 double          xcr = 0.0;
 double          ycl = 0.0;
 double          ycr = 0.0;
 sph_fctable*    fctab = NULL;
 cpl_frameset*   rawframes = cpl_frameset_new();

 cpl_frameset_insert(rawframes, cpl_frame_duplicate(aframe));
 sph_ird_instrument_model_get_centre_left_subframe_coords(model,&xcl,&ycl);
 sph_ird_instrument_model_get_centre_right_subframe_coords(model,&xcr,&ycr);
 fctab = sph_fctable_create_fctable(rawframes,NULL,1,xcl,ycl,xcr,ycr);
 *cent_lx = sph_fctable_get_centre_x_left(fctab);
 *cent_ly = sph_fctable_get_centre_y_left(fctab);
 *cent_rx = sph_fctable_get_centre_x_right(fctab);
 *cent_ry = sph_fctable_get_centre_y_right(fctab);
 cpl_frameset_delete(rawframes); rawframes = NULL;
 sph_fctable_delete(fctab); fctab = NULL;
 }

 */

cpl_frameset*
sph_test_ird_fixtures_create_star_centre_frames_ascii_no_dither(int nf, int np,
        sph_ird_instrument_model* model, cpl_frameset* rawframes) {
    double lxc0 = 0.0;
    double lyc0 = 0.0;
    double rxc0 = 0.0;
    double ryc0 = 0.0;
    sph_fctable* fctab = NULL;
    cpl_frameset* scframes = NULL;
    int ff = 0;
    int pp = 0;
    cpl_frame* aframe = NULL;
    char* fname = NULL;
    scframes = cpl_frameset_new();
    cpl_ensure(model, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rawframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cpl_frameset_get_size(rawframes)==nf, CPL_ERROR_ILLEGAL_INPUT,
            NULL);

    sph_ird_instrument_model_get_centre_left_subframe_coords(model, &lxc0,
            &lyc0);
    sph_ird_instrument_model_get_centre_left_subframe_coords(model, &rxc0,
            &ryc0);
    for (ff = 0; ff < nf; ++ff) {
        fname = sph_fctable_construct_filename(
                cpl_frameset_get_position(rawframes, ff), 1);
        fctab = sph_fctable_new();
        for (pp = 0; pp < np; ++pp) {
            sph_fctable_add_row_double(fctab, 0.0, lxc0, lyc0, rxc0, ryc0, 0.0);
        }
        sph_fctable_save_ascii(fctab, fname);
        aframe = cpl_frame_new();
        cpl_frame_set_filename(aframe, fname);
        cpl_frame_set_tag(aframe, SPH_IRD_TAG_STAR_CENTER_CALIB);
        cpl_frame_set_group(aframe, CPL_FRAME_GROUP_CALIB);
        cpl_free(fname);
        fname = NULL;
        cpl_frameset_insert(scframes, aframe);
        sph_fctable_delete(fctab);
        fctab = NULL;
    }
    return scframes;
}

cpl_frameset*
sph_test_ird_fixtures_create_science_frames_no_dither(int nf, int np,
        sph_ird_instrument_model* model, cpl_frame* actual_dark,
        cpl_frame* actual_flat, cpl_frame* actual_distortion) {
    cpl_frameset* inframes = NULL;
    int ff = 0;
    cpl_frame* aframe = NULL;

    cpl_test_lt(0, np);
    cpl_test_lt(0, nf);
    inframes = cpl_frameset_new();
    for (ff = 0; ff < nf; ++ff) {
        aframe = sph_test_ird_fixtures_create_science_frame_stable_fc(np, model,
                0.0, 0.0, actual_dark, actual_flat, actual_distortion);
        if (aframe)
            cpl_frameset_insert(inframes, aframe);
    }
    cpl_test_error(CPL_ERROR_NONE);
    return inframes;
}
sph_ird_instrument_model*
sph_test_ird_fixtures_instrument_model(int nx, int ny) {
    sph_ird_instrument_model* insmodel = NULL;

    insmodel = sph_ird_instrument_model_new();
    insmodel->detsize_pixels_x = nx;
    insmodel->detsize_pixels_y = ny;
    insmodel->split_pixel_x = nx / 2;
    insmodel->window_size_y = ny;
    return insmodel;
}

cpl_frame*
sph_test_ird_fixtures_instrument_model_frame(int nx, int ny) {
    sph_ird_instrument_model* insmodel = NULL;
    cpl_frame* aframe = NULL;

    aframe = sph_filemanager_create_temp_frame("IRD_MODEL",
            SPH_IRD_TAG_INSTRUMENT_MODEL, CPL_FRAME_GROUP_CALIB);
    insmodel = sph_test_ird_fixtures_instrument_model(nx, ny);
    sph_ird_instrument_model_save(insmodel, cpl_frame_get_filename(aframe));

    sph_ird_instrument_model_delete(insmodel);
    insmodel = NULL;
    return aframe;
}

cpl_frameset*
sph_test_ird_fixtures_create_science_frames_stable_dither(int nf, int np,
        sph_ird_instrument_model* model, cpl_frame* actual_dark,
        cpl_frame* actual_flat, cpl_frame* acutal_distortion, double* dithx,
        double* dithy) {
    cpl_frameset* inframes = NULL;
    int ff = 0;
    cpl_frame* aframe = NULL;

    cpl_test_lt(0, np);
    cpl_test_lt(0, nf);
    inframes = cpl_frameset_new();
    for (ff = 0; ff < nf; ++ff) {
        aframe = sph_test_ird_fixtures_create_science_frame_stable_fc(np, model,
                dithx[ff], dithy[ff], actual_dark, actual_flat,
                acutal_distortion);
        cpl_frameset_insert(inframes, aframe);
    }
    return inframes;
}

/**@}*/
