
/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <cpl.h>
#include "sph_error.h"
#include "sph_test.h"

#include "sph_zpl_framegroups.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_exposure.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_test_image_tools.h"



/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_zpl_framegroups  Unit test of sph_zpl_framegroups
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_zpl_framegroups_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static int cutest_clean_zpl_framegroups_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_framegroups_new function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_framegroups_new(void) {
    sph_zpl_framegroups* fgs = sph_zpl_framegroups_new();

    cpl_test_nonnull( fgs );

    sph_zpl_framegroups_delete(fgs);
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_framegroups_create_Q function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_framegroups_create_Q(void) {
    sph_zpl_framegroups* fgs = NULL;
    cpl_frame* zexpcube_plus = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frameset* inframes = NULL;
    cpl_propertylist* pl = NULL;
    int nframes = 3;

    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube_plus, nframes);
    sph_test_simulator_create_zplpreproc_cube_minus(&zexpcube_minus, nframes);

    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, zexpcube_plus);
    cpl_frameset_insert(inframes, zexpcube_minus);

    fgs = sph_zpl_framegroups_create_Q(inframes);
    cpl_test_nonnull( fgs );

    cpl_test_eq_string( cpl_property_get_string( fgs->propertyminus ),
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS);
    cpl_test_eq_string( cpl_property_get_string( fgs->propertyplus ),
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QPLUS);
    pl = cpl_propertylist_load(
            cpl_frame_get_filename(cpl_frameset_get_first(fgs->framesminus)),
            0);
    cpl_test_eq_string(
            cpl_propertylist_get_string( pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME),
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS);
    cpl_propertylist_delete(pl);
    pl = cpl_propertylist_load(
            cpl_frame_get_filename(cpl_frameset_get_first(fgs->framesplus)), 0);
    cpl_test_eq_string(
            cpl_propertylist_get_string( pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME ),
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QPLUS);
    cpl_propertylist_delete(pl);

    sph_zpl_framegroups_delete(fgs);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_frameset_delete(inframes);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_zpl_framegroups",
            cutest_init_zpl_framegroups_testsuite,
            cutest_clean_zpl_framegroups_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "Testing sph_zpl_framegroups_new",
                    cutest_zpl_framegroups_new)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing cutest_zpl_framegroups_create_Q",
                    cutest_zpl_framegroups_create_Q)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}
