/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_zpl_intensity_flat.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <string.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_intensity_flat_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static int cutest_clean_intensity_flat_testsuite(void) {
    return sph_end_test();
}

#ifdef SPH_ZPL_INTENSITY_FLAT_1
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat
 @note It is fed by raw frame and master bias calibration frame
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_intensity_flat_run_1(void) {
    sph_zpl_intensity_flat* zpl_intensity_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 100.0;

    double zero_odd = 2500.0;
    double zero_even = 2000.0;
    double pi_odd = 3000.0;
    double pi_even = 3500.0;
    cpl_propertylist* pl = NULL;

    inparams = sph_zpl_intensity_flat_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set( &zexpcube, zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_IFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    rerr = cpl_frameset_insert( inframes, zexpcube );

    zpl_intensity_flat = sph_zpl_intensity_flat_new( inframes, inparams);
    cpl_test_nonnull( zpl_intensity_flat );

    rerr = sph_zpl_intensity_flat_run( zpl_intensity_flat );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_intensity_flat->intensity_flat_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN), 0.0, 1e-10);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 1.0, 1e-10);

    rerr = sph_zpl_intensity_flat_delete( zpl_intensity_flat );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}
#endif
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat
 @note It is fed by the pre-processed frames with noise (simple collapse method and normalization)
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_intensity_flat_run_2(void) {
    sph_zpl_intensity_flat* zpl_intensity_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 100.0;

    double zero_odd = 2500.0;
    double zero_even = 2000.0;
    double pi_odd = 3000.0;
    double pi_even = 3500.0;
    cpl_propertylist* pl = NULL;
    cpl_parameter* parameter_collapse = NULL;
    cpl_parameter* keep_intermediate = NULL;
    char* outfilename = NULL;

    //create zpl intensity flat parameter list
    inparams = sph_zpl_intensity_flat_create_paramlist();

    //set "zpl.intensity_flat.collapse" parameter to TRUE
    parameter_collapse = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat.collapse");
    keep_intermediate = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat.keep_intermediate");
    cpl_parameter_set_bool(parameter_collapse, 1);
    cpl_parameter_set_bool(keep_intermediate, 1);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set(&zexpcube, zero_odd,
            zero_even, pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_IFF_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube);

    sph_test_simulator_create_zplpreproc_cube_set(&zexpcube1, zero_odd,
            zero_even, pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube1, SPH_ZPL_TAG_IFF_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube1, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube1);

    zpl_intensity_flat = sph_zpl_intensity_flat_new(inframes, inparams);
    cpl_test_nonnull( zpl_intensity_flat );

    rerr = sph_zpl_intensity_flat_run(zpl_intensity_flat);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    outfilename =
        cpl_sprintf("%s-qc-non-normalized.fits",
                    sph_filemanager_get_basename(zpl_intensity_flat->intensity_flat_outfilename_cam1));

    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load(outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD),
            2750.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN),
            2250.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD),
            3250.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN),
            3750.0, 1e-10);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_propertylist_delete(pl);

    pl = cpl_propertylist_load(
            zpl_intensity_flat->intensity_flat_outfilename_cam1, 0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN),
            1.0, 1e-10);
    cpl_propertylist_delete(pl);

    outfilename =
        cpl_sprintf("%s-qc-non-normalized.fits",
                    sph_filemanager_get_basename(
                          zpl_intensity_flat->intensity_flat_outfilename_cam2));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load(outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;

    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD),
            2750.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN),
            2250.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD),
            3250.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN),
            3750.0, 1e-10);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_propertylist_delete(pl);

    pl = cpl_propertylist_load(
            zpl_intensity_flat->intensity_flat_outfilename_cam2, 0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN),
            1.0, 1e-10);
    cpl_propertylist_delete(pl);

    rerr = sph_zpl_intensity_flat_delete(zpl_intensity_flat);
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);
    cpl_test_error(CPL_ERROR_NONE);

    return;
}

#ifdef SPH_ZPL_INTENSITY_FLAT_3456
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat
 @note It is fed by the pre-processed frames (simple collapse method and normalization)
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_intensity_flat_run_3(void) {
    sph_zpl_intensity_flat* zpl_intensity_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    sph_quad_image* qi = NULL;
    sph_double_image* di0 = NULL;
    sph_double_image* dipi = NULL;
    cpl_frame* master_bias_frame = NULL;
    double iframe_value = 1000.0;
    double pframe_value = 2000.0;
    double sigma_iframe = 10.0;
    double sigma_pframe = 10.0;
    double dV = 100.0;
    double dS = 0.0;

    double zero_odd = 2500.0;
    double zero_even = 2000.0;
    double pi_odd = 3000.0;
    double pi_even = 3500.0;
    cpl_propertylist* pl = NULL;
    cpl_parameter* parameter_collapse = NULL;
    cpl_parameter* parameter_sigma_clip = NULL;
    cpl_frame* zexpcube1 = NULL;
    char* outfilename = NULL;

    //create zpl intensity flat parameter list
    inparams = sph_zpl_intensity_flat_create_paramlist();

    //set "zpl.intensity_flat.collapse" parameter to TRUE
    parameter_collapse = cpl_parameterlist_find( inparams, "zpl.intensity_flat.collapse");
    cpl_parameter_set_bool( parameter_collapse, 1);
    parameter_sigma_clip = cpl_parameterlist_find( inparams, "zpl.intensity_flat.sigma_clip" );
    cpl_parameter_set_double(parameter_sigma_clip, 3.0);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set( &zexpcube, zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_IFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    rerr = cpl_frameset_insert( inframes, zexpcube );

    sph_test_simulator_create_zplpreproc_cube_set_filename( &zexpcube1,"test_iff_cube1.tmp.fits", 20000.0, 20000.0, 20000.0, 20000.0, dV, 1 );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_IFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );

    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    //create master bias frame (quad image from double images)
    //zero double image with noise
    di0 = sph_test_zpl_simulator_create_double_image_noise_uniform( nx, ny, iframe_value, pframe_value, sigma_iframe, sigma_pframe );
    cpl_test_nonnull( di0  );

    //pi double image with noise
    dipi = sph_test_zpl_simulator_create_double_image_noise_uniform( nx, ny, iframe_value+dV, pframe_value+dV,
            sigma_iframe+dS, sigma_pframe+dS );
    cpl_test_nonnull( di0  );

    qi = sph_quad_image_new_from_double_images( di0, dipi );
    cpl_test_nonnull( qi  );

    sph_quad_image_save( qi, "test_if_sph_quad_image_mask_sigma.fits.tmp", NULL );
    master_bias_frame = cpl_frame_new();
    cpl_frame_set_filename( master_bias_frame, "test_if_sph_quad_image_mask_sigma.fits.tmp");
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB);

    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    zpl_intensity_flat = sph_zpl_intensity_flat_new( inframes, inparams);
    cpl_test_nonnull( zpl_intensity_flat );

    rerr = sph_zpl_intensity_flat_run( zpl_intensity_flat );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    outfilename =
        cpl_sprintf("%s-qc-non-normalized.fits",
                    zpl_intensity_flat->intensity_flat_outfilename );
    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load( outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;

    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD)));
    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN)));
    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD)));
    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN)));

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), 5214.286-iframe_value, 1.0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN), 4785.714-pframe_value, 1.0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 5642.857-iframe_value-dV, 1.0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN), 6071.4286-pframe_value-dV, 1.0);

    cpl_test_error(CPL_ERROR_NONE );
    pl = cpl_propertylist_load( zpl_intensity_flat->intensity_flat_outfilename, 0);
    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD)));
    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN)));
    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD)));
    cpl_test_lt(1e-10, fabs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN)));

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), 1.0, 1e-2);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN), 1.0, 1e-2);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 1.0, 1e-2);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN), 1.0, 1e-2);

    rerr = sph_zpl_intensity_flat_delete( zpl_intensity_flat );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat
 @note It is fed by raw frame with ovsercan table
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_intensity_flat_run_4(void) {
    sph_zpl_intensity_flat* zpl_intensity_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nz = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    cpl_propertylist* pl = NULL;
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    double ovsc[8] = {5.0, 1.0, 6.0, 2.0, 7.0, 3.0, 8.0, 4.0};

    int zero_odd_1 = 49;
    int zero_even_1 = 149;
    int pi_odd_1 = 149;
    int pi_even_1 = 49;
    double ovsc_1[8] = {15.0, 1.0, 16.0, 2.0, 17.0, 3.0, 18.0, 4.0};
    //double 					ovsc_1[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};
    double adu1_zero = 0;
    double adu2_zero = 0;
    char* outfilename = NULL;

    inparams = sph_zpl_intensity_flat_create_paramlist();
    inframes = cpl_frameset_new();

    //zexpcube
    //zpl frame cube with overscan table
    zexpcube = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
            "test_iff_zplexp_ovsc_cube", zero_odd, zero_even, pi_odd, pi_even, ovsc,
            1, 1);
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_IFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    //zexpcube1
    // zpl frame cube with overscan table
    zexpcube1 = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
            "test_iff_zplexp_ovsc_cube_1", zero_odd_1, zero_even_1, pi_odd_1,
            pi_even_1, ovsc_1, 1, 0 );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_IFF_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    zpl_intensity_flat = sph_zpl_intensity_flat_new( inframes, inparams);
    cpl_test_nonnull( zpl_intensity_flat );

    rerr = sph_zpl_intensity_flat_run( zpl_intensity_flat );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    outfilename = cpl_sprintf("%s-qc-non-normalized.fits",
                              zpl_intensity_flat->intensity_flat_outfilename );
    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load( outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN), 0.0, 1e-10);

    adu1_zero = (169.-ovsc[0] +170.-(ovsc[0]+1)+171.-(ovsc[0]+2)+49.-ovsc_1[0]+50.-ovsc_1[0]+51.-ovsc_1[0])/6.;
    adu2_zero = (169.-ovsc[2] +170.-(ovsc[2]+1)+171.-(ovsc[2]+2)+49.-ovsc_1[2]+50.-ovsc_1[2]+51.-ovsc_1[2])/6.;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD),
            0.5*(adu1_zero+adu2_zero), 1e-10);

    adu1_zero = (29.-ovsc[0] +30.-(ovsc[0]+1)+31.-(ovsc[0]+2)+149.-ovsc_1[0]+150.-ovsc_1[0]+151.-ovsc_1[0])/6.;
    adu2_zero = (29.-ovsc[2] +30.-(ovsc[2]+1)+31.-(ovsc[2]+2)+149.-ovsc_1[2]+150.-ovsc_1[2]+151.-ovsc_1[2])/6.;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN),
            0.5*(adu1_zero+adu2_zero), 1e-10);

    adu1_zero = (29.-ovsc[4] +30.-(ovsc[4]+1)+31.-(ovsc[4]+2)+149.-ovsc_1[4]+150.-ovsc_1[4]+151.-ovsc_1[4])/6.;
    adu2_zero = (29.-ovsc[6] +30.-(ovsc[6]+1)+31.-(ovsc[6]+2)+149.-ovsc_1[6]+150.-ovsc_1[6]+151.-ovsc_1[6])/6.;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD),
            0.5*(adu1_zero+adu2_zero), 1e-10);

    adu1_zero = (169.-ovsc[4] +170.-(ovsc[4]+1)+171.-(ovsc[4]+2)+49.-ovsc_1[4]+50.-ovsc_1[4]+51.-ovsc_1[4])/6.;
    adu2_zero = (169.-ovsc[6] +170.-(ovsc[6]+1)+171.-(ovsc[6]+2)+49.-ovsc_1[6]+50.-ovsc_1[6]+51.-ovsc_1[6])/6.;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN),
            0.5*(adu1_zero+adu2_zero), 1e-10);
    cpl_propertylist_delete(pl);

    pl = cpl_propertylist_load( zpl_intensity_flat->intensity_flat_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN), 0.0, 1e-10);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 1.0, 1e-10);
    cpl_propertylist_delete(pl);

    rerr = sph_zpl_intensity_flat_delete( zpl_intensity_flat );
    cpl_test_error(CPL_ERROR_NONE );

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete( inframes );
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat to test master frame product
 @note It is fed by the pre-processed frames  (simple collapse method + standard combining )
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_intensity_flat_run_5(void) {
    sph_zpl_intensity_flat* zpl_intensity_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 100.0;

    double zero_odd = 2500.0;
    double zero_even = 2000.0;
    double pi_odd = 3000.0;
    double pi_even = 3500.0;
    cpl_propertylist* pl = NULL;
    cpl_parameter* parameter_collapse = NULL;
    char* outfilename = NULL;

    //create zpl intensity flat parameter list
    inparams = sph_zpl_intensity_flat_create_paramlist();

    //set "zpl.intensity_flat.collapse" parameter to TRUE
    parameter_collapse = cpl_parameterlist_find( inparams, "zpl.intensity_flat.collapse");
    cpl_parameter_set_bool( parameter_collapse, 1);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set( &zexpcube, zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_IFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    rerr = cpl_frameset_insert( inframes, zexpcube );

    zpl_intensity_flat = sph_zpl_intensity_flat_new( inframes, inparams);
    cpl_test_nonnull( zpl_intensity_flat );

    rerr = sph_zpl_intensity_flat_run( zpl_intensity_flat );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    outfilename = cpl_sprintf("%s-qc-non-normalized.fits",
                              zpl_intensity_flat->intensity_flat_outfilename );

    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load( outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN), 0.0, 1e-10);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), 2750.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN), 2250.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 3250.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN), 3750.0, 1e-10);

    cpl_test_error(CPL_ERROR_NONE );
    pl = cpl_propertylist_load( zpl_intensity_flat->intensity_flat_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN), 0.0, 1e-10);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN), 1.0, 1e-10);

    //master frame product test with simple collapse method and standard combining mean
    outfilename = cpl_sprintf("%s-qc-non-normalized.fits",
                              zpl_intensity_flat->master_intensity_flat_outfilename );
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load( outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;

    cpl_test_abs(cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME),
            (2750.+2250.+3250.+3750.)/4., 1e-10 );

    rerr = sph_zpl_intensity_flat_delete( zpl_intensity_flat );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat to test master frame product
 @note It is fed by the pre-processed frames  (simple collapse method + weighted mead for quad image)
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_intensity_flat_run_6(void) {
    sph_zpl_intensity_flat* zpl_intensity_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 100.0;

    double zero_odd = 2500.0;
    double zero_even = 2000.0;
    double pi_odd = 3000.0;
    double pi_even = 3500.0;
    double rms_zero_odd, rms_zero_even, rms_pi_odd, rms_pi_even;
    double value1, value2, value3, value4;
    double mean, rms;
    cpl_propertylist* pl = NULL;
    sph_master_frame* mframe = NULL;
    cpl_parameter* parameter_collapse = NULL;
    cpl_parameter* parameter_weight = NULL;
    char* outfilename = NULL;

    //create zpl intensity flat parameter list
    inparams = sph_zpl_intensity_flat_create_paramlist();

    //set "zpl.intensity_flat.collapse" parameter to TRUE
    parameter_collapse = cpl_parameterlist_find( inparams, "zpl.intensity_flat.collapse");
    cpl_parameter_set_bool( parameter_collapse, 1);

    //set "zpl.intensity_flat.quadimage_weight_mean" parameter to TRUE
    parameter_weight = cpl_parameterlist_find( inparams, "zpl.intensity_flat.quadimage_weight_mean");
    cpl_parameter_set_bool( parameter_weight, 1);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_iff( &zexpcube, zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_IFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    zpl_intensity_flat = sph_zpl_intensity_flat_new( inframes, inparams);
    cpl_test_nonnull( zpl_intensity_flat );

    rerr = sph_zpl_intensity_flat_run( zpl_intensity_flat );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    outfilename = cpl_sprintf("%s-qc-non-normalized.fits",
                              zpl_intensity_flat->intensity_flat_outfilename );
    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load( outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN), 0.0, 1e-10);

    value1=2750.0;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), value1, 1e-10);
    value2 = (2000.+2200.+2400.+2600.+2800.+3000.)/6.;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN),
            value2, 1e-10);
    value3 = (3000.0+3300.+3600.+3900.+4200.+4500.)/6.;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD),
            value3, 1e-10);
    value4 = (3500.0+3900.+4300.+4700.+5100+5500.)/6.;
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN),
            value4, 1e-10);

    cpl_test_error(CPL_ERROR_NONE );
    pl = cpl_propertylist_load( zpl_intensity_flat->intensity_flat_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD), 0.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN), 0.0, 1e-10);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD), 1.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN), 1.0, 1e-10);

    //master frame product test with simple collapse method and standard combining mean
    rms_zero_odd = sqrt(((2500.-2750)*(2500.-2750)+(2600.-2750.)*(2600.-2750.)+
                    (2700.-2750)*(2700.-2750.)+(2800.-2750.)*(2800.-2750.)+
                    (2900.-2750.)*(2900.-2750.)+(3000.-2750.)*(3000.-2750.))/(nframes-1.5));
    // the correction of 1.5 in the line before and below
    // is an implementation in the sphere mean algorithm. See also:
    //http://en.wikipedia.org/wiki/Unbiased_estimation_of_standard_deviation

    rms_zero_even = sqrt(((2000.-2500)*(2000.-2500)+(2200.-2500.)*(2200.-2500.)+
                    (2400.-2500)*(2400.-2500.)+(2600.-2500.)*(2600.-2500.)+
                    (2800.-2500.)*(2800.-2500.)+(3000.-2500.)*(3000.-2500.))/(nframes-1.5));

    rms_pi_odd = sqrt(((3000.-value3)*(3000.-value3)+(3300.-value3)*(3300.-value3)+
                    (3600.-value3)*(3600.-value3)+(3900.-value3)*(3900.-value3)+
                    (4200.-value3)*(4200.-value3)+(4500.-value3)*(4500.-value3))/(nframes-1.5));

    rms_pi_even = sqrt(((3500.-value4)*(3500.-value4)+(3900.-value4)*(3900.-value4)+
                    (4300.-value4)*(4300.-value4)+(4700.-value4)*(4700.-value4)+
                    (5100.-value4)*(5100.-value4)+(5500.-value4)*(5500.-value4))/(nframes-1.5));

    //printf("%f\n", rms_zero_odd);
    //printf("%f\n", rms_zero_even);
    //printf("%f\n", rms_pi_odd);
    //printf("%f\n", rms_pi_even);

    mean = (value1/(rms_zero_odd*rms_zero_odd) + value2/(rms_zero_even*rms_zero_even)+ value3/(rms_pi_odd*rms_pi_odd) + value4/(rms_pi_even*rms_pi_even))/
    (1./(rms_zero_odd*rms_zero_odd) + 1./(rms_zero_even*rms_zero_even) + 1./(rms_pi_odd*rms_pi_odd) + 1./(rms_pi_even*rms_pi_even));
    rms = sqrt( 1./(rms_zero_odd*rms_zero_odd) + 1./(rms_zero_even*rms_zero_even) + 1./(rms_pi_odd*rms_pi_odd) + 1./(rms_pi_even*rms_pi_even));

    //printf("mean = %f\n", mean);
    //printf("rms = %f\n", rms);

    outfilename =
        cpl_sprintf("%s-qc-non-normalized.fits",
                    zpl_intensity_flat->master_intensity_flat_outfilename );
    cpl_test_error(CPL_ERROR_NONE );
    cpl_test_nonnull(outfilename);
    pl = cpl_propertylist_load( outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME),
            mean, 1e-3 );
    mframe = sph_master_frame_load(outfilename, 0);
    cpl_free(outfilename); outfilename = NULL;

    cpl_test_abs(cpl_image_get_mean( mframe->rmsmap), rms, 1e-3 );
    sph_master_frame_delete( mframe );

    rerr = sph_zpl_intensity_flat_delete( zpl_intensity_flat );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_intensity_flat recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("intensity_flat_test",
            cutest_init_intensity_flat_testsuite,
            cutest_clean_intensity_flat_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

#ifdef SPH_ZPL_INTENSITY_FLAT_1

     if ( NULL == sph_test_do(pSuite, "sph_intensity_flat_run_1: linear fitting method ",
     cutest_zpl_intensity_flat_run_1 ) )
     {
     return sph_test_get_error();
     }
#endif

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_intensity_flat_run_2: simple collapse method and normalization",
                    cutest_zpl_intensity_flat_run_2)) {
        return sph_test_get_error();
    }
#ifdef SPH_ZPL_INTENSITY_FLAT_345
     if ( NULL == sph_test_do(pSuite, "sph_intensity_flat_run_3: simple collapse method and normalization + noisy bias",
     cutest_zpl_intensity_flat_run_3 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_intensity_flat_run_4: simple collapse method on the data with ovsc table",
     cutest_zpl_intensity_flat_run_4 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_intensity_flat_run_5: master frame product test - simple collapse method ",
     cutest_zpl_intensity_flat_run_5 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_intensity_flat_run_6: master frame product test - quad image weighted mean ",
     cutest_zpl_intensity_flat_run_6 ) )
     {
     return sph_test_get_error();
     }
#endif

    return sph_test_end();
}

/**@}*/
