/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "cutest_zpl_master_dark.h"
#include "sph_zpl_master_dark.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_master_dark_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    //sph_init_debug();
    sph_test_nop_code();
    return CPL_ERROR_NONE;
}

static int cutest_clean_master_dark_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_master_dark
 @note It is fed only by raw frame (no master bias calibration)
 */
/*----------------------------------------------------------------------------*/

void cutest_zpl_master_dark_run_1(void) {
    sph_zpl_master_dark* zpl_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    inparams = sph_zpl_master_dark_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_DARK_PREPROC_CAM1);
    rerr = cpl_frameset_insert(inframes, zexpcube);
    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube1, nframes);
    cpl_frame_set_tag(zexpcube1, SPH_ZPL_TAG_DARK_PREPROC_CAM2);
    rerr = cpl_frameset_insert(inframes, zexpcube1);

    zpl_master_dark = sph_zpl_master_dark_new(inframes, inparams);
    cpl_test_nonnull( zpl_master_dark );

    rerr = sph_zpl_master_dark_run(zpl_master_dark);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs(
            cpl_image_get_absflux(zpl_master_dark->master_dark_quadimage_cam1->zero_image->iframe->rmsmap)/(nx*ny),
            sqrt(2./(nframes-1.5)), 1e-5);
    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam1->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ),
            170.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam1->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ),
            30.0, 1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam1->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),
            30.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam1->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),
            170., 1e-20);

    cpl_test_abs(
            cpl_image_get_absflux(zpl_master_dark->master_dark_quadimage_cam2->zero_image->iframe->rmsmap)/(nx*ny),
            sqrt(2./(nframes-1.5)), 1e-5);
    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ),
            170.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ),
            30.0, 1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),
            30.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage_cam2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),
            170., 1e-20);

    rerr = sph_zpl_master_dark_delete(zpl_master_dark);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

#if 0
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_master_dark
 @note It is fed by raw frame and master dark calibration frame
 */
/*----------------------------------------------------------------------------*/

void cutest_zpl_master_dark_run_2(void) {
    sph_zpl_master_dark* zpl_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    sph_quad_image* qi = NULL;
    sph_double_image* di0 = NULL;
    sph_double_image* dipi = NULL;
    cpl_frame* master_bias_frame = NULL;
    double iframe_value = 11.5;
    double pframe_value = 10.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 2.0;
    double dV = 10.0;
    double dS = 0.0;

    inparams = sph_zpl_master_dark_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_plus( &zexpcube, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_DARK_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    rerr = cpl_frameset_insert( inframes, zexpcube );

    //create master bias frame (quad image from double images)
    //zero double image with noise
    di0 = sph_test_zpl_simulator_create_double_image_noise_uniform( nx, ny, iframe_value, pframe_value, sigma_iframe, sigma_pframe );
    cpl_test_nonnull( di0  );

    //pi double image with noise
    dipi = sph_test_zpl_simulator_create_double_image_noise_uniform( nx, ny, iframe_value+dV, pframe_value+dV,
            sigma_iframe+dS, sigma_pframe+dS );
    cpl_test_nonnull( di0  );

    qi = sph_quad_image_new_from_double_images( di0, dipi );
    cpl_test_nonnull( qi  );

    sph_quad_image_save( qi, "test_sph_quad_image_mask_sigma.fits.tmp", NULL );
    sph_quad_image_delete( qi );
    master_bias_frame = cpl_frame_new();
    cpl_frame_set_filename( master_bias_frame, "test_sph_quad_image_mask_sigma.fits.tmp");
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB);

    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    zpl_master_dark = sph_zpl_master_dark_new( inframes, inparams );
    cpl_test_nonnull( zpl_master_dark  );

    //printf("raw frame: %s\n", cpl_frame_get_filename(cpl_frameset_get_first( zpl_master_dark->rawframes )));
    //printf("rawframes #: %d\n", cpl_frameset_get_size( zpl_master_dark->rawframes ));

    rerr = sph_zpl_master_dark_run( zpl_master_dark );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ), 158.5, 5*sigma_iframe/sqrt(nx*ny) );

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ), 8.5, 5*(sigma_iframe+dS)/sqrt(nx*ny) );
    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ), 149.1, 5*(sigma_pframe+dS)/sqrt(nx*ny));

    rerr = sph_zpl_master_dark_delete( zpl_master_dark );
    cpl_test_error(CPL_ERROR_NONE );

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete( inframes );
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_master_dark
 @note It is fed by raw pre-processed frame with overscan table
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_master_dark_run_3(void) {
    sph_zpl_master_dark* zpl_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frame* zexpcube_plus = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nz = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    double ovsc[8] = {5.0, 1.0, 6.0, 2.0, 7.0, 3.0, 8.0, 4.0};
    //double 					ovsc[8]		= {1.0, 1.0, 1.0, 2.0, 0.0, 3.0, 0.0, 4.0};

    int zero_odd_1 = 49;
    int zero_even_1 = 149;
    int pi_odd_1 = 149;
    int pi_even_1 = 49;
    double ovsc_1[8] = {15.0, 1.0, 16.0, 2.0, 17.0, 3.0, 18.0, 4.0};
    //double 					ovsc_1[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};
    double adu1_zero = 0;
    double adu2_zero = 0;

    inparams = sph_zpl_master_dark_create_paramlist();
    inframes = cpl_frameset_new();

    // zpl frame cube with overscan table
    zexpcube_plus = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
            "test_master_dark_zplexp_ovsc_cube", zero_odd, zero_even, pi_odd, pi_even, ovsc,
            1, 1);
    cpl_frame_set_tag( zexpcube_plus, SPH_ZPL_TAG_DARK_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_plus );

    // zpl frame cube with overscan table
    zexpcube_minus = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
            "test_master_dark_zplexp_ovsc_cube_1", zero_odd_1, zero_even_1, pi_odd_1,
            pi_even_1, ovsc_1, 1, 0 );
    cpl_frame_set_tag( zexpcube_minus, SPH_ZPL_TAG_DARK_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_minus );

    zpl_master_dark = sph_zpl_master_dark_new( inframes, inparams );
    cpl_test_nonnull( zpl_master_dark  );

    rerr = sph_zpl_master_dark_run( zpl_master_dark );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    adu1_zero = (169.-ovsc[0] +170.-(ovsc[0]+1)+171.-(ovsc[0]+2)+49.-ovsc_1[0]+50.-ovsc_1[0]+51.-ovsc_1[0])/6.;
    adu2_zero = (169.-ovsc[2] +170.-(ovsc[2]+1)+171.-(ovsc[2]+2)+49.-ovsc_1[2]+50.-ovsc_1[2]+51.-ovsc_1[2])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ), 0.5*(adu1_zero+adu2_zero), 1e-10);

    adu1_zero = (29.-ovsc[0] +30.-(ovsc[0]+1)+31.-(ovsc[0]+2)+149.-ovsc_1[0]+150.-ovsc_1[0]+151.-ovsc_1[0])/6.;
    adu2_zero = (29.-ovsc[2] +30.-(ovsc[2]+1)+31.-(ovsc[2]+2)+149.-ovsc_1[2]+150.-ovsc_1[2]+151.-ovsc_1[2])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ), 0.5*(adu1_zero+adu2_zero), 1e-10 );

    adu1_zero = (29.-ovsc[4] +30.-(ovsc[4]+1)+31.-(ovsc[4]+2)+149.-ovsc_1[4]+150.-ovsc_1[4]+151.-ovsc_1[4])/6.;
    adu2_zero = (29.-ovsc[6] +30.-(ovsc[6]+1)+31.-(ovsc[6]+2)+149.-ovsc_1[6]+150.-ovsc_1[6]+151.-ovsc_1[6])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ), 0.5*(adu1_zero+adu2_zero) , 1e-10 );

    adu1_zero = (169.-ovsc[4] +170.-(ovsc[4]+1)+171.-(ovsc[4]+2)+49.-ovsc_1[4]+50.-ovsc_1[4]+51.-ovsc_1[4])/6.;
    adu2_zero = (169.-ovsc[6] +170.-(ovsc[6]+1)+171.-(ovsc[6]+2)+49.-ovsc_1[6]+50.-ovsc_1[6]+51.-ovsc_1[6])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ), 0.5*(adu1_zero+adu2_zero), 1e-10 );

    rerr = sph_zpl_master_dark_delete( zpl_master_dark );
    cpl_test_error(CPL_ERROR_NONE );

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete( inframes );
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_master_dark
 @note It is fed by raw pre-processed frame with overscan table and bias calibration
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_master_dark_run_4(void) {
    sph_zpl_master_dark* zpl_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frame* zexpcube_plus = NULL;
    cpl_frame* master_bias_frame = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nz = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    double ovsc[8] = {5.0, 1.0, 6.0, 2.0, 7.0, 3.0, 8.0, 4.0};
    //double 					ovsc[8]		= {1.0, 1.0, 1.0, 2.0, 0.0, 3.0, 0.0, 4.0};

    int zero_odd_1 = 49;
    int zero_even_1 = 149;
    int pi_odd_1 = 149;
    int pi_even_1 = 49;
    double ovsc_1[8] = {15.0, 1.0, 16.0, 2.0, 17.0, 3.0, 18.0, 4.0};
    //double 					ovsc_1[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};
    double adu1_zero = 0;
    double adu2_zero = 0;

    double zero_odd_bias, zero_even_bias, pi_odd_bias, pi_even_bias;

    inparams = sph_zpl_master_dark_create_paramlist();
    inframes = cpl_frameset_new();

    // zpl frame cube with overscan table
    zexpcube_plus = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
            "test_master_dark_zplexp_ovsc_cube", zero_odd, zero_even, pi_odd, pi_even, ovsc,
            1, 1);
    cpl_frame_set_tag( zexpcube_plus, SPH_ZPL_TAG_DARK_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_plus );

    // zpl frame cube with overscan table
    zexpcube_minus = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
            "test_master_dark_zplexp_ovsc_cube_1", zero_odd_1, zero_even_1, pi_odd_1,
            pi_even_1, ovsc_1, 1, 0 );
    cpl_frame_set_tag( zexpcube_minus, SPH_ZPL_TAG_DARK_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_minus );

    // simulate bias quad image
    zero_odd_bias = 10; zero_even_bias = 11; pi_odd_bias = 12; pi_even_bias = 14;
    master_bias_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_bias_calib.fits.tmp", nx, ny,
            zero_odd_bias, zero_even_bias, pi_odd_bias, pi_even_bias);
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB );
    cpl_frame_set_group( master_bias_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    zpl_master_dark = sph_zpl_master_dark_new( inframes, inparams );
    cpl_test_nonnull( zpl_master_dark  );

    rerr = sph_zpl_master_dark_run( zpl_master_dark );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    adu1_zero = (169.-ovsc[0] +170.-(ovsc[0]+1)+171.-(ovsc[0]+2)+49.-ovsc_1[0]+50.-ovsc_1[0]+51.-ovsc_1[0])/6.;
    adu2_zero = (169.-ovsc[2] +170.-(ovsc[2]+1)+171.-(ovsc[2]+2)+49.-ovsc_1[2]+50.-ovsc_1[2]+51.-ovsc_1[2])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ), 0.5*(adu1_zero+adu2_zero) -zero_odd_bias, 1e-10);

    adu1_zero = (29.-ovsc[0] +30.-(ovsc[0]+1)+31.-(ovsc[0]+2)+149.-ovsc_1[0]+150.-ovsc_1[0]+151.-ovsc_1[0])/6.;
    adu2_zero = (29.-ovsc[2] +30.-(ovsc[2]+1)+31.-(ovsc[2]+2)+149.-ovsc_1[2]+150.-ovsc_1[2]+151.-ovsc_1[2])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ), 0.5*(adu1_zero+adu2_zero)-zero_even_bias, 1e-10 );

    adu1_zero = (29.-ovsc[4] +30.-(ovsc[4]+1)+31.-(ovsc[4]+2)+149.-ovsc_1[4]+150.-ovsc_1[4]+151.-ovsc_1[4])/6.;
    adu2_zero = (29.-ovsc[6] +30.-(ovsc[6]+1)+31.-(ovsc[6]+2)+149.-ovsc_1[6]+150.-ovsc_1[6]+151.-ovsc_1[6])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ), 0.5*(adu1_zero+adu2_zero)-pi_odd_bias, 1e-10 );

    adu1_zero = (169.-ovsc[4] +170.-(ovsc[4]+1)+171.-(ovsc[4]+2)+49.-ovsc_1[4]+50.-ovsc_1[4]+51.-ovsc_1[4])/6.;
    adu2_zero = (169.-ovsc[6] +170.-(ovsc[6]+1)+171.-(ovsc[6]+2)+49.-ovsc_1[6]+50.-ovsc_1[6]+51.-ovsc_1[6])/6.;

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ), 0.5*(adu1_zero+adu2_zero)-pi_even_bias, 1e-10 );

    rerr = sph_zpl_master_dark_delete( zpl_master_dark );
    cpl_test_error(CPL_ERROR_NONE );

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete( inframes );
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_master_dark
 @note It is fed by raw frame and master dark calibration frame
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_master_dark_run_5(void) {
    sph_zpl_master_dark* zpl_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    sph_quad_image* qi = NULL;
    sph_double_image* di0 = NULL;
    sph_double_image* dipi = NULL;
    cpl_frame* master_bias_frame = NULL;
    double iframe_value = 11.5;
    double pframe_value = 10.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 2.0;
    double dV = 10.0;
    double dS = 10.0;
    double sigma_clip = 2.0;
    double nbad = 0.0;
    cpl_parameter* par_sigma_clip = NULL;
    int badcheck = 0;

    //set up parameters
    inparams = sph_zpl_master_dark_create_paramlist();
    //set sigma clipping parameter (default is 0, which means -- no sigma clipping)
    par_sigma_clip = cpl_parameterlist_find( inparams, "zpl.master_dark.sigma_clip");
    cpl_parameter_set_double( par_sigma_clip, sigma_clip);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_plus( &zexpcube, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_DARK_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    rerr = cpl_frameset_insert( inframes, zexpcube );

    //create master bias frame (quad image from double images)
    //zero double image with noise
    di0 = sph_test_zpl_simulator_create_double_image_noise_uniform( nx, ny, iframe_value, pframe_value, sigma_iframe, sigma_pframe );
    cpl_test_nonnull( di0  );

    //pi double image with noise
    dipi = sph_test_zpl_simulator_create_double_image_noise_uniform( nx, ny, iframe_value+dV, pframe_value+dV,
            sigma_iframe+dS, sigma_pframe+dS );
    cpl_test_nonnull( di0  );

    qi = sph_quad_image_new_from_double_images( di0, dipi );
    cpl_test_nonnull( qi  );

    sph_quad_image_save( qi, "test_sph_quad_image_mask_sigma.fits.tmp", NULL );
    sph_quad_image_delete( qi );
    master_bias_frame = cpl_frame_new();
    cpl_frame_set_filename( master_bias_frame, "test_sph_quad_image_mask_sigma.fits.tmp");
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB);

    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    zpl_master_dark = sph_zpl_master_dark_new( inframes, inparams );
    cpl_test_nonnull( zpl_master_dark  );

    //printf("raw frame: %s\n", cpl_frame_get_filename(cpl_frameset_get_first( zpl_master_dark->rawframes )));
    //printf("rawframes #: %d\n", cpl_frameset_get_size( zpl_master_dark->rawframes ));

    rerr = sph_zpl_master_dark_run( zpl_master_dark );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ), 158.5, 5*sigma_iframe/sqrt(nx*ny) );
    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ), 8.5, 5*(sigma_iframe+dS)/sqrt(nx*ny) );
    cpl_test_abs( cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ), 149.1, 5*(sigma_pframe+dS)/sqrt(nx*ny));

    //verify that number of bad pixels is not 0.
    nbad = cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                                        SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_lt(0 , fabs(nbad));

    nbad = cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                                        SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_lt(0 , fabs(nbad));

    nbad = cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                                        SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_lt(0 , fabs(nbad));
    nbad = cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                                        SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_lt(0 , fabs(nbad));

    badcheck=0;
    if ( cpl_image_get_flux(zpl_master_dark->master_dark_quadimage->zero_image->iframe->image) -
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ) > 0.0) {
        badcheck = 1;
    }
    cpl_test_eq(badcheck, 1);

    badcheck = 0;
    if ( cpl_image_get_flux(zpl_master_dark->master_dark_quadimage->zero_image->pframe->image) -
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ) > 0.0) {
        badcheck = 1;
    }
    cpl_test_eq(badcheck, 1);

    badcheck = 0;
    if ( cpl_image_get_flux(zpl_master_dark->master_dark_quadimage->pi_image->iframe->image) -
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ) > 0.0) {
        badcheck = 1;
    }
    cpl_test_eq(badcheck, 1);

    badcheck = 0;
    if ( cpl_image_get_flux(zpl_master_dark->master_dark_quadimage->pi_image->pframe->image) -
            cpl_propertylist_get_double( zpl_master_dark->master_dark_quadimage->qclist,
                    SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ) > 0.0) {
        badcheck = 1;
    }
    cpl_test_eq(badcheck, 1);

    rerr = sph_zpl_master_dark_delete( zpl_master_dark );
    cpl_test_error(CPL_ERROR_NONE );

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete( inframes );
    cpl_parameterlist_delete(inparams);

    return;
}

#endif

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("master_dark_test", cutest_init_master_dark_testsuite,
            cutest_clean_master_dark_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite,
                    "sph_master_dark_run_1: pre-processed frames",
                    cutest_zpl_master_dark_run_1)) {
        return sph_test_get_error();
    }

    /*
     if ( NULL == sph_test_do(pSuite, "sph_master_dark_run_2: pre-processed frames and bias calibration frame",
     cutest_zpl_master_dark_run_2 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_master_dark_run_3: pre-processed frames with overscan table",
     cutest_zpl_master_dark_run_3 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_master_dark_run_4: pre-processed frames with overscan table and bias calibration frame",
     cutest_zpl_master_dark_run_4 ) )
     {
     return sph_test_get_error();
     }


     if ( NULL == sph_test_do(pSuite, "sph_master_dark_run_5: check means (excluding badpixels) & badpixels itself",
     cutest_zpl_master_dark_run_5 ) )
     {
     return sph_test_get_error();
     }
     */

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
