/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_distortion_model.h"
#include "sph_distortion_map.h"
#include "sph_utils.h"
#include "sph_fitting.h"
#include "sph_test_image_tools.h"
#include "sph_time.h"
#include "sph_common_keywords.h"

#include <math.h>
#include <stdio.h>

static sph_distortion_model*
cutest_sph_distortion_model_fixture(void);

static sph_distortion_model*
cutest_sph_distortion_model_fixture2(void);

cpl_image*
cutest_sph_distortion_model_fixture_tough(int nx, int ny,
        int points_per_side, double fwhm,
        sph_distortion_model** distmodel_out,
        sph_point_pattern* pp_ideal);

static sph_distortion_model*
cutest_sph_distortion_model_fixture2inv(void);
static
void
cutest_sph_polyonmial_distortion_model_from_grid(void);

static
void
cutest_sph_polyonmial_distortion_model_from_grid_stable(void);

static
void
cutest_sph_polyonmial_distortion_model_from_grid_no_dist(void);

static
int
cutest_sph_polyonmial_distortion_model_saveit(void);

static
int cutest_init_testsuite(void);

static
int cutest_clean_testsuite(void);

static
void cutest_sph_polyonmial_distortion_model_test_fit_distortion_map_temp(void);

static
void cutest_sph_polyonmial_distortion_model_test_save(void);

static
void cutest_sph_polyonmial_distortion_model_test_load(void);

static
void cutest_sph_polyonmial_distortion_model_apply_timing(void);

static
void cutest_sph_polyonmial_distortion_model_apply_poly(void);

static
void cutest_sph_polyonmial_distortion_model_apply(void);

static
void cutest_sph_polyonmial_distortion_model_apply_highorder(void);
static
void cutest_sph_polyonmial_distortion_model_check_cpl_bug(void);
static
void cutest_sph_polyonmial_distortion_model_check_cpl_leakage(void);

static
void cutest_sph_polyonmial_distortion_model_apply_tough(void);
static
void cutest_sph_polyonmial_distortion_model_apply_tough_im(void);
static
void cutest_sph_polyonmial_distortion_cpl_warp_test(void);

static sph_point_pattern*
sph_distortion_model_apply_pp2(sph_distortion_model*,
                                          sph_point_pattern*);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_test_fit_distortion_map_temp(void) {
    cpl_polynomial* polyx = NULL;
    cpl_polynomial* polyy = NULL;
    sph_distortion_model* model_dist = NULL;
    sph_point_pattern* pp_dist = NULL;
    sph_distortion_model* model_no_dist = NULL;
    sph_point_pattern* pp_no_dist = NULL;
    sph_distortion_map* distmap = NULL;
    sph_distortion_map* newmap = NULL;
    cpl_size pows[2] = { 2, 0 };
    int v;

    cpl_error_reset();

    polyx = cpl_polynomial_new(2);
    polyy = cpl_polynomial_new(2);

    model_no_dist = sph_distortion_model_new(polyx, polyy);
    pp_no_dist = sph_distortion_model_create_point_pattern(
            model_no_dist, 256, 256, 10);

    pows[0] = 2;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, 0.00001);
    pows[0] = 0;
    pows[1] = 2;
    cpl_polynomial_set_coeff(polyy, pows, 0.00001);

    model_dist = sph_distortion_model_new(polyx, polyy);
    pp_dist = sph_distortion_model_create_point_pattern(model_dist,
            256, 256, 10);

    distmap = sph_distortion_map_new(pp_no_dist, pp_dist);
    sph_distortion_map_delete_multi(distmap);
    newmap = sph_distortion_model_fit_distortion_map(model_no_dist,
            distmap, 2, NULL, NULL);
    cpl_test_eq(sph_distortion_map_get_size(newmap), 100);
    cpl_test_eq(sph_distortion_map_get_size(distmap), 100);

    for (v = 0; v < sph_distortion_map_get_size(distmap); ++v) {
            sph_distortion_vector dv;
            const cpl_error_code mycode =
                sph_distortion_map_get_one(distmap, v, &dv);
            cpl_test_eq_error(mycode, CPL_ERROR_NONE);

        cpl_test( fabs( dv.dx)>0.001);
        cpl_test( fabs( dv.dy)>0.001);
        cpl_test( fabs( dv.dx)<1.0);
        cpl_test( fabs( dv.dy)<1.0);
    }
    for (v = 0; v < sph_distortion_map_get_size(newmap); ++v) {
            sph_distortion_vector dv;
            const cpl_error_code mycode =
                sph_distortion_map_get_one(newmap, v, &dv);
            cpl_test_eq_error(mycode, CPL_ERROR_NONE);

        cpl_test( fabs( dv.dx)<0.0001);
        cpl_test( fabs( dv.dy)<0.0001);
    }

    sph_distortion_model_delete(model_dist);
    sph_distortion_model_delete(model_no_dist);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    sph_point_pattern_delete(pp_dist);
    sph_point_pattern_delete(pp_no_dist);
    sph_distortion_map_delete(distmap);
    sph_distortion_map_delete(newmap);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_from_grid(void) {
    int nx = 512;
    int ny = 512;
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    sph_distortion_model* model_dist = NULL;
    sph_distortion_model* result = NULL;
    sph_point_pattern* pp_dist = NULL;
    sph_distortion_model* model_no_dist =
            sph_distortion_model_new(polyx, polyy);
    sph_point_pattern* pp_no_dist =
            sph_distortion_model_create_point_pattern(model_no_dist,
                    nx, ny, 9);
    sph_distortion_map* distmap = NULL;
    sph_distortion_map* newmap = NULL;
    cpl_image* pimage = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* dedistort_image = NULL;
    cpl_image* dx = NULL;
    cpl_image* dy = NULL;
    sph_master_frame* mframe = NULL;
    cpl_table       * dist_table = NULL;
    cpl_size pows[2] = { 2, 0 };
    double avg_centx, avg_centy;
    cpl_error_code code;

    // Setup....
    pows[0] = 1;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, 0.01);
    pows[0] = 0;
    pows[1] = 2;
    cpl_polynomial_set_coeff(polyy, pows, -0.0001);

    sph_point_pattern_centralise_wrt_central_point(pp_no_dist);
    model_dist = sph_distortion_model_new(polyx, polyy);

    sph_polynomial_distortion_set_offset(model_dist, (nx + 1.0) / 2.0,
            (ny + 1.0) / 2.0);

    pp_dist = sph_distortion_model_create_point_pattern(model_dist,
            nx, ny, 9);

    sph_point_pattern_add_offset(pp_dist, model_dist->offx, model_dist->offy);

    pimage = sph_point_pattern_create_image(pp_dist, nx, ny, 3.0);

    mframe = sph_master_frame_new_from_cpl_image(pimage);
    // Exercise...
    result = sph_distortion_model_from_grid_image(mframe, pp_no_dist,
                                                             3,
                                                             3.0,
                                                             5.0, NULL, 0.0, 0.0,
                                                             NULL, NULL,
                                                             &avg_centx, &avg_centy,
                                                             NULL, &dist_table);

    cpl_table_delete(dist_table);

    // Verification ...
    cpl_test_nonnull( result );

    dx = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    dy = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    code = sph_distortion_model_fill_image_xy(result, dx, dy);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_lt(2.0, fabs(cpl_image_get_max(dx)));
    cpl_test_lt(2.0, fabs(cpl_image_get_max(dy)));
    cpl_image_save(dx, "dx.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dy, "dy.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(dx);
    cpl_image_delete(dy);
    sph_master_frame_delete(mframe);

    pl = sph_distortion_model_get_proplist(result);
    cpl_image_save(pimage, "original.fits", CPL_TYPE_UNSPECIFIED, pl,
            CPL_IO_CREATE);
    cpl_propertylist_delete(pl);
    pl = NULL;
    pows[0] = 1;
    pows[1] = 0;
    cpl_test_abs( cpl_polynomial_get_coeff(result->polyx, pows),
            -0.01, 0.001);
    pows[0] = 0;
    pows[1] = 2;
    cpl_test_abs( cpl_polynomial_get_coeff(result->polyy, pows),
            0.0001, 0.00003);

    dedistort_image = sph_distortion_model_apply(result, pimage,
                                                 CPL_KERNEL_DEFAULT,
                                                 CPL_KERNEL_DEF_WIDTH,
                                                 NULL, NULL, NULL);

    mframe = sph_master_frame_new_from_cpl_image(dedistort_image);
    cpl_test_nonnull( dedistort_image );

    sph_distortion_model_delete(result);
    result = sph_distortion_model_from_grid_image(mframe, pp_no_dist,
                                                             3,
                                                             5.0,
                                                             1.0, NULL, 0.0, 0.0,
                                                             NULL, NULL,
                                                             &avg_centx, &avg_centy,
                                                             NULL, &dist_table);

    cpl_table_delete(dist_table);

    cpl_test_nonnull( result );
    pl = sph_distortion_model_get_proplist(result);
    cpl_image_save(dedistort_image, "dedistort.fits", CPL_TYPE_UNSPECIFIED, pl,
            CPL_IO_CREATE);

    cpl_propertylist_delete(pl);
    pl = NULL;

    dx = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    dy = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    code = sph_distortion_model_fill_image_xy(result, dx, dy);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_max(dx), 0.0, 0.2);
    cpl_test_abs( cpl_image_get_min(dx), 0.0, 0.2);
    cpl_test_abs( cpl_image_get_max(dy), 0.0, 0.5);
    cpl_test_abs( cpl_image_get_min(dy), 0.0, 0.5);
    cpl_image_save(dx, "dx0.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dy, "dy0.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(dx);
    cpl_image_delete(dy);
    sph_distortion_model_delete(model_dist);
    sph_distortion_model_delete(model_no_dist);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    sph_point_pattern_delete(pp_dist);
    sph_point_pattern_delete(pp_no_dist);
    sph_distortion_map_delete(distmap);
    sph_distortion_map_delete(newmap);
    cpl_image_delete(dedistort_image);
    cpl_image_delete(pimage);
    sph_master_frame_delete(mframe);
    sph_distortion_model_delete(result);
    cpl_error_reset();
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_from_grid_stable(void) {
    int nx = 512;
    int ny = 512;
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    sph_distortion_model* model_dist = NULL;
    sph_distortion_model* result = NULL;
    sph_point_pattern* pp_dist = NULL;
    sph_distortion_model* model_no_dist =
            sph_distortion_model_new(polyx, polyy);
    sph_point_pattern* pp_no_dist =
            sph_distortion_model_create_point_pattern(model_no_dist,
                    nx, ny, 9);
    sph_distortion_map* distmap = NULL;
    sph_distortion_map* newmap = NULL;
    cpl_image* pimage = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* dedistort_image = NULL;
    cpl_image* dx = NULL;
    cpl_image* dy = NULL;
    sph_master_frame* mframe = NULL;
    cpl_table       * dist_table = NULL;
    cpl_size pows[2] = { 2, 0 };
    double avg_centx, avg_centy;
    cpl_error_code code;

    // Setup....
    pows[0] = 1;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, 0.01);
    pows[0] = 0;
    pows[1] = 2;
    cpl_polynomial_set_coeff(polyy, pows, -0.0001);

    sph_point_pattern_centralise_wrt_central_point(pp_no_dist);
    model_dist = sph_distortion_model_new(polyx, polyy);

    sph_polynomial_distortion_set_offset(model_dist, (nx + 1.0) / 2.0,
            (ny + 1.0) / 2.0);

    pp_dist = sph_distortion_model_create_point_pattern(model_dist,
            nx, ny, 9);

    sph_point_pattern_add_offset(pp_dist, model_dist->offx, model_dist->offy);

    pimage = sph_point_pattern_create_image(pp_dist, nx, ny, 3.0);

    mframe = sph_master_frame_new_from_cpl_image(pimage);
    // Exercise...
    result = sph_distortion_model_from_grid_image(mframe, pp_no_dist,
                                                             3,
                                                             3.0,
                                                             5.0, NULL, 0.0, 0.0,
                                                             NULL, NULL,
                                                             &avg_centx, &avg_centy,
                                                             NULL, &dist_table);

    cpl_table_delete(dist_table);

    // Verification ...
    cpl_test_nonnull( result );

    dx = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    dy = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    code = sph_distortion_model_fill_image_xy(result, dx, dy);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_lt(2.0, fabs(cpl_image_get_max(dx)));
    cpl_test_lt(2.0, fabs(cpl_image_get_max(dy)));
    cpl_image_save(dx, "dx.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dy, "dy.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(dx);
    cpl_image_delete(dy);
    sph_master_frame_delete(mframe);

    pl = sph_distortion_model_get_proplist(result);
    cpl_image_save(pimage, "original.fits", CPL_TYPE_UNSPECIFIED, pl,
            CPL_IO_CREATE);
    cpl_propertylist_delete(pl);
    pl = NULL;
    pows[0] = 1;
    pows[1] = 0;
    cpl_test_abs( cpl_polynomial_get_coeff(result->polyx, pows),
            -0.01, 0.001);
    pows[0] = 0;
    pows[1] = 2;
    cpl_test_abs( cpl_polynomial_get_coeff(result->polyy, pows),
            0.0001, 0.00003);

    dedistort_image = sph_distortion_model_apply(result, pimage,
                                                 CPL_KERNEL_DEFAULT,
                                                 CPL_KERNEL_DEF_WIDTH,
                                                 NULL, NULL, NULL);

    mframe = sph_master_frame_new_from_cpl_image(dedistort_image);
    cpl_test_nonnull( dedistort_image );

    sph_distortion_model_delete(result);
    result = sph_distortion_model_from_grid_image(mframe, pp_no_dist,
                                                             3,
                                                             5.0,
                                                             1.0, NULL, 0.0, 0.0,
                                                             NULL, NULL,
                                                             &avg_centx, &avg_centy,
                                                             NULL, &dist_table);

    cpl_table_delete(dist_table);

    cpl_test_nonnull( result );
    pl = sph_distortion_model_get_proplist(result);
    cpl_image_save(dedistort_image, "dedistort.fits", CPL_TYPE_UNSPECIFIED, pl,
            CPL_IO_CREATE);

    cpl_propertylist_delete(pl);
    pl = NULL;

    dx = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    dy = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    code = sph_distortion_model_fill_image_xy(result, dx, dy);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_max(dx), 0.0, 0.2);
    cpl_test_abs( cpl_image_get_min(dx), 0.0, 0.2);
    cpl_test_abs( cpl_image_get_max(dy), 0.0, 0.5);
    cpl_test_abs( cpl_image_get_min(dy), 0.0, 0.5);
    cpl_image_save(dx, "dx0.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dy, "dy0.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(dx);
    cpl_image_delete(dy);
    sph_distortion_model_delete(model_dist);
    sph_distortion_model_delete(model_no_dist);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    sph_point_pattern_delete(pp_dist);
    sph_point_pattern_delete(pp_no_dist);
    sph_distortion_map_delete(distmap);
    sph_distortion_map_delete(newmap);
    cpl_image_delete(dedistort_image);
    cpl_image_delete(pimage);
    sph_master_frame_delete(mframe);
    sph_distortion_model_delete(result);
    cpl_error_reset();
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_from_grid_no_dist(void) {
    int nx = 1024;
    int ny = 1024;
    int np = 21;
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    sph_distortion_model* result = NULL;
    sph_distortion_model* model_no_dist =
            sph_distortion_model_new(polyx, polyy);
    sph_point_pattern* pp_no_dist =
            sph_distortion_model_create_point_pattern(model_no_dist,
                    nx, ny, np);
    sph_distortion_map* distmap = NULL;
    sph_distortion_map* newmap = NULL;
    cpl_image* pimage = NULL;
    cpl_image* dximage = NULL;
    cpl_image* dyimage = NULL;
    sph_master_frame* mframe = NULL;
    cpl_table       * dist_table = NULL;
    cpl_size pows[2] = { 2, 0 };
    double avg_centx, avg_centy;
    cpl_error_code code;

    // Setup....
    pows[0] = 2;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, 0.000);
    pows[0] = 0;
    pows[1] = 2;
    cpl_polynomial_set_coeff(polyy, pows, 0.000);

    pimage = sph_point_pattern_create_image(pp_no_dist, nx, ny, 6.0);
    mframe = sph_master_frame_new_from_cpl_image(pimage);
    sph_point_pattern_centralise_wrt_central_point(pp_no_dist);

    // Exercise...
    result = sph_distortion_model_from_grid_image(mframe, pp_no_dist,
                                                             1,
                                                             3.0,
                                                             1.0, NULL, 0.0, 0.0,
                                                             NULL, NULL,
                                                             &avg_centx, &avg_centy,
                                                             NULL, &dist_table);

    cpl_table_delete(dist_table);

    // Verification ...
    cpl_test_nonnull( result );

    pows[0] = 2;
    pows[1] = 0;
    cpl_test_abs( cpl_polynomial_get_coeff(result->polyx, pows),
            0.00000, 0.000002);
    pows[0] = 0;
    pows[1] = 2;
    cpl_test_abs( cpl_polynomial_get_coeff(result->polyy, pows),
            0.00000, 0.000002);

    cpl_image_delete(pimage);
    pimage = NULL;

    dximage = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    dyimage = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    code = sph_distortion_model_fill_image_xy(result, dximage, dyimage);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_absflux(dximage), 0.0, 0.01);
    cpl_test_abs( cpl_image_get_absflux(dyimage), 0.0, 0.01);

    sph_distortion_model_delete(model_no_dist);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    sph_point_pattern_delete(pp_no_dist);
    sph_distortion_map_delete(distmap);
    sph_distortion_map_delete(newmap);
    cpl_image_delete(dximage);
    cpl_image_delete(dyimage);
    cpl_image_delete(pimage);
    sph_distortion_model_delete(result);
    sph_master_frame_delete(mframe);
    cpl_error_reset();
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_test_save(void) {
    FILE* f = NULL;
    sph_double_image* double_im = NULL;
    cpl_propertylist* proplist = NULL;
    double val = 0.0;
    cutest_sph_polyonmial_distortion_model_saveit();
    f = fopen("test_quadrdistortion.fits", "r");
    cpl_test_nonnull( f );
    fclose(f);
    double_im = sph_double_image_load("test_quadrdistortion.fits", 0);
    cpl_test_nonnull( double_im );
    proplist = cpl_propertylist_load("test_quadrdistortion.fits", 0);
    val = cpl_propertylist_get_double(proplist, "ESO DRS DIST X COEFF 2_0");
    cpl_test_abs(val, 0.0004, 0.000001);
    val = cpl_propertylist_get_double(proplist, "ESO DRS DIST Y COEFF 0_2");
    cpl_test_abs(val, 0.0004, 0.000001);
    val = cpl_propertylist_get_double(proplist, "ESO DRS DIST X COEFF 2_2");
    cpl_test_abs(val, 1.0e-8, 1.0e-10);
    val = cpl_propertylist_get_double(proplist, "ESO DRS DIST Y COEFF 2_2");
    cpl_test_abs(val, 1.0e-8, 1.0e-10);
    sph_double_image_delete(double_im);
    cpl_propertylist_delete(proplist);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_test_load(void) {
    FILE* f = NULL;
    double val = 0.0;
    cpl_size pows[2] = { 2, 0 };
    sph_distortion_model* model = NULL;
    cutest_sph_polyonmial_distortion_model_saveit();
    f = fopen("test_quadrdistortion.fits", "r");
    cpl_test_nonnull( f );
    fclose(f);
    model = sph_distortion_model_load("test_quadrdistortion.fits",
                                                 0,
                                                 SPH_COMMON_KEYWORD_DISTMAP_COEFFX,
                                                 SPH_COMMON_KEYWORD_DISTMAP_COEFFY);

    cpl_test_nonnull( model );
    val = cpl_polynomial_get_coeff(
            sph_distortion_model_get_poly_x(model), pows);
    cpl_test_abs(val, 0.0004, 0.000001);
    pows[0] = 0;
    pows[1] = 2;
    val = cpl_polynomial_get_coeff(
            sph_distortion_model_get_poly_y(model), pows);
    cpl_test_abs(val, 0.0004, 0.000001);
    pows[0] = 2;
    pows[1] = 2;
    val = cpl_polynomial_get_coeff(
            sph_distortion_model_get_poly_x(model), pows);
    cpl_test_abs(val, 1.0e-8, 1.0e-10);
    pows[0] = 2;
    pows[1] = 2;
    val = cpl_polynomial_get_coeff(
            sph_distortion_model_get_poly_y(model), pows);
    cpl_test_abs(val, 1.0e-8, 1.0e-10);
    sph_distortion_model_delete(model);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_apply_timing(void) {
    cpl_image* image = NULL;
    cpl_image* distim = NULL;
    double xcen = 0.0;
    double ycen = 0.0;
    double sizx = 0.0;
    double sizy = 0.0;
    double rms = 0.0;
    double redchi = 0.0;
    cpl_image* imerr = NULL;
    cpl_array* params = cpl_array_new(7, CPL_TYPE_DOUBLE);
    cpl_array* paramserr = cpl_array_new(7, CPL_TYPE_DOUBLE);
    int nx = 256;
    int ny = 256;

    sph_distortion_model* model =
            cutest_sph_distortion_model_fixture2();
    sph_distortion_model* modelinv =
            cutest_sph_distortion_model_fixture2inv();
    cpl_test_nonnull( model );
    image = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    sph_test_image_tools_add_gauss(image, nx / 2, ny / 2, 2.0, 2.0);

    imerr = sph_test_image_tools_create_flat_image_double(nx, ny, 0.1);

    cpl_fit_image_gaussian(image, NULL, nx / 2, ny / 2, 55, 55, params, NULL,
            NULL, &rms, NULL, NULL, NULL, NULL, NULL, NULL);

    SPH_RAISE_CPL;
    xcen = cpl_array_get_double(params, 3, NULL);
    ycen = cpl_array_get_double(params, 4, NULL);
    sizx = cpl_array_get_double(params, 5, NULL);
    sizy = cpl_array_get_double(params, 6, NULL);

    cpl_test_abs(xcen, nx/2, 0.1);
    cpl_test_abs(ycen, ny/2, 0.1);
    cpl_test_abs(sizx, 2.0, 0.1);
    cpl_test_abs(sizy, 2.0, 0.1);

    sph_time_start_timer(1);
    distim = sph_distortion_model_apply(model, image,
                                        CPL_KERNEL_DEFAULT, 10.0,
                                        NULL, NULL, NULL);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Time to dedistort %d by %d image: %f sec", nx, ny, sph_time_stop_timer(1));

    cpl_test_nonnull( distim );

    cpl_image_save(image, "undistorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_save(distim, "distorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_delete(image);
    image = NULL;

    image = sph_distortion_model_apply(modelinv, distim,
                                       CPL_KERNEL_DEFAULT, 10.0,
                                       NULL, NULL, NULL);

    cpl_test_nonnull( image );
    cpl_image_save(image, "dedistorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);

    cpl_fit_image_gaussian(image, imerr, nx / 2, ny / 2, 55, 55, params,
            paramserr, NULL, &rms, &redchi, NULL, NULL, NULL, NULL, NULL);

    SPH_RAISE_CPL;

    xcen = cpl_array_get_double(params, 3, NULL);
    ycen = cpl_array_get_double(params, 4, NULL);
    sizx = cpl_array_get_double(params, 5, NULL);
    sizy = cpl_array_get_double(params, 6, NULL);

    // TODO: The odd +0.7 down there is due to a weird bug. 
    // On some hardware the cpl_arrag_get_double returns 0 ?!? 
    cpl_test_abs(xcen, nx/2.0,
            cpl_array_get_double(paramserr,3,NULL) + 0.7);
    cpl_test_abs(ycen, ny/2.0,
            cpl_array_get_double(paramserr,4,NULL) + 0.7);
    cpl_test_abs(sizx, 2.0,
            cpl_array_get_double(paramserr,5,NULL) + 0.7);
    cpl_test_abs(sizy, 2.0,
            cpl_array_get_double(paramserr,6,NULL)+ 0.7);

    cpl_image_delete(image);
    cpl_image_delete(imerr);
    cpl_image_delete(distim);
    sph_distortion_model_delete(model);
    sph_distortion_model_delete(modelinv);
    cpl_array_delete(params);
    cpl_array_delete(paramserr);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_apply_poly(void) {
    sph_polygon* poly = NULL;

    sph_distortion_model* model =
            cutest_sph_distortion_model_fixture2();

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, -1.0, -1.0);
    sph_polygon_add_point(poly, -1.0, 1.0);
    sph_polygon_add_point(poly, 1.0, 1.0);
    sph_polygon_add_point(poly, 1.0, -1.0);

    sph_distortion_model_apply_poly(model, poly);
    cpl_test_abs(poly->points[0].x, -1 + 0.0004, 0.0001);
    cpl_test_abs(poly->points[0].y, -1.00, 0.0001);
    cpl_test_abs(poly->points[1].x, -1 + 0.0004, 0.0001);
    cpl_test_abs(poly->points[1].y, 1.00, 0.0001);
    cpl_test_abs(poly->points[2].x, 1.0004, 0.0001);
    cpl_test_abs(poly->points[2].y, 1.00, 0.0001);
    sph_polygon_delete(poly);
    poly = NULL;
    sph_distortion_model_delete(model);
    model = NULL;

}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_apply_tough(void) {
    cpl_image* image = NULL;
    sph_distortion_model* distmodel = NULL;
    sph_point_pattern* pp_obs = NULL;
    sph_point_pattern* pp = NULL;
    sph_point_pattern* pp_corrected = NULL;
    sph_distortion_map* dmap = NULL;
    cpl_image* nim = NULL;
    cpl_image* residx = NULL;
    cpl_image* residy = NULL;
    int ii = 0;
    int nx = 1024;
    int ny = 1024;
    int np = 63;
    int nn = 0;
    int bpix = 0;
    double cx = 0.0;
    double cy = 0.0;
    cpl_error_code code;

    pp = sph_point_pattern_new();
    image = cutest_sph_distortion_model_fixture_tough(nx, ny, np,
            2.0, &distmodel, pp);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(image);

    residx = cpl_image_new(np + 1, np + 1, CPL_TYPE_DOUBLE);
    residy = cpl_image_new(np + 1, np + 1, CPL_TYPE_DOUBLE);
    nim = cpl_image_new(np + 1, np + 1, CPL_TYPE_INT);
    cpl_test_nonnull(nim);

    cpl_test_eq(sph_point_pattern_get_size(pp), 63*63);
    cpl_test_nonnull( distmodel );

    code = cpl_image_save(image, "tough_image.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_distortion_model_save(distmodel, "dmap.fits", nx, ny,
                                                CPL_IO_CREATE,
                                                NULL, NULL, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    pp_obs = sph_point_pattern_new_from_image(image, NULL, 5.0, 3.0,
                                              3.0, 3.0, 0, NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( pp_obs );

    code = sph_point_pattern_centralise_wrt_central_point(pp_obs);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_point_pattern_centralise_wrt_central_point(pp);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    pp_corrected = sph_distortion_model_apply_pp2(distmodel, pp_obs);
    cpl_test_nonnull( pp_corrected );

    code = sph_point_pattern_centralise_wrt_central_point(pp_corrected);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    dmap = sph_distortion_map_new(pp_corrected, pp);
    sph_distortion_map_delete_multi(dmap);
    cpl_test_nonnull( dmap );

    cx = nx / 2.0;
    cy = ny / 2.0;
    for (ii = 0; ii < sph_distortion_map_get_size(dmap); ++ii) {
        sph_distortion_vector dv;
        const cpl_error_code mycode =
            sph_distortion_map_get_one(dmap, ii, &dv);
        cpl_test_eq_error(mycode, CPL_ERROR_NONE);

        code = cpl_image_set(residx, (dv.x + cx) * np / nx + 1,
                             (dv.y + cy) * np / ny + 1, dv.dx);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        code = cpl_image_set(residy, (dv.x + cx) * np / nx + 1,
                             (dv.y + cy) * np / ny + 1, dv.dy);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        nn = cpl_image_get(nim, (dv.x + cx) * np / nx + 1,
                           (dv.y + cy) * np / ny + 1, &bpix);

        code = cpl_image_set(nim, (dv.x + cx) * np / nx + 1,
                             (dv.y + cy) * np / ny + 1, nn + 1);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

    }
    cpl_image_threshold(nim, -0.5, 0.5, 1.0, 1.0);
    cpl_image_divide(residx, nim);
    cpl_image_divide(residy, nim);

    cpl_image_save(residx, "residx.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(residy, "residy.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_test_abs( cpl_image_get_max(residx), 0.0, 0.1);
    cpl_test_abs( cpl_image_get_min(residx), 0.0, 0.1);
    cpl_test_abs( cpl_image_get_max(residy), 0.0, 0.1);
    cpl_test_abs( cpl_image_get_min(residy), 0.0, 0.1);

    sph_distortion_model_delete(distmodel);
    cpl_image_delete(image);
    sph_point_pattern_delete(pp);
    sph_point_pattern_delete(pp_obs);
    sph_point_pattern_delete(pp_corrected);
    cpl_image_delete(residx);
    cpl_image_delete(residy);
    cpl_image_delete(nim);
    sph_distortion_map_delete(dmap);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_apply_tough_im(void) {
    cpl_image* image = NULL;
    cpl_image* image_corrected = NULL;
    sph_distortion_model* distmodel = NULL;
    sph_point_pattern* pp_obs = NULL;
    sph_point_pattern* pp = NULL;
    sph_point_pattern* pp_corrected = NULL;
    sph_distortion_map* dmap = NULL;
    cpl_image* nim = NULL;
    cpl_image* residx = NULL;
    cpl_image* residy = NULL;
    int ii = 0;
    int nx = 1024;
    int ny = 1024;
    int np = 63;
    int nn = 0;
    int bpix = 0;
    double cx = 0.0;
    double cy = 0.0;

    pp = sph_point_pattern_new();
    image = cutest_sph_distortion_model_fixture_tough(nx, ny, np,
            2.0, &distmodel, pp);
    residx = cpl_image_new(np + 1, np + 1, CPL_TYPE_DOUBLE);
    residy = cpl_image_new(np + 1, np + 1, CPL_TYPE_DOUBLE);
    nim = cpl_image_new(np + 1, np + 1, CPL_TYPE_INT);

    cpl_test_eq(sph_point_pattern_get_size(pp), 63*63);
    cpl_test_nonnull( image );
    cpl_test_nonnull( distmodel );

    distmodel->offx = nx / 2.0; // + 0.5;
    distmodel->offy = nx / 2.0; // + 0.5;
    image_corrected = sph_distortion_model_apply(distmodel, image,
                                                 CPL_KERNEL_DEFAULT,
                                                 2 * CPL_KERNEL_DEF_WIDTH,
                                                 NULL, NULL, NULL);

    cpl_test_nonnull( image_corrected );
    cpl_image_save(image_corrected, "tough_image_corrected.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    sph_distortion_model_save(distmodel, "dmap.fits", nx, ny,
            CPL_IO_CREATE, NULL, NULL, NULL, NULL);

    pp_corrected = sph_point_pattern_new_from_image(image_corrected, NULL, 5.0,
                                                    3.0, 3.0, 3.0, 0, NULL);

    cpl_test_nonnull( pp_corrected );

    sph_point_pattern_centralise_wrt_central_point(pp_corrected);
    sph_point_pattern_centralise_wrt_central_point(pp);

    dmap = sph_distortion_map_new(pp_corrected, pp);
    sph_distortion_map_delete_multi(dmap);

    cx = nx / 2.0;
    cy = ny / 2.0;
    for (ii = 0; ii < sph_distortion_map_get_size(dmap); ++ii) {
        sph_distortion_vector dv;
        const cpl_error_code mycode =
            sph_distortion_map_get_one(dmap, ii, &dv);
        cpl_test_eq_error(mycode, CPL_ERROR_NONE);

        cpl_image_set(residx, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, dv.dx);
        cpl_image_set(residy, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, dv.dy);

        nn = cpl_image_get(nim, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, &bpix);

        cpl_image_set(nim, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, nn + 1);

    }
    cpl_image_threshold(nim, -0.5, 0.5, 1.0, 1.0);
    cpl_image_divide(residx, nim);
    cpl_image_divide(residy, nim);

    cpl_image_save(residx, "residx_im.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_save(residy, "residy_im.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_test_abs( cpl_image_get_max(residx), 0.0, 0.1);
    cpl_test_abs( cpl_image_get_min(residx), 0.0, 0.1);
    cpl_test_abs( cpl_image_get_max(residy), 0.0, 0.1);
    cpl_test_abs( cpl_image_get_min(residy), 0.0, 0.1);

    sph_distortion_model_delete(distmodel);
    cpl_image_delete(image);
    cpl_image_delete(image_corrected);
    sph_point_pattern_delete(pp);
    sph_point_pattern_delete(pp_obs);
    sph_point_pattern_delete(pp_corrected);
    cpl_image_delete(residx);
    cpl_image_delete(residy);
    cpl_image_delete(nim);
    sph_distortion_map_delete(dmap);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_apply(void) {
    cpl_image* image = NULL;
    cpl_image* distim = NULL;
    double xcen = 0.0;
    double ycen = 0.0;
    double sizx = 0.0;
    double sizy = 0.0;
    double rms = 0.0;
    double redchi = 0.0;
    cpl_image* imerr = NULL;
    cpl_array* params = cpl_array_new(7, CPL_TYPE_DOUBLE);
    cpl_array* paramserr = cpl_array_new(7, CPL_TYPE_DOUBLE);
    int nx = 256;
    int ny = 256;

    sph_distortion_model* model =
            cutest_sph_distortion_model_fixture2();
    sph_distortion_model* modelinv =
            cutest_sph_distortion_model_fixture2inv();
    cpl_test_nonnull( model );
    image = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    sph_test_image_tools_add_gauss(image, nx / 2, ny / 2, 2.0, 2.0);
    //imerr = cpl_image_duplicate(image);
    //cpl_image_power(imerr,0.5);
    imerr = sph_test_image_tools_create_flat_image_double(nx, ny, 0.1);
    cpl_fit_image_gaussian(image, NULL, nx / 2, ny / 2, 55, 55, params, NULL,
            NULL, &rms, NULL, NULL, NULL, NULL, NULL, NULL);
    SPH_RAISE_CPL;
    xcen = cpl_array_get_double(params, 3, NULL);
    ycen = cpl_array_get_double(params, 4, NULL);
    sizx = cpl_array_get_double(params, 5, NULL);
    sizy = cpl_array_get_double(params, 6, NULL);
    cpl_test_abs(xcen, nx/2, 0.1);
    cpl_test_abs(ycen, ny/2, 0.1);
    cpl_test_abs(sizx, 2.0, 0.1);
    cpl_test_abs(sizy, 2.0, 0.1);

    distim = sph_distortion_model_apply(model, image,
                                        CPL_KERNEL_DEFAULT, 10.0,
                                        NULL, NULL, NULL);
    cpl_test_nonnull( distim );
    cpl_image_save(image, "undistorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_save(distim, "distorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_delete(image);
    image = NULL;
    image = sph_distortion_model_apply(modelinv, distim,
                                       CPL_KERNEL_DEFAULT, 10.0,
                                       NULL, NULL, NULL);
    cpl_test_nonnull( image );
    cpl_image_save(image, "dedistorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);

    cpl_fit_image_gaussian(image, imerr, nx / 2, ny / 2, 55, 55, params,
            paramserr, NULL, &rms, &redchi, NULL, NULL, NULL, NULL, NULL);
    //SPH_RAISE_CPL;
    xcen = cpl_array_get_double(params, 3, NULL);
    ycen = cpl_array_get_double(params, 4, NULL);
    sizx = cpl_array_get_double(params, 5, NULL);
    sizy = cpl_array_get_double(params, 6, NULL);
    // TODO: The odd +0.7 down there is due to a weird bug. 
    // On some hardware the cpl_arrag_get_double returns 0 ?!? 
    cpl_test_abs(xcen, nx/2,
            cpl_array_get_double(paramserr,3,NULL) + 0.7);
    cpl_test_abs(ycen, ny/2,
            cpl_array_get_double(paramserr,4,NULL) + 0.7);
    cpl_test_abs(sizx, 2.0,
            cpl_array_get_double(paramserr,5,NULL) + 0.7);
    cpl_test_abs(sizy, 2.0,
            cpl_array_get_double(paramserr,6,NULL) + 0.7);

    cpl_image_delete(image);
    cpl_image_delete(imerr);
    cpl_image_delete(distim);
    sph_distortion_model_delete(model);
    sph_distortion_model_delete(modelinv);
    cpl_array_delete(params);
    cpl_array_delete(paramserr);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_cpl_warp_test(void) {
    cpl_image* image = NULL;
    cpl_image* image_flat = NULL;
    cpl_image* image_corrected = NULL;
    sph_distortion_model* distmodel = NULL;
    sph_point_pattern* pp = NULL;
    int nx = 1024;
    int ny = 1024;
    int np = 63;
    int ii = 0;
    int pix = 0;
    int piy = 0;
    double x = 0.0;
    double y = 0.0;
    double dx = 0.0;
    double dy = 0.0;
    double xc = 0.0;
    double yc = 0.0;
    double deltax = 0.0;
    double deltay = 0.0;
    double x00 = 0.0; //0.48122307;
    double y00 = 0.0; //0.57683;
    double x01 = -0.00178;
    double y01 = 0.001658;
    double x10 = 0.00765;
    double y10 = 0.001536;
    double x11 = -2.002e-7;
    double y11 = -1.499e-6;
    cpl_vector* profile = NULL;
    double rad = CPL_KERNEL_DEF_WIDTH;
    sph_point_pattern* pp_corrected = NULL;
    sph_distortion_map* dmap = NULL;
    double cx = 0.0;
    double cy = 0.0;
    cpl_image* residx = NULL;
    cpl_image* residy = NULL;
    cpl_image* nim = NULL;
    int nn = 0;
    int bpix = 0;

    image = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    image_flat = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    pp = sph_point_pattern_new_(np * np);
    deltax = (cpl_image_get_size_x(image) - 50.0) / np;
    deltay = (cpl_image_get_size_y(image) - 50.0) / np;
    distmodel = sph_distortion_model_new(NULL, NULL);
    residx = cpl_image_new(np + 1, np + 1, CPL_TYPE_DOUBLE);
    residy = cpl_image_new(np + 1, np + 1, CPL_TYPE_DOUBLE);
    nim = cpl_image_new(np + 1, np + 1, CPL_TYPE_INT);

    for (pix = 0; pix < np; ++pix) {
        for (piy = 0; piy < np; ++piy) {
            x = pix * deltax + 25.0;
            y = piy * deltay + 25.0;

            sph_point_pattern_add_point(pp, x, y);

            dx = x00 + (x - xc) * x10 + (y - yc) * x01
                    + (x - xc) * (y - yc) * x11;
            dy = y00 + (x - xc) * y10 + (y - yc) * y01
                    + (x - xc) * (y - yc) * y11;

            sph_test_image_tools_add_gauss(image, x + dx, y + dy, 1.0,
                    100000.0);
            sph_test_image_tools_add_gauss(image_flat, x, y, 1.0, 100000.0);
        }
    }
    sph_distortion_model_set_coeff_x(distmodel, 0, 0, x00);
    sph_distortion_model_set_coeff_x(distmodel, 0, 1, x01);
    sph_distortion_model_set_coeff_x(distmodel, 1, 0, 1.0 + x10);
    sph_distortion_model_set_coeff_x(distmodel, 1, 1, x11);
    sph_distortion_model_set_coeff_y(distmodel, 0, 0, y00);
    sph_distortion_model_set_coeff_y(distmodel, 0, 1, 1.0 + y01);
    sph_distortion_model_set_coeff_y(distmodel, 1, 0, y10);
    sph_distortion_model_set_coeff_y(distmodel, 1, 1, y11);

    cpl_test_eq(sph_point_pattern_get_size(pp), np * np);
    image_corrected = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( image_corrected );
    profile = cpl_vector_new(1000 * (int) rad);

    cpl_vector_fill_kernel_profile(profile, CPL_KERNEL_DEFAULT,
            CPL_KERNEL_DEF_WIDTH);

    cpl_test_eq(
            cpl_image_warp_polynomial( image_corrected, image, distmodel->polyx, distmodel->polyy, profile,2 * rad,profile, 2 * rad),
            CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_image_save(image, "cpl_warp_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_save(image_flat, "cpl_warp_image_flat.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_CREATE);
    cpl_image_save(image_corrected, "cpl_warp_image_corrected.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    pp_corrected = sph_point_pattern_new_from_image(image_corrected, NULL, 5.0,
                                                    1.0, 3.0, 3.0, 0, NULL);
    dmap = sph_distortion_map_new(pp_corrected, pp);
    sph_distortion_map_delete_multi(dmap);

    cx = 0.0;
    cy = 0.0;
    for (ii = 0; ii < sph_distortion_map_get_size(dmap); ++ii) {
        sph_distortion_vector dv;
        const cpl_error_code mycode =
            sph_distortion_map_get_one(dmap, ii, &dv);
        cpl_test_eq_error(mycode, CPL_ERROR_NONE);

        cpl_image_set(residx, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, dv.dx);
        cpl_image_set(residy, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, dv.dy);

        nn = cpl_image_get(nim, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, &bpix);

        cpl_image_set(nim, (dv.x + cx) * np / nx + 1,
                (dv.y + cy) * np / ny + 1, nn + 1);

    }
    cpl_image_threshold(nim, -0.5, 0.5, 1.0, 1.0);
    cpl_image_divide(residx, nim);
    cpl_image_divide(residy, nim);

    cpl_image_save(residx, "residx_cpl.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_save(residy, "residy_cpl.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);

    cpl_test_abs( cpl_image_get_max(residx), 0.0, 0.2);
    cpl_test_abs( cpl_image_get_min(residx), 0.0, 0.2);
    cpl_test_abs( cpl_image_get_max(residy), 0.0, 0.2);
    cpl_test_abs( cpl_image_get_min(residy), 0.0, 0.2);
    cpl_image_delete(image);
    cpl_image_delete(image_flat);
    cpl_image_delete(image_corrected);
    sph_distortion_model_delete(distmodel);
    cpl_vector_delete(profile);
    sph_point_pattern_delete(pp);
    cpl_image_delete(residx);
    cpl_image_delete(residy);
    cpl_image_delete(nim);
    sph_distortion_map_delete(dmap);
    sph_point_pattern_delete(pp_corrected);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_polyonmial_distortion_model_apply_highorder(void) {
    cpl_image* image = NULL;
    cpl_image* distim = NULL;
    cpl_array* params = cpl_array_new(7, CPL_TYPE_DOUBLE);
    cpl_array* paramserr = cpl_array_new(7, CPL_TYPE_DOUBLE);
    int nx = 256;
    int ny = 256;

    sph_distortion_model* model =
            cutest_sph_distortion_model_fixture2();
    cpl_test_nonnull( model );
    image = sph_test_image_tools_create_grid_image(nx, ny, 10, 10,
            CPL_TYPE_DOUBLE, 1000.0, 2.0);
    distim = sph_distortion_model_apply(model, image,
                                        CPL_KERNEL_DEFAULT,
                                        CPL_KERNEL_DEF_WIDTH,
                                        NULL, NULL, NULL);
    cpl_test_nonnull( distim );
    cpl_image_save(image, "undistorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_save(distim, "distorted_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_delete(image);
    cpl_image_delete(distim);
    sph_distortion_model_delete(model);
    cpl_array_delete(params);
    params = NULL;
    cpl_array_delete(paramserr);
    paramserr = NULL;
    return;
}
static
void cutest_sph_polyonmial_distortion_model_check_cpl_bug(void) {
    cpl_image* result = NULL;
    cpl_vector* profile = NULL;
    double rad = 10.0;
    double coeff = 0.0;
    double radin = 10.0;
    int nx = 256;
    int ny = 256;
    cpl_image* image = NULL;
    cpl_size pows[2] = { 0, 0 };
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);

    pows[0] = 1;
    pows[1] = 0;
    //cpl_polynomial_set_coeff(polyx,pows,1.0000);
    pows[0] = 0;
    pows[1] = 1;
    //cpl_polynomial_set_coeff(polyy,pows,1.0000);
    pows[0] = 2;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, -0.0004);
    pows[0] = 0;
    pows[1] = 2;
    //cpl_polynomial_set_coeff(polyy,pows,0.0004);
    pows[0] = 2;
    pows[1] = 2;
    //cpl_polynomial_set_coeff(polyy,pows,0.00000001);
    //cpl_polynomial_set_coeff(polyx,pows,0.00000001);

    image = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0000001);
    cpl_test_nonnull( image );
    sph_test_image_tools_add_gauss(image, nx / 2, ny / 2, 2.0, 2.0);

    profile = cpl_vector_new(2 * (int) radin);
    pows[0] = 1;
    pows[1] = 0;
    coeff = cpl_polynomial_get_coeff(polyx, pows);
    cpl_polynomial_set_coeff(polyx, pows, coeff + 1.0000);
    pows[0] = 0;
    pows[1] = 1;
    coeff = cpl_polynomial_get_coeff(polyy, pows);
    cpl_polynomial_set_coeff(polyy, pows, coeff + 1.0000);

    cpl_vector_fill_kernel_profile(profile, CPL_KERNEL_DEFAULT, radin);

    result = cpl_image_new(cpl_image_get_size_x(image),
            cpl_image_get_size_y(image), CPL_TYPE_DOUBLE);

    cpl_image_warp_polynomial(result, image, polyx, polyy, profile, rad,
            profile, rad);
    cpl_image_save(image, "undistorted_image2.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_image_save(result, "distorted_image2.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    cpl_vector_delete(profile);
    cpl_image_delete(image);
    cpl_image_delete(result);
}
static
void cutest_sph_polyonmial_distortion_model_check_cpl_leakage(void) {
    cpl_image* image = NULL;
    double rms = 0.0;
    double redchi = 0.0;
    cpl_image* imerr = NULL;
    cpl_array* params = cpl_array_new(7, CPL_TYPE_DOUBLE);
    cpl_array* paramserr = cpl_array_new(7, CPL_TYPE_DOUBLE);

    image = cpl_image_new(512, 512, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(image, 256, 256, 20.0, 20.0, 20.0);
    sph_test_image_tools_add_gauss(image, 256, 256, 20.0, 20.0);
    imerr = cpl_image_new(512, 512, CPL_TYPE_DOUBLE);
    cpl_image_fill_noise_uniform(imerr, 1.0, 1.00001);
    cpl_fit_image_gaussian(image, imerr, 256, 256, 55, 55, params, paramserr,
            NULL, &rms, &redchi, NULL, NULL, NULL, NULL, NULL);
    cpl_image_delete(image);
    cpl_image_delete(imerr);
    cpl_array_delete(params);
    cpl_array_delete(paramserr);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        printf("CPL Error rasied:%s", cpl_error_get_message());
    }
    return;
}

/*----------------------------------------------------------------------------*/
//                      FIXTURES AND HELPERS
/*----------------------------------------------------------------------------*/

static sph_distortion_model*
cutest_sph_distortion_model_fixture(void) {
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    sph_distortion_model* distmod = NULL;
    cpl_size pows[2] = { 2, 0 };
    pows[0] = 2;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, 0.0004);
    pows[0] = 0;
    pows[1] = 2;
    cpl_polynomial_set_coeff(polyy, pows, 0.0004);
    pows[0] = 2;
    pows[1] = 2;
    cpl_polynomial_set_coeff(polyy, pows, 0.00000001);
    cpl_polynomial_set_coeff(polyx, pows, 0.00000001);
    distmod = sph_distortion_model_new(polyx, polyy);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    return distmod;
}

static sph_distortion_model*
cutest_sph_distortion_model_fixture2(void) {
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    sph_distortion_model* distmod = NULL;
    cpl_size pows[2] = { 2, 0 };
    pows[0] = 1;
    pows[1] = 0;
    //cpl_polynomial_set_coeff(polyx,pows,1.0000);
    pows[0] = 0;
    pows[1] = 1;
    //cpl_polynomial_set_coeff(polyy,pows,1.0000);
    pows[0] = 2;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, 0.0004);
    pows[0] = 0;
    pows[1] = 2;
    //cpl_polynomial_set_coeff(polyy,pows,0.0004);
    pows[0] = 2;
    pows[1] = 2;
    //cpl_polynomial_set_coeff(polyy,pows,0.00000001);
    //cpl_polynomial_set_coeff(polyx,pows,0.00000001);
    distmod = sph_distortion_model_new(polyx, polyy);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    return distmod;
}
static sph_distortion_model*
cutest_sph_distortion_model_fixture2inv(void) {
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    sph_distortion_model* distmod = NULL;
    cpl_size pows[2] = { 2, 0 };
    pows[0] = 1;
    pows[1] = 0;
    //cpl_polynomial_set_coeff(polyx,pows,1.0000);
    pows[0] = 0;
    pows[1] = 1;
    //cpl_polynomial_set_coeff(polyy,pows,1.0000);
    pows[0] = 2;
    pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, -0.0004);
    pows[0] = 0;
    pows[1] = 2;
    //cpl_polynomial_set_coeff(polyy,pows,0.0004);
    pows[0] = 2;
    pows[1] = 2;
    //cpl_polynomial_set_coeff(polyy,pows,0.00000001);
    //cpl_polynomial_set_coeff(polyx,pows,0.00000001);
    distmod = sph_distortion_model_new(polyx, polyy);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    return distmod;
}
static
int cutest_sph_polyonmial_distortion_model_saveit(void) {
    sph_distortion_model* distmod = NULL;
    distmod = cutest_sph_distortion_model_fixture();

    sph_distortion_model_save(distmod, "test_quadrdistortion.fits",
            256, 256, CPL_IO_CREATE, NULL, NULL, NULL, NULL);
    sph_distortion_model_delete(distmod);
    return 0;
}

cpl_image*
cutest_sph_distortion_model_fixture_tough(int nx, int ny,
        int points_per_side, double fwhm,
        sph_distortion_model** distmodel_out,
        sph_point_pattern* pp_ideal) {
    int pix = 0;
    int piy = 0;
    cpl_image* im = NULL;
    double x = 0;
    double y = 0;
    double dx = 0;
    double dy = 0;
    double deltax = 0.0;
    double deltay = 0.0;
    double xc = 512.0;
    double yc = 512.0;
    double x00 = 0.0; //0.48122307;
    double y00 = 0.0; //0.57683;
    double x01 = -0.00178;
    double y01 = 0.001658;
    double x10 = 0.00765;
    double y10 = 0.001536;
    double x11 = -2.002e-7;
    double y11 = -1.499e-6;
    double x02 = 0.0; //-5.14e-7;
    double y02 = 0.0; //1.1939e-6;
    double x20 = 0.0; //-1.1657e-6;
    double y20 = 0.0; //1.81e-6;

    cpl_test_lt(0.0, fwhm);

    im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    cpl_test_lt(255, cpl_image_get_size_x(im));
    cpl_test_lt(255, cpl_image_get_size_y(im));
    xc = cpl_image_get_size_x(im) / 2.0;
    yc = cpl_image_get_size_y(im) / 2.0;
    deltax = (cpl_image_get_size_x(im) - 50.0) / points_per_side;
    deltay = (cpl_image_get_size_y(im) - 50.0) / points_per_side;

    for (pix = 0; pix < points_per_side; ++pix) {
        for (piy = 0; piy < points_per_side; ++piy) {
            x = pix * deltax + 25.0;
            y = piy * deltay + 25.0;

            if (pp_ideal)
                sph_point_pattern_add_point(pp_ideal, x, y);

            dx = x00 + (x - xc) * x10 + (y - yc) * x01 + (x - xc) * x20
                    + (y - yc) * x02 + (x - xc) * (y - yc) * x11;
            dy = y00 + (x - xc) * y10 + (y - yc) * y01 + (x - xc) * y20
                    + (y - yc) * y02 + (x - xc) * (y - yc) * y11;

            sph_test_image_tools_add_gauss(im, x + dx, y + dy, 2.0, 100000.0);
        }
    }

    cpl_image_add_scalar(im, 100.0);
    sph_test_image_tools_apply_poisson_noise(im, NULL);

    if (distmodel_out) {
        *distmodel_out = sph_distortion_model_new(NULL, NULL);
        sph_distortion_model_set_coeff_x(*distmodel_out, 0, 0, -x00);
        sph_distortion_model_set_coeff_y(*distmodel_out, 0, 0, -y00);

        sph_distortion_model_set_coeff_x(*distmodel_out, 1, 0, -x10);
        sph_distortion_model_set_coeff_y(*distmodel_out, 1, 0, -y10);

        sph_distortion_model_set_coeff_x(*distmodel_out, 0, 1, -x01);
        sph_distortion_model_set_coeff_y(*distmodel_out, 0, 1, -y01);

        sph_distortion_model_set_coeff_x(*distmodel_out, 1, 1, -x11);
        sph_distortion_model_set_coeff_y(*distmodel_out, 1, 1, -y11);
    }

    return im;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Polynomial distortion model testsuite",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if ( NULL == sph_test_do(pSuite,
                             "Testing apply polynom fit (tough)",
                             cutest_sph_polyonmial_distortion_model_apply_tough) )
        {
            return sph_test_get_error();
        }
    if ( NULL == sph_test_do(pSuite,
                             "Testing apply polynom  cpl_ warp",
                             cutest_sph_polyonmial_distortion_cpl_warp_test) )
        {
            return sph_test_get_error();
        }
    if ( NULL == sph_test_do(pSuite,
                             "Testing apply polynom fit image (tough)",
                             cutest_sph_polyonmial_distortion_model_apply_tough_im) )
        {
            return sph_test_get_error();
        }
    if (NULL
            == sph_test_do(pSuite, "Testing new from image grid (no dist)",
                    cutest_sph_polyonmial_distortion_model_from_grid_no_dist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing new from image grid",
                    cutest_sph_polyonmial_distortion_model_from_grid)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing new from image grid is stable to small changes",
                    cutest_sph_polyonmial_distortion_model_from_grid_stable)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "Testing fit distortion map",
                    cutest_sph_polyonmial_distortion_model_test_fit_distortion_map_temp)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing save",
                    cutest_sph_polyonmial_distortion_model_test_save)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing load",
                    cutest_sph_polyonmial_distortion_model_test_load)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply",
                    cutest_sph_polyonmial_distortion_model_apply)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply timing",
                    cutest_sph_polyonmial_distortion_model_apply_timing)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cpl leakage",
                    cutest_sph_polyonmial_distortion_model_check_cpl_leakage)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cpl new version",
                    cutest_sph_polyonmial_distortion_model_check_cpl_bug)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply highorder",
                    cutest_sph_polyonmial_distortion_model_apply_highorder)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply poly",
                    cutest_sph_polyonmial_distortion_model_apply_poly)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/

static sph_point_pattern*
sph_distortion_model_apply_pp2(
        sph_distortion_model* self,
        sph_point_pattern* pp )
{
    cpl_vector* profile = NULL;
    cpl_polynomial* polyx = NULL;
    cpl_polynomial* polyy = NULL;
    cpl_image*      dxim = NULL;
    cpl_image*      dyim = NULL;
    sph_point_pattern*  result = NULL;
    sph_point*      p = NULL;
    int             nn   = 0;
    cpl_vector*     posv = NULL;
    double          dx = 0.0;
    double          dy = 0.0;
    double          posvxy[2];

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(pp,CPL_ERROR_NULL_INPUT,NULL);

    polyx = cpl_polynomial_duplicate(self->polyx);
    polyy = cpl_polynomial_duplicate(self->polyy);
    cpl_ensure(polyx,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    cpl_ensure(polyy,CPL_ERROR_ILLEGAL_OUTPUT,NULL);

    result = sph_point_pattern_new_(sph_point_pattern_get_size(pp));
    posv = cpl_vector_wrap(2, posvxy);
    for (nn = 0; nn < sph_point_pattern_get_size(pp); ++nn) {
        p = sph_point_pattern_get_point(pp,nn);
        posvxy[0] = p->x;
        posvxy[1] = p->y;
        dx = cpl_polynomial_eval(polyx,posv);
        dy = cpl_polynomial_eval(polyy,posv);

        sph_point_pattern_add_point(result,p->x + dx, p->y + dy);

        sph_point_delete(p); p = NULL;
    }
    cpl_vector_unwrap(posv);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);
    cpl_vector_delete(profile);
    cpl_image_delete(dxim);
    cpl_image_delete(dyim);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

