/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_fctable.h"
#include "sph_time.h"
#include "sph_ird_instrument_model.h"
#include "sph_test_zpl_simulator.h"
#include "sph_test_image_tools.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_irdis_model.h"
#include "sph_utils.h"
#include "sph_skycalc.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    //sph_init_debug_verbose();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

static cpl_frameset*
cutest_sph_fctable_create_frames(int nf, int nx, int ny,
        sph_distortion_model* distmap) {
    cpl_frameset* inframes = NULL;
    sph_test_ngc_ir_simulator* ngc = NULL;
    sph_test_irdis_model* testmodel = NULL;
    int ff = 0;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;

    ngc = sph_test_ngc_ir_simulator_new(nx, ny);
    testmodel = sph_test_irdis_model_new(ngc);
    inframes = cpl_frameset_new();
    for (ff = 0; ff < nf; ++ff) {
        im = sph_test_image_tools_create_flat_image_double(testmodel->nx,
                testmodel->ny, 1.0);
        sph_test_image_tools_add_gauss(im, nx / 4 + 0.5, ny / 4 + 0.5, 2.0,
                1000000.0);
        sph_test_image_tools_add_gauss(im, nx / 4 + 30.5, ny / 4 + 0.5, 2.0,
                1000000.0);

        im2 = sph_test_irdis_model_process_image(testmodel, im);
        cpl_image_delete(im);
        im = NULL;
        im = sph_test_ngc_ir_simulator_create_raw_image(ngc, im2, 0.0, 0.0);
        cpl_image_delete(im2);
        im2 = NULL;
        sph_test_image_tools_apply_poisson_noise(im, ngc->pRNG);
        if (distmap) {
            im2 = sph_distortion_model_apply(distmap, im,
                                             CPL_KERNEL_DEFAULT,
                                             CPL_KERNEL_DEF_WIDTH,
                                             NULL, NULL, NULL);
            cpl_image_delete(im);
            im = im2;
        }
        cpl_frameset_insert(
                inframes,
                sph_test_frame_image(im, CPL_BPP_32_SIGNED, "SCIENCETEST",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(im);
        im = NULL;
    }
    sph_test_ngc_ir_simulator_delete(ngc);
    ngc = NULL;
    sph_test_irdis_model_delete(testmodel);
    testmodel = NULL;

    return inframes;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_new(void) {
    sph_fctable* fctable = NULL;

    fctable = sph_fctable_new();
    cpl_test_nonnull( fctable );
    cpl_test_eq(fctable->mode, SPH_FCTAB_UNDEFINED);
    cpl_test_zero(sph_fctable_get_size(fctable));
    sph_fctable_delete(fctable);
    fctable = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_add_row_single(void) {
    sph_fctable* fctable = NULL;

    fctable = sph_fctable_new();
    cpl_test_nonnull( fctable );
    sph_fctable_add_row_single(fctable, 0.0, 1.0, 2.0, 3.0);
    cpl_test_eq(sph_fctable_get_size(fctable), 1);
    cpl_test_abs(cpl_vector_get(fctable->angles,0), 3.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,0), 1.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,0), 2.0, 0.00001);

    cpl_test_eq(
            sph_fctable_add_row_double(fctable, 0.0,1.0,2.0,0.1,0.2,1.1),
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();
    sph_fctable_delete(fctable);
    fctable = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_add_row_double(void) {
    sph_fctable* fctable = NULL;

    fctable = sph_fctable_new();
    cpl_test_nonnull( fctable );
    sph_fctable_add_row_double(fctable, 0.0, 1.0, 2.0, 0.1, 0.2, 1.1);
    cpl_test_eq(sph_fctable_get_size(fctable), 1);
    cpl_test_abs(cpl_vector_get(fctable->angles,0), 1.1, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,0), 1.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,0), 2.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_x,0), 0.1, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_y,0), 0.2, 0.00001);

    cpl_test_eq(sph_fctable_add_row_single(fctable,0.0,1.0,2.0, 3.0),
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();

    sph_fctable_delete(fctable);
    fctable = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_get(void) {
    sph_fctable* fctable = NULL;
    cpl_vector* angles = NULL;
    cpl_vector* xc = NULL;
    cpl_vector* yc = NULL;
    fctable = sph_fctable_new();
    cpl_test_nonnull( fctable );
    sph_fctable_add_row_single(fctable, 0.0, 1.0, 2.0, 3.0);

    xc = sph_fctable_get_centre_x_left(fctable);
    cpl_test_nonnull( xc );
    yc = sph_fctable_get_centre_y_left(fctable);
    cpl_test_nonnull( yc );
    angles = sph_fctable_get_angles(fctable);
    cpl_test_nonnull( angles );
    cpl_test_eq(cpl_vector_get_size(angles), 1);
    cpl_test_eq(cpl_vector_get_size(xc), 1);
    cpl_test_eq(cpl_vector_get_size(yc), 1);
    cpl_test_abs(cpl_vector_get(angles,0), 3.0, 0.00001);
    cpl_test_abs(cpl_vector_get(xc,0), 1.0, 0.00001);
    cpl_test_abs(cpl_vector_get(yc,0), 2.0, 0.00001);

    cpl_test_null(sph_fctable_get_centre_x_right(fctable));
    cpl_test_null(sph_fctable_get_centre_y_right(fctable));

    cpl_vector_delete(angles);
    angles = NULL;
    cpl_vector_delete(xc);
    xc = NULL;
    cpl_vector_delete(yc);
    yc = NULL;
    sph_fctable_delete(fctable);
    fctable = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_load_fits_single(void) {
    sph_fctable* fctable = NULL;

    cpl_table* tab = NULL;

    fctable = sph_fctable_new();
    cpl_test_nonnull( fctable );
    sph_fctable_add_row_single(fctable, 0.0, 1.0, 2.0, 3.0);

    tab = sph_fctable_get_as_cpl_table(fctable);
    cpl_test_nonnull( tab );
    cpl_table_save(tab, NULL, NULL, "testtab.fits", CPL_IO_CREATE);
    cpl_table_delete(tab);
    tab = NULL;
    sph_fctable_delete(fctable);
    fctable = NULL;

    fctable = sph_fctable_load_fits("testtab.fits");
    cpl_test_nonnull( fctable );
    cpl_test_eq(fctable->mode, SPH_FCTAB_SINGLE);
    cpl_test_eq(sph_fctable_get_size(fctable), 1);
    cpl_test_abs(cpl_vector_get(fctable->angles,0), 3.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,0), 1.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,0), 2.0, 0.00001);

    sph_fctable_delete(fctable);
    fctable = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_load_ascii_single(void) {
    sph_fctable* fctable = NULL;
    FILE* f = NULL;

    f = fopen("fctable.txt", "w");
    fprintf(f, "%s\n", "# Hello this is a comment");
    fprintf(f, "%s\n", "# Hello this is a comment too");
    fprintf(f, "  2000-01-01T20:56:22.34 5.0     2.0      11.0\n");
    fprintf(f, "  2000-01-01T20:56:22.34 6.0     3.0      12.0\n");
    fprintf(f, "  2000-01-01T20:56:22.34 7.0     4.0      13.0\n");
    fprintf(f, "  2000-01-01T20:56:22.34 8.0     5.0      14.0\n");
    fclose(f);
    fctable = sph_fctable_load_ascii("fctable.txt");
    cpl_test_nonnull( fctable );
    cpl_test_eq(fctable->mode, SPH_FCTAB_SINGLE);
    cpl_test_eq(sph_fctable_get_size(fctable), 4);
    cpl_test_abs(cpl_vector_get(fctable->angles,0), 11.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,0), 5.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,0), 2.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->angles,1), 12.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,1), 6.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,1), 3.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->angles,2), 13.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,2), 7.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,2), 4.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->angles,3), 14.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,3), 8.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,3), 5.0, 0.00001);

    sph_fctable_delete(fctable);
    fctable = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_load_ascii_double(void) {
    sph_fctable* fctable = NULL;
    FILE* f = NULL;

    f = fopen("fctable.txt", "w");
    fprintf(f, "%s\n", "# Hello this is a comment");
    fprintf(f, "%s\n", "# Hello this is a comment too");
    fprintf(f, " 2000-01-01T20:56:22.34 5.0     2.0      -2.0 -7.0 11.0\n");
    fprintf(f, " 2000-01-01T20:56:22.34 6.0     3.0      -3.0 -8.0 12.0\n");
    fprintf(f, " 2000-01-01T20:56:22.34 7.0     4.0      -4.0 -9.0 13.0\n");
    fprintf(f, " 2000-01-01T20:56:22.34 8.0     5.0      -5.0 -10.0 14.0\n");
    fclose(f);
    fctable = sph_fctable_load_ascii("fctable.txt");
    cpl_test_nonnull( fctable );
    cpl_test_eq(fctable->mode, SPH_FCTAB_DOUBLE);
    cpl_test_eq(sph_fctable_get_size(fctable), 4);
    cpl_test_abs(cpl_vector_get(fctable->angles,0), 11.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,0), 5.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,0), 2.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_x,0), -2.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_y,0), -7.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(fctable->angles,1), 12.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,1), 6.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,1), 3.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_x,1), -3.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_y,1), -8.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(fctable->angles,2), 13.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,2), 7.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,2), 4.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_x,2), -4.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_y,2), -9.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(fctable->angles,3), 14.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,3), 8.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,3), 5.0, 0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_x,3), -5.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_y,3), -10.0,
            0.00001);

    sph_fctable_delete(fctable);
    fctable = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_save_ascii_single(void) {
    sph_fctable* fctable = NULL;
    fctable = sph_fctable_new();

    cpl_test_nonnull( fctable );
    sph_fctable_add_row_single(fctable, 0.0, 1.0, 2.0, 3.0);
    sph_fctable_add_row_single(fctable, 0.0, 11.0, 22.0, 33.0);
    sph_fctable_add_row_single(fctable, 0.0, 111.0, 222.0, 333.0);
    cpl_test_eq(sph_fctable_save_ascii(fctable, "fctable.txt"),
            CPL_ERROR_NONE);
    sph_fctable_delete(fctable);
    fctable = NULL;

    fctable = sph_fctable_load_ascii("fctable.txt");
    cpl_test_nonnull( fctable );
    cpl_test_eq(fctable->mode, SPH_FCTAB_SINGLE);
    cpl_test_eq(sph_fctable_get_size(fctable), 3);

    sph_fctable_delete(fctable);
    fctable = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_save_ascii_double(void) {
    sph_fctable* fctable = NULL;
    fctable = sph_fctable_new();

    cpl_test_nonnull( fctable );
    sph_fctable_add_row_double(fctable, 0.0, 1.0, 2.0, 3.0, 4.0, 5.0);
    sph_fctable_add_row_double(fctable, 0.0, 11.0, 22.0, 33.0, 44.0, 55.0);
    sph_fctable_add_row_double(fctable, 0.0, 111.0, 222.0, 333.0, 444.0, 555.0);
    cpl_test_eq(sph_fctable_save_ascii(fctable, "fctable.txt"),
            CPL_ERROR_NONE);
    sph_fctable_delete(fctable);
    fctable = NULL;

    fctable = sph_fctable_load_ascii("fctable.txt");
    cpl_test_nonnull( fctable );
    cpl_test_eq(fctable->mode, SPH_FCTAB_DOUBLE);
    cpl_test_eq(sph_fctable_get_size(fctable), 3);

    sph_fctable_delete(fctable);
    fctable = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_set_cent_double(void) {
    sph_fctable* fctable = NULL;
    fctable = sph_fctable_new();

    cpl_test_nonnull( fctable );
    sph_fctable_add_row_double(fctable, 0.0, 1.0, 2.0, 3.0, 4.0, 5.0);
    sph_fctable_add_row_double(fctable, 0.0, 11.0, 22.0, 33.0, 44.0, 55.0);
    sph_fctable_add_row_double(fctable, 0.0, 111.0, 222.0, 333.0, 444.0, 555.0);
    cpl_test_eq(sph_fctable_get_size(fctable), 3);
    sph_fctable_set_cent_double(fctable, 0, 1.0, 2.0, 3.0, 4.0);
    cpl_test_abs(cpl_vector_get(fctable->centresL_x,0), 1.0, 0.0001);
    cpl_test_abs(cpl_vector_get(fctable->centresL_y,0), 2.0, 0.0001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_x,0), 3.0, 0.0001);
    cpl_test_abs(cpl_vector_get(fctable->centresR_y,0), 4.0, 0.0001);

    sph_fctable_delete(fctable);
    fctable = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_get_as_cpl_table(void) {
    sph_fctable* fctable = NULL;
    cpl_table* tab = NULL;
    fctable = sph_fctable_new();

    cpl_test_nonnull( fctable );
    sph_fctable_add_row_double(fctable, 0.0, 1.0, 2.0, 3.0, 4.0, 5.0);
    sph_fctable_add_row_double(fctable, 0.0, 11.0, 22.0, 33.0, 44.0, 55.0);
    sph_fctable_add_row_double(fctable, 0.0, 111.0, 222.0, 333.0, 444.0, 555.0);
    tab = sph_fctable_get_as_cpl_table(fctable);
    cpl_test_nonnull( tab );
    sph_fctable_delete(fctable);
    fctable = NULL;

    cpl_test_eq(cpl_table_get_nrow(tab), 3);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_table_delete(tab);
    tab = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fctable_get_as_cpl_table_single(void) {
    sph_fctable* fctable = NULL;
    cpl_table* tab = NULL;
    fctable = sph_fctable_new();

    cpl_test_nonnull( fctable );
    sph_fctable_add_row_single(fctable, 0.0, 1.0, 2.0, 5.0);
    sph_fctable_add_row_single(fctable, 0.0, 11.0, 22.0, 55.0);
    sph_fctable_add_row_single(fctable, 0.0, 111.0, 222.0, 555.0);
    tab = sph_fctable_get_as_cpl_table(fctable);
    cpl_test_nonnull( tab );
    sph_fctable_delete(fctable);
    fctable = NULL;

    cpl_test_eq(cpl_table_get_nrow(tab), 3);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_table_delete(tab);
    tab = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_fctable_create_fctable(void) {
    sph_fctable* fctab = NULL;
    cpl_frame* zexpcube = NULL;
    int nframes = 10;
    int curid   = 4;
    double mjd_obs = 0;
    double dit = 1.0;
    double utc = 0.0;
    //double				frame_ut = 0;
    const char* date_obs = "1990-10-20T23:31:40.0";



    sph_time_mjd_from_string(&mjd_obs, "1990-10-20T00:00:0.0");
    //printf( "mjd_obs = %f", mjd_obs);
    //sph_skycalc_set_mjd( skyinp, mjd);

    //sph_skycalc_setup(skyinp, "19 19 19", "2 2 2", 2000, "1990 10 20", "00 00 00");
    //sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT: pa = %f\n", skyout.pa);

    sph_test_simulator_create_zplpreproc_imaging_cube_ra_dec_time(&zexpcube,
            nframes, 191919.0, 20202.0, mjd_obs, date_obs, utc, dit, curid);

    //single
    fctab = sph_fctable_create_fctable(zexpcube, SPH_FCTAB_SINGLE);
    cpl_test_nonnull( fctab );
    cpl_test_abs(sph_fctable_get_angle( fctab, 0 ), -136.09, 0.1);
    sph_fctable_save_ascii(fctab, "create_fctab_single.txt");
    sph_fctable_delete(fctab);
    fctab = NULL;

/*
    //double
    fctab = sph_fctable_create_fctable(zexpcube, SPH_FCTAB_DOUBLE);
    cpl_test_nonnull( fctab );
    cpl_test_abs(sph_fctable_get_angle( fctab, 0 ), 143.2, 0.1);
    sph_fctable_save_ascii(fctab, "fctab_double.txt");

    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_frame_delete(zexpcube);
    zexpcube = NULL;

    //single without needed keywords
    sph_test_simulator_create_zplpreproc_imaging_cube(&zexpcube, nframes);
    fctab = sph_fctable_create_fctable(zexpcube, SPH_FCTAB_SINGLE);
    cpl_test_nonnull( fctab );
    sph_fctable_save_ascii(fctab, "fctab_single_empty.txt");
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_frame_delete(zexpcube);
    zexpcube = NULL;

    //single without needed keywords
    sph_test_simulator_create_zplpreproc_imaging_cube(&zexpcube, nframes);
    fctab = sph_fctable_create_fctable(zexpcube, SPH_FCTAB_DOUBLE);
    cpl_test_nonnull( fctab );
    sph_fctable_save_ascii(fctab, "fctab_double_empty.txt");
    sph_fctable_delete(fctab);
    fctab = NULL;
*/

    cpl_frame_delete(zexpcube);
    zexpcube = NULL;

}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_fctable_find_fctable(void) {
    sph_fctable* fctab = NULL;
    sph_ird_instrument_model* insmodel = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frameset* fcframes = NULL;
    int nx = 512;
    int ny = 512;
    int ff = 0;
    double xcl = 0.0;
    double xcr = 0.0;
    double ycl = 0.0;
    double ycr = 0.0;
    char* fname = NULL;
    char* fname2 = NULL;
    cpl_frame* aframe = NULL;
    fcframes = cpl_frameset_new();
    insmodel = sph_ird_instrument_model_new();
    insmodel->detsize_pixels_x = nx;
    insmodel->detsize_pixels_y = ny;
    insmodel->split_pixel_x = nx / 2;
    insmodel->window_size_y = ny / 2;
    inframes = cutest_sph_fctable_create_frames(2, insmodel->detsize_pixels_x,
            insmodel->detsize_pixels_y, NULL);
    sph_ird_instrument_model_get_centre_left_subframe_coords(insmodel, &xcl,
            &ycl);
    sph_ird_instrument_model_get_centre_right_subframe_coords(insmodel, &xcr,
            &ycr);

    for (ff = 0; ff < 2; ++ff) {
        fctab = sph_fctable_new();
        sph_fctable_add_row_double(fctab, 0.0, xcl, ycl, xcr, ycr, 0.0);
        fname2 = sph_filemanager_filename_new_ext(
                cpl_frame_get_filename(cpl_frameset_get_position(inframes, ff)),
                "txt");
        fname = sph_filemanager_new_filename_from_base(fname2, "fctable");
        sph_fctable_save_ascii(fctab, fname);
        aframe = cpl_frame_new();
        cpl_frame_set_filename(aframe, fname);
        cpl_frame_set_tag(aframe, "ATAG");
        cpl_frame_set_group(aframe, CPL_FRAME_GROUP_CALIB);
        cpl_frameset_insert(fcframes, aframe);
        cpl_free(fname2);
        fname2 = NULL;
        cpl_free(fname);
        fname = NULL;
        sph_fctable_delete(fctab);
        fctab = NULL;
    }

    fctab = sph_fctable_find_fctable(cpl_frameset_get_position(inframes, 1),
            fcframes);
    cpl_test_nonnull( fctab );
    sph_fctable_save_ascii(fctab, "fctab.txt");
    sph_fctable_delete(fctab);
    fctab = NULL;

    sph_ird_instrument_model_delete(insmodel);
    insmodel = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    cpl_frameset_delete(fcframes);
    fcframes = NULL;
}
static
void cutest_sph_fctable_construct_filename(void) {
    cpl_frame* aframe = NULL;
    const char* fname = "hello.fits";
    char* result = NULL;
    aframe = cpl_frame_new();
    cpl_test_nonnull( aframe );
    cpl_frame_set_group(aframe, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_tag(aframe, "NONE");
    cpl_frame_set_filename(aframe, fname);
    result = sph_fctable_construct_filename(aframe, 0);
    cpl_test_nonnull( result );
    cpl_test_eq_string(result, "hello_fctable.fits");
    cpl_free(result);
    result = NULL;
    result = sph_fctable_construct_filename(aframe, 1);
    cpl_test_nonnull( result );
    cpl_test_eq_string(result, "hello_fctable.txt");
    cpl_free(result);
    result = NULL;
    cpl_frame_delete(aframe);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A testsuite", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "Testing constructor",
                    cutest_sph_fctable_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Adding row (single view)",
                    cutest_sph_fctable_add_row_single)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Adding row (double view)",
                    cutest_sph_fctable_add_row_double)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Loading from ASCII (single)",
                    cutest_sph_fctable_load_ascii_single)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Loading from FITS (single)",
                    cutest_sph_fctable_load_fits_single)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Loading from ASCII (double)",
                    cutest_sph_fctable_load_ascii_double)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Setting row centres double",
                    cutest_sph_fctable_set_cent_double)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "get vectors", cutest_sph_fctable_get)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Saving to ASCII (single)",
                    cutest_sph_fctable_save_ascii_single)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Saving to ASCII (double)",
                    cutest_sph_fctable_save_ascii_double)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Get as cpl_table (double)",
                    cutest_sph_fctable_get_as_cpl_table)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Get as cpl_table (single)",
                    cutest_sph_fctable_get_as_cpl_table_single)) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing create fctable constructor",
                    cutest_sph_fctable_create_fctable)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing find fctable, same filename base",
                    cutest_sph_fctable_find_fctable)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing constructing ffilename",
                    cutest_sph_fctable_construct_filename)) {
        return sph_test_get_error();
    }


    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
