/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>

#include <string.h>
#include <stdio.h>
#ifdef HAVE_UNISTD_H
/* Needed for access() */
#include <unistd.h>
#endif

#include "cutest_sph_filter_generation.h"
/*-----------------------------------------------------------------------------
 Define
 -----------------------------------------------------------------------------*/


#ifndef SPH_IRD_FILTER_TABLE_IRDIS
#define SPH_IRD_FILTER_TABLE_IRDIS "spher-calib/ird/sph_ird_filt_table.fits"
#endif


#ifndef SPH_IRD_FILTER_TABLE_ZIMPOL
#define SPH_IRD_FILTER_TABLE_ZIMPOL "spher-calib/zpl/sph_zpl_filt_table.fits"
#endif

/*-----------------------------------------------------------------------------
 Function declarations
 -----------------------------------------------------------------------------*/

static void sph_ird_filter_doit(const char *, const char *, const char *,
                                cpl_boolean);

static void sph_zpl_filter_doit(const char *, const char *, const char *,
                                cpl_boolean);



/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char* argv[]) {

    char * base;
    char *end;
    int ndir = 5; /* Depth of source directory */

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    cpl_test_lt(0, argc);

    base = cpl_strdup(argv[0]);

    while ((end = strrchr(base, '/')) > base && --ndir >= 0) {
        *end = 0;
    }
    if (end != NULL) {
        const char *pipeid = strrchr(argv[0], '/');

        end[1] = 0; /* Truncate after '/' */

        const cpl_boolean doit = argc == 2;

        if(doit)
        	cpl_msg_info(cpl_func, "Writing of the fits files enabled");
        else
        	cpl_msg_info(cpl_func, "Writing of the fits files not enabled");

        sph_ird_filter_doit(base, SPH_IRD_FILTER_TABLE_IRDIS,
                            pipeid ? pipeid + 1 : NULL,
                            doit);

        sph_zpl_filter_doit(base, SPH_IRD_FILTER_TABLE_ZIMPOL,
                            pipeid ? pipeid + 1 : NULL,
							doit);
    }

    cpl_free(base);

    return cpl_test_end(0);
}

/**@}*/



/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Append Neutral Density filter magnitudes to the specified FITS file for
 IRDIS
 @param path   Initial path to filter frame
 @param base   Name of FITS file
 @param pipeid Name of FITS producing "recipe"
 @param doit   Perform the write operation
 @return   void

 */
/*----------------------------------------------------------------------------*/
static void sph_ird_filter_doit(const char * path,
                                const char * base,
                                const char * pipeid,
                                cpl_boolean doit)
{

#define CL_SZ 4
#define ND_SZ 3
        const char* ndlab[ND_SZ] = {"ND_1.0", "ND_2.0", "ND_3.5"};
        const char* cllab[CL_SZ] = {"B_Y", "B_J", "B_H", "B_Ks"};

        const double * nd[ND_SZ] = {NULL, NULL, NULL};
        const double nd1[CL_SZ] = {-2.62, -3.00, -2.29, -2.04};
        const double nd2[CL_SZ] = {-5.32, -6.17, -4.64, -4.09};
        const double nd3_5[CL_SZ] = {-9.01, -10.47, -7.61, -7.25};
        nd[0] = nd1;
        nd[1] = nd2;
        nd[2] = nd3_5;

        const size_t  nd_sz = sizeof(ndlab) / sizeof(ndlab[0]);

        sph_filter_doit(path, base, pipeid, ndlab, nd_sz, cllab, CL_SZ,
        		nd, doit, "IRD_FILTER_TABLE", SPH_FILTER_GENERATION_ARM_IRDIS);
#undef CL_SZ
#undef ND_SZ
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Append Neutral Density filter magnitudes to the specified FITS file for
 ZIMPOL
 @param path   Initial path to filter frame
 @param base   Name of FITS file
 @param pipeid Name of FITS producing "recipe"
 @param doit   Perform the write operation
 @return   void

 */
/*----------------------------------------------------------------------------*/
static void sph_zpl_filter_doit(const char * path,
                                const char * base,
                                const char * pipeid,
                                cpl_boolean doit)
{

#define CL_SZ 3
#define ND_SZ 2
        const char* ndlab[ND_SZ] = {"ND_1.0", "ND_2.0"};
        const char* cllab[CL_SZ] = {"V", "N_I", "N_R"};

        const double * nd[ND_SZ] = {NULL, NULL};
        const double nd1[CL_SZ] = {-2.65, -2.70, -2.67};
        const double nd2[CL_SZ] = {-5.43, -5.01, -5.27};
        nd[0] = nd1;
        nd[1] = nd2;

        sph_filter_doit(path, base, pipeid, ndlab, ND_SZ, cllab, CL_SZ, nd, doit,
        		"ZPL_FILTER_TABLE", SPH_FILTER_GENERATION_ARM_ZIMPOL);
#undef CL_SZ
}
