/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_image_grid.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_image_grid_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_image_grid_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_new(void) {
    sph_image_grid* p = NULL;

    p = sph_image_grid_new();

    cpl_test_nonnull( p );

    cpl_test_zero(p->nx);
    cpl_test_zero(p->ny);

    sph_image_grid_delete(p);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_rotate_and_shift(void) {
    sph_image_grid* p = NULL;
    sph_image_grid* target = NULL;
    cpl_image* image = NULL;
    cpl_image* timage = NULL;
    cpl_image* result_image = NULL;
    int xx = 0;
    int yy = 0;
    sph_error_code rerr = 0;

    image = cpl_image_new(48, 48, CPL_TYPE_DOUBLE);
    timage = cpl_image_new(48, 48, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( image );
    cpl_test_nonnull( timage );
    for (xx = 10; xx < 28; ++xx) {
        for (yy = 10; yy < 28; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 1.2);
        }
    }
    p = sph_image_grid_new_from_image(image, 12.0, 260.0, 12.0, 260.0);
    cpl_test_nonnull( p );
    target = sph_image_grid_new_from_image(timage, 0.0, 248.0, 0.0, 248.0);
    cpl_test_nonnull( target );

    rerr = sph_image_grid_rotate_and_shift(target, 25.0, 25.0, 15.0, 00.0,
            00.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_image_grid_map_to_target(p, target);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( p->polys );
    cpl_test_nonnull( p->values );

    cpl_test_eq(target->nx, 48);
    cpl_test_eq(target->ny, 48);
    for (xx = 10; xx < 28; ++xx) {
        for (yy = 10; yy < 28; ++yy) {
            const sph_polygon* poly = target->polys[yy * target->nx + xx];
            cpl_test_nonnull( poly );
            cpl_test_eq(poly->npoints, 4);
        }
    }
    result_image = sph_image_grid_extract_image(target);
    cpl_test_nonnull( result_image );
    cpl_image_save(result_image, "rotated.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_test_abs( cpl_image_get_flux(image),
            cpl_image_get_flux(result_image), 0.01);
    sph_image_grid_delete(p);
    sph_image_grid_delete(target);
    cpl_image_delete(image);
    cpl_image_delete(timage);
    cpl_image_delete(result_image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_mark_point function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_mark_point(void) {
    sph_image_grid* target = NULL;
    cpl_image* timage = NULL;
    cpl_image* result_image = NULL;
    double tv = 0.0;
    int bad = 0;
    sph_error_code rerr = CPL_ERROR_NONE;

    timage = cpl_image_new(64, 64, CPL_TYPE_DOUBLE);
    target = sph_image_grid_new_from_image(timage, 0.0, 16.0, 0.0, 16.0);
    cpl_test_nonnull( target );

    rerr = sph_image_grid_mark_point(target, 8.0, 8.0, 12.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result_image = sph_image_grid_extract_image(target);
    cpl_test_nonnull( result_image );
    tv = cpl_image_get(result_image, 33, 33, &bad);
    cpl_test_zero(bad);
    cpl_test_abs( tv, 12.0, 0.0000000000001);
    cpl_image_delete(timage);
    cpl_image_delete(result_image);
    sph_image_grid_delete(target);

    return;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_get_xy function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_get_xy(void) {
    sph_image_grid* target = NULL;
    cpl_image* timage = NULL;
    int xx = 0;
    int yy = 0;
    sph_error_code rerr = CPL_ERROR_NONE;

    timage = cpl_image_new(64, 64, CPL_TYPE_DOUBLE);
    target = sph_image_grid_new_from_image(timage, 0.0, 16.0, 0.0, 16.0);
    cpl_test_nonnull( target );

    rerr = sph_image_grid_get_xy(target, 0.000001, 0.00001, &xx, &yy);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(xx);
    cpl_test_zero(yy);
    rerr = sph_image_grid_get_xy(target, 8.00000, 8.000, &xx, &yy);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(xx, 32);
    cpl_test_eq(yy, 32);
    rerr = sph_image_grid_get_xy(target, 7.99999999, 0.0, &xx, &yy);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(xx, 31);
    cpl_test_zero(yy);
    rerr = sph_image_grid_get_xy(target, 7.99999999, 7.9999999, &xx, &yy);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(xx, 31);
    cpl_test_eq(yy, 31);
    sph_image_grid_rotate_and_shift(target, 0.0, 0.0, -90.0, 0.0, 0.0);
    rerr = sph_image_grid_get_xy(target, 7.99999999, -7.9999999, &xx, &yy);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(xx, 31);
    cpl_test_eq(yy, 31);
    sph_image_grid_rotate_and_shift(target, 0.0, 0.0, 0.0, 8.0, 0.0);
    rerr = sph_image_grid_get_xy(target, 8.000000, -8.00000, &xx, &yy);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(xx, 32);
    cpl_test_zero(yy);
    sph_image_grid_delete(target);
    cpl_image_delete(timage);

    return;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_map_polygon function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_map_polygon(void) {
    sph_image_grid* target = NULL;
    cpl_image* timage = NULL;
    cpl_image* result_image = NULL;
    sph_polygon* poly = NULL;

    timage = cpl_image_new(48, 48, CPL_TYPE_DOUBLE);
    target = sph_image_grid_new_from_image(timage, -8.0, 8.0, 0.0, 16.0);
    sph_image_grid_rotate_and_shift(target, -8.0, 0.0, -45.0, 0.0, 0.0);
    cpl_test_nonnull( target );

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, -4.0, -4.0);
    sph_polygon_add_point(poly, 4.0, -4.0);
    sph_polygon_add_point(poly, 7.0, 0.0);
    sph_polygon_add_point(poly, 4.0, 4.0);
    sph_polygon_add_point(poly, -4.0, 4.0);
    sph_polygon_add_point(poly, -7.0, 0.0);
    sph_image_grid_map_polygon(target, poly, 10.0);
    result_image = sph_image_grid_extract_image(target);
    cpl_test_nonnull( result_image );
    cpl_image_save(result_image, "mappedpoly.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    sph_image_grid_delete(target);
    cpl_image_delete(timage);
    cpl_image_delete(result_image);
    sph_polygon_delete(poly);
    return;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_map_to_target(void) {
    sph_image_grid* p = NULL;
    sph_image_grid* target = NULL;
    cpl_image* image = NULL;
    cpl_image* timage = NULL;
    cpl_image* result_image = NULL;
    int xx = 0;
    int yy = 0;

    image = cpl_image_new(48, 48, CPL_TYPE_DOUBLE);
    timage = cpl_image_new(48, 48, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( image );
    cpl_test_nonnull( timage );
    for (xx = 10; xx < 28; ++xx) {
        for (yy = 10; yy < 28; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 1.2);
        }
    }
    p = sph_image_grid_new_from_image(image, 12.0, 260.0, 12.0, 260.0);
    cpl_test_nonnull( p );
    target = sph_image_grid_new_from_image(timage, 0.0, 248.0, 0.0, 248.0);
    cpl_test_nonnull( target );

    sph_image_grid_map_to_target(p, target);
    cpl_test_nonnull( p->polys );
    cpl_test_nonnull( p->values );

    cpl_test_eq(target->nx, 48);
    cpl_test_eq(target->ny, 48);
    for (xx = 10; xx < 28; ++xx) {
        for (yy = 10; yy < 28; ++yy) {
            const sph_polygon* poly = target->polys[yy * target->nx + xx];
            cpl_test_nonnull( poly );
            cpl_test_eq(poly->npoints, 4);
        }
    }
    result_image = sph_image_grid_extract_image(target);
    cpl_test_nonnull( result_image );
    cpl_image_save(result_image, "mapped.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    sph_image_grid_delete(p);
    sph_image_grid_delete(target);
    cpl_image_delete(image);
    cpl_image_delete(timage);
    cpl_image_delete(result_image);
    return;
}

static
void cutest_image_grid_map_to_target_caseB(void) {
    sph_image_grid* p = NULL;
    sph_image_grid* target = NULL;
    cpl_image* image = NULL;
    cpl_image* result_image = NULL;
    int xx = 0;
    int yy = 0;

    image = cpl_image_new(50, 50, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( image );
    target = sph_image_grid_new_from_image(image, 1.0, 11.0, 1.0, 11.0);
    cpl_test_nonnull( target );
    for (xx = 10; xx < 28; ++xx) {
        for (yy = 10; yy < 28; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 1.2);
        }
    }
    p = sph_image_grid_new_from_image(image, 0.0, 10.0, 0.0, 10.0);
    cpl_test_nonnull( p );

    sph_image_grid_map_to_target(p, target);

    cpl_test_eq(target->nx, 50);
    cpl_test_eq(target->ny, 50);
    for (xx = 10; xx < 28; ++xx) {
        for (yy = 10; yy < 28; ++yy) {
            const sph_polygon* poly = target->polys[yy * target->nx + xx];
            cpl_test_nonnull( poly );
            cpl_test_eq(poly->npoints, 4);
        }
    }
    result_image = sph_image_grid_extract_image(target);
    cpl_test_nonnull( result_image );
    cpl_image_save(result_image, "mappedB.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    sph_image_grid_delete(target);
    sph_image_grid_delete(p);
    cpl_image_delete(image);
    cpl_image_delete(result_image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_new_from_image(void) {
    sph_image_grid* p = NULL;
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;

    image = cpl_image_new(248, 248, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( image );
    for (xx = 20; xx < 228; ++xx) {
        for (yy = 20; yy < 228; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 1.2);
        }
    }
    p = sph_image_grid_new_from_image(image, 12.0, 260.0, 12.0, 260.0);
    cpl_test_nonnull( p );

    cpl_test_nonnull( p->polys );
    cpl_test_nonnull( p->values );

    cpl_test_eq(p->nx, 248);
    cpl_test_eq(p->ny, 248);
    for (xx = 20; xx < 228; ++xx) {
        for (yy = 20; yy < 228; ++yy) {
            const sph_polygon* poly = p->polys[yy * p->nx + xx];
            cpl_test_nonnull( poly );
            cpl_test_eq(poly->npoints, 4);
        }
    }cpl_test_abs( p->polys[0]->points[0].x, 12.0, 1.0e-15);
    cpl_test_abs( p->polys[0]->points[0].y, 12.0, 1.0e-15);
    sph_image_grid_delete(p);
    cpl_image_delete(image);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_extract_image(void) {
    sph_image_grid* p = NULL;
    cpl_image* image = NULL;
    cpl_image* result = NULL;
    int xx = 0;
    int yy = 0;
    double sum = 0.0;

    image = cpl_image_new(248, 248, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( image );
    for (xx = 20; xx < 228; ++xx) {
        for (yy = 20; yy < 228; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 1.2);
        }
    }
    p = sph_image_grid_new_from_image(image, 12.0, 260.0, 12.0, 260.0);
    cpl_test_nonnull( p );
    sum = cpl_image_get_flux(image);
    cpl_test_abs( sum, 208 * 208 * 1.2, 1.0e-6);

    cpl_test_nonnull( p->polys );
    cpl_test_nonnull( p->values );

    cpl_test_eq(p->nx, 248);
    cpl_test_eq(p->ny, 248);
    for (xx = 20; xx < 228; ++xx) {
        for (yy = 20; yy < 228; ++yy) {
            const sph_polygon* poly = p->polys[yy * p->nx + xx];
            cpl_test_nonnull( poly );
            cpl_test_eq(poly->npoints, 4);
        }
    }cpl_test_abs( p->polys[0]->points[0].x, 12.0, 1.0e-6);
    cpl_test_abs( p->polys[0]->points[0].y, 12.0, 1.0e-6);

    result = sph_image_grid_extract_image(p);
    cpl_test_eq(cpl_image_get_size_x(result), 248);
    cpl_test_eq(cpl_image_get_size_y(result), 248);
    sum = cpl_image_get_flux(result);
    cpl_test_abs( sum, 208 * 208 * 1.2, 1.0e-7);
    cpl_image_subtract(result, image);
    sum = cpl_image_get_flux(result);
    cpl_test_abs( sum, 0.0, 1.0e-12);

    sph_image_grid_delete(p);
    cpl_image_delete(result);
    cpl_image_delete(image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_image_grid_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_image_grid_delete(void) {
    sph_image_grid* p = NULL;
    p = sph_image_grid_new();

    cpl_test_nonnull( p );

    sph_image_grid_delete(p);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("image_grid_test", cutest_init_image_grid_testsuite,
            cutest_clean_image_grid_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_new", cutest_image_grid_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_new_from_image",
                    cutest_image_grid_new_from_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_extract_image",
                    cutest_image_grid_extract_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_get_xy",
                    cutest_image_grid_get_xy)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_mark_point",
                    cutest_image_grid_mark_point)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_map_to_target",
                    cutest_image_grid_map_to_target)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_map_polygon",
                    cutest_image_grid_map_polygon)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_map_to_target_caseB",
                    cutest_image_grid_map_to_target_caseB)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_rotate_and_shift",
                    cutest_image_grid_rotate_and_shift)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_image_grid_delete",
                    cutest_image_grid_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
