/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_shared_irdifs.h"

#include <assert.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_IRDIFS_BASE "sph_irdifs"
#define SPH_IRDIFS_NX   10
#define SPH_IRDIFS_NY   10

/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/

static void sph_irdifs_compute_ron_test(void);
static double sph_irdifs_frameset_insert_one(cpl_frameset*, cpl_size)
    CPL_HAVE_ATTR_NONNULL;
static double sph_irdifs_frameset_insert(cpl_frameset*, cpl_size)
    CPL_HAVE_ATTR_NONNULL;

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of IRDIFS functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */

    sph_irdifs_compute_ron_test();

    return cpl_test_end(0);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit-test the function
  @return   void
 */
/*----------------------------------------------------------------------------*/
static void sph_irdifs_compute_ron_test(void)
{

    cpl_frameset*  frames = cpl_frameset_new();
    cpl_frame*     frame  = cpl_frame_new();
    double         value;
    double         stdev;
    cpl_error_code code;

    /* Null input */
    value = sph_irdifs_compute_ron(NULL, 5, 10, 1, 1, 100, 100);

    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_lt(value, 0.0);

    /* Empty frameset */
    value = sph_irdifs_compute_ron(frames, 5, 10, 1, 1, 100, 100);

    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_lt(value, 0.0);

    /* One bad frame */
    code = cpl_frame_set_tag(frame, "RON");
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_frameset_insert(frames, frame);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    value = sph_irdifs_compute_ron(frames, 5, 10, 1, 1, 100, 100);

    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_lt(value, 0.0);

    code = cpl_frame_set_filename(frame, "/dev/null");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    value = sph_irdifs_compute_ron(frames, 5, 10, 1, 1, 100, 100);

    cpl_test_error(CPL_ERROR_BAD_FILE_FORMAT);
    cpl_test_lt(value, 0.0);

    /* Various incorrect integer parameters */
    value = sph_irdifs_compute_ron(frames, -1, 10, 1, 1, 100, 100);

    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_lt(value, 0.0);

    value = sph_irdifs_compute_ron(frames, 11, 10, 1, 1, 100, 100);

    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_test_lt(value, 0.0);

    value = sph_irdifs_compute_ron(frames, 5, 10, 0, 1, 100, 100);

    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_lt(value, 0.0);

    value = sph_irdifs_compute_ron(frames, 5, 10, 1, 0, 100, 100);

    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_lt(value, 0.0);

    value = sph_irdifs_compute_ron(frames, 5, 10, 100, 1, 99, 100);

    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_test_lt(value, 0.0);

    value = sph_irdifs_compute_ron(frames, 5, 10, 1, 100, 1, 99);

    cpl_test_error(CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_test_lt(value, 0.0);

    /* Valid files */
    stdev = sph_irdifs_frameset_insert(frames, 4);
    cpl_test_eq(cpl_frameset_get_size(frames), 4);

    value = sph_irdifs_compute_ron(frames, 5, 10,
                                   1, SPH_IRDIFS_NX,
                                   1, SPH_IRDIFS_NY);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_lt(value, 0.0);

    stdev = sph_irdifs_frameset_insert_one(frames, 10);
    cpl_test_eq(cpl_frameset_get_size(frames), 1);

    value = sph_irdifs_compute_ron(frames, 5, 10,
                                   1, SPH_IRDIFS_NX,
                                   1, SPH_IRDIFS_NY);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, stdev, 0.0);

    stdev = sph_irdifs_frameset_insert(frames, 5);
    cpl_test_eq(cpl_frameset_get_size(frames), 5);

    value = sph_irdifs_compute_ron(frames, 5, 10,
                                   1, SPH_IRDIFS_NX,
                                   1, SPH_IRDIFS_NY);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(value, stdev, 0.0);

    cpl_frameset_delete(frames);
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Fill a frameset with one frame with a n-cube
  @param    n   The number of image planes, more than 5
  @return   The standard deviation
 */
/*----------------------------------------------------------------------------*/
static double sph_irdifs_frameset_insert_one(cpl_frameset* self, cpl_size n)
{
    cpl_imagelist* imglist = cpl_imagelist_new();
    cpl_frame*     frame   = cpl_frame_new();
    cpl_vector*    vector  = cpl_vector_new(n - 5);
    double         stdev;
    cpl_error_code code;

    assert( n > 5);

    (void)cpl_frameset_erase(self, "RON");

    for (int i = 0; i < n; i++) {
        cpl_image* img = cpl_image_new(SPH_IRDIFS_NX, SPH_IRDIFS_NY,
                                       CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(img, (double)i);
        cpl_imagelist_set(imglist, img, i);

        if (i >= 5)
            cpl_vector_set(vector, i-5, (double)i);
    }

    cpl_imagelist_save(imglist, SPH_IRDIFS_BASE ".fits", CPL_TYPE_UNSPECIFIED,
                       NULL, CPL_IO_CREATE);
    cpl_frame_set_tag(frame, "RON");
    cpl_frame_set_filename(frame, SPH_IRDIFS_BASE ".fits");
    cpl_frameset_insert(self, frame);

    stdev = cpl_vector_get_stdev(vector);

    cpl_imagelist_delete(imglist);
    cpl_vector_delete(vector);

    code = cpl_error_get_code();
    assert( code == CPL_ERROR_NONE);

    return stdev;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Fill a frameset with some frames
  @param    n   The number of frames
  @return   The standard deviation
 */
/*----------------------------------------------------------------------------*/
static double sph_irdifs_frameset_insert(cpl_frameset* self, cpl_size n)
{
    cpl_vector*    vector  = cpl_vector_new(n);
    double         stdev;
    cpl_error_code code;

    assert( n > 0);

    (void)cpl_frameset_erase(self, "RON");

    for (int i = 0; i < n; i++) {
        cpl_frame*     frame  = cpl_frame_new();
        cpl_image* img = cpl_image_new(SPH_IRDIFS_NX, SPH_IRDIFS_NY,
                                       CPL_TYPE_DOUBLE);
        char* file     = cpl_sprintf(SPH_IRDIFS_BASE "_%d.fits", i);

        cpl_image_add_scalar(img, (double)i);
        cpl_image_save(img, file, CPL_TYPE_UNSPECIFIED, NULL, CPL_IO_CREATE);

        cpl_frame_set_tag(frame, "RON");
        cpl_frame_set_filename(frame, file);
        cpl_frameset_insert(self, frame);

        cpl_vector_set(vector, i, (double)i);

        cpl_image_delete(img);
        cpl_free(file);
    }

    stdev = cpl_vector_get_stdev(vector);
    cpl_vector_delete(vector);

    code = cpl_error_get_code();
    assert( code == CPL_ERROR_NONE);

    return stdev;
}
/**@}*/
