/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_keyword_reader.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"
#include <float.h>
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and 
 *                                  associated functions.               
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static sph_keyword_reader*
cutest_util_keyword_reader_new_load(void)
    CPL_ATTR_ALLOC;

static sph_keyword_reader*
cutest_util_keyword_reader_new_load(void) {
    sph_keyword_reader* reader = NULL;
    FILE* filehandle = NULL;

    filehandle = fopen("keylist_test.txt", "w");
    fprintf(filehandle, "#Table of keywords\n");
    fprintf(filehandle, "#TAG        KEYWORD            TYPE        VALUE\n");
    fprintf(
            filehandle,
            "TEST_TAG_A        STRING_KEYWORD        STRING        \"A VALUE\"\n");
    fprintf(filehandle,
            "TEST_TAG_B        \"INT KEYWORD\"        INT            234\n");
    fprintf(filehandle,
            "TEST_TAG_C        DOUBLE_KEYWORD        DOUBLE        112.991\n");
    fprintf(filehandle,
            "TEST_TAG_D        \"ESO DPR TYPE\"    STRING      DARK\n");
    fprintf(filehandle,
            "TEST_TAG_E        \"ESO NREADS\"      INT         12\n");
    fprintf(filehandle,
            "TEST_TAG_F        \"ESO DET DIT\"         DOUBLE      1.0\n");

    fclose(filehandle);

    reader = sph_keyword_reader_new_load("keylist_test.txt");
    return reader;
}

static
int cutest_init_keyword_reader_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_keyword_reader_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_reader_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keyword_reader_new(void) {
    sph_keyword_reader* reader = NULL;

    reader = sph_keyword_reader_new();
    cpl_test_nonnull( reader );
    sph_keyword_reader_delete(reader);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_reader_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keyword_reader_new_load(void) {
    sph_keyword_reader* reader = NULL;

    reader = cutest_util_keyword_reader_new_load();

    cpl_test_nonnull( reader );
    cpl_test_eq(cpl_table_get_nrow(reader->table ), 6);
    sph_keyword_reader_delete(reader);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_reader_get_value_string function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keyword_reader_set_proplist_from_file(void) {
    cpl_frameset* frames = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_keyword_reader* reader = NULL;
    const cpl_frame* iframe;
    const char*      filename;

    reader = cutest_util_keyword_reader_new_load();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( reader );

    frames = cpl_frameset_new();

    rerr = sph_test_create_ird_raw_dark_frameset(frames, 1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    iframe = cpl_frameset_get_first_const(frames);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(iframe);

    filename = cpl_frame_get_filename(iframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(filename);

    rerr = sph_keyword_reader_set_proplist_from_file(reader, filename, 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    sph_keyword_reader_delete(reader);
    sph_test_frameset_delete(frames);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_reader_get_value_string function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keyword_reader_get_value_string(void) {
    cpl_frameset* frames = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_keyword_reader* reader = NULL;
    char* result = NULL;
    const cpl_frame* iframe;
    const char*      filename;
    int cmp = 0;

    reader = cutest_util_keyword_reader_new_load();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( reader );

    frames = cpl_frameset_new();

    rerr = sph_test_create_ird_raw_dark_frameset(frames, 1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    iframe = cpl_frameset_get_first_const(frames);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(iframe);

    filename = cpl_frame_get_filename(iframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(filename);

    rerr = sph_keyword_reader_set_proplist_from_file(reader, filename, 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    //result = sph_keyword_reader_get_value_string( reader, "TEST_TAG_A" );
    //cpl_test_nonnull( result  );
    //cmp = strncmp( result, "A VALUE", 6);
    //cpl_test_zero(cmp);
    //cpl_free( result );

    result = sph_keyword_reader_get_value_string(reader, "TEST_TAG_D");
    cpl_test_nonnull( result );
    cmp = strncmp(result, "DARK", 4);
    cpl_test_zero(cmp);
    cpl_free(result);

    sph_keyword_reader_delete(reader);
    sph_test_frameset_delete(frames);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_reader_get_value_int function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keyword_reader_get_value_int(void) {
    cpl_frameset* frames = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_keyword_reader* reader = NULL;
    const cpl_frame* iframe;
    const char*      filename;
    int result = 0;

    reader = cutest_util_keyword_reader_new_load();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( reader );

    frames = cpl_frameset_new();

    rerr = sph_test_create_ird_raw_dark_frameset(frames, 1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    iframe = cpl_frameset_get_first_const(frames);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(iframe);

    filename = cpl_frame_get_filename(iframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(filename);

    rerr = sph_keyword_reader_set_proplist_from_file(reader, filename, 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    result = sph_keyword_reader_get_value_int(reader, "TEST_TAG_E");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 1);

    sph_test_frameset_delete(frames);
    sph_keyword_reader_delete(reader);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_reader_get_value_double function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keyword_reader_get_value_double(void) {
    cpl_frameset* frames = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_keyword_reader* reader = NULL;
    const cpl_frame* iframe;
    const char*      filename;
    double result = 0.0;

    reader = cutest_util_keyword_reader_new_load();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( reader );

    frames = cpl_frameset_new();

    rerr = sph_test_create_ird_raw_dark_frameset(frames, 1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    iframe = cpl_frameset_get_first_const(frames);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(iframe);

    filename = cpl_frame_get_filename(iframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(filename);

    rerr = sph_keyword_reader_set_proplist_from_file(reader, filename, 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    result = sph_keyword_reader_get_value_double(reader,
            SPH_COMMON_KEYWORD_EXPTIME);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(result, 1.3, DBL_EPSILON);

    sph_test_frameset_delete(frames);
    sph_keyword_reader_delete(reader);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_reader_delete function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keyword_reader_delete(void) {
    sph_keyword_reader* reader = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    reader = sph_keyword_reader_new();
    cpl_test_nonnull( reader );

    rerr = sph_keyword_reader_delete(reader);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("keyword_reader_test",
            cutest_init_keyword_reader_testsuite,
            cutest_clean_keyword_reader_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_keyword_reader_new",
                    cutest_keyword_reader_new)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_keyword_reader_new_load",
                    cutest_keyword_reader_new_load)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_reader_set_proplist_from_file",
                    cutest_keyword_reader_set_proplist_from_file)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_reader_get_values_int",
                    cutest_keyword_reader_get_value_int)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_reader_get_value_double",
                    cutest_keyword_reader_get_value_double)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_keyword_reader_get_value_string",
                    cutest_keyword_reader_get_value_string)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_keyword_reader_delete",
                    cutest_keyword_reader_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
