/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>
#include <float.h>

#include "sph_point.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"
#include "sph_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and 
 *                                  associated functions.               
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_point_testsuite(void) {
    return sph_test_nop_code();
}

static
int cutest_clean_point_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_point_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_point_new(void) {
    sph_point* p = NULL;

    p = sph_point_new(-121233.1212, -1234245.4578);

    cpl_test_nonnull( p );

    cpl_test_eq(sizeof(*p), 2 * sizeof(double));

    cpl_test_rel( p->x, -121233.1212, DBL_EPSILON);
    cpl_test_rel( p->y, -1234245.4578, DBL_EPSILON);

    sph_point_delete(p);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_point_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_point_ccw_test(void) {
    sph_point* p1 = NULL;
    sph_point* p2 = NULL;
    sph_point* p3 = NULL;
    sph_point* p4 = NULL;
    sph_point* p5 = NULL;
    sph_point* p6 = NULL;
    int result = 0;

    p1 = sph_point_new(0.0, 0.0);
    p2 = sph_point_new(1.0, 0.0);
    p3 = sph_point_new(0.0, 1.0);
    p4 = sph_point_new(0.0, 0.0);
    p5 = sph_point_new(-1.0, 0.0);
    p6 = sph_point_new(0.0, 2.0);

    cpl_test_nonnull( p1 );
    cpl_test_nonnull( p2 );
    cpl_test_nonnull( p3 );
    cpl_test_nonnull( p4 );
    cpl_test_nonnull( p5 );
    cpl_test_nonnull( p6 );

    result = sph_point_ccw_test(*p1, *p2, *p3);
    cpl_test_eq(result, 1);
    result = sph_point_ccw_test(*p1, *p3, *p2);
    cpl_test_zero(result);
    result = sph_point_ccw_test(*p1, *p3, *p6);
    cpl_test_zero(result);
    result = sph_point_ccw_test(*p1, *p4, *p1);
    cpl_test_zero(result);
    result = sph_point_ccw_test(*p1, *p5, *p3);
    cpl_test_zero(result);

    /* Horizontal line */

    /* p3 and p6 on same side */
    result = sph_point_same_side(p1, p2, p3, p6);
    cpl_test_eq(result, CPL_TRUE);

    /* p3 and p6 on same side */
    result = sph_point_same_side(p2, p1, p3, p6);
    cpl_test_eq(result, CPL_TRUE);

    /* p3 and p6 on same side */
    result = sph_point_same_side(p2, p1, p6, p3);
    cpl_test_eq(result, CPL_TRUE);

    /* p3 and p6 on same side */
    result = sph_point_same_side(p1, p2, p6, p3);
    cpl_test_eq(result, CPL_TRUE);

    /* p3 and p3 on same side */
    result = sph_point_same_side(p1, p2, p3, p3);
    cpl_test_eq(result, CPL_TRUE);

    /* p3 and p3 on same side */
    result = sph_point_same_side(p2, p1, p3, p3);
    cpl_test_eq(result, CPL_TRUE);

    /* p3 and p3 on same side */
    result = sph_point_same_side(p1, p2, p3, p3);
    cpl_test_eq(result, CPL_TRUE);

    /* p1 and p2 on line */
    result = sph_point_same_side(p1, p2, p1, p2);
    cpl_test_eq(result, CPL_TRUE);

    /* p1 and p1 on line */
    result = sph_point_same_side(p1, p2, p1, p1);
    cpl_test_eq(result, CPL_TRUE);

    /* p2 and p2 on line */
    result = sph_point_same_side(p2, p1, p2, p2);
    cpl_test_eq(result, CPL_TRUE);

    /* p1 and p2 on line */
    result = sph_point_same_side(p2, p1, p1, p2);
    cpl_test_eq(result, CPL_TRUE);

    /* p1 and p1 on line */
    result = sph_point_same_side(p2, p1, p1, p1);
    cpl_test_eq(result, CPL_TRUE);

    /* p2 and p2 on line */
    result = sph_point_same_side(p2, p1, p2, p2);
    cpl_test_eq(result, CPL_TRUE);

    /* p3 and p6 on same side of a single point */
    result = sph_point_same_side(p1, p1, p3, p6);
    cpl_test_eq(result, CPL_TRUE);

    /* p2 and p5 on same side of a single point */
    result = sph_point_same_side(p1, p1, p2, p5);
    cpl_test_eq(result, CPL_TRUE);

    /* Vertical line */

    /* p2 and p5 on opposite sides */
    result = sph_point_same_side(p3, p6, p2, p5);
    cpl_test_zero(result);

    /* p2 and p5 on opposite sides */
    result = sph_point_same_side(p6, p3, p2, p5);
    cpl_test_zero(result);

    /* p2 and p5 on opposite sides */
    result = sph_point_same_side(p6, p3, p5, p2);
    cpl_test_zero(result);

    /* p2 and p5 on opposite sides */
    result = sph_point_same_side(p6, p3, p2, p5);
    cpl_test_zero(result);

    /* p2 and p2 on same side */
    result = sph_point_same_side(p6, p3, p2, p2);
    cpl_test_eq(result, 1);

    /* p2 and p2 on same side */
    result = sph_point_same_side(p3, p6, p2, p2);
    cpl_test_eq(result, 1);

    /* p2 and p5 on same side of single point */
    result = sph_point_same_side(p6, p6, p2, p5);
    cpl_test_eq(result, 1);

    /* p2 and p5 on same side of single point */
    result = sph_point_same_side(p6, p6, p5, p2);
    cpl_test_eq(result, 1);

    sph_point_delete(p1);
    sph_point_delete(p2);
    sph_point_delete(p3);
    sph_point_delete(p4);
    sph_point_delete(p5);
    sph_point_delete(p6);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_point_rotate_around function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_point_rotate_around(void) {
    sph_point* p1 = NULL;
    sph_point* p2 = NULL;
    sph_point* p3 = NULL;
    sph_point* p4 = NULL;
    sph_point* p5 = NULL;
    sph_point* p6 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int ii = 40;

    p1 = sph_point_new(0.0, 0.0);
    p2 = sph_point_new(1.0, 0.0);
    p3 = sph_point_new(0.0, 1.0);
    p4 = sph_point_new(0.0, 0.0);
    p5 = sph_point_new(-1.0, 0.0);
    p6 = sph_point_new(0.0, 2.0);

    cpl_test_nonnull( p1 );
    cpl_test_nonnull( p2 );
    cpl_test_nonnull( p3 );
    cpl_test_nonnull( p4 );
    cpl_test_nonnull( p5 );
    cpl_test_nonnull( p6 );

    rerr = sph_point_rotate_around(p1, 0.0, 0.0, 0.12, -0.145, 0.0, 0.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs( p1->x, 0.0, 1.0e-15);
    cpl_test_abs( p1->y, 0.0, 1.0e-15);
    rerr = sph_point_rotate_around(p5, -1.0, 0.0, 0.12, -0.145, 0.0, 0.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs( p5->x, -1.0, 1.0e-15);
    cpl_test_abs( p5->y, 0.0, 1.0e-15);
    rerr = sph_point_rotate_around(p1, 0.0, 0.0, 0.12, -0.145, 12.0, -4.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs( p1->x, 12.0, 1.0e-15);
    cpl_test_abs( p1->y, -4.0, 1.0e-15);
    rerr = sph_point_rotate_around(p3, 0.0, 0.0, cos(CPL_MATH_PI_2),
            sin(CPL_MATH_PI_2), 1.0, 0.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs( p3->x, 0.0, 1.0e-15);
    cpl_test_abs( p3->y, 0.0, 1.0e-15);
    for (ii = 0; ii < 40; ++ii) {
        rerr |= sph_point_rotate_around(p6, 0.0, 0.0, cos(CPL_MATH_PI_2),
                sin(CPL_MATH_PI_2), 1.0, 0.0);
    }cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs( p6->x, 0.0, 1.0e-14);
    cpl_test_abs( p6->y, 2.0, 1.0e-14);

    sph_point_delete(p1);
    sph_point_delete(p2);
    sph_point_delete(p3);
    sph_point_delete(p4);
    sph_point_delete(p5);
    sph_point_delete(p6);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_point_delete function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_point_delete(void) {
    sph_point* p = NULL;
    p = sph_point_new(-121233.1212, -1234245.4578);

    cpl_test_nonnull( p );

    sph_point_delete(p);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("point_test", cutest_init_point_testsuite,
            cutest_clean_point_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL == sph_test_do(pSuite, "sph_point_new", cutest_point_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_point_rotate_around",
                    cutest_point_rotate_around)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_point_ccw_test", cutest_point_ccw_test)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_point_delete", cutest_point_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
