/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_polygon.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_time.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"

#include <cpl.h>

#ifdef _OPENMP
#include <omp.h>
#endif

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <limits.h>


/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_polygon_testsuite(void) {
    return sph_test_nop_code();
}

static
int cutest_clean_polygon_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_new(void) {
    sph_polygon* result = NULL;

    result = sph_polygon_new();

    cpl_test_nonnull( result );

    sph_polygon_delete(result);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_duplicate(void) {
    sph_polygon* poly_old = NULL;
    sph_polygon* poly = NULL;
    sph_error_code rerr = 0;

    poly_old = sph_polygon_new();
    sph_polygon_add_point(poly_old, 1.0, 2.0);
    sph_polygon_add_point(poly_old, -1.0, -2.0);

    cpl_test_nonnull( poly_old );
    poly = sph_polygon_duplicate(poly_old);
    cpl_test_nonnull( poly );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(poly->npoints, 2);
    cpl_test_abs( poly->points[0].x, 1.0, 1.0e-15);
    cpl_test_abs( poly->points[0].y, 2.0, 1.0e-15);
    cpl_test_abs( poly->points[1].x, -1.0, 1.0e-15);
    cpl_test_abs( poly->points[1].y, -2.0, 1.0e-15);
    sph_polygon_delete(poly);
    sph_polygon_delete(poly_old);
    poly = sph_polygon_duplicate(NULL);
    cpl_test_null( poly);
    cpl_error_reset();
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_add_point(void) {
    sph_polygon* poly = NULL;
    sph_error_code rerr = 0;
    poly = sph_polygon_new();

    cpl_test_nonnull( poly );

    cpl_test_zero(poly->npoints);

    rerr = sph_polygon_add_point(poly, 1.0, 2.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs( poly->points[0].x, 1.0, 1.0e-15);
    cpl_test_abs( poly->points[0].y, 2.0, 1.0e-15);

    rerr = sph_polygon_add_point(poly, 3.0, 4.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(poly->npoints, 2);
    cpl_test_abs( poly->points[1].x, 3.0, 1.0e-15);
    cpl_test_abs( poly->points[1].y, 4.0, 1.0e-15);

    sph_polygon_delete(poly);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_get_edge(void) {
    sph_polygon* poly = NULL;
    sph_error_code rerr = 0;
    sph_line* edgeA = NULL;
    sph_line* edgeB = NULL;
    sph_line* edgeC = NULL;
    sph_line* edgeD = NULL;

    poly = sph_polygon_new();

    cpl_test_nonnull( poly );

    cpl_test_zero(poly->npoints);

    rerr = sph_polygon_add_point(poly, 0.0, 0.0);
    cpl_test_eq(poly->npoints, 1);
    rerr |= sph_polygon_add_point(poly, 1.0, 0.0);
    cpl_test_eq(poly->npoints, 2);
    rerr |= sph_polygon_add_point(poly, 1.0, 1.0);
    cpl_test_eq(poly->npoints, 3);
    rerr |= sph_polygon_add_point(poly, 0.0, 1.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_eq(poly->npoints, 4);

    edgeA = sph_polygon_get_edge(poly, 1);
    cpl_test_nonnull( edgeA );
    edgeB = sph_polygon_get_edge(poly, 2);
    cpl_test_nonnull( edgeB );
    edgeC = sph_polygon_get_edge(poly, 3);
    cpl_test_nonnull( edgeC );
    edgeD = sph_polygon_get_edge(poly, 4);
    cpl_test_nonnull( edgeD );
    cpl_test_abs( edgeA->p1->x, 0.0, 1.0e-15);
    cpl_test_abs( edgeA->p1->y, 0.0, 1.0e-15);
    cpl_test_abs( edgeA->p2->x, 1.0, 1.0e-15);
    cpl_test_abs( edgeA->p2->y, 0.0, 1.0e-15);

    cpl_test_abs( edgeB->p1->x, 1.0, 1.0e-15);
    cpl_test_abs( edgeB->p1->y, 0.0, 1.0e-15);
    cpl_test_abs( edgeB->p2->x, 1.0, 1.0e-15);
    cpl_test_abs( edgeB->p2->y, 1.0, 1.0e-15);

    cpl_test_abs( edgeC->p1->x, 1.0, 1.0e-15);
    cpl_test_abs( edgeC->p1->y, 1.0, 1.0e-15);
    cpl_test_abs( edgeC->p2->x, 0.0, 1.0e-15);
    cpl_test_abs( edgeC->p2->y, 1.0, 1.0e-15);

    cpl_test_abs( edgeD->p1->x, 0.0, 1.0e-15);
    cpl_test_abs( edgeD->p1->y, 1.0, 1.0e-15);
    cpl_test_abs( edgeD->p2->x, 0.0, 1.0e-15);
    cpl_test_abs( edgeD->p2->y, 0.0, 1.0e-15);
    sph_line_delete(edgeA);
    sph_line_delete(edgeB);
    sph_line_delete(edgeC);
    sph_line_delete(edgeD);

    sph_polygon_delete(poly);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_test_inside(void) {
    sph_polygon* poly = NULL;
    sph_error_code rerr = 0;
    poly = sph_polygon_new();

    cpl_test_nonnull( poly );
    rerr = sph_polygon_add_point(poly, 0.0, 0.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 0.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 0.0, 1.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_eq(sph_polygon_test_inside( poly, 0.0, 0.0 ), 1);
    cpl_test_eq(sph_polygon_test_inside( poly, 1.0, 0.0 ), 1);
    //cpl_test_eq(sph_polygon_test_inside( poly, 0.0, 1.0 ), 1 );
    cpl_test_eq(sph_polygon_test_inside( poly, 1.0, 1.0 ), 1);
    cpl_test_zero(sph_polygon_test_inside( poly, -1.0, 0.0 ));
    cpl_test_zero(sph_polygon_test_inside( poly, 1.0, -1.0 ));
    cpl_test_zero(sph_polygon_test_inside( poly, -1.0, 1.0 ));
    cpl_test_eq(sph_polygon_test_inside( poly, 0.5, 0.5 ), 1);

    sph_polygon_delete(poly);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_test_inside_im_hex_case(void) {
    sph_polygon* poly = NULL;
    sph_polygon* pixelpoly = NULL;
    sph_ifs_lenslet_model* model = NULL;

    int nx = 582;
    int ny = 582;
    int xx = 0;
    int yy = 0;
    int minx = 0;
    int miny = 0;
    int maxx = 0;
    int maxy = 0;
    double minx2 = 0;
    double miny2 = 0;
    double maxx2 = 0;
    double maxy2 = 0;
    cpl_image* image = NULL;
    double area = 0.0;
    double overlap = 0.0;
    int nn = 0;
    double cosan = cos(CPL_MATH_PI * 0.0 / 180.0);
    double sinan = sin(CPL_MATH_PI * 0.0 / 180.0);
    double pixsize = 0.0;
    sph_polygon_workspace* ws = NULL;

    ws = sph_polygon_workspace_new();
    cpl_test_nonnull( ws );

    model = sph_ifs_lenslet_model_new();
    poly = sph_ifs_lenslet_model_get_poly(model, 0, 0);
    sph_polygon_rotate_around(poly, 0.0, 0.0, cosan, sinan, 0.0, 0.0);
    pixsize = 4.0 * model->lensize_microns * model->lenslets_per_side / nx;

    for (nn = 0; nn < poly->npoints; ++nn) {
        poly->points[nn].x /= pixsize;
        poly->points[nn].y /= pixsize;
        poly->points[nn].x += nx / 2.0;
        poly->points[nn].y += ny / 2.0;
    }
    //sph_polygon_enlarge(poly,2.0);
    image = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    area = sph_polygon_area(poly);
    maxx = sph_polygon_get_right(poly);
    maxy = sph_polygon_get_top(poly);
    minx = sph_polygon_get_left(poly);
    miny = sph_polygon_get_bottom(poly);

    sph_polygon_get_extent(poly, &maxy2, &miny2, &minx2, &maxx2);
    cpl_test_eq(maxx, (int)maxx2);
    cpl_test_eq(maxy, (int)maxy2);
    cpl_test_eq(minx, (int)minx2);
    cpl_test_eq(miny, (int)miny2);

    if (maxx >= nx)
        maxx = nx - 1;
    if (maxy >= ny)
        maxy = ny - 1;
    if (minx < 0)
        minx = 0;
    if (miny < 0)
        miny = 0;
    for (xx = minx; xx <= maxx; ++xx) {
        for (yy = miny; yy <= maxy; ++yy) {
            pixelpoly = sph_polygon_new();
            sph_polygon_add_point(pixelpoly, xx, yy);
            sph_polygon_add_point(pixelpoly, xx + 1.0, yy);
            sph_polygon_add_point(pixelpoly, xx + 1.0, yy + 1.0);
            sph_polygon_add_point(pixelpoly, xx, yy + 1.0);
            sph_polygon_clip_to_target(pixelpoly, poly, ws);
            overlap = sph_polygon_area(pixelpoly);
            if (overlap > 0.0) {
                if (sph_polygon_test_inside(poly, xx + 1.0, yy) < 1
                        && sph_polygon_test_inside(poly, xx + 1.0, yy + 1.0) < 1
                        && sph_polygon_test_inside(poly, xx, yy + 1.0) < 1
                        && sph_polygon_test_inside(poly, xx, yy) < 1) {
                    if (sph_polygon_test_inside(poly, xx, yy) < 1)
                        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                                __LINE__, SPH_ERROR_ERROR, "Point was: %f, %f",
                                xx + 0.0, yy + 0.0);
                    if (sph_polygon_test_inside(poly, xx + 1.0, yy) < 1)
                        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                                __LINE__, SPH_ERROR_ERROR, "Point was: %f, %f",
                                xx + 1.0, yy + 0.0);
                    if (sph_polygon_test_inside(poly, xx + 1.0, yy + 1.0) < 1)
                        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                                __LINE__, SPH_ERROR_ERROR, "Point was: %f, %f",
                                xx + 1.0, yy + 1.0);
                    if (sph_polygon_test_inside(poly, xx, yy + 1.0) < 1)
                        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__,
                                __LINE__, SPH_ERROR_ERROR, "Point was: %f, %f",
                                xx + 0.0, yy + 1.0);
                    sph_test_later(
                            "Failed to have inside point with positive overlap!");
                }
                cpl_image_set(image, xx + 1, yy + 1, overlap);
            }
            sph_polygon_delete(pixelpoly);
            pixelpoly = NULL;
        }
    }
    cpl_image_save(image, "testim.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    sph_polygon_workspace_delete(ws);
    sph_polygon_delete(poly);
    sph_ifs_lenslet_model_delete(model);
    cpl_image_delete(image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_same_side(void) {
    sph_polygon* poly = NULL;
    sph_error_code rerr = 0;
    sph_line* workline = NULL;
    sph_point* point1 = NULL;
    sph_point* point2 = NULL;
    sph_point* point3 = NULL;
    sph_point* point4 = NULL;
    sph_point* point5 = NULL;
    sph_point* point6 = NULL;
    sph_point* point7 = NULL;
    poly = sph_polygon_new();
    workline = sph_line_new(0.0, 0.0, 0.0, 0.0);

    cpl_test_nonnull( poly );
    cpl_test_nonnull( workline );
    rerr = sph_polygon_add_point(poly, 0.0, 0.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 0.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 0.0, 1.0);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    point1 = sph_point_new(0.0, 0.0);
    point2 = sph_point_new(1.0, 0.0);
    point3 = sph_point_new(1.0, 1.0);
    point4 = sph_point_new(0.1, 0.1);
    point5 = sph_point_new(1.1, 1.1);
    point6 = sph_point_new(-0.1, -0.1);
    point7 = sph_point_new(0.0, 0.5);

    cpl_test_eq(sph_polygon_same_side( poly, 1, point1),
            1);
    cpl_test_eq(sph_polygon_same_side( poly, 1, point2),
            1);
    cpl_test_eq(sph_polygon_same_side( poly, 1, point3),
            1);
    cpl_test_eq(sph_polygon_same_side( poly, 2, point3),
            1);
    cpl_test_eq(sph_polygon_same_side( poly, 3, point4),
            1);
    cpl_test_eq(sph_polygon_same_side( poly, 3, point5),
            0);
    cpl_test_eq(sph_polygon_same_side( poly, 4, point6),
            0);
    cpl_test_eq(sph_polygon_same_side( poly, 4, point7),
            1);

    sph_polygon_delete(poly);
    poly = NULL;
    sph_line_delete(workline);
    workline = NULL;
    sph_point_delete(point1);
    point1 = NULL;
    sph_point_delete(point2);
    point2 = NULL;
    sph_point_delete(point3);
    point3 = NULL;
    sph_point_delete(point4);
    point4 = NULL;
    sph_point_delete(point5);
    point5 = NULL;
    sph_point_delete(point6);
    point6 = NULL;
    sph_point_delete(point7);
    point7 = NULL;

    cpl_test_error(CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_clipping function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_clip_to_target_caseA(void) {
    sph_polygon* poly = NULL;
    sph_polygon* target = NULL;
    sph_polygon_workspace* ws = NULL;
    const sph_polygon* areapoly;
    sph_error_code rerr = 0;

    poly = sph_polygon_new();
    ws = sph_polygon_workspace_new();
    cpl_test_nonnull( ws );
    cpl_test_nonnull( poly );
    target = sph_polygon_new();

    cpl_test_nonnull( target );

    rerr = sph_polygon_add_point(target, 0.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 2.0);
    rerr |= sph_polygon_add_point(target, 0.0, 2.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_polygon_add_point(poly, 1.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 3.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 3.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    areapoly = sph_polygon_clip_to_target(poly, target, ws);
    cpl_test_nonnull( areapoly );

    cpl_test_eq(areapoly->npoints, 4);
    cpl_test_abs( areapoly->points[0].x, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[0].y, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[1].x, 2.0, 1.0e-15);
    cpl_test_abs( areapoly->points[1].y, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[2].x, 2.0, 1.0e-15);
    cpl_test_abs( areapoly->points[2].y, 2.0, 1.0e-15);
    cpl_test_abs( areapoly->points[3].x, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[3].y, 2.0, 1.0e-15);
    sph_polygon_workspace_delete(ws);
    sph_polygon_delete(poly);
    sph_polygon_delete(target);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_clipping function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_clip_to_target_caseB(void) {
    sph_polygon* poly = NULL;
    sph_polygon* target = NULL;
    sph_error_code rerr = 0;
    sph_polygon_workspace* ws = NULL;
    const sph_polygon* areapoly;

    poly = sph_polygon_new();
    ws = sph_polygon_workspace_new();
    cpl_test_nonnull( ws );

    cpl_test_nonnull( poly );
    target = sph_polygon_new();

    cpl_test_nonnull( target );

    rerr = sph_polygon_add_point(poly, 0.0, 0.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 0.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 1.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_polygon_add_point(target, -20.0, -20.0);
    rerr |= sph_polygon_add_point(target, 2.0, 0.0);
    rerr |= sph_polygon_add_point(target, 10.0, 10.0);
    rerr |= sph_polygon_add_point(target, 0.0, 3.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    areapoly = sph_polygon_clip_to_target(poly, target, ws);
    cpl_test_nonnull( areapoly );

    cpl_test_eq(areapoly->npoints, 3);
    cpl_test_abs( areapoly->points[0].x, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[0].y, 0.0, 1.0e-15);
    cpl_test_abs( areapoly->points[1].x, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[1].y, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[2].x, 0.0, 1.0e-15);
    cpl_test_abs( areapoly->points[2].y, 0.0, 1.0e-15);
    sph_polygon_workspace_delete(ws);
    sph_polygon_delete(poly);
    sph_polygon_delete(target);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_clipping function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_clip_to_target_caseC(void) {
    sph_polygon* poly = NULL;
    sph_polygon* target = NULL;
    sph_error_code rerr = 0;
    sph_polygon_workspace* ws = NULL;
    const sph_polygon* areapoly;

    poly = sph_polygon_new();
    ws = sph_polygon_workspace_new();
    cpl_test_nonnull( ws );

    cpl_test_nonnull( poly );
    target = sph_polygon_new();

    cpl_test_nonnull( target );

    rerr = sph_polygon_add_point(target, 0.0, 0.0);
    rerr |= sph_polygon_add_point(target, 1.0, 0.0);
    rerr |= sph_polygon_add_point(target, 1.0, 1.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_polygon_add_point(poly, -20.0, -20.0);
    rerr |= sph_polygon_add_point(poly, 2.0, 0.0);
    rerr |= sph_polygon_add_point(poly, 10.0, 10.0);
    rerr |= sph_polygon_add_point(poly, 0.0, 3.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    areapoly = sph_polygon_clip_to_target(poly, target, ws);
    cpl_test_nonnull( areapoly );

    cpl_test_lt(areapoly->npoints, 4);
    cpl_test_lt(2, areapoly->npoints);
    cpl_test_abs( areapoly->points[0].x, 0.0, 1.0e-15);
    cpl_test_abs( areapoly->points[0].y, 0.0, 1.0e-15);
    cpl_test_abs( areapoly->points[1].x, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[1].y, 0.0, 1.0e-15);
    cpl_test_abs( areapoly->points[2].x, 1.0, 1.0e-15);
    cpl_test_abs( areapoly->points[2].y, 1.0, 1.0e-15);
    sph_polygon_workspace_delete(ws);
    sph_polygon_delete(poly);
    sph_polygon_delete(target);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_clipping function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_calculate_overlap(void) {
    sph_polygon* poly = NULL;
    sph_polygon* target = NULL;
    sph_error_code rerr = 0;
    double result = 0.0;
    sph_polygon_workspace* ws = NULL;

    poly = sph_polygon_new();
    ws = sph_polygon_workspace_new();
    cpl_test_nonnull( ws );

    cpl_test_nonnull( poly );
    target = sph_polygon_new();

    cpl_test_nonnull( target );

    rerr = sph_polygon_add_point(target, 0.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 2.0);
    rerr |= sph_polygon_add_point(target, 0.0, 2.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_polygon_add_point(poly, 1.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 3.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 3.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    result = sph_polygon_calculate_overlap(poly, target, ws);

    cpl_test_abs( result, 1.0, 1.0e-10);

    sph_polygon_delete(poly);
    sph_polygon_delete(target);
    sph_polygon_workspace_delete(ws);
    return;
}
static
void cutest_polygon_calculate_overlap2(void) {
    sph_polygon* poly = NULL;
    sph_polygon* target = NULL;
    double offset_x = 0.0;
    double offset_y = 0.0;
    sph_polygon_workspace* ws = NULL;

    ws = sph_polygon_workspace_new();
    cpl_test_nonnull( ws );

    poly = sph_polygon_new();
    cpl_test_nonnull( poly );
    sph_polygon_add_point(poly, offset_x + 0.0, offset_y + 0.0);
    sph_polygon_add_point(poly, offset_x + 1.0, offset_y + 0.0);
    sph_polygon_add_point(poly, offset_x + 1.0, offset_y + 1.0);
    sph_polygon_add_point(poly, offset_x + 0.0, offset_y + 1.0);

    target = sph_polygon_new();

    cpl_test_nonnull( target );
    sph_polygon_add_point(target, 0.0, 0.0);
    sph_polygon_add_point(target, 1.01, 0.0);
    sph_polygon_add_point(target, 1.01, 1.01);
    sph_polygon_add_point(target, 0.0, 1.01);
    cpl_test_abs( sph_polygon_calculate_overlap(poly,target,ws), 1.0,
            0.1);
    cpl_test_abs( sph_polygon_calculate_overlap(target,poly,ws), 1.0,
            0.1);
    cpl_test_abs( sph_polygon_calculate_overlap(poly,poly,ws), 1.0,
            0.1);
    cpl_test_abs( sph_polygon_calculate_overlap(target,target,ws),
            1.0, 0.1);
    sph_polygon_delete(poly);
    poly = NULL;
    sph_polygon_delete(target);
    target = NULL;
    sph_polygon_workspace_delete(ws);
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_clipping function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_calculate_overlap_parallel(void) {
    sph_polygon* target = NULL;
    sph_error_code rerr = 0;
    double result = 0.0;
#ifdef _OPENMP
    sph_polygon_workspace* ws[omp_get_max_threads()];
#else
    sph_polygon_workspace* ws[4];
#endif
    int ii = 0;
    int npolys = 1000000;
    sph_polygon** polys = (sph_polygon**)cpl_malloc(npolys * sizeof(*polys));
    int tn = 0;

    for (ii = 0; ii < npolys; ++ii) {
        polys[ii] = sph_polygon_new();
        rerr = sph_polygon_add_point(polys[ii], 1.0, 1.0);
        rerr |= sph_polygon_add_point(polys[ii], 3.0, 1.0);
        rerr |= sph_polygon_add_point(polys[ii], 3.0, 3.0);
        rerr |= sph_polygon_add_point(polys[ii], 1.0, 3.0);
    }

    target = sph_polygon_new();

    cpl_test_nonnull( target );

    rerr = sph_polygon_add_point(target, 0.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 2.0);
    rerr |= sph_polygon_add_point(target, 0.0, 2.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Start timer %d: ", sph_time_start_timer(1));

#ifdef _OPENMP
    for (ii = 0; ii < omp_get_max_threads(); ++ii) {
#else
    for (ii = 0; ii < 4; ++ii) {
#endif
        ws[ii] = sph_polygon_workspace_new();
    }
#ifdef _OPENMP
#pragma omp parallel for shared(ws) private(ii,tn)
#endif
    for (ii = 0; ii < npolys; ++ii) {
#ifdef _OPENMP
        tn = omp_get_thread_num();
#endif
        result = sph_polygon_calculate_overlap(polys[ii], target, ws[tn]);
//        if ( fabs (result - 1.0 ) > 1.0e-10 ) {
        //sph_test_fail("Incorrect");
//        }
    }SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Stop time %f: ", sph_time_stop_timer(1));
    for (ii = 0; ii < npolys; ++ii) {
        sph_polygon_delete(polys[ii]);
    }
#ifdef _OPENMP
    for (ii = 0; ii < omp_get_max_threads(); ++ii) {
#else
    for (ii = 0; ii < 4; ++ii) {
#endif
        sph_polygon_workspace_delete(ws[ii]);
    }
    cpl_free(polys);
    sph_polygon_delete(target);
    target = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_rotate function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_rotate_around(void) {
    sph_polygon* poly = NULL;
    sph_error_code rerr = 0;

    poly = sph_polygon_new();

    cpl_test_nonnull( poly );
    rerr = sph_polygon_add_point(poly, 1.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 3.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 3.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    rerr = sph_polygon_rotate_around(poly, 1.0, 1.0, cos(90.0 * CPL_MATH_RAD_DEG),
            sin(90.0 * CPL_MATH_RAD_DEG), 0.0, 0.0);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_eq(poly->npoints, 4);
    cpl_test_abs( poly->points[0].x, 1.0, 1.0e-15);
    cpl_test_abs( poly->points[0].y, 1.0, 1.0e-15);
    cpl_test_abs( poly->points[1].x, 1.0, 1.0e-15);
    cpl_test_abs( poly->points[1].y, 3.0, 1.0e-15);
    cpl_test_abs( poly->points[2].x, -1.0, 1.0e-15);
    cpl_test_abs( poly->points[2].y, 3.0, 1.0e-15);
    cpl_test_abs( poly->points[3].x, -1.0, 1.0e-15);
    cpl_test_abs( poly->points[3].y, 1.0, 1.0e-15);
    sph_polygon_delete(poly);
    poly = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_rotate function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_area(void) {
    sph_polygon*   poly;
    double         area;
    sph_error_code code;
    cpl_size nz;

    nz = sph_polygon_get_size(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_lt(nz, 0);

    poly = sph_polygon_new();
    cpl_test_nonnull( poly );

    nz = sph_polygon_get_size(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(nz);

    area = sph_polygon_area(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(area, 0.0, DBL_EPSILON);

    code = sph_polygon_add_point(poly, 1.0, 1.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    nz = sph_polygon_get_size(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(nz, 1);

    area = sph_polygon_area(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(area, 0.0, DBL_EPSILON);

    code = sph_polygon_add_point(poly, 3.0, 1.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq_error(code, CPL_ERROR_NONE);
    nz = sph_polygon_get_size(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(nz, 2);

    area = sph_polygon_area(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(area, 0.0, DBL_EPSILON);

    code = sph_polygon_add_point(poly, 3.0, 3.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    nz = sph_polygon_get_size(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(nz, 3);

    area = sph_polygon_area(poly);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_abs( area, 2.0, DBL_EPSILON);

    code = sph_polygon_add_point(poly, 1.0, 3.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    nz = sph_polygon_get_size(poly);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(nz, 4);

    area = sph_polygon_area(poly);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_abs( area, 4.0, DBL_EPSILON);

    sph_polygon_delete(poly);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_rotate function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_rotate_around_performance(void) {
    sph_polygon* poly = NULL;
    sph_error_code rerr = 0;
    double cosang = cos(90.0 * CPL_MATH_RAD_DEG);
    double sinang = sin(90.0 * CPL_MATH_RAD_DEG);
    int maxi = 4 * 4200000 + 1;
    int ii = 0;

    poly = sph_polygon_new();

    cpl_test_nonnull( poly );
    rerr = sph_polygon_add_point(poly, 1.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 1.0);
    rerr |= sph_polygon_add_point(poly, 3.0, 3.0);
    rerr |= sph_polygon_add_point(poly, 1.0, 3.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    for (ii = 0; ii < maxi; ++ii) {
        rerr = sph_polygon_rotate_around(poly, 1.0, 1.0, cosang, sinang, 0.0,
                0.0);
    }cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_eq(poly->npoints, 4);
    cpl_test_abs( poly->points[0].x, 1.0, 1.0e-8);
    cpl_test_abs( poly->points[0].y, 1.0, 1.0e-8);
    cpl_test_abs( poly->points[1].x, 1.0, 1.0e-8);
    cpl_test_abs( poly->points[1].y, 3.0, 1.0e-8);
    cpl_test_abs( poly->points[2].x, -1.0, 1.0e-8);
    cpl_test_abs( poly->points[2].y, 3.0, 1.0e-8);
    cpl_test_abs( poly->points[3].x, -1.0, 1.0e-8);
    cpl_test_abs( poly->points[3].y, 1.0, 1.0e-8);
    sph_polygon_delete(poly);
    poly = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_delete(void) {
    sph_polygon* result = NULL;

    result = sph_polygon_new();

    cpl_test_nonnull( result );

    sph_polygon_delete(result);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("polygon_test", cutest_init_polygon_testsuite,
            cutest_clean_polygon_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL == sph_test_do(pSuite, "sph_polygon_new", cutest_polygon_new)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_polygon_add_point",
                    cutest_polygon_add_point)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_get_edge",
                    cutest_polygon_get_edge)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_test_inside",
                    cutest_polygon_test_inside)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_same_side",
                    cutest_polygon_same_side)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_clip_to_target_caseA",
                    cutest_polygon_clip_to_target_caseA)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_clip_to_target_caseB",
                    cutest_polygon_clip_to_target_caseB)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_clip_to_target_caseC",
                    cutest_polygon_clip_to_target_caseC)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_calculate_overlap",
                    cutest_polygon_calculate_overlap)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_rotate_around",
                    cutest_polygon_rotate_around)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_rotate_around_performance",
                    cutest_polygon_rotate_around_performance)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_polygon_area", cutest_polygon_area)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_inside_im_hex",
                    cutest_polygon_test_inside_im_hex_case)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_duplicate",
                    cutest_polygon_duplicate)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_calculate_overlap2",
                    cutest_polygon_calculate_overlap2)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_calculate_overlap parralel",
                    cutest_polygon_calculate_overlap_parallel)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_polygon_delete", cutest_polygon_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
