/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_psfcube.h"
#include "sph_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_psfcube_new(void) {
    cpl_imagelist* imlist = NULL;
    sph_psfcube* psfcube = NULL;
    cpl_vector* shiftsx = NULL;
    cpl_vector* shiftsy = NULL;

    imlist = cpl_imagelist_new();
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 1.0), 0);
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 2.0), 1);
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 3.0), 2);

    shiftsx = cpl_vector_new(3);
    shiftsy = cpl_vector_new(3);
    cpl_vector_set(shiftsx, 0, 0.0);
    cpl_vector_set(shiftsy, 0, 0.0);
    cpl_vector_set(shiftsx, 1, 3.0);
    cpl_vector_set(shiftsy, 1, 1.0);
    cpl_vector_set(shiftsx, 2, 4.0);
    cpl_vector_set(shiftsy, 2, 2.0);
    psfcube = sph_psfcube_new(imlist, shiftsx, shiftsy);
    cpl_test_assert( psfcube);

    sph_psfcube_delete(psfcube);
    return;
}
static
void cutest_sph_psfcube_get(void) {
    cpl_imagelist* imlist = NULL;
    sph_psfcube* psfcube = NULL;
    cpl_vector* shiftsx = NULL;
    cpl_vector* shiftsy = NULL;
    const cpl_image* im = NULL;

    imlist = cpl_imagelist_new();
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 1.0), 0);
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 2.0), 1);
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 3.0), 2);

    shiftsx = cpl_vector_new(3);
    shiftsy = cpl_vector_new(3);
    cpl_vector_set(shiftsx, 0, 0.0);
    cpl_vector_set(shiftsy, 0, 0.0);
    cpl_vector_set(shiftsx, 1, 3.0);
    cpl_vector_set(shiftsy, 1, 1.0);
    cpl_vector_set(shiftsx, 2, 4.0);
    cpl_vector_set(shiftsy, 2, 2.0);
    psfcube = sph_psfcube_new(imlist, shiftsx, shiftsy);
    cpl_test_nonnull( psfcube);
    im = sph_psfcube_get_exact(psfcube, 3.0, 1.0);
    cpl_test_nonnull( im);
    im = sph_psfcube_get_exact(psfcube, 3.0, 0.0);
    cpl_test_null( im);
    sph_psfcube_delete(psfcube);
    return;
}
static
void cutest_sph_psfcube_insert(void) {
    cpl_imagelist* imlist = NULL;
    sph_psfcube* psfcube = NULL;
    cpl_vector* shiftsx = NULL;
    cpl_vector* shiftsy = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    const cpl_image* im = NULL;

    imlist = cpl_imagelist_new();
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 1.0), 0);
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 2.0), 1);
    cpl_imagelist_set(imlist,
            sph_test_image_tools_create_flat_image_double(32, 32, 3.0), 2);

    shiftsx = cpl_vector_new(3);
    shiftsy = cpl_vector_new(3);
    cpl_vector_set(shiftsx, 0, 0.0);
    cpl_vector_set(shiftsy, 0, 0.0);
    cpl_vector_set(shiftsx, 1, 3.0);
    cpl_vector_set(shiftsy, 1, 1.0);
    cpl_vector_set(shiftsx, 2, 4.0);
    cpl_vector_set(shiftsy, 2, 2.0);
    psfcube = sph_psfcube_new(imlist, shiftsx, shiftsy);
    cpl_test_nonnull( psfcube );
    rerr = sph_psfcube_insert(psfcube,
            sph_test_image_tools_create_flat_image_double(32, 32, 4.0), 1.5,
            2.5);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    im = sph_psfcube_get_exact(psfcube, 1.5, 2.5);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( im);
    sph_psfcube_delete(psfcube);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing cutest_sph_psfcube", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "cutest_sph_psfcube_new",
                    cutest_sph_psfcube_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "cutest_sph_psfcube_get",
                    cutest_sph_psfcube_get)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "cutest_sph_psfcube_insert",
                    cutest_sph_psfcube_insert)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();

    return result;
}

/**@}*/
