/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_test.h"
#include "sph_fits.h"
#include "sph_simple_adi.h"
#include "sph_fctable.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_common_science.h"
static
void cutest_sph_simple_adi_fixture_fill_imlist__(sph_simple_adi* adi, int n);
static
void cutest_sph_simple_adi_fixture_fill_bmaplist__(sph_simple_adi* adi, int n);
static
void cutest_sph_simple_adi_fixture_fill_wmaplist__(sph_simple_adi* adi, int n);
static
void cutest_sph_simple_adi_fixture_fill_manyframes__(sph_simple_adi* adi, int n,
        int bmaps_flag, int wmaps_flag);
static
void cutest_sph_simple_adi_fixture_fill_cubeframe__(sph_simple_adi* adi, int n,
        int bmaps_flag, int wmaps_flag);
static cpl_frame*
cutest_sph_simple_adi_fixture_planet_cube__(cpl_vector** angles);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_new_and_delete(void) {
    sph_simple_adi* adi = NULL;
    sph_transform* transform = NULL;

    transform = sph_transform_new_default();
    cpl_test_null( sph_simple_adi_new(NULL));
    cpl_error_reset();

    adi = sph_simple_adi_new(transform);
    cpl_test_nonnull( adi );
    sph_simple_adi_delete(NULL);
    cpl_test_error(CPL_ERROR_NONE);
    sph_simple_adi_delete(adi);
    cpl_test_error(CPL_ERROR_NONE);
    sph_transform_delete(transform);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_set_angles(void) {
    sph_simple_adi* adi = NULL;
    cpl_vector* angles = NULL;

    adi = cpl_calloc(1, sizeof(sph_simple_adi));
    cpl_test_assert(adi);

    // checking for correct NULL behaviour...
    cpl_test_eq(sph_simple_adi_set_angles(NULL, NULL),
            CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    cpl_test_eq_error(sph_simple_adi_set_angles(adi,NULL), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    angles = cpl_vector_new(1);
    // Checking for correct error when no imageset set...
    cpl_test_eq(sph_simple_adi_set_angles(adi, angles),
            CPL_ERROR_DATA_NOT_FOUND);
    cpl_error_reset();

    // Setting up frameset (single frame)
    cutest_sph_simple_adi_fixture_fill_cubeframe__(adi, 2, 0, 0);

    // checking for error on incorrect length
    cpl_test_eq(sph_simple_adi_set_angles(adi, angles),
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();
    cpl_vector_set_size(angles, 3);
    cpl_test_eq(sph_simple_adi_set_angles(adi, angles),
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();

    // Checking correctly set...
    cpl_vector_set_size(angles, 2);
    cpl_test_eq_error(sph_simple_adi_set_angles(adi,angles), CPL_ERROR_NONE);
    cpl_test_assert(adi->current_cube_angles);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);

    // Creating ADI with frameset (many frames)
    cutest_sph_simple_adi_fixture_fill_manyframes__(adi, 2, 0, 0);
    cpl_test_eq_error(sph_simple_adi_set_angles(adi,angles), CPL_ERROR_NONE);
    cpl_test_assert(adi->current_cube_angles);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);

    // Creating ADI with imlist
    cutest_sph_simple_adi_fixture_fill_imlist__(adi, 3);
    cpl_vector_set_size(angles, 3);
    cpl_test_eq_error(sph_simple_adi_set_angles(adi,angles), CPL_ERROR_NONE);
    cpl_test_assert(adi->current_cube_angles);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 3);

    cpl_vector_delete(angles);
    cpl_imagelist_delete(adi->current_imlist);
    cpl_imagelist_delete(adi->current_bmaplist);
    cpl_imagelist_delete(adi->current_wmaplist);
    cpl_frameset_delete(adi->current_frameset);
    cpl_vector_delete(adi->current_cube_angles);
    cpl_free(adi);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_set_current_cube_frame(void) {
    sph_simple_adi* adi = NULL;
    cpl_frame* aframe = NULL;

    adi = cpl_calloc(1, sizeof(sph_simple_adi));
    cpl_test_nonnull( adi );

    // Test with NULL... should give error
    cutest_sph_simple_adi_fixture_fill_manyframes__(adi, 3, 0, 0);
    cpl_test_eq(sph_simple_adi_set_current_cube_frame( adi,NULL, 0,-1, -1),
            CPL_ERROR_NULL_INPUT);
    cpl_error_reset();

    // Test with cube set -- should be ok
    cutest_sph_simple_adi_fixture_fill_cubeframe__(adi, 3, 0, 0);

    aframe = cpl_frame_duplicate(cpl_frameset_get_first(adi->current_frameset));
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    cpl_test_eq(
            sph_simple_adi_set_current_cube_frame( adi,aframe, 0,-1,-1),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_frameset );
    cpl_test_eq(cpl_frameset_get_size(adi->current_frameset), 1);

    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 3);

    // Test with manyframes -- should be ok
    cutest_sph_simple_adi_fixture_fill_manyframes__(adi, 3, 0, 0);

    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    cpl_test_eq(
            sph_simple_adi_set_current_cube_frame( adi,aframe, 0,-1,-1),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_frameset );
    cpl_test_eq(cpl_frameset_get_size(adi->current_frameset), 1);

    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 3);

    // Test with imagelist.... should be ok.
    cutest_sph_simple_adi_fixture_fill_imlist__(adi, 3);
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    cpl_test_eq(
            sph_simple_adi_set_current_cube_frame( adi,aframe, 0,-1,-1),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_frameset );
    cpl_test_eq(cpl_frameset_get_size(adi->current_frameset), 1);

    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 3);

    cpl_frame_delete(aframe);
    aframe = NULL;
    cpl_frameset_delete(adi->current_frameset);
    cpl_vector_delete(adi->current_cube_angles);
    cpl_free(adi);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_set_current_frameset(void) {
    sph_simple_adi* adi = NULL;
    cpl_frame* aframe = NULL;
    cpl_frameset* aframeset = NULL;

    adi = cpl_calloc(1, sizeof(sph_simple_adi));
    cpl_test_nonnull( adi );

    // Test with NULL... should give error
    cutest_sph_simple_adi_fixture_fill_manyframes__(adi, 3, 0, 0);
    cpl_test_eq(sph_simple_adi_set_current_frameset( adi, NULL, 0,-1, -1),
            CPL_ERROR_NULL_INPUT);
    cpl_error_reset();

    // Test with cube set -- should be ok
    cutest_sph_simple_adi_fixture_fill_cubeframe__(adi, 3, 0, 0);

    aframe = cpl_frame_duplicate(cpl_frameset_get_first(adi->current_frameset));
    aframeset = cpl_frameset_new();
    cpl_frameset_insert(aframeset, aframe);
    aframe = cpl_frame_duplicate(cpl_frameset_get_first(adi->current_frameset));
    cpl_frameset_insert(aframeset, aframe);

    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    cpl_test_eq(
            sph_simple_adi_set_current_frameset( adi,aframeset, 0,-1,-1),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_frameset );
    cpl_test_eq(cpl_frameset_get_size(adi->current_frameset), 2);

    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);

    // Test with manyframes -- should be ok
    cutest_sph_simple_adi_fixture_fill_manyframes__(adi, 3, 0, 0);

    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    cpl_test_eq(
            sph_simple_adi_set_current_frameset( adi,aframeset, 0,-1,-1),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_frameset );
    cpl_test_eq(cpl_frameset_get_size(adi->current_frameset), 2);

    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);

    // Test with imagelist.... should be ok.
    cutest_sph_simple_adi_fixture_fill_imlist__(adi, 3);
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    cpl_test_eq(
            sph_simple_adi_set_current_frameset( adi,aframeset, 0,-1,-1),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_frameset );
    cpl_test_eq(cpl_frameset_get_size(adi->current_frameset), 2);

    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);

    cpl_frameset_delete(aframeset);
    aframeset = NULL;
    cpl_frameset_delete(adi->current_frameset);
    cpl_vector_delete(adi->current_cube_angles);
    cpl_free(adi);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_append_image(void) {
    sph_simple_adi* adi = NULL;
    cpl_image* im = NULL;

    adi = cpl_calloc(1, sizeof(sph_simple_adi));
    im = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    cutest_sph_simple_adi_fixture_fill_imlist__(adi, 3);
    // Checking NULL input gives error...
    cpl_test_eq(sph_simple_adi_append_image( adi,NULL,NULL,NULL, 0.0),
            CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    // Checking providing unmatching bpix gives error
    cpl_test_eq(sph_simple_adi_append_image( adi,im,im, NULL, 0.0),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();
    // Checking providing unmatching wmap gives error
    cpl_test_eq(sph_simple_adi_append_image( adi,im,NULL,im, 0.0),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();
    // Checking not providing matching bpix gives error
    cutest_sph_simple_adi_fixture_fill_bmaplist__(adi, 3);
    cpl_test_eq(sph_simple_adi_append_image( adi,im,NULL, NULL, 0.0),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();
    // Checking not providing matching wmap gives error
    cutest_sph_simple_adi_fixture_fill_wmaplist__(adi, 3);
    cpl_test_eq(sph_simple_adi_append_image( adi,im,im, NULL, 0.0),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();

    cpl_imagelist_delete(adi->current_bmaplist);
    adi->current_bmaplist = NULL;
    cpl_imagelist_delete(adi->current_wmaplist);
    adi->current_wmaplist = NULL;
    cpl_imagelist_delete(adi->current_imlist);
    adi->current_imlist = NULL;
    // Checking works on empty ADI
    cpl_test_eq(sph_simple_adi_append_image( adi, im, NULL, NULL, 0.0),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_imlist );
    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_imagelist_get_size(adi->current_imlist), 1);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 1);

    cpl_imagelist_delete(adi->current_imlist);
    adi->current_imlist = NULL;

    // Checking works on empty ADI, when bpix specified
    cpl_test_eq(sph_simple_adi_append_image( adi, im, im, NULL, 0.0),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_imlist );
    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_nonnull( adi->current_bmaplist );
    cpl_test_eq(cpl_imagelist_get_size(adi->current_imlist), 1);
    cpl_test_eq(cpl_imagelist_get_size(adi->current_bmaplist), 1);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 1);

    cpl_imagelist_delete(adi->current_imlist);
    adi->current_imlist = NULL;
    cpl_imagelist_delete(adi->current_bmaplist);
    adi->current_bmaplist = NULL;
    // Checking works on empty ADI, when wmap specified
    cpl_test_eq(sph_simple_adi_append_image( adi, im, NULL, im, 0.0),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_imlist );
    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_nonnull( adi->current_wmaplist );
    cpl_test_eq(cpl_imagelist_get_size(adi->current_imlist), 1);
    cpl_test_eq(cpl_imagelist_get_size(adi->current_wmaplist), 1);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 1);

    cpl_imagelist_delete(adi->current_imlist);
    cpl_imagelist_delete(adi->current_bmaplist);
    cpl_imagelist_delete(adi->current_wmaplist);
    cpl_vector_delete(adi->current_cube_angles);
    cpl_free(adi);
    cpl_image_delete(im);
    im = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_append_imagelist(void) {
    sph_simple_adi* adi = NULL;
    cpl_image* dumim = NULL;
    cpl_imagelist* imlist = NULL;
    cpl_vector* angles = NULL;

    adi = cpl_calloc(1, sizeof(sph_simple_adi));
    dumim = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    imlist = cpl_imagelist_new();

    cpl_imagelist_set(imlist, cpl_image_duplicate(dumim), 0);
    cpl_imagelist_set(imlist, cpl_image_duplicate(dumim), 1);

    angles = cpl_vector_new(cpl_imagelist_get_size(imlist));
    cpl_vector_fill(angles, 0.0);
    cutest_sph_simple_adi_fixture_fill_imlist__(adi, 3);
    // Checking NULL input gives error...
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi,NULL,NULL,NULL,angles),
            CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    // Checking providing unmatching bpix gives error
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi,imlist,imlist, NULL,angles),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();
    // Checking providing unmatching wmap gives error
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi,imlist,NULL,imlist, angles),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();
    // Checking not providing matching bpix gives error
    cutest_sph_simple_adi_fixture_fill_bmaplist__(adi, 3);
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi,imlist,NULL, NULL,angles),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();
    // Checking not providing matching wmap gives error
    cutest_sph_simple_adi_fixture_fill_wmaplist__(adi, 3);
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi,imlist,imlist, NULL,angles),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_error_reset();

    cpl_imagelist_delete(adi->current_bmaplist);
    adi->current_bmaplist = NULL;
    cpl_imagelist_delete(adi->current_wmaplist);
    adi->current_wmaplist = NULL;
    cpl_imagelist_delete(adi->current_imlist);
    adi->current_imlist = NULL;
    // Checking works on empty ADI
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi, imlist, NULL, NULL, angles),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_imlist );
    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_imagelist_get_size(adi->current_imlist), 2);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);
    // Checking works on ADI with existing list
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi, imlist, NULL, NULL, angles),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_imlist );
    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_eq(cpl_imagelist_get_size(adi->current_imlist), 4);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 4);

    cpl_imagelist_delete(adi->current_imlist);
    adi->current_imlist = NULL;

    // Checking works on empty ADI, when bpix specified
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi, imlist, imlist, NULL, angles),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_imlist );
    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_nonnull( adi->current_bmaplist );
    cpl_test_eq(cpl_imagelist_get_size(adi->current_imlist), 2);
    cpl_test_eq(cpl_imagelist_get_size(adi->current_bmaplist), 2);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);

    cpl_imagelist_delete(adi->current_imlist);
    adi->current_imlist = NULL;
    cpl_imagelist_delete(adi->current_bmaplist);
    adi->current_bmaplist = NULL;
    // Checking works on empty ADI, when wmap specified
    cpl_test_eq(
            sph_simple_adi_append_imagelist( adi, imlist, NULL, imlist, angles),
            CPL_ERROR_NONE);
    cpl_test_nonnull( adi->current_imlist );
    cpl_test_nonnull( adi->current_cube_angles );
    cpl_test_nonnull( adi->current_wmaplist );
    cpl_test_eq(cpl_imagelist_get_size(adi->current_imlist), 2);
    cpl_test_eq(cpl_imagelist_get_size(adi->current_wmaplist), 2);
    cpl_test_eq(cpl_vector_get_size(adi->current_cube_angles), 2);

    cpl_imagelist_delete(adi->current_imlist);
    cpl_imagelist_delete(adi->current_bmaplist);
    cpl_imagelist_delete(adi->current_wmaplist);
    cpl_vector_delete(adi->current_cube_angles);
    cpl_free(adi);
    cpl_vector_delete(angles);
    cpl_imagelist_delete(imlist);
    cpl_image_delete(dumim);
    dumim = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_calculate_speckle_frame(void) {
    sph_simple_adi* adi = NULL;
    cpl_image* im = NULL;
    cpl_imagelist* imlist = NULL;
    cpl_frame* cube_frame = NULL;
    cpl_frameset* aframeset = NULL;
    double rms = 0;
    aframeset = cpl_frameset_new();
    imlist = cpl_imagelist_new();
    cube_frame = sph_filemanager_create_temp_frame("speck_cube", "SPECKIN",
            CPL_FRAME_GROUP_CALIB);
    adi = cpl_calloc(1, sizeof(sph_simple_adi));
    adi->ext_im = 0;
    adi->ext_bp = -1;
    adi->ext_wm = -1;
    cpl_test_noneq( sph_simple_adi_calculate_speckle_frame(adi),
            CPL_ERROR_NONE);
    cpl_error_reset();

    // First test with an imagelist of constant value ims...
    im = sph_test_image_tools_create_flat_image_double(32, 32, 1000.0);
    cpl_imagelist_set(imlist, cpl_image_duplicate(im), 0);
    cpl_frameset_insert(
            aframeset,
            sph_test_frame_image(im, CPL_TYPE_DOUBLE, "SPECKIN",
                    CPL_FRAME_GROUP_CALIB));
    cpl_frameset_insert(
            aframeset,
            sph_test_frame_image(im, CPL_TYPE_DOUBLE, "SPECKIN",
                    CPL_FRAME_GROUP_CALIB));
    cpl_frameset_insert(
            aframeset,
            sph_test_frame_image(im, CPL_TYPE_DOUBLE, "SPECKIN",
                    CPL_FRAME_GROUP_CALIB));

    cpl_imagelist_set(imlist, cpl_image_duplicate(im), 1);
    cpl_imagelist_set(imlist, cpl_image_duplicate(im), 2);
    cpl_imagelist_set(imlist, cpl_image_duplicate(im), 3);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(cube_frame),
            CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    // Check with ADI using imlist...
    adi->current_imlist = cpl_imagelist_duplicate(imlist);
    cpl_test_eq( sph_simple_adi_calculate_speckle_frame(adi),
            CPL_ERROR_NONE);
    cpl_test_abs( sph_master_frame_get_mean( adi->speckleframe, &rms),
            1000.0, 0.001);
    cpl_test_abs( rms, 0.0, 0.001);

    cpl_imagelist_delete(adi->current_imlist);
    adi->current_imlist = NULL;
    // Check with ADI using cube_frame...
    adi->current_frameset = cpl_frameset_new();
    cpl_frameset_insert(adi->current_frameset, cube_frame);
    cpl_test_eq( sph_simple_adi_calculate_speckle_frame(adi),
            CPL_ERROR_NONE);
    cpl_test_abs( sph_master_frame_get_mean( adi->speckleframe, &rms),
            1000.0, 0.001);
    cpl_test_abs( rms, 0.0, 0.001);
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    // Check with ADI using many frames...
    adi->current_frameset = cpl_frameset_duplicate(aframeset);
    cpl_test_eq( sph_simple_adi_calculate_speckle_frame(adi),
            CPL_ERROR_NONE);
    cpl_test_abs( sph_master_frame_get_mean( adi->speckleframe, &rms),
            1000.0, 0.001);
    cpl_test_abs( rms, 0.0, 0.001);

    cpl_frameset_delete(adi->current_frameset);
    cpl_imagelist_delete(adi->current_imlist);
    cpl_imagelist_delete(imlist);
    cpl_image_delete(im);
    cpl_frameset_delete(aframeset);
    sph_master_frame_delete(adi->speckleframe);
    cpl_free(adi);
    adi = NULL;
}

static
void cutest_sph_simple_adi_fixture_fill_imlist__(sph_simple_adi* adi, int n) {
    cpl_imagelist* imlist = NULL;
    int ii = 0;

    imlist = cpl_imagelist_new();
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    for (ii = 0; ii < n; ++ii) {
        cpl_imagelist_set(imlist, cpl_image_new(32, 32, CPL_TYPE_DOUBLE),
                cpl_imagelist_get_size(imlist));
    }
    if (adi->current_cube_angles == NULL)
        adi->current_cube_angles = cpl_vector_new(n);
    else
        cpl_vector_set_size(adi->current_cube_angles, n);
    adi->current_imlist = imlist;
}

static
void cutest_sph_simple_adi_fixture_fill_bmaplist__(sph_simple_adi* adi, int n) {
    cpl_imagelist* imlist = NULL;
    int ii = 0;

    imlist = cpl_imagelist_new();
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    for (ii = 0; ii < n; ++ii) {
        cpl_imagelist_set(imlist, cpl_image_new(32, 32, CPL_TYPE_DOUBLE),
                cpl_imagelist_get_size(imlist));
    }
    if (adi->current_cube_angles == NULL)
        adi->current_cube_angles = cpl_vector_new(n);
    else
        cpl_vector_set_size(adi->current_cube_angles, n);
    adi->current_bmaplist = imlist;
}

static
void cutest_sph_simple_adi_fixture_fill_wmaplist__(sph_simple_adi* adi, int n) {
    cpl_imagelist* imlist = NULL;
    int ii = 0;

    imlist = cpl_imagelist_new();
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = NULL;
    for (ii = 0; ii < n; ++ii) {
        cpl_imagelist_set(imlist, cpl_image_new(32, 32, CPL_TYPE_DOUBLE),
                cpl_imagelist_get_size(imlist));
    }
    if (adi->current_cube_angles == NULL)
        adi->current_cube_angles = cpl_vector_new(n);
    else
        cpl_vector_set_size(adi->current_cube_angles, n);
    adi->current_wmaplist = imlist;
}

static
void cutest_sph_simple_adi_fixture_fill_cubeframe__(sph_simple_adi* adi, int n,
        int bmaps_flag, int wmaps_flag) {
    cpl_imagelist* imlist = NULL;
    int ii = 0;
    cpl_frame* aframe = NULL;
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = cpl_frameset_new();
    imlist = cpl_imagelist_new();
    aframe = sph_filemanager_create_temp_frame("tmp", "NOTAG",
            CPL_FRAME_GROUP_RAW);
    for (ii = 0; ii < n; ++ii) {
        cpl_imagelist_set(imlist, cpl_image_new(32, 32, CPL_TYPE_DOUBLE),
                cpl_imagelist_get_size(imlist));
    }
    cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe), CPL_TYPE_DOUBLE,
            NULL, CPL_IO_CREATE);
    if (bmaps_flag) {
        cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe),
                CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    }
    if (wmaps_flag) {
        cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe),
                CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    }
    cpl_frameset_insert(adi->current_frameset, aframe);
    if (adi->current_cube_angles == NULL)
        adi->current_cube_angles = cpl_vector_new(n);
    else
        cpl_vector_set_size(adi->current_cube_angles, n);
    cpl_imagelist_delete(imlist);
}

static
void cutest_sph_simple_adi_fixture_fill_manyframes__(sph_simple_adi* adi, int n,
        int bmaps_flag, int wmaps_flag) {
    cpl_imagelist* imlist = NULL;
    int ii = 0;
    cpl_frame* aframe = NULL;
    cpl_frameset_delete(adi->current_frameset);
    adi->current_frameset = cpl_frameset_new();
    for (ii = 0; ii < n; ++ii) {
        imlist = cpl_imagelist_new();
        cpl_imagelist_set(imlist, cpl_image_new(32, 32, CPL_TYPE_DOUBLE),
                cpl_imagelist_get_size(imlist));
        aframe = sph_filemanager_create_temp_frame("tmp", "NOTAG",
                CPL_FRAME_GROUP_RAW);
        cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe),
                CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        if (bmaps_flag) {
            cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe),
                    CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
        }
        if (wmaps_flag) {
            cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe),
                    CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
        }
        cpl_frameset_insert(adi->current_frameset, aframe);
        cpl_imagelist_delete(imlist);
        imlist = NULL;
    }
    if (adi->current_cube_angles == NULL)
        adi->current_cube_angles = cpl_vector_new(n);
    else
        cpl_vector_set_size(adi->current_cube_angles, n);
    cpl_imagelist_delete(imlist);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_process_cubes(void) {
    cpl_frame* input_cube_frame = NULL;
    int np = 10;
    int nf = 3;
    int nx = 64;
    int ny = 64;
    int pp = 0;
    cpl_image* im = NULL;
    sph_transform* transform = NULL;
    cpl_vector* angles = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_frameset* inframeset = NULL;
    cpl_frameset* result_frameset = NULL;
    int ff = 0;
    double angle = 0.0;
    sph_point* p = NULL;
    sph_cube* acube = NULL;
    transform = sph_transform_new_default();

    cpl_test_error(CPL_ERROR_NONE);
    inframeset = cpl_frameset_new();
    angles = cpl_vector_new(nf * np);
    p = sph_point_new(20.0, 0.0);
    for (ff = 0; ff < nf; ++ff) {
        input_cube_frame = sph_filemanager_create_temp_frame("testcube",
                "IRDIS_SCI", CPL_FRAME_GROUP_CALIB);
        acube = sph_cube_new(cpl_frame_get_filename(input_cube_frame));
        cpl_test_error(CPL_ERROR_NONE);
        for (pp = 0; pp < np; ++pp) {
            im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
            sph_test_image_tools_add_gauss(im, nx / 2.0, ny / 2.0, 10.0,
                    1000.0);
            p->x = 20.0;
            p->y = 0.0;
            angle += 10.0 * CPL_MATH_RAD_DEG;
            sph_point_rotate_around(p, 0.0, 0.0, cos(angle), sin(angle), 0.0,
                    0.0);
            sph_test_image_tools_add_gauss(im, nx / 2.0 + p->x, ny / 2.0 + p->y,
                    1.0, 10.0);
            sph_cube_append_image(cpl_frame_get_filename(input_cube_frame), im,
                    NULL, 0);
            cpl_vector_set(angles, ff * np + pp, angle * CPL_MATH_DEG_RAD);
            cpl_image_delete(im);
            im = NULL;
        }cpl_test_error(CPL_ERROR_NONE);
        sph_cube_finalise_file(cpl_frame_get_filename(input_cube_frame));
        sph_cube_delete(acube);
        acube = NULL;
        cpl_frameset_insert(inframeset, input_cube_frame);
    }

    result_frameset = sph_simple_adi_process_cubes(inframeset, inframeset,
            angles, transform, model, 0, -1, -1, 0);
    cpl_test_eq(cpl_frameset_get_size(result_frameset), nf);
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
    cpl_frameset_delete(result_frameset);
    result_frameset = NULL;
    sph_ird_instrument_model_delete(model);
    model = NULL;
    sph_point_delete(p);
    cpl_vector_delete(angles);
    sph_transform_delete(transform);
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_process_cubes_full_frameset(void) {
    cpl_frame* input_cube_frame = NULL;
    int np = 10;
    int nf = 3;
    int nx = 64;
    int ny = 64;
    int pp = 0;
    cpl_image* im = NULL;
    sph_transform* transform = NULL;
    cpl_vector* angles = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_frameset* inframeset = NULL;
    cpl_frameset* result_frameset = NULL;
    int ff = 0;
    double angle = 0.0;
    sph_point* p = NULL;
    sph_cube* acube = NULL;
    transform = sph_transform_new_default();

    cpl_test_error(CPL_ERROR_NONE);
    inframeset = cpl_frameset_new();
    angles = cpl_vector_new(nf * np);
    p = sph_point_new(20.0, 0.0);
    for (ff = 0; ff < nf; ++ff) {
        input_cube_frame = sph_filemanager_create_temp_frame("testcube",
                "IRDIS_SCI", CPL_FRAME_GROUP_CALIB);
        acube = sph_cube_new(cpl_frame_get_filename(input_cube_frame));
        cpl_test_error(CPL_ERROR_NONE);
        for (pp = 0; pp < np; ++pp) {
            im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
            sph_test_image_tools_add_gauss(im, nx / 2.0, ny / 2.0, 10.0,
                    1000.0);
            p->x = 20.0;
            p->y = 0.0;
            angle += 10.0 * CPL_MATH_RAD_DEG;
            sph_point_rotate_around(p, 0.0, 0.0, cos(angle), sin(angle), 0.0,
                    0.0);
            sph_test_image_tools_add_gauss(im, nx / 2.0 + p->x, ny / 2.0 + p->y,
                    1.0, 10.0);
            sph_cube_append_image(cpl_frame_get_filename(input_cube_frame), im,
                    NULL, 0);
            cpl_vector_set(angles, ff * np + pp, angle * CPL_MATH_DEG_RAD);
            cpl_image_delete(im);
            im = NULL;
        }cpl_test_error(CPL_ERROR_NONE);
        sph_cube_finalise_file(cpl_frame_get_filename(input_cube_frame));
        sph_cube_delete(acube);
        acube = NULL;
        cpl_frameset_insert(inframeset, input_cube_frame);
    }

    result_frameset = sph_simple_adi_process_cubes(inframeset, inframeset,
            angles, transform, model, 0, -1, -1, 1);
    cpl_test_eq(cpl_frameset_get_size(result_frameset), nf);
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
    cpl_frameset_delete(result_frameset);
    result_frameset = NULL;
    sph_ird_instrument_model_delete(model);
    model = NULL;
    sph_point_delete(p);
    cpl_vector_delete(angles);
    sph_transform_delete(transform);
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_simple_adi_process_single_cube(void) {
    cpl_frame* input_cube_frame = NULL;
    int nx = 128;
    int ny = 128;
    cpl_vector* angles = NULL;
    cpl_frame* result_cube = NULL;
    sph_transform* transform = NULL;
    sph_master_frame* result_mframe = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_apertures* aps = NULL;

    transform = sph_transform_new_default();
    input_cube_frame = cutest_sph_simple_adi_fixture_planet_cube__(&angles);
    result_cube = sph_simple_adi_process_single_cube(input_cube_frame, NULL,
            angles, transform, NULL, 0, -1, -1);
    cpl_test_nonnull( result_cube );
    cpl_vector_delete(angles);
    angles = NULL;
    cpl_frame_delete(input_cube_frame);
    input_cube_frame = NULL;
    sph_ird_instrument_model_delete(model);
    model = NULL;
    sph_transform_delete(transform);
    transform = NULL;

    result_mframe = sph_common_science_combine_single_cube(result_cube,
            SPH_COLL_ALG_MEDIAN, 0, -1, -1);
    cpl_test_nonnull( result_mframe );

    aps = cpl_apertures_extract_sigma(result_mframe->image, 3.0);
    cpl_test_nonnull( aps );
    cpl_test_eq(cpl_apertures_get_size(aps), 1);
    cpl_test_abs( cpl_apertures_get_centroid_x(aps,1), nx/2.0 + 20.0,
            0.5);
    cpl_test_abs( cpl_apertures_get_centroid_y(aps,1), ny/2.0, 0.5);

    cpl_apertures_delete(aps);
    sph_master_frame_delete(result_mframe);
    result_mframe = NULL;
    cpl_frame_delete(result_cube);
    result_cube = NULL;
    cpl_test_error(CPL_ERROR_NONE);
}

static cpl_frame*
cutest_sph_simple_adi_fixture_planet_cube__(cpl_vector** angles) {
    cpl_frame* input_cube_frame = NULL;
    int np = 10;
    int nx = 128;
    int ny = 128;
    int pp = 0;
    cpl_image* im = NULL;
    sph_cube* incube = NULL;
    double angle = 0.0;
    sph_point* p = NULL;

    input_cube_frame = sph_filemanager_create_temp_frame("testcube", "SOMETAG",
            CPL_FRAME_GROUP_CALIB);
    incube = sph_cube_new(cpl_frame_get_filename(input_cube_frame));
    *angles = cpl_vector_new(np);
    for (pp = 0; pp < np; ++pp) {
        im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
        sph_test_image_tools_add_gauss(im, nx / 2.0, ny / 2.0, 10.0, 1000.0);
        p = sph_point_new(20.0, 0.0);
        angle = pp * 10.0 * CPL_MATH_RAD_DEG;
        sph_point_rotate_around(p, 0.0, 0.0, cos(angle), sin(angle), 0.0, 0.0);
        sph_test_image_tools_add_gauss(im, nx / 2.0 + p->x, ny / 2.0 + p->y,
                1.0, 10.0);
        sph_cube_append_image(cpl_frame_get_filename(input_cube_frame), im,
                NULL, 0);
        cpl_vector_set(*angles, pp, angle * CPL_MATH_DEG_RAD);
        cpl_image_delete(im);
        im = NULL;
        sph_point_delete(p);
        p = NULL;
    }

    sph_cube_finalise_file(cpl_frame_get_filename(input_cube_frame));
    im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    cpl_image_delete(im);
    im = NULL;
    sph_cube_delete(incube);
    incube = NULL;
    return input_cube_frame;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing simple ADI", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing new and delete",
                    cutest_sph_simple_adi_new_and_delete)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing set angles",
                    cutest_sph_simple_adi_set_angles)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing set current cube frame",
                    cutest_sph_simple_adi_set_current_cube_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing set current frameset",
                    cutest_sph_simple_adi_set_current_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing append image",
                    cutest_sph_simple_adi_append_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing append imagelist",
                    cutest_sph_simple_adi_append_imagelist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing calculate speckle frame",
                    cutest_sph_simple_adi_calculate_speckle_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing process single cube",
                    cutest_sph_simple_adi_process_single_cube)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing process IRDIS cubes",
                    cutest_sph_simple_adi_process_cubes)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing process IRDIS cubes full frameset",
                    cutest_sph_simple_adi_process_cubes_full_frameset)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
