/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_point.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_fitting.h"
#include "sph_unsharp_mask.h"

/*----------------------------------------------------------------------------*/
/**
 */
/*----------------------------------------------------------------------------*/

typedef struct _test_center_ {
    double cx;
    double cy;
    int npix;
} test_center;

static sph_error_code sph_ird_star_center_fit_center__(cpl_apertures* aps,
        int aa, cpl_image* im, cpl_image* errim, test_center* sim) {
    double xfit = 0;
    double yfit = 0;
    double wx = 0;
    double wy = 0;

    cpl_ensure_code( aps, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( errim, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( sim, CPL_ERROR_NULL_INPUT);
    sim->cx = cpl_apertures_get_centroid_x(aps, aa + 1) - 0.5;
    sim->cy = cpl_apertures_get_centroid_y(aps, aa + 1) - 0.5;
    sim->npix = cpl_apertures_get_npix(aps, aa + 1);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Peak found: centroid: %f, %f and npix %d", sim->cx, sim->cy, sim->npix);

    xfit = sim->cx + 0.5;
    yfit = sim->cy + 0.5;
    wx = wy = 3.0; //SPH_IRD_STAR_CENTER_FIT_FWHM_GUESS;
    if (sph_fitting_fit_gauss2D(im, errim, &xfit, &yfit, &wx, &wy, 6, 3)
            == CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Centroid: %f, %f. "
        "Gauss fit :%f, %f.", sim->cx, sim->cy, xfit - 0.5, yfit - 0.5);

        sim->cx = xfit - 0.5;
        sim->cy = yfit - 0.5;
    } else {
        cpl_error_reset();
        SPH_ERROR_RAISE_WARNING( SPH_ERROR_GENERAL,
                "Could not do gauss fit. Am using centroid.");
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static int images_are_identical(cpl_image* imActual, cpl_image* imageExcpected) {
    cpl_size xx = 0;
    cpl_size yy = 0;
    int bpixA = 0;
    int bpixB = 0;
    if (cpl_image_get_size_x(imActual)
            != cpl_image_get_size_x(imageExcpected)) {
        return 0;
    }
    if (cpl_image_get_size_y(imActual)
            != cpl_image_get_size_y(imageExcpected)) {
        return 0;
    }
    if (cpl_image_count_rejected(imActual)
            != cpl_image_count_rejected(imageExcpected)) {
        return 0;
    }
    for (xx = 0; xx < cpl_image_get_size_x(imActual); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(imActual); ++yy) {
            if (cpl_image_get(imActual, xx + 1, yy + 1, &bpixA)
                    != cpl_image_get(imageExcpected, xx + 1, yy + 1, &bpixB)) {
                return 0;
            }
            if (bpixA != bpixB) {
                return 0;
            }
        }

    }
    return 1;
}

/**@{*/
static int cutest_init_unsharp_mask_testsuite(void) {
    return sph_test_nop_code();
}

static int cutest_clean_unsharp_mask_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/** Unit tests for unsharp_mask on cpl_image
 */
/*----------------------------------------------------------------------------*/
static void cutest_unsharp_mask_cplimage_gives_null_on_null_image(void) {
    const cpl_image* nullimage = sph_unsharp_mask(NULL, 2);

    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null( nullimage );
    return;
}
/*
static void cutest_unsharp_mask_cplimage_gives_null_on_pre_existing_error(void) {
    cpl_image* animage = NULL;
    cpl_error_code errcode = CPL_ERROR_NONE;

    animage = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);

    cpl_image_set(animage, 11, 11, 111.0); // this should set cpl_error
    errcode = cpl_error_get_code();
    cpl_test_null( sph_unsharp_mask(animage, 2));
    cpl_test_error(errcode);
    cpl_error_reset();

    cpl_image_delete(animage);
    return;
}
*/
static void cutest_unsharp_mask_cplimage_gives_no_null_on_valid_image(void) {
    cpl_image* animage = NULL;
    cpl_image* result = NULL;
    animage = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);

    cpl_test_assert( result = sph_unsharp_mask(animage, 2));
    cpl_test_noneq( cpl_error_get_code(), CPL_ERROR_NULL_INPUT);

    cpl_image_delete(animage);
    cpl_image_delete(result);

    return;
}
static void cutest_unsharp_mask_cplimage_gives_zero_image_on_fully_masked_image(void) {
    cpl_image* animage = NULL;
    cpl_image* result = NULL;
    animage = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);

    cpl_test_assert( result = sph_unsharp_mask(animage, 2));
    cpl_test_noneq( cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_test( images_are_identical(result,animage));
    cpl_image_delete(animage);
    cpl_image_delete(result);

    return;
}
static void cutest_unsharp_mask_cplimage_gives_zero_image_on_constant_value_image(void) {
    cpl_image* animage = NULL;
    cpl_image* result = NULL;
    cpl_image* zeroimage = NULL;
    animage = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    zeroimage = cpl_image_duplicate(animage);
    cpl_image_add_scalar(animage, 10.0);

    cpl_test_assert( result = sph_unsharp_mask(animage, 2));
    cpl_test_noneq( cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_test( images_are_identical(result,zeroimage));
    cpl_image_delete(animage);
    cpl_image_delete(result);
    cpl_image_delete(zeroimage);

    return;
}
static void cutest_unsharp_mask_cplimage_gives_different_image_on_nonconstant_value_image(void) {
    cpl_image* animage = NULL;
    cpl_image* result = NULL;
    animage = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(animage, 10.0);
    cpl_image_set(animage, 5, 5, 10000.0);
    cpl_image_set(animage, 5, 6, 20000.0);
    cpl_image_set(animage, 6, 5, 30000.0);
    cpl_image_set(animage, 6, 6, 40000.0);
    cpl_image_set(animage, 4, 5, 1000.0);
    cpl_image_set(animage, 5, 4, 2000.0);
    cpl_test_assert( result = sph_unsharp_mask(animage, 2));
    cpl_test_noneq( cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_test_zero( images_are_identical(result,animage));
    cpl_image_delete(animage);
    cpl_image_delete(result);

    return;
}

static void cutest_unsharp_mask_cplimage_gives_unsharped_image(void) {
    cpl_image* animage = NULL;
    cpl_image* result = NULL;
    cpl_image* errim = NULL;
    test_center sim = {0.0, 0.0, 0};
    cpl_apertures* aps = NULL;
    cpl_error_code code;

    animage = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    sph_test_image_tools_add_gauss(animage, 128.0 - 64.0, 128.0, 2.0, 1000.0);
    sph_test_image_tools_add_gauss(animage, 128.0 + 64.0, 128.0, 2.0, 1000.0);
    sph_test_image_tools_add_gauss(animage, 128.0, 128.0 - 64.0, 2.0, 1000.0);
    sph_test_image_tools_add_gauss(animage, 128.0, 128.0 + 64.0, 2.0, 1000.0);
    sph_test_image_tools_add_gauss(animage, 128.0, 128.0, 50.0, 6000000.0);
    code = cpl_image_save(animage, "test_unsharp_in.fits", CPL_TYPE_UNSPECIFIED,
                          NULL, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    result = sph_unsharp_mask(animage, 8);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( result );
    code = cpl_image_save(result, "test_unsharp_out.fits", CPL_TYPE_UNSPECIFIED,
                          NULL, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    errim = cpl_image_power_create(animage, 2.0);
    cpl_test_error(CPL_ERROR_NONE);

    code = cpl_image_power(errim, 0.25);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    aps = cpl_apertures_extract_sigma(result, 15.0);
    cpl_test_error(CPL_ERROR_NONE);

    code = sph_ird_star_center_fit_center__( aps, 0, result, errim, &sim );
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs( sim.cx, 127.5, 0.5);
    cpl_test_abs( sim.cy, 128.0 - 64.0 - 0.5, 0.5);

    cpl_image_delete(animage);
    cpl_image_delete(result);
    cpl_image_delete(errim);
    cpl_apertures_delete(aps);
    return;
}

/*----------------------------------------------------------------------------*/
/** Main test function
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("unsharp_mask_test",
            cutest_init_unsharp_mask_testsuite,
            cutest_clean_unsharp_mask_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite,
                    "cutest_unsharp_mask_cplimage_gives_null_on_null_image",
                    cutest_unsharp_mask_cplimage_gives_null_on_null_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_unsharp_mask_cplimage_gives_no_null_on_valid_image",
                    cutest_unsharp_mask_cplimage_gives_no_null_on_valid_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_unsharp_mask_cplimage_gives_zero_image_on_fully_masked_image",
                    cutest_unsharp_mask_cplimage_gives_zero_image_on_fully_masked_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_unsharp_mask_cplimage_gives_zero_image_on_constant_value_image",
                    cutest_unsharp_mask_cplimage_gives_zero_image_on_constant_value_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_unsharp_mask_cplimage_gives_different_image_on_nonconstant_value_image",
                    cutest_unsharp_mask_cplimage_gives_different_image_on_nonconstant_value_image)) {
        return sph_test_get_error();
    }
/*
    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_unsharp_mask_cplimage_gives_null_on_pre_existing_error",
                    cutest_unsharp_mask_cplimage_gives_null_on_pre_existing_error)) {
        return sph_test_get_error();
    }
*/
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_unsharp_mask_cplimage_gives_unsharped_image",
                    cutest_unsharp_mask_cplimage_gives_unsharped_image)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/

