/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_dataset_peak.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup Requirement Test: sph_dataset_peak_find_subpeaks
 * 
 * The requirement tests for the function sph_dataset_peak_find_subpeaks              
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static sph_dataset*
reqtest_util_dataset_new_gauss(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = cpl_vector_get(lams, ii) - 50.0;
        val = exp(-val * val / 25.0);
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}
static sph_dataset*
reqtest_util_dataset_new_double_gauss(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    double valB = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = cpl_vector_get(lams, ii) - 25.0;
        val = 5.0 * exp(-val * val / 400.0);
        valB = cpl_vector_get(lams, ii) - 75.0;
        valB = exp(-valB * valB / 400.0);
        val = val + valB;
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}
static sph_dataset*
reqtest_util_dataset_new_no_peak(gsl_rng* pRNG) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = gsl_ran_gaussian(pRNG, 1.0);
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}
static
int reqtest_util_dataset_add_noise(sph_dataset* self, gsl_rng* pRNG,
        double noise_sigma) {
    int ii = 0;
    double oldv = 0.0;

    cpl_test_nonnull( self );

    for (ii = 0; ii < cpl_vector_get_size(self->yvalues); ++ii) {
        oldv = cpl_vector_get(self->yvalues, ii);
        oldv = oldv + gsl_ran_gaussian(pRNG, noise_sigma);
        cpl_vector_set(self->yvalues, ii, oldv);
    }

    return CPL_ERROR_NONE;
}
static
int reqtest_util_write_subpeaks(FILE* resfile, sph_dataset_peak* peak, int nc) {
    sph_dataset_peak* apeak = NULL;
    int ii = 0;
    int cc = 0;

    apeak = peak;

    while (apeak) {
        cc++;
        for (ii = 0; ii < nc; ii++)
            fprintf(resfile, "--");
        fprintf(resfile,
                "Subpeak: %d, level: %f, maxpos: %f, meanpos: %f, totval: %f\n",
                cc, apeak->frac_threshold, apeak->maxpos, apeak->meanpos,
                apeak->totval);
        for (ii = 0; ii < nc; ii++)
            fprintf(resfile, "--");
        fprintf(resfile, "Number of subpeaks: %d\n",
                sph_dataset_peak_get_number_subpeaks(apeak));
        reqtest_util_write_subpeaks(resfile, apeak->firstchild, nc + 1);
        apeak = apeak->next;
    }
    return 0;
}

static
int reqtest_util_write_result_file(const char* fname, sph_dataset_peak* root) {
    FILE* resfile = NULL;

    resfile = fopen(fname, "w");
    fprintf(resfile, "================= Result of equirements test ====\n");
    fprintf(resfile, "\n");
    reqtest_util_write_subpeaks(resfile, root, 0);

    fclose(resfile);

    return 0;
}

static
int reqtest_init_dataset_peak_find_subpeaks_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int reqtest_clean_dataset_peak_find_subpeaks_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_peak_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_dataset_peak_find_subpeaks_one_peak_no_noise(void) {
    sph_dataset_peak* root = NULL;
    sph_dataset* dat = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    dat = reqtest_util_dataset_new_gauss();
    cpl_test_nonnull( dat );

    root = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( root );

    root->frac_threshold = 0.0;
    root->frac_acc = 0.01;
    root->minpixels = 1;

    rerr = sph_dataset_peak_find_subpeaks(root);

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_null( root->firstchild);
    cpl_test_abs( root->maxpos, 50.0, 0.05);

    reqtest_util_write_result_file(
            "reqtest_dataset_peak_find_subpeaks_one_peak_no_noise_web.txt",
            root);

    rerr = sph_dataset_peak_delete(root);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
static
void reqtest_dataset_peak_find_subpeaks_no_noise(void) {
    sph_dataset_peak* root = NULL;
    sph_dataset* dat = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    dat = reqtest_util_dataset_new_double_gauss();
    cpl_test_nonnull( dat );

    root = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( root );

    root->frac_threshold = 0.0;
    root->frac_acc = 0.01;
    root->minpixels = 3;

    rerr = sph_dataset_peak_find_subpeaks(root);

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( root->firstchild );
    cpl_test_nonnull( root->firstchild->next );
    cpl_test_null( root->firstchild->next->next);
    cpl_test_abs( root->firstchild->maxpos, 25.0, 0.05);
    cpl_test_abs( root->firstchild->next->maxpos, 75.0, 1.15);

    reqtest_util_write_result_file(
            "reqtest_dataset_peak_find_subpeaks_no_noise_web.txt", root);

    rerr = sph_dataset_peak_delete(root);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
static
void reqtest_dataset_peak_find_subpeaks_no_peak(void) {
    sph_dataset_peak* root = NULL;
    sph_dataset* dat = NULL;
    gsl_rng* pRNG = NULL;
    int rerr = CPL_ERROR_NONE;
    unsigned long ulSeed = 1;

    /* Setup and run ...*/
    pRNG = gsl_rng_alloc(gsl_rng_taus);

    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);

    dat = reqtest_util_dataset_new_no_peak(pRNG);
    cpl_test_nonnull( dat );
    gsl_rng_free(pRNG);

    root = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( root );

    root->frac_threshold = 0.5;
    root->frac_acc = 0.01;
    root->minpixels = 3;

    rerr = sph_dataset_peak_find_subpeaks(root);

    /*Verify */

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_null( root->firstchild);
    reqtest_util_write_result_file(
            "reqtest_dataset_peak_find_subpeaks_no_peak_web.txt", root);

    rerr = sph_dataset_peak_delete(root);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
static
void reqtest_dataset_peak_find_subpeaks_two_peaks_noisy(void) {
    sph_dataset_peak* root = NULL;
    sph_dataset* dat = NULL;
    gsl_rng* pRNG = NULL;
    int rerr = CPL_ERROR_NONE;
    unsigned long ulSeed = 1;

    /* Setup and run ...*/
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);

    dat = reqtest_util_dataset_new_double_gauss();

    reqtest_util_dataset_add_noise(dat, pRNG, 0.1);
    cpl_test_nonnull( dat );
    gsl_rng_free(pRNG);

    root = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( root );

    root->frac_threshold = 0.17;
    root->frac_acc = 0.01;
    root->minpixels = 3;

    rerr = sph_dataset_peak_find_subpeaks(root);

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( root->firstchild );
    cpl_test_nonnull( root->firstchild->next );
    cpl_test_null( root->firstchild->next->next);
    cpl_test_abs( root->firstchild->maxpos, 25.0, 3.05);
    cpl_test_abs( root->firstchild->next->maxpos, 75.0, 1.15);
    cpl_test_abs( root->firstchild->meanpos, 25.0, 1.05);
    cpl_test_abs( root->firstchild->next->meanpos, 75.0, 1.25);
    reqtest_util_write_result_file(
            "reqtest_dataset_peak_find_subpeaks_two_peaks_noisy_web.txt", root);

    rerr = sph_dataset_peak_delete(root);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("dataset_peak_find_subpeaks_test",
            reqtest_init_dataset_peak_find_subpeaks_testsuite,
            reqtest_clean_dataset_peak_find_subpeaks_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite,
                    "sph_dataset_find_subpeaks_one_peak_no_noise",
                    reqtest_dataset_peak_find_subpeaks_one_peak_no_noise)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_find_subpeaks_no_noise",
                    reqtest_dataset_peak_find_subpeaks_no_noise)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_find_subpeaks_no_peak",
                    reqtest_dataset_peak_find_subpeaks_no_peak)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_find_subpeaks_two_peaks_noisy",
                    reqtest_dataset_peak_find_subpeaks_two_peaks_noisy)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
