/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_dataset_stats.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup Requirement Test: sph_dataset_peak_find_subpeaks
 * 
 * The requirement tests for the function sph_dataset_peak_find_subpeaks              
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static sph_dataset*
reqtest_util_dataset_new_gauss(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = cpl_vector_get(lams, ii) - 50.0;
        val = exp(-val * val / 25.0);
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}

static sph_dataset*
reqtest_util_dataset_new_double_gauss(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    double valB = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = cpl_vector_get(lams, ii) - 25.0;
        val = 5.0 * exp(-val * val / 400.0);
        valB = cpl_vector_get(lams, ii) - 75.0;
        valB = exp(-valB * valB / 400.0);
        val = val + valB;
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}

static
int reqtest_init_dataset_stats_find_peaks_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/

    return sph_test_nop_code();
}

static
int reqtest_clean_dataset_stats_find_peaks_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_stats_find_peaks function.
 
 This function tests the beahviour for a dataset with one peak and no 
 noise added. 
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_dataset_stats_find_peaks_one_peak_no_noise(void) {
    sph_dataset_stats* dat_stats = NULL;
    sph_dataset* dat = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    dat = reqtest_util_dataset_new_gauss();
    cpl_test_nonnull( dat );

    dat_stats = sph_dataset_stats_new(dat);
    cpl_test_nonnull( dat_stats );

    rerr = sph_dataset_stats_find_peaks(dat_stats, 0.1, 3);

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(dat_stats->npeaks, 1);
    cpl_test_abs( dat_stats->peakloc[0], 50.0, 0.05);

    rerr = sph_dataset_stats_delete(dat_stats);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_stats_find_peaks function.
 
 This function tests the beahviour for a dataset with two peaks and 
 a noise added. 
 
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_dataset_stats_find_peaks_two_peaks_noisy(void) {
    sph_dataset_stats* dat_stats = NULL;
    sph_dataset* dat = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    dat = reqtest_util_dataset_new_double_gauss();
    cpl_test_nonnull( dat );

    dat_stats = sph_dataset_stats_new(dat);
    cpl_test_nonnull( dat_stats );

    rerr = sph_dataset_stats_find_peaks(dat_stats, 0.1, 3);

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(dat_stats->npeaks, 2);
    cpl_test_abs( dat_stats->peakloc[0], 25.0, 2.05);
    cpl_test_abs( dat_stats->peakloc[1], 75.0, 2.2);

    rerr = sph_dataset_stats_delete(dat_stats);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("dataset_stats_find_peaks_test",
            reqtest_init_dataset_stats_find_peaks_testsuite,
            reqtest_clean_dataset_stats_find_peaks_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_dataset_stats_find_peaks_one_peak_no_noise",
                    reqtest_dataset_stats_find_peaks_one_peak_no_noise)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_dataset_stats_find_peaks_two_peaks_noisy",
                    reqtest_dataset_stats_find_peaks_two_peaks_noisy)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
