from edps import (
    task, subworkflow,
    alternative_associated_inputs, match_rules,
    qc0, qc1calib, calchecker, science,
    FilterMode, ReportInput,
    ONE_DAY, TWO_DAYS
)

from .ifs_datasources import *
from .sphere_rules import TEN_DAYS, TWELVE_DAYS
from .sphere_task_functions import (
    is_YJH, which_disperser, distortion_mandatory, distortion_optional,
)

"""
# Naming conventions

- Classification rules for raw frames start with `cls_`
- Classification rules for products have the same name as the PRO.CATG
- DataSources start with `raw_`
- Tasks start with `task_`
"""


########
# TASK #
########
# dark and background

@subworkflow("ifs_dark_background", "")
def process_ifs_dark_background():
    # Determine dark current
    task_ifs_dark = (task("ifs_dark")
                     .with_recipe("sph_ifs_master_dark")
                     .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                     .with_report("sphere_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
                     .with_main_input(ifs_raw_dark)
                     .with_meta_targets([qc1calib])
                     .with_output_filter(IFS_MASTER_DARK)
                     .build())
    # Determine static bad pixel map
    task_ifs_static_bpm = (task("ifs_static_bad_pixel_map")
                           .with_recipe("sph_ifs_master_dark")
                           .with_main_input(ifs_raw_dark)
                           .with_meta_targets([qc1calib])
                           .with_output_filter(IFS_STATIC_BADPIXELMAP)
                           .build())
    # Determine instrumental background
    task_ifs_ins_bg = (task("ifs_instrument_background")
                       .with_recipe("sph_ifs_cal_background")
                       .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                       .with_report("sphere_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
                       .with_main_input(ifs_raw_ins_bg)
                       .with_meta_targets([qc1calib])
                       .build())
    # Determine sky background
    task_ifs_sky_bg = (task("ifs_sky_background")
                       .with_recipe("sph_ifs_cal_background")
                       .with_main_input(ifs_raw_sky_bg)
                       .with_meta_targets([qc0])
                       .build())

    return task_ifs_dark, task_ifs_ins_bg, task_ifs_sky_bg, task_ifs_static_bpm


@subworkflow("ifs_detector_flat", "")
def process_det_flat(task_ifs_static_bpm):
    # Determination of pixel-to-pixel sensitivity variations for 4 different wavelengths  using data illuminated by
    # laser light (task_ifs_det_flat_nb1...nb4), and for white light data (task_ifs_det_flat_bb)
    # nb4 can be observed and used only for the YJH wavelength setting
    # wavelength 1020nm
    task_ifs_det_flat_nb1 = (task("ifs_det_flat_narrow_band1")
                             .with_recipe("sph_ifs_master_detector_flat")
                             .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                             .with_report("sphere_ifu_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                             .with_main_input(ifs_raw_det_flat_nb1)
                             .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                             .with_meta_targets([qc1calib])
                             .build())
    # wavelength 1230nm
    task_ifs_det_flat_nb2 = (task("ifs_det_flat_narrow_band2")
                             .with_recipe("sph_ifs_master_detector_flat")
                             .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                             .with_report("sphere_ifu_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                             .with_main_input(ifs_raw_det_flat_nb2)
                             .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                             .with_meta_targets([qc1calib])
                             .build())
    # wavelength 1300nm
    task_ifs_det_flat_nb3 = (task("ifs_det_flat_narrow_band3")
                             .with_recipe("sph_ifs_master_detector_flat")
                             .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                             .with_report("sphere_ifu_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                             .with_main_input(ifs_raw_det_flat_nb3)
                             .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                             .with_meta_targets([qc1calib])
                             .build())
    # wavelength 1540nm
    task_ifs_det_flat_nb4 = (task("ifs_det_flat_narrow_band4")
                             .with_recipe("sph_ifs_master_detector_flat")
                             .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                             .with_report("sphere_ifu_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                             .with_main_input(ifs_raw_det_flat_nb4)
                             .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                             .with_meta_targets([qc1calib])
                             .build())
    # broad band lamp
    task_ifs_det_flat_bb = (task("ifs_det_flat_broad_band")
                            .with_recipe("sph_ifs_master_detector_flat")
                            .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                            .with_report("sphere_ifu_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                            .with_main_input(ifs_raw_det_flat_bb)
                            .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                            .with_meta_targets([qc1calib])
                            .build())

    return task_ifs_det_flat_nb1, task_ifs_det_flat_nb2, task_ifs_det_flat_nb3, task_ifs_det_flat_nb4, \
        task_ifs_det_flat_bb


@subworkflow("ifs_standard", "")
def process_ifs_standard(task_ifs_dark, task_ifs_ins_bg, task_ifs_sky_bg, task_ifs_static_bpm, task_ifs_wavecal,
                         task_ifs_det_flat_nb1, task_ifs_det_flat_nb2, task_ifs_det_flat_nb3, task_ifs_det_flat_nb4,
                         task_ifs_det_flat_bb, task_ifs_ifu_flat, task_ifs_distortion):
    # Associate suitable mandatory background (sky preferred, next internal background, next dark)
    assoc_dark_to_night_obs = (match_rules()
                               .with_match_keywords(rules.matchkwd_ifs_dark, time_range=RelativeTimeRange(-1.5, 1.5),
                                                    level=0)
                               .with_match_keywords(rules.matchkwd_ifs_dark, time_range=TEN_DAYS, level=1))

    obs_background_mandatory = (alternative_associated_inputs()
                                .with_associated_input(task_ifs_sky_bg, [IFS_CAL_BACKGROUND])
                                .with_associated_input(task_ifs_ins_bg, [IFS_CAL_BACKGROUND])
                                .with_associated_input(task_ifs_dark, [IFS_MASTER_DARK],
                                                       match_rules=assoc_dark_to_night_obs))

    # Process flux standard observations like science data, because there is no catalog to enable the determination
    # of a response or zeropoints
    assoc_ifs_wavecal = (match_rules()
                         .with_match_keywords(rules.matchkwd_ifs_ifu_cal, time_range=TWELVE_DAYS,
                                              level=0)
                         .with_match_keywords(rules.matchkwd_ifs_ifu_cal, time_range=ONE_MONTH, level=1))

    task_ifs_standard_flux = (task("ifs_standard_flux")
                              .with_recipe("sph_ifs_science_dr")
                              .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                              .with_main_input(ifs_raw_standard_flux)
                              .with_alternative_associated_inputs(obs_background_mandatory)
                              .with_dynamic_parameter("IFS_disperser", which_disperser)
                              .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                              .with_associated_input(task_ifs_wavecal, [IFS_WAVECALIB], match_rules=assoc_ifs_wavecal)
                              .with_associated_input(task_ifs_det_flat_nb1, [IFS_MASTER_DFF_LONG1])
                              .with_associated_input(task_ifs_det_flat_nb2, [IFS_MASTER_DFF_LONG2])
                              .with_associated_input(task_ifs_det_flat_nb3, [IFS_MASTER_DFF_LONG3])
                              .with_associated_input(task_ifs_det_flat_nb4, [IFS_MASTER_DFF_LONG4],
                                                     condition=is_YJH)
                              .with_associated_input(task_ifs_det_flat_bb, [IFS_MASTER_DFF_LONGBB])
                              .with_associated_input(task_ifs_ifu_flat, [IFS_IFU_FLAT_FIELD])
                              .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=1,
                                                     condition=distortion_mandatory)
                              .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=0,
                                                     condition=distortion_optional)
                              .with_input_filter(IFS_STATIC_BADPIXELMAP, IFS_WAVECALIB,
                                                 IFS_MASTER_DFF_LONG1, IFS_MASTER_DFF_LONG2,
                                                 IFS_MASTER_DFF_LONG3, IFS_MASTER_DFF_LONG4,
                                                 IFS_MASTER_DFF_LONGBB, IFS_IFU_FLAT_FIELD,
                                                 IFS_MASTER_DARK, IFS_CAL_BACKGROUND,
                                                 mode=FilterMode.SELECT)
                              .with_input_filter(IFS_DISTORTION_MAP, mode=FilterMode.REJECT)
                              .with_meta_targets([qc1calib, calchecker])
                              .build())

    # Process astrometric standard observations like science data, because there is no catalog to enable the
    # determination of astrometric information
    task_ifs_standard_astrometry = (task("ifs_standard_astrometry")
                                    .with_recipe("sph_ifs_science_dr")
                                    .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                                    .with_main_input(ifs_raw_standard_astrometry)
                                    .with_alternative_associated_inputs(obs_background_mandatory)
                                    .with_dynamic_parameter("IFS_disperser", which_disperser)
                                    .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                                    .with_associated_input(task_ifs_wavecal, [IFS_WAVECALIB])
                                    .with_associated_input(task_ifs_det_flat_nb1, [IFS_MASTER_DFF_LONG1])
                                    .with_associated_input(task_ifs_det_flat_nb2, [IFS_MASTER_DFF_LONG2])
                                    .with_associated_input(task_ifs_det_flat_nb3, [IFS_MASTER_DFF_LONG3])
                                    .with_associated_input(task_ifs_det_flat_nb4, [IFS_MASTER_DFF_LONG4],
                                                           condition=is_YJH)
                                    .with_associated_input(task_ifs_det_flat_bb, [IFS_MASTER_DFF_LONGBB])
                                    .with_associated_input(task_ifs_ifu_flat, [IFS_IFU_FLAT_FIELD])
                                    .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=1,
                                                           condition=distortion_mandatory)
                                    .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=0,
                                                           condition=distortion_optional)
                                    .with_input_filter(IFS_STATIC_BADPIXELMAP, IFS_WAVECALIB,
                                                       IFS_MASTER_DFF_LONG1, IFS_MASTER_DFF_LONG2,
                                                       IFS_MASTER_DFF_LONG3, IFS_MASTER_DFF_LONG4,
                                                       IFS_MASTER_DFF_LONGBB, IFS_IFU_FLAT_FIELD,
                                                       IFS_MASTER_DARK, IFS_CAL_BACKGROUND,
                                                       mode=FilterMode.SELECT)
                                    .with_input_filter(IFS_DISTORTION_MAP, mode=FilterMode.REJECT)
                                    .with_meta_targets([qc1calib, calchecker])
                                    .build())

    return task_ifs_standard_flux, task_ifs_standard_astrometry


@subworkflow("ifs_science", "")
def process_ifs_science(task_ifs_dark, task_ifs_ins_bg, task_ifs_sky_bg, task_ifs_wavecal, task_ifs_det_flat_nb1,
                        task_ifs_det_flat_nb2, task_ifs_det_flat_nb3, task_ifs_det_flat_nb4, task_ifs_det_flat_bb,
                        task_ifs_ifu_flat, task_ifs_distortion, task_ifs_static_bpm):
    # sort by keywords only to test vs reflex which uses closest-in-time
    # alternative_associated_inputs(sort_keys=[kwd.mjd_obs])
    # obs_background_mandatory = (alternative_associated_inputs(sort_keys=[kwd.mjd_obs])

    # Associate suitable mandatory background (sky preferred, next internal background, next dark)
    obs_background_mandatory = (alternative_associated_inputs()
                                .with_associated_input(task_ifs_sky_bg, [IFS_CAL_BACKGROUND])
                                .with_associated_input(task_ifs_ins_bg, [IFS_CAL_BACKGROUND])
                                .with_associated_input(task_ifs_dark, [IFS_MASTER_DARK]))

    # Process FLUX data for coronagraphic science observations (telescope moved so that the target can be observed
    # directly; not used for science processing, but useful for later analysis

    # Science tasks require flats with different validity ranges
    match_flat_1day = (match_rules()
                       .with_match_keywords(rules.matchkwd_ifs_ifu_cal, time_range=ONE_DAY, level=0)
                       .with_match_keywords(rules.matchkwd_ifs_ifu_cal, time_range=THREE_WEEKS, level=1))

    match_flat_2days = (match_rules()
                        .with_match_keywords(rules.matchkwd_ifs_ifu_cal, time_range=TWO_DAYS, level=0)
                        .with_match_keywords(rules.matchkwd_ifs_ifu_cal, time_range=THREE_WEEKS, level=1))

    task_ifs_science_flux = (task("ifs_science_flux")
                             .with_recipe("sph_ifs_science_dr")
                             .with_main_input(ifs_raw_science_flux)
                             .with_alternative_associated_inputs(obs_background_mandatory)
                             .with_dynamic_parameter("IFS_disperser", which_disperser)
                             .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                             .with_associated_input(task_ifs_wavecal, [IFS_WAVECALIB])
                             .with_associated_input(task_ifs_det_flat_nb1, [IFS_MASTER_DFF_LONG1],
                                                    match_rules=match_flat_1day)
                             .with_associated_input(task_ifs_det_flat_nb2, [IFS_MASTER_DFF_LONG2],
                                                    match_rules=match_flat_1day)
                             .with_associated_input(task_ifs_det_flat_nb3, [IFS_MASTER_DFF_LONG3],
                                                    match_rules=match_flat_1day)
                             .with_associated_input(task_ifs_det_flat_nb4, [IFS_MASTER_DFF_LONG4],
                                                    condition=is_YJH, match_rules=match_flat_1day)
                             .with_associated_input(task_ifs_det_flat_bb, [IFS_MASTER_DFF_LONGBB],
                                                    match_rules=match_flat_2days)
                             .with_associated_input(task_ifs_ifu_flat, [IFS_IFU_FLAT_FIELD])
                             .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=1,
                                                    condition=distortion_mandatory)
                             .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=0,
                                                    condition=distortion_optional)
                             .with_input_filter(IFS_STATIC_BADPIXELMAP, IFS_WAVECALIB,
                                                IFS_MASTER_DFF_LONG1, IFS_MASTER_DFF_LONG2,
                                                IFS_MASTER_DFF_LONG3, IFS_MASTER_DFF_LONG4,
                                                IFS_MASTER_DFF_LONGBB, IFS_IFU_FLAT_FIELD, IFS_MASTER_DARK,
                                                IFS_CAL_BACKGROUND, mode=FilterMode.SELECT)
                             .with_meta_targets([qc0, science, calchecker])
                             .build())

    # Determine the star's position behind the coronagraph, needed for precise definition of center of
    # rotation in pupil-stabilized data; output data rejected because the recipe does not provide any useful product
    # needed for processing of pupil-stabilized science data
    task_ifs_coronagraph_center = (task("ifs_coronagraph_center")
                                   .with_recipe("sph_ifs_science_dr")
                                   .with_main_input(ifs_raw_coronagraph_center)
                                   .with_alternative_associated_inputs(obs_background_mandatory)
                                   .with_dynamic_parameter("IFS_disperser", which_disperser)
                                   .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                                   .with_associated_input(task_ifs_wavecal, [IFS_WAVECALIB], min_ret=1)
                                   .with_associated_input(task_ifs_det_flat_nb1, [IFS_MASTER_DFF_LONG1])
                                   .with_associated_input(task_ifs_det_flat_nb2, [IFS_MASTER_DFF_LONG2])
                                   .with_associated_input(task_ifs_det_flat_nb3, [IFS_MASTER_DFF_LONG3])
                                   .with_associated_input(task_ifs_det_flat_nb4, [IFS_MASTER_DFF_LONG4],
                                                          condition=is_YJH)
                                   .with_associated_input(task_ifs_det_flat_bb, [IFS_MASTER_DFF_LONGBB])
                                   .with_associated_input(task_ifs_ifu_flat, [IFS_IFU_FLAT_FIELD])
                                   .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=1,
                                                          condition=distortion_mandatory)
                                   .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=0,
                                                          condition=distortion_optional)
                                   .with_input_filter(IFS_STATIC_BADPIXELMAP, IFS_WAVECALIB,
                                                      IFS_MASTER_DFF_LONG1, IFS_MASTER_DFF_LONG2,
                                                      IFS_MASTER_DFF_LONG3, IFS_MASTER_DFF_LONG4,
                                                      IFS_MASTER_DFF_LONGBB, IFS_IFU_FLAT_FIELD,
                                                      IFS_MASTER_DARK, IFS_CAL_BACKGROUND,
                                                      mode=FilterMode.SELECT)
                                   .with_meta_targets([qc0, science, calchecker])
                                   .build())
    # Process the science data to create wavelength-calibrated cubes;
    # currently done file-by-file, because the pipeline does not stack science data;
    # association of science_flux, distortion and coronagraph_center data only for calselector
    task_ifs_science = (task("ifs_science")
                        .with_recipe("sph_ifs_science_dr")
                        .with_main_input(ifs_raw_science_object)
                        .with_alternative_associated_inputs(obs_background_mandatory)
                        .with_dynamic_parameter("IFS_disperser", which_disperser)
                        .with_associated_input(task_ifs_static_bpm, [IFS_STATIC_BADPIXELMAP], min_ret=0)
                        .with_associated_input(task_ifs_wavecal, [IFS_WAVECALIB], min_ret=1)
                        .with_associated_input(task_ifs_det_flat_nb1, [IFS_MASTER_DFF_LONG1])
                        .with_associated_input(task_ifs_det_flat_nb2, [IFS_MASTER_DFF_LONG2])
                        .with_associated_input(task_ifs_det_flat_nb3, [IFS_MASTER_DFF_LONG3])
                        .with_associated_input(task_ifs_det_flat_nb4, [IFS_MASTER_DFF_LONG4], condition=is_YJH)
                        .with_associated_input(task_ifs_det_flat_bb, [IFS_MASTER_DFF_LONGBB])
                        .with_associated_input(task_ifs_ifu_flat, [IFS_IFU_FLAT_FIELD])
                        .with_input_filter(IFS_STATIC_BADPIXELMAP, IFS_WAVECALIB,
                                           IFS_MASTER_DFF_LONG1, IFS_MASTER_DFF_LONG2,
                                           IFS_MASTER_DFF_LONG3, IFS_MASTER_DFF_LONG4,
                                           IFS_MASTER_DFF_LONGBB, IFS_IFU_FLAT_FIELD,
                                           IFS_MASTER_DARK, IFS_CAL_BACKGROUND,
                                           mode=FilterMode.SELECT)
                        .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=1,
                                               condition=distortion_mandatory)
                        .with_associated_input(task_ifs_distortion, [IFS_DISTORTION_MAP], min_ret=0,
                                               condition=distortion_optional)
                        # for calselector only
                        .with_associated_input(task_ifs_science_flux, [IFS_SCIENCE_DR], min_ret=0)
                        .with_associated_input(task_ifs_coronagraph_center, [IFS_SCIENCE_DR], min_ret=0)
                        # filter out unwanted products
                        .with_input_filter(IFS_SCIENCE_DR, IFS_DISTORTION_MAP, mode=FilterMode.REJECT)
                        .with_meta_targets([qc0, science, calchecker])
                        .build())

    return task_ifs_coronagraph_center, task_ifs_science_flux, task_ifs_science
