from edps import (
    task, subworkflow,
    alternative_associated_inputs, match_rules,
    qc0, qc1calib, calchecker, science,
    FilterMode, ReportInput,
    TWO_DAYS
)

from .irdis_datasources import *
from .sphere_task_functions import (
    distortion_optional, distortion_mandatory,
)

"""
# Naming conventions

- Classification rules for raw frames start with `cls_`
- Classification rules for products have the same name as the PRO.CATG
- DataSources start with `raw_`
- Tasks start with `task_`
"""


########
# TASK #
########
# dark and background

@subworkflow("irdis_dark_background", "")
def process_ird_dark_background():
    # Determine dark current
    task_ird_dark = (task("irdis_dark_imaging")
                     .with_recipe("sph_ird_master_dark")
                     .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                     .with_report("sphere_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
                     .with_main_input(ird_raw_dark_img)
                     .with_meta_targets([qc1calib])
                     .with_output_filter(IRD_MASTER_DARK)
                     .build())
    # Determine static bad pixel map
    task_ird_static_bpm = (task("irdis_static_bad_pixel_map")
                           .with_recipe("sph_ird_master_dark")
                           .with_main_input(ird_raw_dark_img)
                           .with_meta_targets([qc1calib])
                           .with_output_filter(IRD_STATIC_BADPIXELMAP)
                           .build())
    # Determine instrumental background
    task_ird_ins_bg = (task("irdis_instrument_background")
                       .with_recipe("sph_ird_ins_bg")
                       .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                       .with_report("sphere_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
                       .with_main_input(ird_raw_ins_bg)
                       .with_meta_targets([qc1calib])
                       .build())
    # Determine sky background
    task_ird_sky_bg = (task("irdis_sky_background")
                       .with_recipe("sph_ird_sky_bg")
                       .with_main_input(ird_raw_sky_bg)
                       .build())

    return task_ird_dark, task_ird_ins_bg, task_ird_sky_bg, task_ird_static_bpm


@subworkflow("irdis_standard_imaging", "")
def process_ird_standard_img(task_ird_dark, task_ird_static_bpm, task_ird_ins_bg, task_ird_sky_bg, task_ird_flat,
                             task_ird_distortion_img):
    # Associate suitable background (sky preferred, next internal background, next dark)
    obs_background = (alternative_associated_inputs()
                      .with_associated_input(task_ird_sky_bg, [IRD_SKY_BG])
                      .with_associated_input(task_ird_ins_bg, [IRD_INS_BG])
                      .with_associated_input(task_ird_dark, [IRD_MASTER_DARK]))
    # Determine zeropoints
    assoc_ird_flat = (match_rules()
                      .with_match_keywords(rules.matchkwd_ird_flat, time_range=TWO_DAYS, level=0)
                      .with_match_keywords(rules.matchkwd_ird_flat, time_range=TEN_DAYS, level=1))

    task_ird_standard_flux = (task("irdis_standard_flux_imaging")
                              .with_recipe("sph_ird_science_imaging")
                              .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                              .with_report("sphere_std_star", ReportInput.RECIPE_INPUTS_OUTPUTS)
                              .with_main_input(ird_raw_standard_flux)
                              .with_alternative_associated_inputs(obs_background)
                              .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1,
                                                     match_rules=assoc_ird_flat)
                              .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                              .with_associated_input(ird_phot_star_table, min_ret=1)
                              .with_associated_input(ird_filter_table, min_ret=1)
                              .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=1,
                                                     condition=distortion_mandatory)
                              .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=0,
                                                     condition=distortion_optional)
                              .with_input_filter(IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                              .with_meta_targets([qc1calib, calchecker])
                              .build())
    # Process astrometric standard star observations (no analysis beyond processing like science imaging data)
    task_ird_standard_astrometry = (task("irdis_standard_astrometry_imaging")
                                    .with_recipe("sph_ird_science_dbi")
                                    .with_main_input(ird_raw_img_standard_astrometry)
                                    .with_alternative_associated_inputs(obs_background)
                                    .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1)
                                    .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                                    .with_input_filter(IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                                    .with_meta_targets([qc1calib, calchecker])
                                    .build())

    return task_ird_standard_flux, task_ird_standard_astrometry


@subworkflow("irdis_on-sky_science_calibrations", "")
def process_ird_science_img_pol(task_ird_dark, task_ird_static_bpm, task_ird_ins_bg, task_ird_sky_bg, task_ird_flat,
                                task_ird_distortion_img):
    # Associate suitable background (sky preferred, next internal background, next dark)
    obs_background = (alternative_associated_inputs()
                      .with_associated_input(task_ird_sky_bg, [IRD_SKY_BG])
                      .with_associated_input(task_ird_ins_bg, [IRD_INS_BG])
                      .with_associated_input(task_ird_dark, [IRD_MASTER_DARK]))
    # Determine the star's position behind the coronagraph, needed for precise definition of center of
    # rotation in pupil-stabilized data
    task_ird_coronagraph_center = (task("irdis_coronagraph_center")
                                   .with_recipe("sph_ird_star_center")
                                   .with_main_input(ird_raw_coronagraph_center)
                                   .with_alternative_associated_inputs(obs_background)
                                   .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1)
                                   .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                                   .with_associated_input(ird_static_wafflemap, min_ret=1)
                                   .with_input_filter(IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                                   .with_meta_targets([qc0, science, calchecker])
                                   .build())

    # Determine flux of science target for coronagraphic data (telescope moved so that the star can be observed without
    # coronagraph)
    # Not used for science processing
    task_ird_science_flux_img = (task("irdis_science_flux_imaging")
                                 .with_recipe("sph_ird_science_dbi")
                                 .with_main_input(ird_raw_science_flux_img)
                                 .with_alternative_associated_inputs(obs_background)
                                 .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1)
                                 .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                                 .with_associated_input(ird_filter_table, min_ret=0)
                                 .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=1,
                                                        condition=distortion_mandatory)
                                 .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=0,
                                                        condition=distortion_optional)
                                 .with_input_filter(IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                                 .with_meta_targets([qc0, science, calchecker])
                                 .build())
    # Determine flux of science target for coronagraphic data (telescope moved so that the star can be observed without
    # coronagraph)
    # Not used for science processing
    match_dist_science = (match_rules()
                          .with_match_function(rules.assoc_ird_distortion_polarimetry, time_range=ONE_WEEK, level=0)
                          .with_match_function(rules.assoc_ird_distortion_polarimetry, time_range=TEN_DAYS, level=1))

    task_ird_science_flux_pol = (task("irdis_science_flux_polarimetry")
                                 .with_recipe("sph_ird_science_dpi")
                                 .with_main_input(ird_raw_science_flux_pol)
                                 .with_alternative_associated_inputs(obs_background)
                                 .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1)
                                 .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                                 .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=1,
                                                        condition=distortion_mandatory, match_rules=match_dist_science)
                                 .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=0,
                                                        condition=distortion_optional, match_rules=match_dist_science)
                                 .with_input_filter(IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                                 .with_meta_targets([qc0, science, calchecker])
                                 .build())

    return task_ird_coronagraph_center, task_ird_science_flux_img, task_ird_science_flux_pol
